/************************************************************************
 *
 *  mot_est.c, part of tmn (TMN encoder)
 *
 *  Copyright (C) 1997  University of BC, Canada
 *
 *  Contacts: 
 *  Michael Gallant                   <mikeg@ee.ubc.ca>
 *  Guy Cote                          <guyc@ee.ubc.ca>
 *  Berna Erol                        <bernae@ee.ubc.ca>
 *
 *  UBC Image Processing Laboratory   http://www.ee.ubc.ca/image
 *  2356 Main Mall                    tel.: +1 604 822 4051
 *  Vancouver BC Canada V6T1Z4        fax.: +1 604 822 5949
 *
 *  Copyright (C) 1995, 1996  Telenor R&D, Norway
 *
 *  Contacts:
 *  Robert Danielsen                  <Robert.Danielsen@nta.no>
 *
 *  Telenor Research and Development  http://www.nta.no/brukere/DVC/
 *  P.O.Box 83                        tel.:   +47 63 84 84 00
 *  N-2007 Kjeller, Norway            fax.:   +47 63 81 00 76
 *
 ************************************************************************/

/* Disclaimer of Warranty
 * 
 * These software programs are available to the user without any license fee
 * or royalty on an "as is" basis. The University of British Columbia
 * disclaims any and all warranties, whether express, implied, or
 * statuary, including any implied warranties or merchantability or of
 * fitness for a particular purpose.  In no event shall the
 * copyright-holder be liable for any incidental, punitive, or
 * consequential damages of any kind whatsoever arising from the use of
 * these programs.
 * 
 * This disclaimer of warranty extends to the user of these programs and
 * user's customers, employees, agents, transferees, successors, and
 * assigns.
 * 
 * The University of British Columbia does not represent or warrant that the
 * programs furnished hereunder are free of infringement of any
 * third-party patents.
 * 
 * Commercial implementations of H.263, including shareware, are subject to
 * royalty fees to patent holders.  Many of these patents are general
 * enough such that they are unavoidable regardless of implementation
 * design.
 * 
 */


#include "sim.h"

/**********************************************************************
 *
 *	Name:          MotionEstimatePicture
 *	Description:   Finds integer and half pel motion estimation
 *                     and chooses 8x8 or 16x16 
 *	
 *	Input:	       current image, previous image, interpolated
 *                     reconstructed previous image, seek_dist,
 *                     motion vector array
 *	Returns:       
 *	Side effects:  Allocates memory for MV structure
 *
 *	Date: 950209   Author: Karl.Lillevold@nta.no
 *
 ***********************************************************************/

void MotionEstimatePicture( unsigned char *curr, unsigned char *prev,
                            unsigned char *next, unsigned char *prev_ipol, 
                            unsigned char *next_ipol, int seek_dist, 
                            MotionVector *MV[7][MBR+1][MBC+2], 
                            int gobsync, int estimation_type)                      
{
  int i,j,k;
  int pmv0,pmv0b,pmv1,pmv1b,xoff,yoff, xoffb, yoffb;
  int curr_mb[16][16];
  int sad8 = INT_MAX, sad16, sad16b, sad0, sad0b;
  int newgob;
  MotionVector *f0,*f1,*f2,*f3,*f4,*fBack;

  /* Do motion estimation and store result in array */
  for ( j = 0; j < lines/MB_SIZE; j++) 
  {
    newgob = 0;
    if (gobsync && j%gobsync == 0) 
    {
      newgob = 1;
    }

    for ( i = 0; i < pels/MB_SIZE; i++) 
    {
      /* Integer pel search */
      f0 = MV[0][j+1][i+1];
      f1 = MV[1][j+1][i+1];
      f2 = MV[2][j+1][i+1];
      f3 = MV[3][j+1][i+1];
      f4 = MV[4][j+1][i+1];
      fBack = MV[5][j+1][i+1];

      /* P-picture or PB-picture prediction. */
      if (B_PICTURE_ESTIMATION != estimation_type)
      {
        /* Here the PMV's are found using integer motion vectors for
         * the forward prediction only. */
        FindPMV(MV,i+1,j+1,&pmv0,&pmv1,0,newgob,0);

        if (long_vectors && (estimation_type != PB_PICTURE_ESTIMATION) )
        {
          /* Always divisable by two. */
          xoff = pmv0/2;
          yoff = pmv1/2;
        }
        else 
        {
          xoff = yoff = 0;
        }
      
        MotionEstimation( curr, prev, i*MB_SIZE, j*MB_SIZE, 
                          xoff, yoff, seek_dist, MV,
                          &sad0,estimation_type, 0,pmv0,pmv1);

        sad16 = f0->min_error;

        if (use_4mv)
        {
          sad8 = f1->min_error + f2->min_error + f3->min_error + f4->min_error;
        }

        f0->Mode = ChooseMode(curr,i*MB_SIZE,j*MB_SIZE, mmin(sad8,sad16));
        if(intra_mb_refresh && intra_refresh[j+1][i+1]>=intra_mb_refresh)
        {
          f0->Mode = MODE_INTRA;
        }

        /* Half pel search. Forward */
        if (f0->Mode != MODE_INTRA) 
        {
          FindMB(i*MB_SIZE,j*MB_SIZE ,curr, curr_mb);
          FindHalfPel(i*MB_SIZE,j*MB_SIZE,f0, prev_ipol, &curr_mb[0][0],16,0);
          sad16 = f0->min_error;

          if (use_4mv) 
          {
            FindHalfPel(i*MB_SIZE,j*MB_SIZE,f1, prev_ipol, &curr_mb[0][0],8,0);
            FindHalfPel(i*MB_SIZE,j*MB_SIZE,f2, prev_ipol, &curr_mb[0][8],8,1);
            FindHalfPel(i*MB_SIZE,j*MB_SIZE,f3, prev_ipol, &curr_mb[8][0],8,2);
            FindHalfPel(i*MB_SIZE,j*MB_SIZE,f4, prev_ipol, &curr_mb[8][8],8,3);

            sad8 = f1->min_error +f2->min_error +f3->min_error +f4->min_error;
            sad8 += PREF_16_VEC;
          
            /* Choose Zero Vector, 8x8 or 16x16 vectors */
            if (sad0 < sad8 && sad0 < sad16) 
            {
              f0->x = f0->y = 0;
              f0->x_half = f0->y_half = 0;
            }
            else 
            {
              if (sad8 < sad16) 
                 f0->Mode = MODE_INTER4V;
            }
          }
          else 
          {
            /* Choose Zero Vector or 16x16 vectors */
            if (sad0 < sad16) 
            {
              f0->x = f0->y = 0;
              f0->x_half = f0->y_half = 0;
            }
          }
        }
        else
        { 
          for (k = 0; k < 5; k++)
          {
            ZeroVec(MV[k][j+1][i+1]);
          }
        }
      }
      /* B-Picture. */
      else
      {
        /* Here the PMV's are found using integer motion vectors 
         * for both backward and forward prediction. */
        FindPMV(MV,i+1,j+1,&pmv0,&pmv1,0,newgob,0);
        FindPMV(MV,i+1,j+1,&pmv0b,&pmv1b,5,newgob,0);
      
        if (long_vectors && (estimation_type != PB_PICTURE_ESTIMATION) )
        {
          /* Always divisable by two. */
          xoff = pmv0/2;
          yoff = pmv1/2;
          xoffb = pmv0b/2;
          yoffb = pmv1b/2;
        }
        else 
        {
          xoff = yoff = xoffb = yoffb = 0;
        }

        /* Forward prediction. */
        MotionEstimation( curr, prev, i*MB_SIZE, j*MB_SIZE, 
                          xoff, yoff, seek_dist, MV,
                          &sad0, estimation_type, 0, pmv0, pmv1);

        sad16 = f0->min_error;

        /* Backward prediction. */
        MotionEstimation( curr, next, i*MB_SIZE, j*MB_SIZE, 
                          xoffb, yoffb, seek_dist, MV,
                          &sad0b, estimation_type, 1, pmv0b, pmv1b);

        sad16b = fBack->min_error;

        FindMB(i*MB_SIZE, j*MB_SIZE, curr, curr_mb);
        FindHalfPel(i*MB_SIZE,j*MB_SIZE, f0, prev_ipol, &curr_mb[0][0],16,0);
        sad16 = f0->min_error;

        /* Choose Zero Vector or 16x16 vectors */
        if (sad0 < sad16) 
        {
          f0->x = f0->y = 0;
          f0->x_half = f0->y_half = 0;
        }
       
        FindHalfPel(i*MB_SIZE,j*MB_SIZE, fBack, next_ipol, &curr_mb[0][0],16,0);
        sad16b = fBack->min_error;

        /* Choose Zero Vector or 16x16 vectors */
        if (sad0b < sad16b) 
        {
          fBack->x = fBack->y = 0;
          fBack->x_half = fBack->y_half = 0;
        }
      }
    }
  }

#ifdef PRINTMV
  fprintf(stdout,"Motion estimation\n");
  fprintf(stdout,"16x16 vectors:\n");

  for ( j = 0; j < lines/MB_SIZE; j++) 
  {
    for ( i = 0; i < pels/MB_SIZE; i++) 
    {
      /* B picture. */
      if (B_PICTURE_ESTIMATION == estimation_type)
      {
        /* Forward MV. */
        fprintf(stdout," forward (B): %3d%3d",
        2*MV[0][j+1][i+1]->x + MV[0][j+1][i+1]->x_half,
        2*MV[0][j+1][i+1]->y + MV[0][j+1][i+1]->y_half);
        fprintf(stdout," backward (B): %3d%3d",
        2*MV[5][j+1][i+1]->x + MV[5][j+1][i+1]->x_half,
        2*MV[5][j+1][i+1]->y + MV[5][j+1][i+1]->y_half);
      }
      /* P picture. */
      else if (MV[0][j+1][i+1]->Mode != MODE_INTRA) 
      {
        fprintf(stdout," %3d%3d",
        2*MV[0][j+1][i+1]->x + MV[0][j+1][i+1]->x_half,
        2*MV[0][j+1][i+1]->y + MV[0][j+1][i+1]->y_half);
      }
      else
      {
        fprintf(stdout,"  .  . ");
      }
    }
    fprintf(stdout,"\n");
  }

  if (use_4mv) 
  {
    fprintf(stdout,"8x8 vectors:\n");
    for (k = 1; k < 5; k++) 
    {
      fprintf(stdout,"Block: %d\n", k-1);
      for ( j = 0; j < lines/MB_SIZE; j++) 
      {
        for ( i = 0; i < pels/MB_SIZE; i++) 
	      {
          if (MV[0][j+1][i+1]->Mode != MODE_INTRA)
            fprintf(stdout," %3d%3d",
            2*MV[k][j+1][i+1]->x + MV[k][j+1][i+1]->x_half,
            2*MV[k][j+1][i+1]->y + MV[k][j+1][i+1]->y_half);
          else
            fprintf(stdout,"  .  . ");
        }
        fprintf(stdout,"\n");
      }
    }
  }
#endif
  return;
}


/**********************************************************************
 *
 *	Name:        MotionEstimation
 *	Description:	Estimate all motion vectors for one MB
 *
 *	Input:        pointers to current and reference (previous)
 *                    image, pointers to current slice and
 *                    current MB
 *	Returns:
 *	Side effects:	motion vector information in MB changed
 *
 *	Date: 930118	Author: Karl.Lillevold@nta.no
 *            940203    Mod. to use PGB's faster search
 *            941208    Mod to use spiral search from mpeg2encode
 *            951001    Mod for extended motion vector range
 *            970526	  Fast motion estimation by mikeg@ee.ubc.ca
 *            971119    Added new motion vector range for RVLC
 *                      modified by guyc@ee.ubc.ca.
 *            980701    Fixed bugs with various MV extrapolation ranges
 *                      and incorporated forward and backward ME into
 *                      same routine by mikeg@ee.ubc.ca
 *
 ***********************************************************************/


void MotionEstimation (unsigned char *curr, unsigned char *reference, int x_curr,
                        int y_curr, int xoff, int yoff, int seek_dist,
                        MotionVector * MV[7][MBR + 1][MBC + 2], int
                       *SAD_0, int estimation_type, int backward_pred,
                       int pmv0, int pmv1)

{

  int Min_FRAME[7];
  MotionVector MV_FRAME[7];
  unsigned char *act_block, *aa, *ii;
  unsigned char *search_area, *adv_search_area = NULL, *zero_area = NULL;
  int sxy, i, k, j, l;
  int xlim, ylim;
  int ihigh, ilow, jhigh, jlow, h_length, v_length;
  int adv_ihigh, adv_ilow, adv_jhigh, adv_jlow, adv_h_length, adv_v_length;
  int xmax, ymax, block, sad, lx;
  int adv_x_curr, adv_y_curr, xvec, yvec;

#ifndef FULLSEARCH
  static htp[4] = {-1, 0, 1, 0};
  static vtp[4] = {0, 1, 0, -1};
  int j_min_now, i_min_now, i_min_next, j_min_next, sad_layr;
  int distortion_0, distortion_1, distortion_2;
  int m;
#endif

  xmax = pels;
  ymax = lines;
  xlim = ylim = seek_dist;

  /* There are 8 scenarios for motion vector extrapolaton range
   * and search window size combinations
   *  1. No optional modes
   *  2. Annex F (H.263 or H.263+) only.
   *  3. Annex D (H.263) only.
   *  4. Annex D (H.263) + Annex F
   *  5. Annex D (H.263+, uui = '01' i.e. unlimited unrestricted MVs) only
   *  6. Annex D (H.263+, uui = '01' i.e. unlimited unrestricted MVs) + Annex F
   *  7. Annex D (H.263+, uui = '1' i.e. MVs based on picture size) only
   *  8. Annex D (H.263+, uui = '1' i.e. MVs based on picture size) + Annex F */

  /* The Unrestricted Motion Vector mode (see Annex D) is disabled. */
  if ( (!long_vectors) || (estimation_type == PB_PICTURE_ESTIMATION) )
  {    
    /* When the Advanced Prediction mode (see Annex F) is used without 
     * the Unrestricted Motion Vector mode, the extrapolation range is 
     * a maximum of 16 pixels outside the coded picture area. */
    if (mv_outside_frame)
    {
      /* Maximum normal search range centered around the predicted vector
       * Special handling for 8x8 vectors, as the 8x8 search may change 
       * the MV predictor for some of the blocks within the macroblock. 
       * When we impose this limitation, we are sure that any 8x8 vector 
       * we might find is possible to transmit. */
      xlim = mmin (15 - (2 * DEF_8X8_WIN + 1), xlim);
      ylim = mmin (15 - (2 * DEF_8X8_WIN + 1), ylim);
      
      xoff = mmin (16, mmax (-16, xoff));
      yoff = mmin (16, mmax (-16, yoff));
      
      ilow = x_curr + xoff - xlim;
      ihigh = x_curr + xoff + xlim;
      
      jlow = y_curr + yoff - ylim;
      jhigh = y_curr + yoff + ylim;
      
      lx = pels + 32;
      
      if (ilow < -15)
        ilow = -15;
      if (ihigh > xmax)
        ihigh = xmax;
      if (jlow < -15)
        jlow = -15;
      if (jhigh > ymax)
        jhigh = ymax;
    }
    /* When the Advanced Prediction mode (see Annex F) and the
     * Unrestricted Motion Vector mode are not used, the 
     * extrapolation range is not permitted to extend
     * outside the coded picture area.  */
    else
    {
      /* Maximum normal search range centered around the predicted vector */
      xlim = mmin (15, xlim);
      ylim = mmin (15, ylim);
      
      ilow = x_curr + xoff - xlim;
      ihigh = x_curr + xoff + xlim;
      
      jlow = y_curr + yoff - ylim;
      jhigh = y_curr + yoff + ylim;
      
      lx = pels;

      if (ilow < 0)
        ilow = 0;
      if (ihigh > xmax - 15)
        ihigh = xmax - 15;
      if (jlow < 0)
        jlow = 0;
      if (jhigh > ymax - 15)
        jhigh = ymax - 15;
    }
  } 
  /* The Unrestricted Motion Vector mode (see Annex D) is enabled.*/
  else
  {
    /* PLUSPTYPE is enabled. If RVLC codes of H.263+ are used, new 
     * motion vector range is used. */
    if(EPTYPE) 
    {
      /* UUI = '01', unlimited unrestricted motion vectors */
      if (unlimited_unrestricted_motion_vectors) 
      {
        /* Motion vector range is unlimited (only limited by the
         * picture boundaries). */
        xlim = xmax;
        ylim = ymax;

        /* If PLUSPTYPE is present, the motion vector range does not 
         * depend on the motion vector prediction value */
        ilow = x_curr - xlim;
        ihigh = x_curr + xlim;
        
        jlow = y_curr - ylim;
        jhigh = y_curr + ylim;

        lx = pels + 64;
      
        if (ilow < -15)
          ilow = -15;
        if (ihigh > xmax)
          ihigh = xmax;
        if (jlow < -15)
          jlow = -15;
        if (jhigh > ymax)
         jhigh = ymax;
      } 
      /* UUI = '1' */
      else
      {
#ifdef FULLSEARCH
        /* Motion vector range is limited by Tables in Annex D. */
        xlim = (pels < 353) ? 31 : (pels < 705) ? 63 : (pels < 1409) ? 127 : 255;
        ylim = (lines < 289) ? 31 : (lines < 577) ? 63 : 127;

        /* If PLUSPTYPE is present, the motion vector range does not 
         * depend on the motion vector prediction value */
        ilow = x_curr - xlim;
        ihigh = x_curr + xlim;
        
        jlow = y_curr - ylim;
        jhigh = y_curr + ylim;
#else   
        /* Limit search range more fore faster RD optimized code.
         * Motion vector range is limited by Tables in Annex D, and we shorten
         * this range further for speed. */
        xlim = (pels < 353) ? 15 : (pels < 705) ? 31 : (pels < 1409) ? 63 : 255;
        ylim = (lines < 289) ? 15 : (lines < 577) ? 31 : 127;

        /* Further limit range if 8x8 vectors allowed, to ensure the
         * obtained vectors can be transmitted. */
        if (mv_outside_frame)
        {
          xlim = mmin (xlim - (2 * DEF_8X8_WIN + 1), xlim);
          ylim = mmin (ylim - (2 * DEF_8X8_WIN + 1), ylim);
        }

        /* If PLUSPTYPE is present, the motion vector range does not 
         * depend on the motion vector prediction value */
        ilow = x_curr - xlim;
        ihigh = x_curr + xlim;
        
        jlow = y_curr - ylim;
        jhigh = y_curr + ylim;
#endif
        /* If PLUSPTYPE is present in the picture header, the motion vector 
         * values are restricted such that no element of the 16x16 (or 8x8) 
         * region shall have a horizontal or vertical distance more than 
         * 15 pixels outside the coded picture area. */
        lx = pels + 64;
        
        if (ilow < -15)
          ilow = -15;
        if (ihigh > xmax)
          ihigh = xmax;
        if (jlow < -15)
          jlow = -15;
        if (jhigh > ymax)
          jhigh = ymax;
      }
    }
    else
    {
      /* When PLUSPTYPE is absent, the extrapolation range is a maximum of 
       * 31.5 pixels outside the coded picture area when the Unrestricted 
       * Motion Vector mode is used */
      xlim = 31;
      ylim = 31;

      /* Further limit range if 8x8 vectors allowed, to ensure the
       * obtained vectors can be transmitted. */
      if (use_4mv)
      {
        xlim = mmin (xlim - (2 * DEF_8X8_WIN + 1), xlim);
        ylim = mmin (ylim - (2 * DEF_8X8_WIN + 1), ylim);
      }
        
      /* The reason for the search window's reduction above with
       * 2*DEF_8X8_WIN+1 is that the 8x8 search may change the MV predictor
       * for some of the blocks within the macroblock. When we impose the
       * limitation above, we are sure that any 8x8 vector we might find is
       * possible to transmit */
      
      /* We have found that with OBMC, DEF_8X8_WIN should be quite small for 
       * two reasons: (i) a good filtering effect, and (ii) not too many
       * bits used for transferring the vectors. As can be seen above this
       * is also useful to avoid a large limitation on the MV search range */
      
      /* It is possible to make sure the motion vectors found are legal in
       * other less limiting ways than above, but this would be more
       * complicated as well as time-consuming. Any good suggestions for 
       * improvement is welcome, though */
      
      xoff = mmin (31, mmax (-31, xoff));
      yoff = mmin (31, mmax (-31, yoff));
      
      /* There is no need to check if (xoff + x_curr) points outside the
       * picture, since the Extended Motion Vector Range is always used
       * together with the Unrestricted MV mode */

      // if (xoff < -15) 
      if (pmv0 < -31)
      {
        ilow = x_curr - xlim;
        ihigh = x_curr - 1;
      }
      else /*if (xoff > 16)*/ if (pmv0 > 32)
      {
        ilow = x_curr + 1;
        ihigh = x_curr + xlim;
      }
      else
      {
        ilow = x_curr + xoff - 15;
        ihigh = x_curr + xoff + 14;
      }

      // if (yoff < -15)
      if (pmv1 < -31)
      {
        jlow = y_curr - ylim;
        jhigh = y_curr - 1;
      }
      else /*if (yoff > 16)*/ if (pmv1 > 32)
      {
        jlow = y_curr + 1;
        jhigh = y_curr + ylim;
      }
      else
      {
        jlow = y_curr + yoff - 15;
        jhigh = y_curr + yoff + 14;
      }

      lx = pels + 64;

      if (ilow < -31)
        ilow = -31;
      if (ihigh > xmax + 15)
        ihigh = xmax + 15;
      if (jlow < -31)
        jlow = -31;
      if (jhigh > ymax + 15)
        jhigh = ymax + 15;
    }
  }
  
  h_length = ihigh - ilow + 16;
  v_length = jhigh - jlow + 16;

  act_block = LoadArea (curr, x_curr, y_curr, 16, 16, pels);
  search_area = LoadArea (reference, ilow, jlow, h_length, v_length, lx);

  for (k = 0; k < 7; k++)
  {
    Min_FRAME[k] = INT_MAX;
    MV_FRAME[k].x = 0;
    MV_FRAME[k].y = 0;
    MV_FRAME[k].x_half = 0;
    MV_FRAME[k].y_half = 0;
  }

  /* Zero vector search */
  if (x_curr - ilow < 0 || y_curr - jlow < 0 ||
      x_curr - ilow + MB_SIZE > h_length || y_curr - jlow + MB_SIZE > v_length)
  {
    /* in case the zero vector is outside the loaded area in search_area */
    zero_area = LoadArea (reference, x_curr, y_curr, 16, 16, lx);
    *SAD_0 = SAD_Macroblock (zero_area, act_block, 16, Min_FRAME[0]) -
      PREF_NULL_VEC;
    free (zero_area);
  } 
  else
  {
    /* the zero vector is within search_area */
    ii = search_area + (x_curr - ilow) + (y_curr - jlow) * h_length;
    *SAD_0 = SAD_Macroblock (ii, act_block, h_length, Min_FRAME[0]) -
      PREF_NULL_VEC;
  }

  if (xoff == 0 && yoff == 0)
  {
    Min_FRAME[0] = *SAD_0;
    MV_FRAME[0].x = 0;
    MV_FRAME[0].y = 0;
  } else
  {
    ii = search_area + (x_curr + xoff - ilow) + (y_curr + yoff - jlow) * h_length;
    Min_FRAME[0] = SAD_Macroblock (ii, act_block, h_length, Min_FRAME[0]);
    MV_FRAME[0].x = xoff;
    MV_FRAME[0].y = yoff;
  }
  /* NB: if xoff or yoff != 0, the Extended MV Range is used. If we allow
   * the zero vector to be chosen prior to the half pel search in this
   * case, the half pel search might lead to a non-transmittable vector
   * (on the wrong side of zero). If SAD_0 turns out to be the best SAD,
   * the zero-vector will be chosen after half pel search instead.  The
   * zero-vector can be transmitted in all modes, no matter what the MV
   * predictor is */

  sxy = mmax(xlim,ylim);

/* Spiral search for full search motion estimation */
#ifdef FULLSEARCH
  for (l = 1; l <= sxy; l++) {
    i = x_curr + xoff - l;
    j = y_curr + yoff - l;
    for (k = 0; k < 8*l; k++) {
      if (i>=ilow && i<=ihigh && j>=jlow && j<=jhigh) {

        /* 16x16 integer pel MV */
        ii = search_area + (i-ilow) + (j-jlow)*h_length;
        sad = SAD_Macroblock(ii, act_block, h_length, Min_FRAME[0]);
        if (sad < Min_FRAME[0]) {
          MV_FRAME[0].x = i - x_curr;
          MV_FRAME[0].y = j - y_curr;
          Min_FRAME[0] = sad;
        }

      }
      if      (k<2*l) i++;
      else if (k<4*l) j++;
      else if (k<6*l) i--;
      else            j--;
    }      
  }
#else
/* Fast search motion estimation */
  i_min_now = x_curr + xoff;
  j_min_now = y_curr + yoff;

  distortion_0 = distortion_1 = distortion_2 = 65536;

  for (l = 1; l <= 2 * sxy; l++)
  {
    sad_layr = 65536;

    for (m = 0; m < 4; m++)
    {

      i = i_min_now + htp[m];
      j = j_min_now + vtp[m];

      if (i >= ilow && i <= ihigh && j >= jlow && j <= jhigh)
      {
        /* 16x16 integer pel MV */
        ii = search_area + (i - ilow) + (j - jlow) * h_length;
        sad = SAD_Macroblock (ii, act_block, h_length, sad_layr);

        /* set search map to 1 for this location */
        if (sad < Min_FRAME[0])
        {
          MV_FRAME[0].x = i - x_curr;
          MV_FRAME[0].y = j - y_curr;
          Min_FRAME[0] = sad;
        }
        if (sad < sad_layr)
        {
          sad_layr = sad;
          i_min_next = i;
          j_min_next = j;
        }
      }
    }

    i_min_now = i_min_next;
    j_min_now = j_min_next;


    distortion_2 = distortion_1;
    distortion_1 = distortion_0;
    distortion_0 = sad_layr;
    if (distortion_1 <= distortion_0 && distortion_2 <= distortion_0)
    {
      break;
    }
  }
#endif

  if (use_4mv)
  {

    /* Center the 8x8 search around the 16x16 vector.  This is different
     * than in TMN5 where the 8x8 search is also a full search. The
     * reasons for this is: (i) it is faster, and (ii) it generally gives
     * better results because of a better OBMC filtering effect and less
     * bits spent for vectors, and (iii) if the Extended MV Range is used,
     * the search range around the motion vector predictor will be less
     * limited */

    xvec = MV_FRAME[0].x;
    yvec = MV_FRAME[0].y;

    if (!long_vectors || estimation_type == PB_PICTURE_ESTIMATION)
    {
      if (xvec > 15 - DEF_8X8_WIN)
      {
        xvec = 15 - DEF_8X8_WIN;
      }
      if (yvec > 15 - DEF_8X8_WIN)
      {
        yvec = 15 - DEF_8X8_WIN;
      }
      if (xvec < -15 + DEF_8X8_WIN)
      {
        xvec = -15 + DEF_8X8_WIN;
      }
      if (yvec < -15 + DEF_8X8_WIN)
      {
        yvec = -15 + DEF_8X8_WIN;
      }
    }
    adv_x_curr = x_curr + xvec;
    adv_y_curr = y_curr + yvec;

    sxy = DEF_8X8_WIN;

    adv_ilow = adv_x_curr - xlim;
    adv_ihigh = adv_x_curr + xlim;

    adv_jlow = adv_y_curr - ylim;
    adv_jhigh = adv_y_curr + ylim;

    adv_h_length = adv_ihigh - adv_ilow + 16;
    adv_v_length = adv_jhigh - adv_jlow + 16;

    adv_search_area = LoadArea (reference, adv_ilow, adv_jlow,
                                adv_h_length, adv_v_length, lx);

    for (block = 0; block < 4; block++)
    {
      ii = adv_search_area + (adv_x_curr - adv_ilow) + ((block & 1) << 3) +
        (adv_y_curr - adv_jlow + ((block & 2) << 2)) * adv_h_length;
      aa = act_block + ((block & 1) << 3) + ((block & 2) << 2) * 16;
      Min_FRAME[block + 1] = SAD_Block (ii, aa, adv_h_length, Min_FRAME[block + 1]);
      MV_FRAME[block + 1].x = MV_FRAME[0].x;
      MV_FRAME[block + 1].y = MV_FRAME[0].y;
    }

    /* Spiral search for full search motion estimation */
#ifdef FULLSEARCH
    for (l = 1; l <= sxy; l++) {
      i = adv_x_curr - l;
      j = adv_y_curr - l;
      for (k = 0; k < 8*l; k++) {
        if (i>=adv_ilow && i<=adv_ihigh && j>=adv_jlow && j<=adv_jhigh) {
          
          /* 8x8 integer pel MVs */
          for (block = 0; block < 4; block++) {
            ii = adv_search_area + (i-adv_ilow) + ((block&1)<<3) + 
              (j-adv_jlow + ((block&2)<<2) )*adv_h_length;
            aa = act_block + ((block&1)<<3) + ((block&2)<<2)*16;
            sad = SAD_Block(ii, aa, adv_h_length, Min_FRAME[block+1]);
            if (sad < Min_FRAME[block+1]) {
              MV_FRAME[block+1].x = i - x_curr;
              MV_FRAME[block+1].y = j - y_curr;
              Min_FRAME[block+1] = sad;
            }
          }
          
        }
        if      (k<2*l) i++;
        else if (k<4*l) j++;
        else if (k<6*l) i--;
        else            j--;
      }      
    }
#else
    /* Fast search motion estimation */
    for (block = 0; block < 4; block++)
    {
      i_min_now = adv_x_curr;
      j_min_now = adv_y_curr;

      distortion_0 = distortion_1 = distortion_2 = 65536;

      for (l = 1; l <= 2 * sxy; l++)
      {
        sad_layr = 65536;
        for (m = 0; m < 4; m++)
        {

          i = i_min_now + htp[m];
          j = j_min_now + vtp[m];

          if (i >= adv_ilow && i <= adv_ihigh && j >= adv_jlow && j <= adv_jhigh)
          {
            /* 8x8 integer pel MVs */
            ii = adv_search_area + (i - adv_ilow) + ((block & 1) << 3) +
              (j - adv_jlow + ((block & 2) << 2)) * adv_h_length;
            aa = act_block + ((block & 1) << 3) + ((block & 2) << 2) * 16;
            sad = SAD_Block (ii, aa, adv_h_length, sad_layr);
            if (sad < Min_FRAME[block + 1])
            {
              MV_FRAME[block + 1].x = i - x_curr;
              MV_FRAME[block + 1].y = j - y_curr;
              Min_FRAME[block + 1] = sad;
            }
            if (sad < sad_layr)
            {
              sad_layr = sad;
              i_min_next = i;
              j_min_next = j;
            }
          }
        }
      }

      i_min_now = i_min_next;
      j_min_now = j_min_next;

      distortion_2 = distortion_1;
      distortion_1 = distortion_0;
      distortion_0 = sad_layr;
      if (distortion_1 <= distortion_0 && distortion_2 <= distortion_0)
      {
        break;
      }
    }
#endif

  }
  i = x_curr / MB_SIZE + 1;
  j = y_curr / MB_SIZE + 1;

  if (!use_4mv)
  {
    if(!backward_pred)
    {
      MV[0][j][i]->x = MV_FRAME[0].x;
      MV[0][j][i]->y = MV_FRAME[0].y;
      MV[0][j][i]->min_error = Min_FRAME[0];
    }
    else
    {
      MV[5][j][i]->x = MV_FRAME[0].x;
      MV[5][j][i]->y = MV_FRAME[0].y;
      MV[5][j][i]->min_error = Min_FRAME[0];
    }
  } 
  else
  {
    for (k = 0; k < 5; k++)
    {
      MV[k][j][i]->x = MV_FRAME[k].x;
      MV[k][j][i]->y = MV_FRAME[k].y;
      MV[k][j][i]->min_error = Min_FRAME[k];
    }
  }

  free (act_block);
  free (search_area);
  if (use_4mv)
    free (adv_search_area);
  return;
}

/**********************************************************************
 *
 *	Name:        FindHalfPel
 *	Description:	Find the optimal half pel prediction
 *
 *	Input:        position, vector, array with current data
 *        pointer to previous interpolated luminance,
 *
 *	Returns:
 *
 *	Date: 930126        Author: Karl.Lillevold@nta.no
 *            950208    Mod: Karl.Lillevold@nta.no
 *
 ***********************************************************************/


void FindHalfPel (int x, int y, MotionVector * fr, unsigned char *prev,
                  int *curr, int bs, int comp)
{
  int i, m, n;
  int half_pel;
  int start_x, start_y, stop_x, stop_y, new_x, new_y, lx;
  int min_pos;
  int AE, AE_min;
  Point search[9];

  start_x = -1;
  stop_x = 1;
  start_y = -1;
  stop_y = 1;

  new_x = x + fr->x;
  new_y = y + fr->y;

  new_x += ((comp & 1) << 3);
  new_y += ((comp & 2) << 2);

  lx = (mv_outside_frame ? pels + (long_vectors ? 64 : 32) : pels);

  /* Make sure that no addressing is outside the frame */
  if (!mv_outside_frame)
  {
    if ((new_x) <= 0)
      start_x = 0;
    if ((new_y) <= 0)
      start_y = 0;
    if ((new_x) >= (pels - bs))
      stop_x = 0;
    if ((new_y) >= (lines - bs))
      stop_y = 0;
  }
  search[0].x = 0;
  search[0].y = 0;
  search[1].x = start_x;
  search[1].y = start_y;        /* 1 2 3   */
  search[2].x = 0;
  search[2].y = start_y;        /* 4 0 5   */
  search[3].x = stop_x;
  search[3].y = start_y;        /* 6 7 8   */
  search[4].x = start_x;
  search[4].y = 0;
  search[5].x = stop_x;
  search[5].y = 0;
  search[6].x = start_x;
  search[6].y = stop_y;
  search[7].x = 0;
  search[7].y = stop_y;
  search[8].x = stop_x;
  search[8].y = stop_y;

  AE_min = INT_MAX;
  min_pos = 0;
  for (i = 0; i < 9; i++)
  {
    AE = 0;
    for (n = 0; n < bs; n++)
    {
      for (m = 0; m < bs; m++)
      {
        /* Find absolute error */
        half_pel = *(prev + 2 * new_x + 2 * m + search[i].x +
                     (2 * new_y + 2 * n + search[i].y) * lx * 2);
        AE += abs (half_pel - *(curr + m + n * 16));
      }
    }
    /* if (i == 0 && fr->x == 0 && fr->y == 0 && bs == 16) AE -=
     * PREF_NULL_VEC; */
    if (AE < AE_min)
    {
      AE_min = AE;
      min_pos = i;
    }
  }

  /* Store optimal values */
  fr->min_error = AE_min;
  fr->x_half = search[min_pos].x;
  fr->y_half = search[min_pos].y;

  return;
}


/**********************************************************************
 *
 *	Name:        ChooseMode
 *	Description:    chooses coding mode
 *
 *	Input:	        pointer to original fram, min_error from
 *                      integer pel search, DQUANT
 *	Returns:        1 for Inter, 0 for Intra
 *
 *	Date: 941130	Author: Karl.Lillevold@nta.no
 *
 ***********************************************************************/

int ChooseMode (unsigned char *curr, int x_pos, int y_pos, int min_SAD)
{
  int i, j;
  int MB_mean = 0, A = 0;
  int y_off;

  for (j = 0; j < MB_SIZE; j++)
  {
    y_off = (y_pos + j) * pels;
    for (i = 0; i < MB_SIZE; i++)
    {
      MB_mean += *(curr + x_pos + i + y_off);
    }
  }
  MB_mean /= (MB_SIZE * MB_SIZE);
  for (j = 0; j < MB_SIZE; j++)
  {
    y_off = (y_pos + j) * pels;
    for (i = 0; i < MB_SIZE; i++)
    {
      A += abs (*(curr + x_pos + i + y_off) - MB_mean);
    }
  }

  if (A < (min_SAD - 500))
    return MODE_INTRA;
  else
    return MODE_INTER;
}

int ModifyMode (int Mode, int dquant, int EPTYPE)
{

  if (Mode == MODE_INTRA)
  {
    if (dquant != 0)
      return MODE_INTRA_Q;
    else
      return MODE_INTRA;
  } else if (Mode == MODE_INTER)
  {
    if (dquant != 0)
      return MODE_INTER_Q;
    else
      return MODE_INTER;
  } else if (Mode == MODE_INTER4V)
  {
    if (dquant != 0 && EPTYPE)
      return MODE_INTER4V_Q;
    else
      return MODE_INTER4V;
  } else
    return Mode;
}


void InitializeMV(MotionVector *MV[7][MBR+1][MBC+2])
{
  int i,j,k;

  for (j=0; j<(lines>>4)+1; ++j)
  {
    for (i=0; i<(pels>>4)+2; ++i)
    {
      for (k=0; k<7; ++k)
      {
        MV[k][j][i] = (MotionVector *)calloc(1,sizeof(MotionVector));
      }
    }
  }
}
