/************************************************************************
 *
 *  coder.c, main coding engine of tmn (TMN encoder)
 *
 *  Copyright (C) 1997  University of BC, Canada
 *
 *  Contacts: 
 *  Michael Gallant                   <mikeg@ee.ubc.ca>
 *  Guy Cote                          <guyc@ee.ubc.ca>
 *  Berna Erol                        <bernae@ee.ubc.ca>
 *
 *  UBC Image Processing Laboratory   http://www.ee.ubc.ca/image
 *  2356 Main Mall                    tel.: +1 604 822 4051
 *  Vancouver BC Canada V6T1Z4        fax.: +1 604 822 5949
 *
 *  Copyright (C) 1995, 1996  Telenor R&D, Norway
 *  
 *  Contacts: 
 *  Robert Danielsen                  <Robert.Danielsen@nta.no>
 *
 *  Telenor Research and Development  http://www.nta.no/brukere/DVC/
 *  P.O.Box 83                        tel.:   +47 63 84 84 00
 *  N-2007 Kjeller, Norway            fax.:   +47 63 81 00 76
 *  
 ************************************************************************/

/*
 * Disclaimer of Warranty
 *
 * These software programs are available to the user without any
 * license fee or royalty on an "as is" basis. The University of
 * British Columbia disclaims any and all warranties, whether
 * express, implied, or statuary, including any implied warranties
 * or merchantability or of fitness for a particular purpose.  In no
 * event shall the copyright-holder be liable for any incidental,
 * punitive, or consequential damages of any kind whatsoever arising
 * from the use of these programs.
 *
 * This disclaimer of warranty extends to the user of these programs
 * and user's customers, employees, agents, transferees, successors,
 * and assigns.
 *
 * The University of British Columbia does not represent or warrant
 * that the programs furnished hereunder are free of infringement of
 * any third-party patents.
 *
 * Commercial implementations of H.263, including shareware, are
 * subject to royalty fees to patent holders.  Many of these patents
 * are general enough such that they are unavoidable regardless of
 * implementation design.
 *
*/


/*****************************************************************
 *
 * Modified by Pat Mulroy, BT Labs to run syntax based arithmetic
 * coding.  SAC option, H.263 (Annex E).
 *
 *****************************************************************/


#include"sim.h"

/**********************************************************************
 *
 *	Name:           CodeOneOrTwo
 *	Description:	code one image normally or two images
 *                      as a PB-frame (CodeTwoPB and CodeOnePred merged)
 *	
 *	Input:          pointer to image, prev_image, prev_recon, Q
 *        
 *	Returns:	pointer to reconstructed image
 *	Side effects:	memory is allocated to recon image
 *
 *	Date: 950221	Author:	<klillevo@mailbox.jf.intel.com>
 *            970831    modified by mikeg@ee.ubc.ca
 *            970815    modified by guyc@ee.ubc.ca
 *
 ***********************************************************************/

void CodeOneOrTwo(PictImage *curr, PictImage *B_image, PictImage *prev, 
          PictImage *pr, int QP, int frameskip, Bits *bits,
          Pict *pic, PictImage *B_recon, PictImage *recon)
{
  unsigned char *prev_ipol,*prev_ipol_woRTYPE, *pi, *pi_woRTYPE;
  PictImage *prev_recon=NULL, *pr_edge = NULL;
  MotionVector *MV[7][MBR+1][MBC+2];
  MotionVector *B_f_MV[7][MBR+1][MBC+2];
  MotionVector ZERO = {0,0,0,0,0};
  MB_Structure *recon_data_P; 
  MB_Structure *recon_data_B=NULL; 
  MB_Structure *diff; 
  int *qcoeff_P, *rcoeff, *coeff;
  int *qcoeff_B=NULL;
  int Mode;
  int CBP, CBPB=0;
  int bquant[] = {5,6,7,8};
  int QP_B;
  int newgob, advanced_temporarily_off = DEF_ADV_MODE;
  int i,j,k,B;

  /* buffer control vars */
  float QP_cumulative = (float)0.0;
  int abs_mb_num = 0, QuantChangePostponed = 0;
  int QP_new, QP_prev, dquant, QP_xmitted=QP;

  /* Rate control variables */
  Bits *bits_prev = (Bits*)calloc(1,sizeof(Bits));

  MB_Structure *pred_P = (MB_Structure *)malloc(sizeof(MB_Structure));
  MB_Structure *pred_B = (MB_Structure *) malloc (sizeof (MB_Structure));
  
  int PB_pred_type;

  /* advanced intra coding variables */
  int *store_rcoeff, *store_coeff, *pcoeff; /* used to do prediction in advanced intra coding */
  int store_pb;
 
  /* Reference Picture Selection variables */
  static int sync_gob = 0;
  int sync_gob_done = 0;
  int encode_this_gob = 1; /* Always on if Annex N not used */
  int sync_gob_this_frame = 0;

  ZeroBits(bits);

  /* Currently, the MV info is stored in 7 MV structs per MB. MV[0]
   * stores the 16x16 MV. MV[1] through MV[4] store the 8x8 MVs. MV[6]
   * stores the PB delta MV or the forward predicted B MV. MV[5]
   * stores the backward predicted MV for true-B pictures. */ 

  InitializeMV(MV);
  InitializeMV(B_f_MV);

  /* Mark PMV's outside the frame. */
  for (i = 1; i < (pels>>4)+1; i++) 
  {
    for (j = 0; j < 7; j++)
    {
      MarkVec(MV[j][0][i]); 
      MarkVec(B_f_MV[j][0][i]); 
    }
    MV[0][0][i]->Mode = MODE_INTRA;
    B_f_MV[0][0][i]->Mode = MODE_INTRA;
  }

  /* Zero out PMV's outside the frame */
  for (i = 0; i < (lines>>4)+1; i++) 
  {
    for (j = 0; j < 7; j++) 
    {
      ZeroVec(MV[j][i][0]);
      ZeroVec(MV[j][i][(pels>>4)+1]);
      ZeroVec(B_f_MV[j][i][0]);
      ZeroVec(B_f_MV[j][i][(pels>>4)+1]);
    }

    MV[0][i][0]->Mode = MODE_INTRA;
    MV[0][i][(pels>>4)+1]->Mode = MODE_INTRA;

    B_f_MV[0][i][0]->Mode = MODE_INTRA;
    B_f_MV[0][i][(pels>>4)+1]->Mode = MODE_INTRA;
  }

  GenerateFrameAndInterpolatedImages(pr, pic, &prev_ipol, &prev_recon, &pi, &pr_edge);

  /* Integer and half pel motion estimation */
  MotionEstimatePicture( curr->lum,prev_recon->lum, NULL, prev_ipol,
                         NULL, pic->seek_dist, MV, pic->use_gobsync,
                         P_PICTURE_ESTIMATION);

  if (successive_B_frames)
  {
    StoreDirectModeVectors(MV, True_B_Direct_Mode_MV);
  }

  if (pic->PB)
  { /* Interpolate the image for B picture prediction without using ETYPE*/
    if (mv_outside_frame) 
    {
      if (long_vectors) B = 16; else  B = 8;
      pi_woRTYPE = InterpolateImage(pr_edge->lum, pels+4*B, lines+4*B,0);
      prev_ipol_woRTYPE = pi_woRTYPE + (2*pels + 8*B) * 4*B + 4*B; 
    }
    else 
    {
      pi_woRTYPE = InterpolateImage(pr->lum,pels,lines,0);
      prev_ipol_woRTYPE = pi_woRTYPE;
    }
  }

  if (PCT_IPB == pic->picture_coding_type)
  {
    if (adv_pred)
    {
      /* Turn off advanced coding since there can be only 1 
       * motion vector for B frames of IPB. */
      advanced_temporarily_off = YES;
      overlapping_MC = OFF;
      adv_pred = OFF;
      use_4mv = OFF;
    }

    /* Forward motion Integer and half pel estimation for improved PB frames.
     * Motion estimation type is set to PB_PICTURE so that long vectors will 
     * not be used. */
    MotionEstimatePicture( B_image->lum,prev_recon->lum, NULL, prev_ipol_woRTYPE,
                           NULL, pic->seek_dist, B_f_MV, pic->use_gobsync, 
                           PB_PICTURE_ESTIMATION);                          

    if (advanced_temporarily_off)
    {
      advanced_temporarily_off = NO;
      overlapping_MC = ON;
      adv_pred = ON;
      use_4mv = ON;
    }
  }

  switch (rate_control_method) 
  {
    case NO:
    case OFFLINE_RC:

      QP_new = QP_xmitted = QP_prev = QP; /* Copy the passed value of QP */
      break;

    case  TMN8_RC:
      
      /* Initialization routine for Frame Layer Rate Control */
      /* Jordi Ribas added more parameters to InitializeRateControlMB */
      InitializeRateControlMB(curr, (float)pic->bit_rate, 
                              (pic->PB ? pic->target_frame_rate/2 : pic->target_frame_rate), MV,
                              prev_recon, prev_ipol, 
                              pred_P, pic->PB, pic->RTYPE);

      /* compute the first QP to be included in the picture header */
      QP_new = Compute_QP(0,0);
      QP_xmitted = QP_prev = QP_new; 
      break;

    case  TMN5_RC:

      /* Initialization routine for MB Layer Rate Control */
      QP_new = InitializeQuantizer(PCT_INTER, (float)pic->bit_rate, 
               (pic->PB ? pic->target_frame_rate/2 : pic->target_frame_rate),
               pic->QP_mean);
      QP_xmitted = QP_prev = QP_new;
      break;

    default:
      break;
  }

  dquant = 0; 

  for ( j = 0; j < lines/MB_SIZE; j++) 
  {

    if (rate_control_method == TMN5_RC) 
    {
      /* QP updated at the beginning of each row for Frame layer rate control */
      AddBitsPicture(bits);
      QP_new =  UpdateQuantizer(abs_mb_num, pic->QP_mean, PCT_INTER, 
                (float)pic->bit_rate, pels/MB_SIZE, lines/MB_SIZE, 
                bits->total);
    }  

    newgob = 0;
    /* Do VRC on GOBs -- just add sync gobs at user defined frequency for now 
     * and only in the sync frames */
    if (pic->reference_picture_selection && pic->sync 
        && (pic->use_gobsync && j%pic->use_gobsync == 0))
    {
      /* Determine if this GOB is a sync GOB */
      if (j == sync_gob && !sync_gob_done)
      {
        ++sync_gob_this_frame;
        sync_gob = (sync_gob+pic->use_gobsync);
        if (sync_gob >= (lines/MB_SIZE)) sync_gob = 0;
        fprintf(stdout, "Coding Sync GOB: %d\n", j);
        encode_this_gob = 1;
        if (sync_gob_this_frame == sync_gobs_per_frame) sync_gob_done = 1;
      }
      else
        encode_this_gob = 0;
    } 

    /* encoder this GOBs (used for VRC on GOBs, otherwise always true) */
    if (encode_this_gob)
    {
    if (j == 0) 
    {
      if (advanced_intra_coding)
      {
         /* store the coeff for the frame */
         if ((store_coeff=(int *)calloc(384*(pels/MB_SIZE)*(lines/MB_SIZE), sizeof(int))) == 0) 
         {
            fprintf(stderr,"coder(): Couldn't allocate store_coeff.\n");
            exit(-1);
         }
         if ((store_rcoeff=(int *)calloc(384*(pels/MB_SIZE)*(lines/MB_SIZE), sizeof(int))) == 0) 
         {
            fprintf(stderr,"coder(): Couldn't allocate store_rcoeff.\n");
            exit(-1);
         }
      }
      pic->QUANT = QP_new;
      bits->header += CountBitsPicture(pic);
      QP_xmitted = QP_prev = QP_new;
    }
    else if (pic->use_gobsync && j%pic->use_gobsync == 0) 
    {
      bits->header += CountBitsGOB(j,QP_new,pic); /* insert gob header */
      QP_xmitted = QP_prev = QP_new;
      newgob = 1;
    }
  
    for ( i = 0; i < pels/MB_SIZE; i++) 
    {     
      if (rate_control_method == TMN8_RC) 
      {
        /* Compute optimized quantizer for the current macroblock */
        QP_new = Compute_QP(i,j);
      }
      /* Update of dquant, check and correct its limit */
      if (PCT_B != pic->picture_coding_type)
      {
        
        dquant = QP_new - QP_prev;
        if ( dquant != 0 && MV[0][j+1][i+1]->Mode == MODE_INTER4V
	     && !EPTYPE) 
        {
          /* It is not possible to change the quantizer and at the same
           * time use 8x8 vectors, unless H.263+ PLUS PTYPE is used. 
           * Turning off 8x8 vectors is not
           * possible at this stage because the previous macroblock
           * encoded assumed this one should use 8x8 vectors. Therefore
           * the change of quantizer is postponed until the first MB
           * without 8x8 vectors */
      	   dquant = 0;
           QP_xmitted = QP_prev;
           QuantChangePostponed = 1;
        }
        else 
        {
          QP_xmitted = QP_new;
          QuantChangePostponed = 0;
        }

        /* Unless modified quantization mode is in use restrict the range of
         * dquant to [-2,+2]*/
        if (!modified_quantization) 
	      {
      	  if (dquant > 2)  
          { 
            dquant =  2; 
            QP_xmitted = QP_prev + dquant;
          }
   	      if (dquant < -2) 
          { 
            dquant = -2; 
            QP_xmitted = QP_prev + dquant;
          }
	      }
        else 
        {
          /* Modified quantization mode is in use. */
#ifdef RESTRICTED_MQ
          /* restrict dquant so that when the modified quantization mode is on quant only uses 2 bits*/
          if (dquant != 0)
          {
            dquant= Get_restricted_MQ(dquant,QP_prev);
            QP_xmitted = QP_prev + dquant;
          }
#endif
          if (dquant!=0) 	
	        {
            if ((QP_prev >= 21 && dquant == -3) ||
                (QP_prev >= 11 && QP_prev <= 20 && dquant == -2) ||
                (QP_prev >= 2  && QP_prev <= 10 && dquant == -1) ||
                (QP_prev == 1  && dquant == 2)) 
	          {
   	          /* dquant will be coded into 2 bits*/
       	      pic->dquant_size = 2;
	            dquant = 2;
            }
            else 
            {
              if ((QP_prev == 31 && dquant == -5) ||
                  (QP_prev == 30 && dquant == 1) ||
                  (QP_prev == 29 && dquant == 2) ||
                  (QP_prev >= 21 && QP_prev <= 28 && dquant == 3) ||
                  (QP_prev >= 11 && QP_prev <= 20 && dquant == 2) ||
                  (QP_prev >= 1 && QP_prev <= 10 && dquant == 1))
	            {
		            /* dquant will be coded into 2 bits*/
	      	      pic->dquant_size = 2;
		            dquant = 3;
	            } 
	            else 
	            {
	              /* dquant will be coded intou 6 bits. */
		            pic->dquant_size=6;

		            /* instead of the difference between current quantizer and
		             * the previous quantizer,current quantizer is coded into dquant parameter*/
                dquant = QP_new;
              }
            }
          }
        }
	      pic->DQUANT = dquant;

        /* modify mode if dquant != 0 (e.g. MODE_INTER -> MODE_INTER_Q  
         * or MODE_INTER4V -> MODE_INTER4V_Q) */
        Mode = ModifyMode(MV[0][j+1][i+1]->Mode,pic->DQUANT, EPTYPE);
        MV[0][j+1][i+1]->Mode = Mode;
      }

      pic->MB = i + j * (pels/MB_SIZE);
      
      /* store the QP for every macroblock */
      quant_map[j+1][i+1] = QP_xmitted;

      if ((rcoeff = (int *)malloc(sizeof(int)*384)) == NULL) 
      {
        fprintf(stderr,"MB_Coder: Could not allocate space for rcoeff\n");
        exit(-1);
      }  

      if ( Mode == MODE_INTER || Mode == MODE_INTER_Q || 
          Mode == MODE_INTER4V || Mode==MODE_INTER4V_Q)
      {
        /* Predict P-MB */
	      diff = Predict_P( curr,prev_recon,prev_ipol, pred_P,
                          i*MB_SIZE,j*MB_SIZE,MV,pic->PB, pic->RTYPE);
      }
      else 
      {
        /* fill prediction array with 0s */
        memset (pred_P, 0, sizeof (MB_Structure));
        /* fill the difference array with the current macroblock pixels */
        diff = (MB_Structure *)malloc(sizeof(MB_Structure));
      	FillLumBlock(i*MB_SIZE, j*MB_SIZE, curr, diff);
        FillChromBlock(i*MB_SIZE, j*MB_SIZE, curr, diff);
      }

      /* P or INTRA Macroblock */
      if ((qcoeff_P=(int *)malloc(sizeof(int)*384)) == 0) 
      {
        fprintf(stderr,"coder(): Couldn't allocate qcoeff_P.\n");
        exit(-1);
      }
      coeff = MB_Encode(diff);
      if (advanced_intra_coding) 
      {
        if (!(Mode == MODE_INTRA || Mode == MODE_INTRA_Q)) 
        {
          for (k=0;k<6;k++) 
          {    
            /* store default coeff if non-intra macroblock */
            store_coeff[(i + j * pels/MB_SIZE) * 384 + k * 64] = 1024;
            store_rcoeff[(i + j * pels/MB_SIZE) * 384 + k * 64] = 1024;
          }
          for (k=0;k<6;k++)
          {
            Quant_blk(coeff,qcoeff_P,QP_xmitted, Mode,k);
          }
          CBP = FindCBP(qcoeff_P, Mode, 64);
          if (CBP == 0 && (Mode == MODE_INTER || Mode == MODE_INTER_Q)) 
          {
            ZeroMBlock(diff);
          }
          else
          {  
            for (k=0;k<6;k++)
            {
              Quant_blk(coeff,qcoeff_P,QP_xmitted, Mode,k);
              Dequant(qcoeff_P, rcoeff, QP_xmitted, Mode,k);
            }
            MB_Decode(rcoeff, diff);
          }
        }
        else 
        {
          if ((pcoeff=(int *)malloc(sizeof(int)*384)) == 0) 
          {
            fprintf(stderr,"coder(): Couldn't allocate pcoeff.\n");
            exit(-1);
          }

          /* store the quantized DCT coefficients */
          memcpy( (void *) (store_coeff + (i + j*pels/MB_SIZE)*384), (void *) coeff, sizeof(int) * 384);

          /* Do Intra mode prediction */
          pic->Intra_Mode = Choose_Intra_Mode(pcoeff, store_coeff, i, j, newgob);
          for (k=0;k<6;k++) 
          { 
            Intra_AC_DC_Encode(coeff, store_rcoeff, pic->Intra_Mode, i, j, newgob,k);
            Quant_blk(coeff,pcoeff,QP_xmitted,Mode,k);
            Dequant(pcoeff, rcoeff, QP_xmitted, Mode,k);
            Intra_AC_DC_Decode(rcoeff, store_rcoeff, pic->Intra_Mode, i, j, newgob,k);
          }
          MB_Decode(rcoeff, diff);
          CBP = FindCBP(pcoeff,Mode,64);
        }    
      }
      else
      {  
        for (k=0;k<6;k++)
          Quant_blk(coeff,qcoeff_P,QP_xmitted, Mode,k);
        CBP = FindCBP(qcoeff_P, Mode, 64);
        if (CBP == 0 && (Mode == MODE_INTER || Mode == MODE_INTER_Q)) 
          ZeroMBlock(diff);
        else
        {
          for (k=0;k<6;k++)  
            Dequant(qcoeff_P, rcoeff, QP_xmitted, Mode,k); 
          MB_Decode(rcoeff, diff);
        }
      }

      recon_data_P = MB_Reconstruct (diff, pred_P);

      Clip(recon_data_P);
      free(diff);
      free(coeff);
        
      /* Predict B-MB using reconstructed P-MB and prev. recon. image */
      if (pic->PB) 
      {
        diff = Predict_B( B_image, prev_recon, prev_ipol_woRTYPE,pred_B,i*MB_SIZE, j*MB_SIZE,
                          MV,B_f_MV, recon_data_P, frameskip, pic->TRB, pic->PB, 
                          &PB_pred_type);
        if (QP_xmitted == 0)
        {
          /* (QP = 0 means no quantization) */
          QP_B = 0;  
        }
        else 
        {
          QP_B = mmax(1,mmin(31,bquant[pic->BQUANT]*QP_xmitted/4));
        }
        if ((qcoeff_B=(int *)malloc(sizeof(int)*384)) == 0) 
        {
		      fprintf(stderr,"coder(): Couldn't allocate qcoeff_B.\n");
          exit(-1);
        }
        coeff = MB_Encode(diff);
        for (k=0;k<6;k++)
          Quant_blk(coeff,qcoeff_B,QP_B, MODE_INTER,k);
        CBPB = FindCBP(qcoeff_B, MODE_INTER, 64);
         
        if (CBPB)
        { 
          for (k=0;k<6;k++)
            Dequant(qcoeff_B, rcoeff, QP_B, MODE_INTER,k);
          MB_Decode(rcoeff, diff);
        }
        else
          ZeroMBlock(diff);

        recon_data_B = MB_Reconstruct(diff,pred_B);
        Clip(recon_data_B);

        /* decide MODB */ 
        if (pic->PB == IM_PB_FRAMES) 
        {
          if (CBPB)
          {
            if (PB_pred_type == FORWARD_PREDICTION)
              pic->MODB = PBMODE_CBPB_FRW_PRED;
            else if (PB_pred_type == BACKWARD_PREDICTION)
            pic->MODB = PBMODE_CBPB_BCKW_PRED;
            else
              pic->MODB = PBMODE_CBPB_BIDIR_PRED;
          } 
          else
          {
            if (PB_pred_type == FORWARD_PREDICTION)
              pic->MODB = PBMODE_FRW_PRED;
             else if (PB_pred_type == BACKWARD_PREDICTION)
              pic->MODB = PBMODE_BCKW_PRED;
            else
              pic->MODB = PBMODE_BIDIR_PRED;
          }
          if (PB_pred_type == FORWARD_PREDICTION)
          {
            /* Store the forward predicted MV instead of PB delta. */
            MV[6][j + 1][i + 1]->x = B_f_MV[0][j + 1][i + 1]->x;
            MV[6][j + 1][i + 1]->y = B_f_MV[0][j + 1][i + 1]->y;
            MV[6][j + 1][i + 1]->x_half = B_f_MV[0][j + 1][i + 1]->x_half;
            MV[6][j + 1][i + 1]->y_half = B_f_MV[0][j + 1][i + 1]->y_half;
          }
          if (PB_pred_type == BACKWARD_PREDICTION)
          {
            MV[6][j + 1][i + 1]->x = 0;
            MV[6][j + 1][i + 1]->y = 0;
            MV[6][j + 1][i + 1]->x_half = 0;
            MV[6][j + 1][i + 1]->y_half = 0;
          }
        }
        else 
        {
          if (CBPB) 
          {
            pic->MODB = PBMODE_CBPB_MVDB;
          }
          else  
          {
            if (MV[6][j+1][i+1]->x == 0 && MV[6][j+1][i+1]->y == 0)
              pic->MODB = PBMODE_NORMAL;
            else
              pic->MODB = PBMODE_MVDB;
          }
        }
        free(diff);
        free(coeff);
        
      }
      else
        ZeroVec(MV[6][j+1][i+1]); /* Zero out PB deltas */

      if ((CBP==0) && (CBPB==0) && (EqualVec(MV[0][j+1][i+1],&ZERO)) && 
          (EqualVec(MV[6][j+1][i+1],&ZERO)) &&
          (Mode == MODE_INTER || Mode == MODE_INTER_Q) &&
          (pic->MODB==0)) 
      { 
        /* Skipped MB : both CBP and CBPB are zero, 16x16 vector is zero,
         * PB delta vector is zero, Mode = MODE_INTER and B picture is
         * predicted bidirectionally. */
        coded_map[j+1][i+1] = 0;
        quant_map[j+1][i+1] = 0;

        if (Mode == MODE_INTER_Q || Mode == MODE_INTER4V_Q) 
        {
          /* DQUANT != 0 but not coded anyway */
          QP_xmitted = QP_prev;
          pic->DQUANT = 0;
          Mode = MODE_INTER;
        }
        if (!syntax_arith_coding)
          CountBitsMB(Mode,1,CBP,CBPB,pic,bits,0);
        else
          Count_sac_BitsMB(Mode,1,CBP,CBPB,pic,bits);
      }
      else 
      {
        /* Normal MB */
        if (!syntax_arith_coding) 
        { 
          /* VLC */
          CountBitsMB(Mode,0,CBP,CBPB,pic,bits,0);
          if (!CBP || !CBPB)
            intra_refresh[j+1][i+1] += 1;

          if (Mode == MODE_INTER  || Mode == MODE_INTER_Q) 
          {
            coded_map[j+1][i+1] = 1;
            quant_map[j+1][i+1] = QP_xmitted;

            bits->no_inter++;
            CountBitsVectors(MV, bits, i, j, Mode, newgob, pic, 0);
          }
          else if (Mode == MODE_INTER4V || Mode == MODE_INTER4V_Q) 
          {
            coded_map[j+1][i+1] = 1;
            quant_map[j+1][i+1] = QP_xmitted;

            bits->no_inter4v++;
            CountBitsVectors(MV, bits, i, j, Mode, newgob, pic, 0);
          }
          else 
          {
            /* MODE_INTRA or MODE_INTRA_Q */
            coded_map[j+1][i+1] = 2;
            quant_map[j+1][i+1] = QP_xmitted;
            intra_refresh[j+1][i+1] = 0;
           
            bits->no_intra++;
            if (pic->PB)
            {
              CountBitsVectors(MV, bits, i, j, Mode, newgob, pic, 0);
            }
          }

          if ( (Mode == MODE_INTRA || Mode == MODE_INTRA_Q) &&
                advanced_intra_coding )
          {
            Scan(pcoeff,pic->Intra_Mode);
            CountBitsCoeff(pcoeff, Mode, CBP, bits, 64);
          }
          else if (CBP || Mode == MODE_INTRA || Mode == MODE_INTRA_Q)
          {
            Scan(qcoeff_P,0);
            CountBitsCoeff(qcoeff_P, Mode, CBP, bits, 64);  
          }        

          if (CBPB)
          {
            Scan(qcoeff_B,0);
            CountBitsCoeff(qcoeff_B, MODE_INTER, CBPB, bits, 64);
          }
        } 
        /* end VLC */
        else 
        { 
          /* SAC */
          Count_sac_BitsMB(Mode,0,CBP,CBPB,pic,bits);
          if (!CBP || !CBPB)
            intra_refresh[j+1][i+1] += 1;

          if (Mode == MODE_INTER  || Mode == MODE_INTER_Q) 
          {
            coded_map[j+1][i+1] = 1;
            quant_map[j+1][i+1] = QP_xmitted;
      
            bits->no_inter++;
            Count_sac_BitsVectors(MV, bits, i, j, Mode, newgob, pic);
          }
          else if (Mode == MODE_INTER4V || Mode == MODE_INTER4V_Q) 
          {
            coded_map[j+1][i+1] = 1;
            quant_map[j+1][i+1] = QP_xmitted;

            bits->no_inter4v++;
            Count_sac_BitsVectors(MV, bits, i, j, Mode, newgob, pic);
          }
          else 
          {
            /* MODE_INTRA or MODE_INTRA_Q */
            coded_map[j+1][i+1] = 2;
            quant_map[j+1][i+1] = QP_xmitted;
            intra_refresh[j+1][i+1] = 0;

            bits->no_intra++;
            if (pic->PB)
	            Count_sac_BitsVectors(MV, bits, i, j, Mode, newgob, pic);
          }

          if ((Mode == MODE_INTRA || Mode == MODE_INTRA_Q) &&
               advanced_intra_coding)
          {
            Scan(pcoeff, pic->Intra_Mode);
            Count_sac_BitsCoeff(pcoeff, Mode, CBP, bits, 64);
          }
          else if ( CBP || Mode == MODE_INTRA || Mode == MODE_INTRA_Q)
          {
            Scan(qcoeff_P,0);           
            Count_sac_BitsCoeff(qcoeff_P, Mode, CBP, bits, 64);
          }

          if (CBPB)
          {
            Scan(qcoeff_B,0);
            Count_sac_BitsCoeff(qcoeff_B, MODE_INTER, CBPB, bits, 64);
          } 
        }
        /* end SAC */
        QP_prev = QP_xmitted;
      }

      /* update the rate control parameters for TMN-8 rate control */
      if (rate_control_method == TMN8_RC) 
      { 
        AddBitsPicture (bits);  
        UpdateRateControlMB((bits->total - bits_prev->total),
                            ((bits->Y + bits->C)-(bits_prev->Y + bits_prev->C)), i, j, QP_xmitted);  
        bits_prev->total = bits->total;
        bits_prev->Y = bits->Y;
        bits_prev->C = bits->C;
      } 

      abs_mb_num++;
      QP_cumulative += QP_xmitted;     
#ifdef PRINTQ 
      /* most useful when quantizer changes within a picture */
      if (QuantChangePostponed)
        fprintf(stdout,"@%2d",QP_xmitted);
      else
        fprintf(stdout," %2d",QP_xmitted);
#endif

      if (pic->PB) 
      {
        ReconImage(i,j,recon_data_B,B_recon);
      }

      ReconImage(i,j,recon_data_P,recon);

      if ( (PCT_IPB == pic->picture_coding_type) || PCT_PB == pic->picture_coding_type )
      {
        free(qcoeff_B);
        free(recon_data_B);
      }

      free(recon_data_P);
      free(qcoeff_P);
      free(rcoeff);

      if (advanced_intra_coding && (Mode == MODE_INTRA || Mode == MODE_INTRA_Q))  
        free(pcoeff);
    }
    }
#ifdef PRINTQ
    fprintf(stdout,"\n");
#endif
  }

  /* Do the deblocking filtering, not used in true B pictures. */
  if (deblocking_filter) 
  {
    store_pb = pic->PB;
    pic->PB = 0;
    EdgeFilter(recon, pic);
    pic->PB = store_pb;
    if (pic->PB)
      EdgeFilter(B_recon,pic);
  }

  pic->QP_mean = QP_cumulative/(float)abs_mb_num;

  /* Free memory */
  free (pred_P);
  free (pred_B);

  if (mv_outside_frame)
  {
    free(prev_recon);
    FreeImage(pr_edge);
  }

  if (PCT_IPB == pic->picture_coding_type || PCT_PB == pic->picture_coding_type)
    free(pi_woRTYPE);

  free(pi);
 
  for (j = 0; j < (lines>>4)+1; j++)
    for (i = 0; i < (pels>>4)+2; i++) 
      for (k = 0; k < 7; k++)
      {
        free(MV[k][j][i]);
        free(B_f_MV[k][j][i]);
      }

  if (advanced_intra_coding)
  {
    free(store_coeff);
    free(store_rcoeff);
  }

  if (advanced_temporarily_off)
  {
    overlapping_MC = ON;
    adv_pred = ON;
    use_4mv = ON;
    advanced_temporarily_off = NO;
  }
  return;
}


/**********************************************************************
 *
 *	Name:        Predict_P
 *	Description:    Predicts P macroblock in advanced or normal
 *                      mode
 *
 *	Input:        pointers to current and previous frames
 *        and previous interpolated image,
 *                      position and motion vector array
 *	Returns:	pointer to MB_Structure of data to be coded
 *	Side effects:	allocates memory to MB_Structure
 *
 *	Date: 9501        Author: <klillevo@mailbox.jf.intel.com>
 *
 ***********************************************************************/

MB_Structure *Predict_P (PictImage * curr_image, PictImage * prev_image,
                         unsigned char *prev_ipol, MB_Structure * pred_macroblock,
                         int x, int y,
                         MotionVector * MV[7][MBR + 1][MBC + 2], int PB, int RTYPE)
{
  int m, n;
  int curr[16][16];
  MotionVector *fr0, *fr1, *fr2, *fr3, *fr4;
  int sum, dx, dy;
  int xmb, ymb;

  MB_Structure *pred_error = (MB_Structure *) malloc (sizeof (MB_Structure));

  xmb = x / MB_SIZE + 1;
  ymb = y / MB_SIZE + 1;

  fr0 = MV[0][ymb][xmb];
  fr1 = MV[1][ymb][xmb];
  fr2 = MV[2][ymb][xmb];
  fr3 = MV[3][ymb][xmb];
  fr4 = MV[4][ymb][xmb];

  /* Find MB in current image */
  FindMB (x, y, curr_image->lum, curr);

  /* Find prediction based on half pel MV    */

  if (overlapping_MC)
  {
    FindPredOBMC (x, y, MV, prev_ipol, &pred_macroblock->lum[0][0], 0, PB);
    FindPredOBMC (x, y, MV, prev_ipol, &pred_macroblock->lum[0][8], 1, PB);
    FindPredOBMC (x, y, MV, prev_ipol, &pred_macroblock->lum[8][0], 2, PB);
    FindPredOBMC (x, y, MV, prev_ipol, &pred_macroblock->lum[8][8], 3, PB);
  } else if (fr0->Mode == MODE_INTER4V || fr0->Mode == MODE_INTER4V_Q)
  {
    /* Luma */
    FindPred (x, y, fr1, prev_ipol, &pred_macroblock->lum[0][0], 8, 0);
    FindPred (x, y, fr2, prev_ipol, &pred_macroblock->lum[0][8], 8, 1);
    FindPred (x, y, fr3, prev_ipol, &pred_macroblock->lum[8][0], 8, 2);
    FindPred (x, y, fr4, prev_ipol, &pred_macroblock->lum[8][8], 8, 3);
  } else
  {
    FindPred (x, y, fr0, prev_ipol, &pred_macroblock->lum[0][0], 16, 0);
  }

  /* Do the actual prediction */
  if (fr0->Mode == MODE_INTER || fr0->Mode == MODE_INTER_Q)
  {
    for (n = 0; n < MB_SIZE; n++)
      for (m = 0; m < MB_SIZE; m++)
        pred_error->lum[n][m] = (int) (curr[n][m] - pred_macroblock->lum[n][m]);

    dx = 2 * fr0->x + fr0->x_half;
    dy = 2 * fr0->y + fr0->y_half;
    dx = (dx % 4 == 0 ? dx >> 1 : (dx >> 1) | 1);
    dy = (dy % 4 == 0 ? dy >> 1 : (dy >> 1) | 1);

    DoPredChrom_P (x, y, dx, dy, curr_image, prev_image, pred_macroblock, pred_error, RTYPE);

  } else if (fr0->Mode == MODE_INTER4V || fr0->Mode == MODE_INTER4V_Q)
  {
    for (n = 0; n < MB_SIZE; n++)
      for (m = 0; m < MB_SIZE; m++)
        pred_error->lum[n][m] = (int) (curr[n][m] - pred_macroblock->lum[n][m]);

    sum = 2 * fr1->x + fr1->x_half + 2 * fr2->x + fr2->x_half +
      2 * fr3->x + fr3->x_half + 2 * fr4->x + fr4->x_half;
    dx = sign (sum) * (roundtab[abs (sum) % 16] + (abs (sum) / 16) * 2);

    sum = 2 * fr1->y + fr1->y_half + 2 * fr2->y + fr2->y_half +
      2 * fr3->y + fr3->y_half + 2 * fr4->y + fr4->y_half;
    dy = sign (sum) * (roundtab[abs (sum) % 16] + (abs (sum) / 16) * 2);

    DoPredChrom_P (x, y, dx, dy, curr_image, prev_image, pred_macroblock, pred_error, RTYPE);
  } else
    fprintf (stderr, "Illegal Mode in Predict_P (pred.c)\n");

  return pred_error;
}


/***********************************************************************
 *
 *	Name:        Predict_B
 *	Description:    Predicts the B macroblock in PB-frame prediction
 *
 *	Input:	        pointers to current frame, previous recon. frame,
 *                      pos. in image, MV-data, reconstructed macroblock
 *                      from image ahead
 *	Returns:        pointer to differential MB data after prediction
 *	Side effects:   allocates memory to MB_structure
 *
 *	Date: 950113        Author: Karl.Lillevold@nta.no
 *
 ***********************************************************************/

MB_Structure *Predict_B (PictImage * curr_image, PictImage * prev_image,
  unsigned char *prev_ipol, MB_Structure * pred_macroblock, int x, int y,
                          MotionVector * MV[7][MBR + 1][MBC + 2], MotionVector * B_f_MV[7][MBR + 1][MBC + 2],
  MB_Structure * recon_P, int TRD, int TRB, int PB, int *im_PB_pred_type)
{
  int i, j, k, xx, yy;
  int dx, dy, sad, sad_min = INT_MAX, curr[16][16], bdx = 0, bdy = 0;
  MB_Structure *p_err = (MB_Structure *) malloc (sizeof (MB_Structure));
  MB_Structure *pred = (MB_Structure *) malloc (sizeof (MB_Structure));
  MB_Structure *frw_pred = (MB_Structure *) malloc (sizeof (MB_Structure));
  MotionVector *f[7];
  MotionVector *frw[7];
  int xvec, yvec, mvx, mvy;
  int SADbidir, SADforw,SADbackw,min_SAD;
  int forward_pred = 0;
  int bidir_pred = 0;
  int backward_pred = 0;

  for (k = 0; k <= 4; k++)
    f[k] = MV[k][y / MB_SIZE + 1][x / MB_SIZE + 1];

  for (k = 0; k <= 4; k++)
    frw[k] = B_f_MV[k][y / MB_SIZE + 1][x / MB_SIZE + 1];

  /* Find MB in current image */
  FindMB (x, y, curr_image->lum, curr);

  if (f[0]->Mode == MODE_INTER4V || f[0]->Mode == MODE_INTER4V_Q)
  {
    /* Mode INTER4V */
    if (PB == IM_PB_FRAMES)
    {
      bdx = 0;
      bdy = 0;
    } else
    {
      /* Find forward prediction. Luma */
      for (j = -DEF_PBDELTA_WIN; j <= DEF_PBDELTA_WIN; j++)
      {
        for (i = -DEF_PBDELTA_WIN; i <= DEF_PBDELTA_WIN; i++)
        {

          FindForwLumPredPB (prev_ipol, x, y, f[1], &pred->lum[0][0], TRD, TRB, i, j, 8, 0);
          FindForwLumPredPB (prev_ipol, x, y, f[2], &pred->lum[0][8], TRD, TRB, i, j, 8, 1);
          FindForwLumPredPB (prev_ipol, x, y, f[3], &pred->lum[8][0], TRD, TRB, i, j, 8, 2);
          FindForwLumPredPB (prev_ipol, x, y, f[4], &pred->lum[8][8], TRD, TRB, i, j, 8, 3);

          sad = SAD_MB_integer (&curr[0][0], &pred->lum[0][0], 16, INT_MAX);
          if (i == 0 && j == 0)
            sad -= PREF_PBDELTA_NULL_VEC;
          if (sad < sad_min)
          {
            sad_min = sad;
            bdx = i;
            bdy = j;
          }
        }
      }
    }

    FindForwLumPredPB (prev_ipol, x, y, f[1], &pred->lum[0][0], TRD, TRB, bdx, bdy, 8, 0);
    FindForwLumPredPB (prev_ipol, x, y, f[2], &pred->lum[0][8], TRD, TRB, bdx, bdy, 8, 1);
    FindForwLumPredPB (prev_ipol, x, y, f[3], &pred->lum[8][0], TRD, TRB, bdx, bdy, 8, 2);
    FindForwLumPredPB (prev_ipol, x, y, f[4], &pred->lum[8][8], TRD, TRB, bdx, bdy, 8, 3);

    /* Find bidirectional prediction */
    FindBiDirLumPredPB (&recon_P->lum[0][0], f[1], &pred->lum[0][0],
                        TRD, TRB, bdx, bdy, 0, 0);
    FindBiDirLumPredPB (&recon_P->lum[0][8], f[2], &pred->lum[0][8],
                        TRD, TRB, bdx, bdy, 1, 0);
    FindBiDirLumPredPB (&recon_P->lum[8][0], f[3], &pred->lum[8][0],
                        TRD, TRB, bdx, bdy, 0, 1);
    FindBiDirLumPredPB (&recon_P->lum[8][8], f[4], &pred->lum[8][8],
                        TRD, TRB, bdx, bdy, 1, 1);

  } else
  {
    /* Mode INTER or INTER_Q */
    if (PB == IM_PB_FRAMES)
    {
      bdx = 0;
      bdy = 0;
    } else
    {
      /* Find forward prediction */
      for (j = -DEF_PBDELTA_WIN; j <= DEF_PBDELTA_WIN; j++)
      {
        for (i = -DEF_PBDELTA_WIN; i <= DEF_PBDELTA_WIN; i++)
        {

          dx = i;
          dy = j;

          /* To keep things simple I turn off PB delta vectors at the
           * edges */
          if (!mv_outside_frame)
          {
            if (x == 0)
              dx = 0;
            if (x == pels - MB_SIZE)
              dx = 0;
            if (y == 0)
              dy = 0;
            if (y == lines - MB_SIZE)
              dy = 0;
          }
          if (f[0]->Mode == MODE_INTRA || f[0]->Mode == MODE_INTRA_Q)
          {
            dx = dy = 0;
          }
          if ((f[0]->x == 0 && f[0]->y == 0) &&
              (f[0]->x_half == 0 && f[0]->y_half == 0))
          {
            dx = dy = 0;
          }
          FindForwLumPredPB (prev_ipol, x, y, f[0], &pred->lum[0][0],
                             TRD, TRB, dx, dy, 16, 0);

          sad = SAD_MB_integer (&curr[0][0], &pred->lum[0][0], 16, INT_MAX);
          if (i == 0 && j == 0)
          {
            sad -= PREF_PBDELTA_NULL_VEC;
          }
          if (sad < sad_min)
          {
            sad_min = sad;
            bdx = dx;
            bdy = dy;
          }
        }
      }
    }

    FindForwLumPredPB (prev_ipol, x, y, f[0], &pred->lum[0][0], TRD, TRB,
                       bdx, bdy, 16, 0);

    /* Find bidirectional prediction */
    FindBiDirLumPredPB (&recon_P->lum[0][0], f[0], &pred->lum[0][0],
                        TRD, TRB, bdx, bdy, 0, 0);
    FindBiDirLumPredPB (&recon_P->lum[0][8], f[0], &pred->lum[0][8],
                        TRD, TRB, bdx, bdy, 1, 0);
    FindBiDirLumPredPB (&recon_P->lum[8][0], f[0], &pred->lum[8][0],
                        TRD, TRB, bdx, bdy, 0, 1);
    FindBiDirLumPredPB (&recon_P->lum[8][8], f[0], &pred->lum[8][8],
                        TRD, TRB, bdx, bdy, 1, 1);
  }

  if (PB == IM_PB_FRAMES)
  {
    SADbidir = SAD_MB_integer (&curr[0][0], &pred->lum[0][0], 16, INT_MAX);

    FindPred (x, y, frw[0], prev_ipol, &frw_pred->lum[0][0], 16, 0);
    SADforw = SAD_MB_integer (&curr[0][0], &frw_pred->lum[0][0], 16, INT_MAX);
    SADbackw = SAD_MB_integer (&curr[0][0], &recon_P->lum[0][0], 16, INT_MAX);

    /* make decision about using forward prediction, backward prediction or bidir pre
diction */
    min_SAD=mmin(mmin(SADforw,SADbackw),(SADbidir-100));

    if (min_SAD == (SADbidir - 100))
    {
      /* bidirectional prediction is chosen */
      bidir_pred = 1;
      (*im_PB_pred_type) = BIDIRECTIONAL_PREDICTION;
    } 
    else if (min_SAD==SADbackw)
    {
      /* backward prediction is chosen */
      backward_pred = 1;
      (*im_PB_pred_type) = BACKWARD_PREDICTION;
    }
    else 
    {
      /* forward prediction is chosen */
      forward_pred = 1;
      (*im_PB_pred_type) = FORWARD_PREDICTION;
    }
  }
  if (PB == IM_PB_FRAMES && forward_pred)
  {
    dx = 2 * frw[0]->x + frw[0]->x_half;
    dy = 2 * frw[0]->y + frw[0]->y_half;
    dx = (dx % 4 == 0 ? dx >> 1 : (dx >> 1) | 1);
    dy = (dy % 4 == 0 ? dy >> 1 : (dy >> 1) | 1);

    /* predict B as if it is P */
    DoPredChrom_P (x, y, dx, dy, curr_image, prev_image, frw_pred, p_err, 0);

    for (j = 0; j < MB_SIZE; j++)
      for (i = 0; i < MB_SIZE; i++)
      {
        p_err->lum[j][i] = *(curr_image->lum + x + i + (y + j) * pels) - frw_pred->lum[j][i];
        pred_macroblock->lum[j][i] = frw_pred->lum[j][i];
      }
    xx = x >> 1;
    yy = y >> 1;
    for (j = 0; j < MB_SIZE >> 1; j++)
      for (i = 0; i < MB_SIZE >> 1; i++)
      {
        p_err->Cr[j][i] = *(curr_image->Cr + xx + i + (yy + j) * cpels) - frw_pred->Cr[j][i];
        p_err->Cb[j][i] = *(curr_image->Cb + xx + i + (yy + j) * cpels) - frw_pred->Cb[j][i];
        pred_macroblock->Cr[j][i] = frw_pred->Cr[j][i];
        pred_macroblock->Cb[j][i] = frw_pred->Cb[j][i];
      }
  } 
  else if (PB == IM_PB_FRAMES && backward_pred)
  {
    dx = 0;
    dy = 0;
    
    for (j = 0; j < MB_SIZE; j++)
      for (i = 0; i < MB_SIZE; i++)
      {
        p_err->lum[j][i] = *(curr_image->lum + x + i + (y + j) * pels) - recon_P->lum
[j][i];
        pred_macroblock->lum[j][i] = recon_P->lum[j][i];
      }

    xx = x >> 1;
    yy = y >> 1;

    for (j = 0; j < MB_SIZE >> 1; j++)
      for (i = 0; i < MB_SIZE >> 1; i++)
      {
        p_err->Cr[j][i] = *(curr_image->Cr + xx + i + (yy + j) * cpels) - recon_P->Cr[j][i];
        p_err->Cb[j][i] = *(curr_image->Cb + xx + i + (yy + j) * cpels) - recon_P->Cb[j][i];

        pred_macroblock->Cr[j][i] = recon_P->Cr[j][i];
        pred_macroblock->Cb[j][i] = recon_P->Cb[j][i];
      }
  }
  else
  {
    /* bidir prediction */
    if (f[0]->Mode == MODE_INTER4V || f[0]->Mode == MODE_INTER4V_Q)
    {
      /* Mode INTER4V chroma vectors are sum of B luma vectors divided and
       * rounded */
      xvec = yvec = 0;
      for (k = 1; k <= 4; k++)
      {
        xvec += TRB * (2 * f[k]->x + f[k]->x_half) / TRD + bdx;
        yvec += TRB * (2 * f[k]->y + f[k]->y_half) / TRD + bdy;
      }

      /* round values according to TABLE 16/H.263 */
      dx = sign (xvec) * (roundtab[abs (xvec) % 16] + (abs (xvec) / 16) * 2);
      dy = sign (yvec) * (roundtab[abs (yvec) % 16] + (abs (yvec) / 16) * 2);
      FindChromBlock_P (x, y, dx, dy, prev_image, pred, 0);

      /* chroma vectors are sum of B luma vectors divided and rounded */
      xvec = yvec = 0;
      for (k = 1; k <= 4; k++)
      {
        mvx = 2 * f[k]->x + f[k]->x_half;
        mvy = 2 * f[k]->y + f[k]->y_half;
        xvec += bdx == 0 ? (TRB - TRD) * mvx / TRD : TRB * mvx / TRD + bdx - mvx;
        yvec += bdy == 0 ? (TRB - TRD) * mvy / TRD : TRB * mvy / TRD + bdy - mvy;
      }

      /* round values according to TABLE 16/H.263 */
      dx = sign (xvec) * (roundtab[abs (xvec) % 16] + (abs (xvec) / 16) * 2);
      dy = sign (yvec) * (roundtab[abs (yvec) % 16] + (abs (yvec) / 16) * 2);

      FindBiDirChrPredPB (recon_P, dx, dy, pred);
    } 
    else
    {
      xvec = 4 * (TRB * (2 * f[0]->x + f[0]->x_half) / TRD + bdx);
      yvec = 4 * (TRB * (2 * f[0]->y + f[0]->y_half) / TRD + bdy);
      /* round values according to TABLE 16/H.263 */
      dx = sign (xvec) * (roundtab[abs (xvec) % 16] + (abs (xvec) / 16) * 2);
      dy = sign (yvec) * (roundtab[abs (yvec) % 16] + (abs (yvec) / 16) * 2);

      /* chroma vectors */
      FindChromBlock_P (x, y, dx, dy, prev_image, pred, 0);

      mvx = 2 * f[0]->x + f[0]->x_half;
      xvec = bdx == 0 ? (TRB - TRD) * mvx / TRD : TRB * mvx / TRD + bdx - mvx;
      xvec *= 4;

      mvy = 2 * f[0]->y + f[0]->y_half;
      yvec = bdy == 0 ? (TRB - TRD) * mvy / TRD : TRB * mvy / TRD + bdy - mvy;
      yvec *= 4;

      /* round values according to TABLE 16/H.263 */
      dx = sign (xvec) * (roundtab[abs (xvec) % 16] + (abs (xvec) / 16) * 2);
      dy = sign (yvec) * (roundtab[abs (yvec) % 16] + (abs (yvec) / 16) * 2);

      FindBiDirChrPredPB (recon_P, dx, dy, pred);
    }

    /* Do the actual prediction */
    for (j = 0; j < MB_SIZE; j++)
      for (i = 0; i < MB_SIZE; i++)
      {
        p_err->lum[j][i] = *(curr_image->lum + x + i + (y + j) * pels) - pred->lum[j][i];
        pred_macroblock->lum[j][i] = pred->lum[j][i];
      }

    xx = x >> 1;
    yy = y >> 1;
    for (j = 0; j < MB_SIZE >> 1; j++)
      for (i = 0; i < MB_SIZE >> 1; i++)
      {
        p_err->Cr[j][i] = *(curr_image->Cr + xx + i + (yy + j) * cpels) - pred->Cr[j][i];
        p_err->Cb[j][i] = *(curr_image->Cb + xx + i + (yy + j) * cpels) - pred->Cb[j][i];

        pred_macroblock->Cr[j][i] = pred->Cr[j][i];
        pred_macroblock->Cb[j][i] = pred->Cb[j][i];
      }
  }

  /* store PB-deltas */
  MV[6][y / MB_SIZE + 1][x / MB_SIZE + 1]->x = bdx; /* is in half pel format */
  MV[6][y / MB_SIZE + 1][x / MB_SIZE + 1]->y = bdy;
  MV[6][y / MB_SIZE + 1][x / MB_SIZE + 1]->x_half = 0;
  MV[6][y / MB_SIZE + 1][x / MB_SIZE + 1]->y_half = 0;

  free (pred);
  free (frw_pred);

  return p_err;
}

/***********************************************************************
 *
 *	Name:        MB_Recon_B
 *	Description:    Reconstructs the B macroblock in PB-frame
 *                      prediction
 *
 *	Input:	        pointers previous recon. frame, pred. diff.,
 *                      pos. in image, MV-data, reconstructed macroblock
 *                      from image ahead
 *	Returns:        pointer to reconstructed MB data
 *	Side effects:   allocates memory to MB_structure
 *
 *	Date: 950114        Author: Karl.Lillevold@nta.no
 *
 ***********************************************************************/

MB_Structure *MB_Recon_B (PictImage * prev_image, MB_Structure * diff,
                           unsigned char *prev_ipol, int x, int y,
                           MotionVector * MV[5][MBR + 1][MBC + 2],
                           MB_Structure * recon_P, int TRD, int TRB)
{
  int i, j, k;
  int dx, dy, bdx, bdy, mvx, mvy, xvec, yvec;
  MB_Structure *recon_B = (MB_Structure *) malloc (sizeof (MB_Structure));
  MB_Structure *pred = (MB_Structure *) malloc (sizeof (MB_Structure));
  MotionVector *f[5];

  for (k = 0; k <= 4; k++)
    f[k] = MV[k][y / MB_SIZE + 1][x / MB_SIZE + 1];

  bdx = MV[6][y / MB_SIZE + 1][x / MB_SIZE + 1]->x;
  bdy = MV[6][y / MB_SIZE + 1][x / MB_SIZE + 1]->y;

  if (f[0]->Mode == MODE_INTER4V || f[0]->Mode == MODE_INTER4V_Q)
  {                             /* Mode INTER4V */
    /* Find forward prediction */

    /* Luma */
    FindForwLumPredPB (prev_ipol, x, y, f[1], &pred->lum[0][0], TRD, TRB, bdx, bdy, 8, 0);
    FindForwLumPredPB (prev_ipol, x, y, f[2], &pred->lum[0][8], TRD, TRB, bdx, bdy, 8, 1);
    FindForwLumPredPB (prev_ipol, x, y, f[3], &pred->lum[8][0], TRD, TRB, bdx, bdy, 8, 2);
    FindForwLumPredPB (prev_ipol, x, y, f[4], &pred->lum[8][8], TRD, TRB, bdx, bdy, 8, 3);

    /* chroma vectors are sum of B luma vectors divided and rounded */
    xvec = yvec = 0;
    for (k = 1; k <= 4; k++)
    {
      xvec += TRB * (2 * f[k]->x + f[k]->x_half) / TRD + bdx;
      yvec += TRB * (2 * f[k]->y + f[k]->y_half) / TRD + bdy;
    }

    /* round values according to TABLE 16/H.263 */
    dx = sign (xvec) * (roundtab[abs (xvec) % 16] + (abs (xvec) / 16) * 2);
    dy = sign (yvec) * (roundtab[abs (yvec) % 16] + (abs (yvec) / 16) * 2);

    FindChromBlock_P (x, y, dx, dy, prev_image, pred, 0);

    /* Find bidirectional prediction */
    FindBiDirLumPredPB (&recon_P->lum[0][0], f[1], &pred->lum[0][0],
                        TRD, TRB, bdx, bdy, 0, 0);
    FindBiDirLumPredPB (&recon_P->lum[0][8], f[2], &pred->lum[0][8],
                        TRD, TRB, bdx, bdy, 1, 0);
    FindBiDirLumPredPB (&recon_P->lum[8][0], f[3], &pred->lum[8][0],
                        TRD, TRB, bdx, bdy, 0, 1);
    FindBiDirLumPredPB (&recon_P->lum[8][8], f[4], &pred->lum[8][8],
                        TRD, TRB, bdx, bdy, 1, 1);

    /* chroma vectors are sum of B luma vectors divided and rounded */
    xvec = yvec = 0;
    for (k = 1; k <= 4; k++)
    {
      mvx = 2 * f[k]->x + f[k]->x_half;
      mvy = 2 * f[k]->y + f[k]->y_half;
      xvec += bdx == 0 ? (TRB - TRD) * mvx / TRD : TRB * mvx / TRD + bdx - mvx;
      yvec += bdy == 0 ? (TRB - TRD) * mvy / TRD : TRB * mvy / TRD + bdy - mvy;
    }

    /* round values according to TABLE 16/H.263 */
    dx = sign (xvec) * (roundtab[abs (xvec) % 16] + (abs (xvec) / 16) * 2);
    dy = sign (yvec) * (roundtab[abs (yvec) % 16] + (abs (yvec) / 16) * 2);

    FindBiDirChrPredPB (recon_P, dx, dy, pred);

  } else
  {                             /* Mode INTER or INTER_Q */
    /* Find forward prediction */

    FindForwLumPredPB (prev_ipol, x, y, f[0], &pred->lum[0][0], TRD, TRB,
                       bdx, bdy, 16, 0);

    xvec = 4 * (TRB * (2 * f[0]->x + f[0]->x_half) / TRD + bdx);
    yvec = 4 * (TRB * (2 * f[0]->y + f[0]->y_half) / TRD + bdy);
    /* round values according to TABLE 16/H.263 */
    dx = sign (xvec) * (roundtab[abs (xvec) % 16] + (abs (xvec) / 16) * 2);
    dy = sign (yvec) * (roundtab[abs (yvec) % 16] + (abs (yvec) / 16) * 2);

    FindChromBlock_P (x, y, dx, dy, prev_image, pred, 0);

    /* Find bidirectional prediction */
    FindBiDirLumPredPB (&recon_P->lum[0][0], f[0], &pred->lum[0][0],
                        TRD, TRB, bdx, bdy, 0, 0);
    FindBiDirLumPredPB (&recon_P->lum[0][8], f[0], &pred->lum[0][8],
                        TRD, TRB, bdx, bdy, 1, 0);
    FindBiDirLumPredPB (&recon_P->lum[8][0], f[0], &pred->lum[8][0],
                        TRD, TRB, bdx, bdy, 0, 1);
    FindBiDirLumPredPB (&recon_P->lum[8][8], f[0], &pred->lum[8][8],
                        TRD, TRB, bdx, bdy, 1, 1);

    /* chroma vectors */
    mvx = 2 * f[0]->x + f[0]->x_half;
    xvec = bdx == 0 ? (TRB - TRD) * mvx / TRD : TRB * mvx / TRD + bdx - mvx;
    xvec *= 4;

    mvy = 2 * f[0]->y + f[0]->y_half;
    yvec = bdy == 0 ? (TRB - TRD) * mvy / TRD : TRB * mvy / TRD + bdy - mvy;
    yvec *= 4;

    /* round values according to TABLE 16/H.263 */
    dx = sign (xvec) * (roundtab[abs (xvec) % 16] + (abs (xvec) / 16) * 2);
    dy = sign (yvec) * (roundtab[abs (yvec) % 16] + (abs (yvec) / 16) * 2);

    FindBiDirChrPredPB (recon_P, dx, dy, pred);

  }

  /* Reconstruction */
  for (j = 0; j < MB_SIZE; j++)
    for (i = 0; i < MB_SIZE; i++)
      recon_B->lum[j][i] = pred->lum[j][i] + diff->lum[j][i];

  for (j = 0; j < MB_SIZE >> 1; j++)
    for (i = 0; i < MB_SIZE >> 1; i++)
    {
      recon_B->Cr[j][i] = pred->Cr[j][i] + diff->Cr[j][i];
      recon_B->Cb[j][i] = pred->Cb[j][i] + diff->Cb[j][i];
    }

  free (pred);
  return recon_B;
}

/**********************************************************************
 *
 *	Name:	       FindForwLumPredPB
 *	Description:   Finds the forward luma  prediction in PB-frame
 *                     pred.
 *
 *	Input:	       pointer to prev. recon. frame, current positon,
 *                     MV structure and pred. structure to fill
 *
 *	Date: 950114        Author: Karl.Lillevold@nta.no
 *
 ***********************************************************************/

void FindForwLumPredPB (unsigned char *prev_ipol, int x_curr, int y_curr,
                         MotionVector * fr, int *pred, int TRD, int TRB,
                         int bdx, int bdy, int bs, int comp)
{
  int i, j;
  int xvec, yvec, lx;

  lx = (mv_outside_frame ? pels + (long_vectors ? 64 : 32) : pels);

  /* Luma */
  xvec = (TRB) * (2 * fr->x + fr->x_half) / TRD + bdx;
  yvec = (TRB) * (2 * fr->y + fr->y_half) / TRD + bdy;

  x_curr += ((comp & 1) << 3);
  y_curr += ((comp & 2) << 2);

  for (j = 0; j < bs; j++)
  {
    for (i = 0; i < bs; i++)
    {
      *(pred + i + j * 16) = *(prev_ipol + (i + x_curr) * 2 + xvec +
                               ((j + y_curr) * 2 + yvec) * lx * 2);
    }
  }

  return;
}

/**********************************************************************
 *
 *	Name:	       FindBiDirLumPredPB
 *	Description:   Finds the bi-dir. luma prediction in PB-frame
 *                     prediction
 *
 *	Input:	       pointer to future recon. data, current positon,
 *                     MV structure and pred. structure to fill
 *
 *	Date: 950115        Author: Karl.Lillevold@nta.no
 *
 ***********************************************************************/

void FindBiDirLumPredPB (int *recon_P, MotionVector * fr, int *pred, int TRD,
                          int TRB, int bdx, int bdy, int nh, int nv)
{
  int xstart, xstop, ystart, ystop;
  int xvec, yvec, mvx, mvy;

  mvx = 2 * fr->x + fr->x_half;
  mvy = 2 * fr->y + fr->y_half;

  xvec = (bdx == 0 ? (TRB - TRD) * mvx / TRD : TRB * mvx / TRD + bdx - mvx);
  yvec = (bdy == 0 ? (TRB - TRD) * mvy / TRD : TRB * mvy / TRD + bdy - mvy);

  /* Luma */

  FindBiDirLimits (xvec, &xstart, &xstop, nh);
  FindBiDirLimits (yvec, &ystart, &ystop, nv);

  BiDirPredBlock (xstart, xstop, ystart, ystop, xvec, yvec, recon_P, pred, 16);

  return;
}

/**********************************************************************
 *
 *	Name:	       FindBiDirChrPredPB
 *	Description:   Finds the bi-dir. chroma prediction in PB-frame
 *                     prediction
 *
 *	Input:	       pointer to future recon. data, current positon,
 *                     MV structure and pred. structure to fill
 *
 *	Date: 950115        Author: Karl.Lillevold@nta.no
 *
 ***********************************************************************/

void FindBiDirChrPredPB (MB_Structure * recon_P, int dx, int dy,
                          MB_Structure * pred)
{
  int xstart, xstop, ystart, ystop;

  FindBiDirChromaLimits (dx, &xstart, &xstop);
  FindBiDirChromaLimits (dy, &ystart, &ystop);

  BiDirPredBlock (xstart, xstop, ystart, ystop, dx, dy,
                  &recon_P->Cb[0][0], &pred->Cb[0][0], 8);
  BiDirPredBlock (xstart, xstop, ystart, ystop, dx, dy,
                  &recon_P->Cr[0][0], &pred->Cr[0][0], 8);

  return;
}

void FindBiDirLimits (int vec, int *start, int *stop, int nhv)
{

  /* limits taken from C loop in section G5 in H.263 */
  *start = mmax (0, (-vec + 1) / 2 - nhv * 8);
  *stop = mmin (7, 15 - (vec + 1) / 2 - nhv * 8);

  return;

}

void FindBiDirChromaLimits (int vec, int *start, int *stop)
{

  /* limits taken from C loop in section G5 in H.263 */
  *start = mmax (0, (-vec + 1) / 2);
  *stop = mmin (7, 7 - (vec + 1) / 2);

  return;
}


void BiDirPredBlock (int xstart, int xstop, int ystart, int ystop,
                      int xvec, int yvec, int *recon, int *pred, int bl)
{
  int i, j, pel;
  int xint, yint;
  int xh, yh;

  xint = xvec >> 1;
  xh = xvec - 2 * xint;
  yint = yvec >> 1;
  yh = yvec - 2 * yint;

  if (!xh && !yh)
  {
    for (j = ystart; j <= ystop; j++)
    {
      for (i = xstart; i <= xstop; i++)
      {
        pel = *(recon + (j + yint) * bl + i + xint);
        *(pred + j * bl + i) = (mmin (255, mmax (0, pel)) + *(pred + j * bl + i)) >> 1;
      }
    }
  } else if (!xh && yh)
  {
    for (j = ystart; j <= ystop; j++)
    {
      for (i = xstart; i <= xstop; i++)
      {
        pel = (*(recon + (j + yint) * bl + i + xint) +
               *(recon + (j + yint + yh) * bl + i + xint) + 1) >> 1;
        *(pred + j * bl + i) = (pel + *(pred + j * bl + i)) >> 1;
      }
    }
  } else if (xh && !yh)
  {
    for (j = ystart; j <= ystop; j++)
    {
      for (i = xstart; i <= xstop; i++)
      {
        pel = (*(recon + (j + yint) * bl + i + xint) +
               *(recon + (j + yint) * bl + i + xint + xh) + 1) >> 1;
        *(pred + j * bl + i) = (pel + *(pred + j * bl + i)) >> 1;
      }
    }
  } else
  {                             /* xh && yh */
    for (j = ystart; j <= ystop; j++)
    {
      for (i = xstart; i <= xstop; i++)
      {
        pel = (*(recon + (j + yint) * bl + i + xint) +
               *(recon + (j + yint + yh) * bl + i + xint) +
               *(recon + (j + yint) * bl + i + xint + xh) +
               *(recon + (j + yint + yh) * bl + i + xint + xh) + 2) >> 2;
        *(pred + j * bl + i) = (pel + *(pred + j * bl + i)) >> 1;
      }
    }
  }
  return;
}

/**********************************************************************
 *
 *	Name:        DoPredChrom_P
 *	Description:	Does the chrominance prediction for P-frames
 *
 *	Input:        motionvectors for each field,
 *        current position in image,
 *        pointers to current and previos image,
 *        pointer to pred_error array,
 *        (int) field: 1 if field coding
 *
 *	Side effects:	fills chrom-array in pred_error structure
 *
 *	Date: 930211	Author: Karl.Lillevold@nta.no
 *
 ***********************************************************************/

void DoPredChrom_P (int x_curr, int y_curr, int dx, int dy,
                    PictImage * curr, PictImage * prev, MB_Structure * prediction,
                    MB_Structure * pred_error, int rtype)
{
  int m, n;

  int x, y, ofx, ofy, pel, lx;
  int xint, yint;
  int xh, yh;

  lx = (mv_outside_frame ? pels / 2 + (long_vectors ? 32 : 16) : pels / 2);

  x = x_curr >> 1;
  y = y_curr >> 1;

  xint = dx >> 1;
  xh = dx & 1;
  yint = dy >> 1;
  yh = dy & 1;

  if (!xh && !yh)
  {
    for (n = 0; n < 8; n++)
    {
      for (m = 0; m < 8; m++)
      {

        ofx = x + xint + m;
        ofy = y + yint + n;
        pel = *(prev->Cr + ofx + (ofy) * lx);
        prediction->Cr[n][m] = pel;
        pred_error->Cr[n][m] = (int) (*(curr->Cr + x + m + (y + n) * cpels) - pel);

        pel = *(prev->Cb + ofx + (ofy) * lx);
        prediction->Cb[n][m] = pel;
        pred_error->Cb[n][m] = (int) (*(curr->Cb + x + m + (y + n) * cpels) - pel);
      }
    }
  } else if (!xh && yh)
  {
    for (n = 0; n < 8; n++)
    {
      for (m = 0; m < 8; m++)
      {

        ofx = x + xint + m;
        ofy = y + yint + n;
        pel = (*(prev->Cr + ofx + (ofy) * lx) +
               *(prev->Cr + ofx + (ofy + yh) * lx) + 1 - rtype) >> 1;
        prediction->Cr[n][m] = pel;
        pred_error->Cr[n][m] =
          (int) (*(curr->Cr + x + m + (y + n) * cpels) - pel);

        pel = (*(prev->Cb + ofx + (ofy) * lx) +
               *(prev->Cb + ofx + (ofy + yh) * lx) + 1 - rtype) >> 1;
        prediction->Cb[n][m] = pel;
        pred_error->Cb[n][m] =
          (int) (*(curr->Cb + x + m + (y + n) * cpels) - pel);

      }
    }
  } else if (xh && !yh)
  {
    for (n = 0; n < 8; n++)
    {
      for (m = 0; m < 8; m++)
      {

        ofx = x + xint + m;
        ofy = y + yint + n;
        pel = (*(prev->Cr + ofx + (ofy) * lx) +
               *(prev->Cr + ofx + xh + (ofy) * lx) + 1 - rtype) >> 1;
        prediction->Cr[n][m] = pel;
        pred_error->Cr[n][m] =
          (int) (*(curr->Cr + x + m + (y + n) * cpels) - pel);

        pel = (*(prev->Cb + ofx + (ofy) * lx) +
               *(prev->Cb + ofx + xh + (ofy) * lx) + 1 - rtype) >> 1;
        prediction->Cb[n][m] = pel;
        pred_error->Cb[n][m] =
          (int) (*(curr->Cb + x + m + (y + n) * cpels) - pel);

      }
    }
  } else
  {                             /* xh && yh */
    for (n = 0; n < 8; n++)
    {
      for (m = 0; m < 8; m++)
      {
        ofx = x + xint + m;
        ofy = y + yint + n;
        pel = (*(prev->Cr + ofx + (ofy) * lx) +
               *(prev->Cr + ofx + xh + (ofy) * lx) +
               *(prev->Cr + ofx + (ofy + yh) * lx) +
               *(prev->Cr + ofx + xh + (ofy + yh) * lx) +
               2 - rtype) >> 2;
        prediction->Cr[n][m] = pel;
        pred_error->Cr[n][m] =
          (int) (*(curr->Cr + x + m + (y + n) * cpels) - pel);

        pel = (*(prev->Cb + ofx + (ofy) * lx) +
               *(prev->Cb + ofx + xh + (ofy) * lx) +
               *(prev->Cb + ofx + (ofy + yh) * lx) +
               *(prev->Cb + ofx + xh + (ofy + yh) * lx) +
               2 - rtype) >> 2;
        prediction->Cb[n][m] = pel;
        pred_error->Cb[n][m] =
          (int) (*(curr->Cb + x + m + (y + n) * cpels) - pel);

      }
    }
  }
  return;
}


/**********************************************************************
 *
 *	Name:        ReconMacroblock_P
 *	Description:	Reconstructs MB after quantization for P_images
 *
 *	Input:        pointers to current and previous image,
 *        current slice and mb, and which mode
 *        of prediction has been used
 *	Returns:
 *	Side effects:
 *
 *	Date: 930122	Author: Karl.Lillevold@nta.no
 *
 ***********************************************************************/

MB_Structure *MB_Recon_P (PictImage * prev_image, unsigned char *prev_ipol,
                          MB_Structure * diff, int x_curr, int y_curr,
                          MotionVector * MV[6][MBR + 1][MBC + 2], int PB, int RTYPE)
{
  MB_Structure *recon_data = (MB_Structure *) malloc (sizeof (MB_Structure));
  MotionVector *fr0, *fr1, *fr2, *fr3, *fr4;
  int pred[16][16];
  int dx, dy, sum;
  int i, j;

  fr0 = MV[0][y_curr / MB_SIZE + 1][x_curr / MB_SIZE + 1];
  /* change advanced to overlapping_MC since * 4 MV can be used without
   * OBMC in        * deblocking filter mode                  */
  if (overlapping_MC)
  {
    if (fr0->Mode == MODE_INTER || fr0->Mode == MODE_INTER_Q)
    {
      FindPredOBMC (x_curr, y_curr, MV, prev_ipol, &pred[0][0], 0, PB);
      FindPredOBMC (x_curr, y_curr, MV, prev_ipol, &pred[0][8], 1, PB);
      FindPredOBMC (x_curr, y_curr, MV, prev_ipol, &pred[8][0], 2, PB);
      FindPredOBMC (x_curr, y_curr, MV, prev_ipol, &pred[8][8], 3, PB);
      for (j = 0; j < MB_SIZE; j++)
        for (i = 0; i < MB_SIZE; i++)
          diff->lum[j][i] += pred[j][i];
      dx = 2 * fr0->x + fr0->x_half;
      dy = 2 * fr0->y + fr0->y_half;
      dx = (dx % 4 == 0 ? dx >> 1 : (dx >> 1) | 1);
      dy = (dy % 4 == 0 ? dy >> 1 : (dy >> 1) | 1);
      ReconChromBlock_P (x_curr, y_curr, dx, dy, prev_image, diff, RTYPE);
    } else if (fr0->Mode == MODE_INTER4V || fr0->Mode == MODE_INTER4V_Q)
    {                           /* Inter 8x8 */

      FindPredOBMC (x_curr, y_curr, MV, prev_ipol, &pred[0][0], 0, PB);
      FindPredOBMC (x_curr, y_curr, MV, prev_ipol, &pred[0][8], 1, PB);
      FindPredOBMC (x_curr, y_curr, MV, prev_ipol, &pred[8][0], 2, PB);
      FindPredOBMC (x_curr, y_curr, MV, prev_ipol, &pred[8][8], 3, PB);
      for (j = 0; j < MB_SIZE; j++)
        for (i = 0; i < MB_SIZE; i++)
          diff->lum[j][i] += pred[j][i];

      fr1 = MV[1][y_curr / MB_SIZE + 1][x_curr / MB_SIZE + 1];
      fr2 = MV[2][y_curr / MB_SIZE + 1][x_curr / MB_SIZE + 1];
      fr3 = MV[3][y_curr / MB_SIZE + 1][x_curr / MB_SIZE + 1];
      fr4 = MV[4][y_curr / MB_SIZE + 1][x_curr / MB_SIZE + 1];

      sum = 2 * fr1->x + fr1->x_half + 2 * fr2->x + fr2->x_half +
        2 * fr3->x + fr3->x_half + 2 * fr4->x + fr4->x_half;
      dx = sign (sum) * (roundtab[abs (sum) % 16] + (abs (sum) / 16) * 2);

      sum = 2 * fr1->y + fr1->y_half + 2 * fr2->y + fr2->y_half +
        2 * fr3->y + fr3->y_half + 2 * fr4->y + fr4->y_half;
      dy = sign (sum) * (roundtab[abs (sum) % 16] + (abs (sum) / 16) * 2);

      ReconChromBlock_P (x_curr, y_curr, dx, dy, prev_image, diff, RTYPE);
    }
  } else
  {
    if (fr0->Mode == MODE_INTER || fr0->Mode == MODE_INTER_Q)
    {
      /* Inter 16x16 */
      ReconLumBlock_P (x_curr, y_curr, fr0, prev_ipol, &diff->lum[0][0], 16, 0);

      dx = 2 * fr0->x + fr0->x_half;
      dy = 2 * fr0->y + fr0->y_half;
      dx = (dx % 4 == 0 ? dx >> 1 : (dx >> 1) | 1);
      dy = (dy % 4 == 0 ? dy >> 1 : (dy >> 1) | 1);
      ReconChromBlock_P (x_curr, y_curr, dx, dy, prev_image, diff, RTYPE);
    }
    if (fr0->Mode == MODE_INTER4V || fr0->Mode == MODE_INTER4V_Q)
    {
      fr1 = MV[1][y_curr / MB_SIZE + 1][x_curr / MB_SIZE + 1];
      fr2 = MV[2][y_curr / MB_SIZE + 1][x_curr / MB_SIZE + 1];
      fr3 = MV[3][y_curr / MB_SIZE + 1][x_curr / MB_SIZE + 1];
      fr4 = MV[4][y_curr / MB_SIZE + 1][x_curr / MB_SIZE + 1];
      FindPred (x_curr, y_curr, fr1, prev_ipol, &pred[0][0], 8, 0);
      FindPred (x_curr, y_curr, fr2, prev_ipol, &pred[0][8], 8, 1);
      FindPred (x_curr, y_curr, fr3, prev_ipol, &pred[8][0], 8, 2);
      FindPred (x_curr, y_curr, fr4, prev_ipol, &pred[8][8], 8, 3);
      for (j = 0; j < MB_SIZE; j++)
        for (i = 0; i < MB_SIZE; i++)
          diff->lum[j][i] += pred[j][i];

      sum = 2 * fr1->x + fr1->x_half + 2 * fr2->x + fr2->x_half +
        2 * fr3->x + fr3->x_half + 2 * fr4->x + fr4->x_half;
      dx = sign (sum) * (roundtab[abs (sum) % 16] + (abs (sum) / 16) * 2);

      sum = 2 * fr1->y + fr1->y_half + 2 * fr2->y + fr2->y_half +
        2 * fr3->y + fr3->y_half + 2 * fr4->y + fr4->y_half;
      dy = sign (sum) * (roundtab[abs (sum) % 16] + (abs (sum) / 16) * 2);

      ReconChromBlock_P (x_curr, y_curr, dx, dy, prev_image, diff, RTYPE);
    }
  }

  memcpy (recon_data, diff, sizeof (MB_Structure));

  return recon_data;
}

/**********************************************************************
 *
 *	Name:        ReconLumBlock_P
 *	Description:	Reconstructs one block of luminance data
 *
 *	Input:        position, vector-data, previous image, data-block
 *	Returns:
 *	Side effects:	reconstructs data-block
 *
 *	Date: 950210	Author: Karl.Lillevold@nta.no
 *
 ***********************************************************************/

void ReconLumBlock_P (int x, int y, MotionVector * fr,
                       unsigned char *prev, int *data, int bs, int comp)
{
  int m, n;
  int x1, y1, lx;

  lx = (mv_outside_frame ? pels + (long_vectors ? 64 : 32) : pels);

  x1 = 2 * (x + fr->x) + fr->x_half;
  y1 = 2 * (y + fr->y) + fr->y_half;

  x1 += ((comp & 1) << 4);
  y1 += ((comp & 2) << 3);

  for (n = 0; n < bs; n++)
  {
    for (m = 0; m < bs; m++)
    {
      *(data + m + n * 16) += (int) (*(prev + x1 + 2 * m + (y1 + 2 * n) * 2 * lx));
    }
  }

  return;
}

/**********************************************************************
 *
 *	Name:        ReconChromBlock_P
 *	Description:        Reconstructs chrominance of one block in P frame
 *
 *	Input:        position, vector-data, previous image, data-block
 *	Returns:
 *	Side effects:	reconstructs data-block
 *
 *	Date: 930203	Author: Karl.Lillevold@nta.no
 *
 ***********************************************************************/

void ReconChromBlock_P (int x_curr, int y_curr, int dx, int dy,
                         PictImage * prev, MB_Structure * data, int rtype)

{
  int m, n;

  int x, y, ofx, ofy, pel, lx;
  int xint, yint;
  int xh, yh;

  lx = (mv_outside_frame ? pels / 2 + (long_vectors ? 32 : 16) : pels / 2);

  x = x_curr >> 1;
  y = y_curr >> 1;

  xint = dx >> 1;
  xh = dx & 1;
  yint = dy >> 1;
  yh = dy & 1;

  if (!xh && !yh)
  {
    for (n = 0; n < 8; n++)
    {
      for (m = 0; m < 8; m++)
      {

        ofx = x + xint + m;
        ofy = y + yint + n;
        pel = *(prev->Cr + ofx + (ofy) * lx);
        data->Cr[n][m] += pel;

        pel = *(prev->Cb + ofx + (ofy) * lx);
        data->Cb[n][m] += pel;
      }
    }
  } else if (!xh && yh)
  {
    for (n = 0; n < 8; n++)
    {
      for (m = 0; m < 8; m++)
      {

        ofx = x + xint + m;
        ofy = y + yint + n;
        pel = (*(prev->Cr + ofx + (ofy) * lx) +
               *(prev->Cr + ofx + (ofy + yh) * lx) + 1 - rtype) >> 1;

        data->Cr[n][m] += pel;

        pel = (*(prev->Cb + ofx + (ofy) * lx) +
               *(prev->Cb + ofx + (ofy + yh) * lx) + 1 - rtype) >> 1;

        data->Cb[n][m] += pel;

      }
    }
  } else if (xh && !yh)
  {
    for (n = 0; n < 8; n++)
    {
      for (m = 0; m < 8; m++)
      {

        ofx = x + xint + m;
        ofy = y + yint + n;
        pel = (*(prev->Cr + ofx + (ofy) * lx) +
               *(prev->Cr + ofx + xh + (ofy) * lx) + 1 - rtype) >> 1;

        data->Cr[n][m] += pel;

        pel = (*(prev->Cb + ofx + (ofy) * lx) +
               *(prev->Cb + ofx + xh + (ofy) * lx) + 1 - rtype) >> 1;

        data->Cb[n][m] += pel;

      }
    }
  } else
  {                             /* xh && yh */
    for (n = 0; n < 8; n++)
    {
      for (m = 0; m < 8; m++)
      {
        ofx = x + xint + m;
        ofy = y + yint + n;
        pel = (*(prev->Cr + ofx + (ofy) * lx) +
               *(prev->Cr + ofx + xh + (ofy) * lx) +
               *(prev->Cr + ofx + (ofy + yh) * lx) +
               *(prev->Cr + ofx + xh + (ofy + yh) * lx) +
               2 - rtype) >> 2;

        data->Cr[n][m] += pel;

        pel = (*(prev->Cb + ofx + (ofy) * lx) +
               *(prev->Cb + ofx + xh + (ofy) * lx) +
               *(prev->Cb + ofx + (ofy + yh) * lx) +
               *(prev->Cb + ofx + xh + (ofy + yh) * lx) +
               2 - rtype) >> 2;

        data->Cb[n][m] += pel;

      }
    }
  }
  return;
}

/**********************************************************************
 *
 *	Name:        FindChromBlock_P
 *	Description:        Finds chrominance of one block in P frame
 *
 *	Input:        position, vector-data, previous image, data-block
 *
 *	Date: 950222	Author: Karl.Lillevold@nta.no
 *
 ***********************************************************************/

void FindChromBlock_P ( int x_curr, int y_curr, int dx, int dy,
                        PictImage * prev, MB_Structure * data, int rtype)

{
  int m, n;

  int x, y, ofx, ofy, pel, lx;
  int xint, yint;
  int xh, yh;

  lx = (mv_outside_frame ? pels / 2 + (long_vectors ? 32 : 16) : pels / 2);

  x = x_curr >> 1;
  y = y_curr >> 1;

  xint = dx >> 1;
  xh = dx & 1;
  yint = dy >> 1;
  yh = dy & 1;

  if (!xh && !yh)
  {
    for (n = 0; n < 8; n++)
    {
      for (m = 0; m < 8; m++)
      {

        ofx = x + xint + m;
        ofy = y + yint + n;
        pel = *(prev->Cr + ofx + (ofy) * lx);
        data->Cr[n][m] = pel;

        pel = *(prev->Cb + ofx + (ofy) * lx);
        data->Cb[n][m] = pel;
      }
    }
  } else if (!xh && yh)
  {
    for (n = 0; n < 8; n++)
    {
      for (m = 0; m < 8; m++)
      {

        ofx = x + xint + m;
        ofy = y + yint + n;
        pel = (*(prev->Cr + ofx + (ofy) * lx) +
               *(prev->Cr + ofx + (ofy + yh) * lx) + 1 - rtype) >> 1;

        data->Cr[n][m] = pel;

        pel = (*(prev->Cb + ofx + (ofy) * lx) +
               *(prev->Cb + ofx + (ofy + yh) * lx) + 1 - rtype) >> 1;

        data->Cb[n][m] = pel;

      }
    }
  } else if (xh && !yh)
  {
    for (n = 0; n < 8; n++)
    {
      for (m = 0; m < 8; m++)
      {

        ofx = x + xint + m;
        ofy = y + yint + n;
        pel = (*(prev->Cr + ofx + (ofy) * lx) +
               *(prev->Cr + ofx + xh + (ofy) * lx) + 1 - rtype) >> 1;

        data->Cr[n][m] = pel;

        pel = (*(prev->Cb + ofx + (ofy) * lx) +
               *(prev->Cb + ofx + xh + (ofy) * lx) + 1 - rtype) >> 1;

        data->Cb[n][m] = pel;

      }
    }
  } else
  {                             /* xh && yh */
    for (n = 0; n < 8; n++)
    {
      for (m = 0; m < 8; m++)
      {
        ofx = x + xint + m;
        ofy = y + yint + n;
        pel = (*(prev->Cr + ofx + (ofy) * lx) +
               *(prev->Cr + ofx + xh + (ofy) * lx) +
               *(prev->Cr + ofx + (ofy + yh) * lx) +
               *(prev->Cr + ofx + xh + (ofy + yh) * lx) +
               2 - rtype) >> 2;

        data->Cr[n][m] = pel;

        pel = (*(prev->Cb + ofx + (ofy) * lx) +
               *(prev->Cb + ofx + xh + (ofy) * lx) +
               *(prev->Cb + ofx + (ofy + yh) * lx) +
               *(prev->Cb + ofx + xh + (ofy + yh) * lx) +
               2 - rtype) >> 2;

        data->Cb[n][m] = pel;

      }
    }
  }
  return;
}


/**********************************************************************
 *
 *	Name:           StoreDirectModeVectors
 *	Description:    Stores info for P-P MVs, to be used with direct 
 *			mode prediction in true B pictures.
 *	
 *	Input:		MV data
 *	Side effects:   allocates memory for 1-D MV structs.
 *
 *      Return:         
 *
 *	Date:970926	Michael Gallant <mikeg@ee.ubc.ca>
 *
 ***********************************************************************/

void StoreDirectModeVectors(MotionVector *ppMV[7][MBR+1][MBC+2], MotionVector *directMV[5][MBR][MBC])
{
  int i, j, k;

  for (j=0; j<(lines>>4)+1; ++j)
  {
    for (i=0; i<(pels>>4)+2; ++i)
    {
      for (k=0; k<5; k++)
      {
        directMV[k][j][i] = (MotionVector *)malloc(sizeof(MotionVector) );
      }
    }
  }

  for (j=0; j<lines/MB_SIZE; j++) 
  {
    for (i=0; i< pels/MB_SIZE; i++) 
    {
      /* Must store all forward MVs found for the P-picture, to use with true B 
       * Direct Mode prediction. */
 
      if ( MODE_INTER4V   == ppMV[0][j+1][i+1]->Mode ||   
	         MODE_INTER4V_Q == ppMV[0][j+1][i+1]->Mode)
      {
        directMV[0][j][i]->Mode = ppMV[0][j+1][i+1]->Mode;
        for (k=1; k<5; k++)
        {
          directMV[k][j][i]->x = ppMV[k][j+1][i+1]->x;
          directMV[k][j][i]->x_half = ppMV[k][j+1][i+1]->x_half;
          directMV[k][j][i]->y = ppMV[k][j+1][i+1]->y;
          directMV[k][j][i]->y_half = ppMV[k][j+1][i+1]->y_half;
          directMV[k][j][i]->min_error = ppMV[k][j+1][i+1]->min_error;
          directMV[k][j][i]->Mode = ppMV[k][j+1][i+1]->Mode;
        }
      }
      else
      {
        directMV[0][j][i]->x =         ppMV[0][j+1][i+1]->x;
        directMV[0][j][i]->x_half =    ppMV[0][j+1][i+1]->x_half;
        directMV[0][j][i]->y =         ppMV[0][j+1][i+1]->y;
        directMV[0][j][i]->y_half =    ppMV[0][j+1][i+1]->y_half;
        directMV[0][j][i]->min_error = ppMV[0][j+1][i+1]->min_error;
        directMV[0][j][i]->Mode =      ppMV[0][j+1][i+1]->Mode;
      }           
    }
  }
  return;
}
