/*****************************************************************************
 *                                                                          
 * This software module was originally developed by 
 *
 * Martina Eckert (UPM-GTI / ACTS-MoMuSys)
 *
 * in the course of development of the MPEG-4 Video (ISO/IEC 14496-2) standard.
 * This software module is an implementation of a part of one or more MPEG-4 
 * Video (ISO/IEC 14496-2) tools as specified by the MPEG-4 Video (ISO/IEC 
 * 14496-2) standard. 
 *
 * ISO/IEC gives users of the MPEG-4 Video (ISO/IEC 14496-2) standard free 
 * license to this software module or modifications thereof for use in hardware
 * or software products claiming conformance to the MPEG-4 Video (ISO/IEC 
 * 14496-2) standard. 
 *
 * Those intending to use this software module in hardware or software products
 * are advised that its use may infringe existing patents. The original 
 * developer of this software module and his/her company, the subsequent 
 * editors and their companies, and ISO/IEC have no liability for use of this 
 * software module or modifications thereof in an implementation. Copyright is 
 * not released for non MPEG-4 Video (ISO/IEC 14496-2) Standard conforming 
 * products. 
 *
 * ACTS-MoMuSys partners retain full right to use the code for his/her own 
 * purpose, assign or donate the code to a third party and to inhibit third 
 * parties from using the code for non MPEG-4 Video (ISO/IEC 14496-2) Standard
 * conforming products. This copyright notice must be included in all copies or
 * derivative works. 
 *
 * Copyright (c) 1997
 *
 *****************************************************************************/

/***********************************************************HeaderBegin*******
 *                                                                         
 * File:        rc_upm3.c
 *
 * Author:      Martina Eckert, UPM-GTI
 *
 * Created:     28-04-98
 *                                                                         
 * Description: UPM3 control functions
 *
 * Notes:       
 *
 * Flags:       -D_RC_DEBUG_  -  RC debugging   
 *
 * Modified:
 *                                 
 ***********************************************************HeaderEnd*********/

/************************    INCLUDE FILES    ********************************/

#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "rc.h"
#include "momusys.h"
#include "mom_vop.h"
#include "vm_config.h"
#include "mom_vol.h"
#include "mom_structs.h"
#include "vm_enc_defs.h"


#define MAX_DIFF 1E-3

static RC_HIST      rc_hist_upm[MAX_NUM_VOS][MAX_NUM_VOLS][3];  /* History of each VOL */
static RC_CFG_PARAM rc_cfg_upm[MAX_NUM_VOS][MAX_NUM_VOLS];   /* RC parameter */

static Int first_vo, first_vol;

/***********************************************************CommentBegin******
 *
 * -- min2_rld --
 *
 * Author : 
 *      Martina Eckert, UPM-GTI     
 *
 * Created :            
 *      28-04-98
 *
 * Purpose :
 *      Minimization Function for lambda
 * 
 * Arguments in :       
 *     Int    *vo_list   -  To access all VOLs
 *     Double lambda_r   -  Search variable
 *     Double lambda_d   -  Search variable
 *     Int   (* rate)(RC_HIST *rch, Int vo_i, Int vol_i, Int q, Double mad, 
 *                          Int frame_type)  
 *                       -  Rate functions 
 *     Double (* dist)(RC_HIST *rch, Int vo_i, Int vol_i, Int q, Int mode, 
 *                          Int frame_type) 
 *                       -  Distortion functions 
 *     Double mad        -  MAD 
 *     Int    mode       -  0 = calculation of distortion per pixel,
 *                          1 = calculation of total distortion 
 *     Int    frame_type -  I=0, P=1, B=2
 *     Float   c_1       -  Weight for weighted sum of d_i
 *     Float   c_2       -  Weight for weighted sum of differences 
 *                          (d_i - beta*d_0)
 *
 * Arguments in/out :   
 *     Int    qp[][]   -  Decisions 
 *     Double *R       -  Sum of estimated rates 
 *     Double *D       -  Sum of weighted distortions 
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      -
 *
 * Description :        
 *
 *
 * See also :
 *
 *
 ***********************************************************CommentEnd********/

static void min2_rld(
                     VOConfig *vo_list,
                     Double lambda_r,
                     Double lambda_d,
                     Int   (* rate)(RC_HIST *rch,
                                    Int vo_i, Int vol_i, Int q, Double mad,
                                    Int frame_type), 
                     Double (* dist)(RC_HIST *rch,
                                     Int vo_i, Int vol_i, 
                                     Int q, Int mode,
                                     Int frame_type), 
                     Double mad, 
                     Int    qp[][MAX_NUM_VOLS],     
                     Int    qp0,       
                     Double *R,       
                     Double *D,
                     Double *D_diff,
                     Int    mode,
                     Int    frame_type,
                     Float  c_1,
                     Float  c_2
                     )
{
  RC_HIST *rch;
  RC_CFG_PARAM *rcc;
  Int    vo_i, vol_i, q, min_qp, max_qp;
  Double r_iq, d_iq, dp_iq, d_diff, c, dist0;
  Double r_min[MAX_NUM_VOS][MAX_NUM_VOLS], d_min[MAX_NUM_VOS][MAX_NUM_VOLS],
    d_diff_min[MAX_NUM_VOS][MAX_NUM_VOLS];
  VOConfig   *curr_vo;
  VolConfig  *curr_vol;

  Float   next_coding_time, end_time;

  *D=0;
  *R=0; 
  *D_diff=0;

  curr_vo = vo_list;
  while(curr_vo != NULL)
    {
      vo_i = GetVOConfigId(curr_vo);
      curr_vol =  GetVOConfigLayers(curr_vo);
      while(curr_vol != NULL)
        {
	  /* meb 03.08.99: */
	  next_coding_time = GetVolConfigNextCodingTime(curr_vol);
	  end_time = GetVolConfigEndTime(curr_vol);
	  if(next_coding_time <= end_time){ /* VO in process */
	  
	    Double c_min=1E99; 
	    vol_i = GetVolConfigId(curr_vol);
          
	    /* Search qp in full range [1..31]: */
	    /* for (q = MIN_QUANT; q <= MAX_QUANT; q++)*/
          
	    /* qp range control within the loop, search in range of last_qp+-5: */
	    /* If the last n qp's have been the same, exclude this value
	       as possibility! */
          
	    rch = &rc_hist_upm[vo_i][vol_i][frame_type];
	    rcc = &rc_cfg_upm[vo_i][vol_i];
          
	    if (first_frame(rch)) 
	      {
		min_qp = rc_upm_get_QPfirst(&rc_cfg_upm[vo_i][vol_i], frame_type);
		max_qp = min_qp + 1;
	      }
	    else
	      {
		min_qp = rch_min_qp(rch);
		max_qp = rch_max_qp(rch);
	      }
          

	    /* First VO: */
	    if (vo_i == first_vo && vol_i == first_vol)
            {
              /* Get rate, distortion*pixels (dp_iq) and distortion per 
                 pixel (d_iq): */
              r_iq = (Double) (*rate)(rch, vo_i, vol_i, qp0, mad, frame_type); 
              dp_iq = (*dist)(rch, vo_i, vol_i, qp0, 1, frame_type);
              d_iq = (*dist)(rch, vo_i, vol_i, qp0, 0, frame_type);
              /* mode = 1 for total distortion, 
                 mode = 0 for pixel distortion 
                 If the two values would be calculated in equal mode, the 
                 absolute difference (d_diff) could cause a hangup because 
                 it can happen that it is not strictly increasing */

              /* If VOP0 has already been coded, distortion is PSNR value. */
              /* With PSNR=10*log10(255/dist_val): */
              if (!rch_proc_stat(rch))
		/* dp_iq = d_iq = 255. / pow (10., d_iq/20.); */
		/* Dist is quadratic: */
		dp_iq = d_iq = 65025. / pow (10., d_iq/10.); 

              /* Reference distortion, normalized: */
              if (rc_get_RCParam(rcc, frame_type) == 0.0) 
                error_exit("RC_UPM3 - min2_rld: RC Parameter 1 has to be != 0 !!!\n");  
              dist0 = d_iq / rc_get_RCParam(rcc, frame_type);
              
              qp[vo_i][vol_i] = qp0;
              r_min[vo_i][vol_i] = r_iq;
              d_min[vo_i][vol_i] = d_iq;
              d_diff_min[vo_i][vol_i] = 0;
            }
          
	    /* Other VOs: */
	    else
            {
              for (q = min_qp; q <= max_qp; q++)
                {
                  if (!first_frame(rch)) 
                    if (q == rch_exclude_qp(rch)){
#ifdef _RC_DEBUG_
                      printf("!!!! The last %d qp were equal, %d is excluded \n",
                             QP_INC, q);
#endif
                      continue;
                    }  /* A return value != 0 means to exclude this qp */
                  
                  if (rc_get_RCParam(rcc, frame_type) == 0.0) 
                    error_exit("RC_UPM3 - min2_rld: RC Parameter 1 has to be != 0 !!!\n");  
                  
                  /* Get rate, distortion*pixels and distortion per pixel: */
                  r_iq = (Double) (*rate)(rch, vo_i, vol_i, q, mad, frame_type); 
                  dp_iq = (*dist)(rch, vo_i, vol_i, q, 1, frame_type);
                  d_iq = (*dist)(rch, vo_i, vol_i, q, 0, frame_type);
                  
                  /* If VOP0 has already been coded, distortion is PSNR value. */
                  /* With PSNR=10*log10(255/dist_val): */
                  if (!rch_proc_stat(rch))
		    /* dp_iq = d_iq = 255. / pow (10., d_iq/20.); */
		    /* Dist is quadratic: */
		    dp_iq = d_iq = 65025. / pow (10., d_iq/10.); 

                  d_diff = ABS_DOUBLE( d_iq - rc_get_RCParam(rcc, frame_type) * dist0 );
                  /* rc_get_RCParam is here beta (in UPM2 it is alpha), 
                     alpha is set to one. */
                  
                  c = lambda_r * r_iq + lambda_d * (c_1*dp_iq + c_2*d_diff); 
                  
#ifdef _RC_DEBUG_
                  fprintf(stdout,"r[%d]=%f, c1_*d[%d]=%f, c_2*d'[%d]=%f, c[%d]=%f \n", 
                          q, r_iq, q, c_1*dp_iq, q, c_2*d_diff, q, c);
                  fflush(stdout);
                  
#endif
                  if (c < c_min)
                    {
                      c_min = c;
                      qp[vo_i][vol_i] = q;
                      
                      r_min[vo_i][vol_i] = r_iq;
                      d_min[vo_i][vol_i] = d_iq;
                      d_diff_min[vo_i][vol_i] = d_diff;
                    }
                } /* rof */
            } /* fi */
          
          *R += r_min[vo_i][vol_i];
          *D += d_min[vo_i][vol_i];
          *D_diff += d_diff_min[vo_i][vol_i];
          
#ifdef _RC_DEBUG_
          fprintf(stdout,"Smallest: r([%d,%d],%d) = %f, d([%d,%d],%d) = %f \n",
                  vo_i, vol_i, qp[vo_i][vol_i], r_min[vo_i][vol_i], 
                  vo_i, vol_i, qp[vo_i][vol_i], d_min[vo_i][vol_i]);
          fflush(stdout);
#endif
	  } /* end if meb 03.08.99 */
          curr_vol = GetVolConfigNext(curr_vol);
        }
      curr_vo = GetVOConfigNext(curr_vo);
    }
}


/***********************************************************CommentBegin******
 *
 * -- MinOptimizationRatios --
 *
 * Author : 
 *      Martina Eckert, UPM-GTI     
 *
 * Created :            
 *      28-04-98
 *
 * Purpose :
 *      Calculates a distribution of the global target bitrate between the
 *      different VOs by minimization of the sum of weighted distortions.
 * 
 * Arguments in :       
 *  Int     *vo_list   -  To access all VOLs
 *  Double  R          -  Global target bit rate
 *  Int     (* rate)(RC_HIST *rch, Int vo_i, Int vol_i, Int q, Double mad, Int frame_type) 
 *                     -  Rate functions 
 *  Double  (* dist)(RC_HIST *rch, Int vo_i, Int vol_i, Int q, Int mode, Int frame_type) 
 *                     -  Distortion functions 
 *  Double  mad        -  MAD 
 *  Double  dist0      -  distortion of VO 0 (for ratio)
 *  Int     mode       -  0 = calculation of distortion per pixel,
 *                        1 = calculation of total distortion 
 *  Int     frame_type -  I=0, P=1, B=2
 *  Float   c_1        -  Weight for weighted sum of d_i
 *  Float   c_2        -  Weight for weighted sum of differences 
 *                          (d_i - beta*d_0)
 *
 * Arguments in/out :   
 *  Int     qp[][]     -  Decisions 
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *  Double  r_lambda   -  Predicted bitrate sum for all VOs
 *
 * Side effects :       
 *      -
 *
 * Description :        
 *
 *
 * See also :
 *
 *
 * Modified :           
 *
 *
 ***********************************************************CommentEnd********/

static Double MinOptimizationRatios(          
                                    VOConfig *vo_list,
                                    Double R,          
                                    Int   (* rate)(RC_HIST *rch, 
                                                   Int vo_i, Int vol_i, 
                                                   Int q, Double mad,
                                                   Int frame_type), 
                                    Double (* dist)(RC_HIST *rch,
                                                    Int vo_i, Int vol_i, 
                                                    Int q, Int mode,
                                                    Int frame_type), 
                                    Double mad,       
                                    Int    qp[][MAX_NUM_VOLS],
                                    Int    qp0,
                                    Int    mode,
                                    Int    frame_type,
                                    Double *d_tot,
                                    Short  *valid,
                                    Float  c_1,
                                    Float  c_2
                                    )
{

  Double lambda1=0, lambda2, lambda_mid; /* search variables lambda */
  Double r_lambda, d_lambda, d_diff_lambda; /* r and d for lambda */


  /* Search for the upper limit of lambda */
  
#ifdef _RC_DEBUG_
  fprintf(stdout,"calculate minimal rate \n");fflush(stdout);
#endif

  /* Calculate r_lambda_min */ 
  
  min2_rld(vo_list, 1, 0, 
           rate, dist, 
           mad, qp, qp0,
           &r_lambda, &d_lambda, /* sums of r and d for lambda_r = 1, 
                                    lambda_d = 0 */
           &d_diff_lambda, 
           mode, frame_type,
           c_1, c_2);
  

  if (r_lambda > R)
    {
#ifdef _RC_DEBUG_
      fprintf(stdout,"--> R_lambda = %f, D = %f \n", r_lambda, d_lambda); 
      fprintf(stdout," R min > R. Exiting.\n");fflush(stdout);
#endif
      *d_tot = 0.;
      *valid = 0;
      return r_lambda;
    }
  
  /* Calculate r_lambda_inf */
  
#ifdef _RC_DEBUG_
  fprintf(stdout,"calculate maximal rate (minimal distortion) \n");fflush(stdout);
#endif

  min2_rld(vo_list, 
           0, 1, 
           rate, dist, 
           mad, qp, qp0,
           &r_lambda, &d_lambda, /* sums of r and d for lambda_r = 0, 
                                    lambda_d = 1 */
           &d_diff_lambda, 
           mode, frame_type,
           c_1, c_2); 
  
  if (r_lambda <= R)
    {
#ifdef _RC_DEBUG_
      fprintf(stdout,"--> R_lambda = %f, D = %f \n", r_lambda, d_lambda); 
      fprintf(stdout," R max < R. Exiting. \n");fflush(stdout);
#endif
      
      *d_tot = c_1 * d_lambda + c_2 * d_diff_lambda;
      *valid = 1;
      return r_lambda;
    }
  else
    {
      lambda2 = 1;
      do
        {
#ifdef _RC_DEBUG_
          fprintf(stdout,"lambda = %f \n", lambda2);
#endif
          
          min2_rld(vo_list, 
                   1, lambda2, 
                   rate, dist, 
                   mad, qp, qp0,
                   &r_lambda, &d_lambda,
                   &d_diff_lambda, 
                   mode, frame_type,
                   c_1, c_2);
          
#ifdef _RC_DEBUG_
          fprintf(stdout,"--> R_lambda = %f, D = %f \n", r_lambda, d_lambda);
          fflush(stdout);
#endif
          lambda2 *= 2;
          
        }
      while (r_lambda < R);
    }
  

  /* Middle point search */
  
#ifdef _RC_DEBUG_
  fprintf(stdout,"Middle point search \n");fflush(stdout);
#endif
  
  while ( (lambda2 - lambda1 > MAX_DIFF) || (r_lambda > R) )
    {
      lambda_mid = (lambda1 + lambda2) / 2;
      
      min2_rld(vo_list, 
               1, lambda_mid, 
               rate, dist, 
               mad, qp, qp0,
               &r_lambda, &d_lambda,
               &d_diff_lambda, 
               mode, frame_type,
               c_1, c_2);
      
#ifdef _RC_DEBUG_
      fprintf(stdout,"lambda = %f -> R_lambda = %f, D = %f \n", 
              lambda_mid, r_lambda, d_lambda);
      fflush(stdout);
#endif
      
      if (r_lambda < R)
        lambda1 = lambda_mid;
      else
        lambda2 = lambda_mid;
    }
  
  /*   *d_tot_lambda = lambda_mid *( d_lambda + d_diff_lambda);*/
  *d_tot = c_1 * d_lambda + c_2 * d_diff_lambda;
  *valid = 1;

  return r_lambda;
}

/***********************************************************CommentBegin******
 *
 * -- RC_UPM3_QuantAdjust --
 *
 * Author : 
 *      Martina Eckert, UPM-GTI     
 *
 * Created :            
 *      28-04-98
 *
 * Purpose :
 *      3. UPM control algorithm for global rate distribution with the 
 *      selected prediction model.
 *      Minimization of distortion and distortion ratios with the restriction
 *      sum(rate_i) <= R_target.      
 *      Returns QP for current VO.
 * 
 * Arguments in :       
 *      Int    vo_id          -  VO Id
 *      Int    vol_id         -  VOL Id
 *      VOConfig *vo_list     -  Access to all VOLs
 *      Double R              -  Global target bitrate 
 *      Double mad            -  Current MAD
 *      UInt   num_pels       -  Current number of pixels in VO
 *      Int    rc_rate_model  -  Used rate prediction model
 *      Int    rc_dist_model  -  Used distortion prediction model
 *      Int    mode           -  0 = calculation of distortion per pixel,
 *                               1 = calculation of total distortion 
 *      Int    frame_type     -  I=0, P=1, B=2
 *      Float  c_1            -  Weight for weighted sum of d_i
 *      Float  c_2            -  Weight for weighted sum of differences 
 *                               (d_i - beta*d_0)
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      Int    qp       - Quantization parameter for current VO
 *
 * Side effects :       
 *      -
 *
 * Description :        
 *
 *
 * See also :
 *      file "README"
 *
 * Modified : 
 *      28.04.98  M. Eckert: Modification of UPM 1 algorithm
 *      27.01.99  M. Eckert: Pass weighting parameters c_1, c_2
 *      24.09.99  M.Eckert: Introduce variables "next_coding_time" and
 *                          "end_time" to facilitate coding of VOLs with
 *                          different length
 *
 ***********************************************************CommentEnd********/

Int RC_UPM3_QuantAdjust(
                        Int       vo_id, 
                        Int       vol_id,
                        VOConfig *vo_list,
                        Double    R,         
                        Double    mad,       
                        UInt      num_pels_vop,
                        Int       rc_rate_model,
                        Int       rc_dist_model,
                        Int       mode,
                        Int       frame_type,
                        Float     c_1,
                        Float     c_2
                        )
                        
{
  RC_HIST *rch_0, *rch = &rc_hist_upm[vo_id][vol_id][frame_type];
  RC_CFG_PARAM *rc_cfg_0, *rc_cfg = &rc_cfg_upm[vo_id][vol_id];
  Int    (* rate_model) (RC_HIST*, Int, Int, Int, Double, Int) = NULL;
  Double (* dist_model) (RC_HIST*, Int, Int, Int, Int, Int) = NULL;
  
  Int       qp[MAX_NUM_VOS][MAX_NUM_VOLS], qp0[MAX_NUM_VOS][MAX_NUM_VOLS], q0;
  Short     vol_i, vo_i, min_qp_0, max_qp_0, set_break=0, valid;                
  VOConfig  *curr_vo;
  VolConfig *curr_vol;
  Double    r_sum, c_min, c_tot, d_tot;

  Int     first = 0;
  Float   next_coding_time, end_time;

#ifdef _RC_DEBUG_
  static Int  last_vo_id, last_vol_id;
 
  fprintf(stdout,"Global target rate: %f \n\n", R); fflush(stdout); 
#endif

 
   /* Initialize process flags for global control: */

#if 0
   curr_vo   = vo_list;
   curr_vol  = GetVOConfigLayers(curr_vo);
   if (!init){
     first_vo  = GetVOConfigId(curr_vo);
     first_vol = GetVolConfigId(curr_vol);
   }

   if (vo_id == first_vo && vol_id == first_vol)
      while(curr_vo != NULL)
      {
         vo_i = GetVOConfigId(curr_vo);
         curr_vol = GetVOConfigLayers(curr_vo);
         while(curr_vol != NULL)
         {
            vol_i = GetVolConfigId(curr_vol);
            /* Status of all VOs: unprocessed: */
            rch_set_process_flag(&rc_hist_upm[vo_i][vol_i][frame_type], 2);    
            curr_vol = GetVolConfigNext(curr_vol);
         }
         curr_vo = GetVOConfigNext(curr_vo);
      }

   if (!init){
#ifdef _RC_DEBUG_
     last_vo_id = vo_i;
     last_vol_id = vol_i;
#endif
     init = 1;
   }
#endif
   /* meb 03.08.99: change dependence on VO 0: */

      curr_vo = vo_list;
      while(curr_vo != NULL)
      {
         curr_vol =  GetVOConfigLayers(curr_vo);
         while(curr_vol != NULL)
         {
            next_coding_time = GetVolConfigNextCodingTime(curr_vol);
            end_time = GetVolConfigEndTime(curr_vol);
	    if(next_coding_time <= end_time) /* VO in process */
	    {
	      if (!first){
		first_vo  = GetVOConfigId(curr_vo);
		first_vol = GetVolConfigId(curr_vol);
		first = 1;
	      }
	      vol_i = GetVolConfigId(curr_vol);
	      /* Status of all VOs: unprocessed: */
	      rch_set_process_flag(&rc_hist_upm[vo_i][vol_i][frame_type], 2);    
	    }
	    curr_vol = GetVolConfigNext(curr_vol);
         }
         curr_vo = GetVOConfigNext(curr_vo);
      }



   /* Set process flag for currently processed VOL = [vo_id,vol_id]: */

   rch_set_process_flag(rch, 1);   


   for(vo_i = 0; vo_i < MAX_NUM_VOS; vo_i++)
      for(vol_i = 0; vol_i < MAX_NUM_VOLS; vol_i++)
      {
         qp[vo_i][vol_i] = 31;
      }


   rch_store(rch, (Int)num_pels_vop, mad);

   if (first_frame(rch)) 
     {
     /* Code  all VOs of first frame with initial qp: */
     qp[vo_id][vol_id] = rc_upm_get_QPfirst(rc_cfg, frame_type); 
#ifdef _RC_DEBUG_
      printf("first qp: %d\n",qp[vo_id][vol_id]);
#endif
     }

   else
     {
       /* Calculate QPs of optimal global distribution of target bit rate: */

       switch (rc_rate_model) 
         {
         case RC_MODEL_UPM1:
         case RC_MODEL_UPM2:
           rate_model = RC_UPM_pred_bits;
           break; 
         default:
           error_exit("RC_UPM1_QuantAdjust: Error: Mode not supported\n");
         }     
       
       switch (rc_dist_model) 
         {
         case RC_MODEL_UPM1:
         case RC_MODEL_UPM2:
           dist_model = RC_UPM_pred_dist;
           break; 
         default:
           error_exit("RC_UPM1_QuantAdjust: Error: Mode not supported\n");
         }     
       

       /* If q0 still not fixed, test for all q0, 
          which one minimizes the function: */
     
       c_min = 999999999.;
       rch_0 = &rc_hist_upm[first_vo][first_vol][frame_type];     
       rc_cfg_0 = &rc_cfg_upm[first_vo][first_vol];

       if (first_frame(rch_0)) 
         {
           min_qp_0 = rc_upm_get_QPfirst(rc_cfg_0, frame_type);
           max_qp_0 = min_qp_0 + 1;
         }
       else
         {
           min_qp_0 = rch_min_qp(rch_0);
           max_qp_0 = rch_max_qp(rch_0);
         }
       
       for (q0 = min_qp_0; q0 <= max_qp_0; q0++){
         
         if (!first_frame(rch_0)) 
           if (q0 == rch_exclude_qp(rch_0))
             {
               
#ifdef _RC_DEBUG_
               printf("!!!! The last %d qp were equal, %d is excluded \n",QP_INC,q0);
#endif
               continue;
             }  /* A return value != 0 means to exclude this qp */
         
         /* If VOP0 has already been coded, q0 is known: */
         
         if (!rch_proc_stat(rch_0)){
           q0 = rch_get_last_qp(rch_0);
           set_break = 1;
         }

         printf("Minimization for q0 = %d \n", q0);
         
         r_sum = MinOptimizationRatios(vo_list, R,
                                       rate_model, dist_model, 
                                       mad, qp0, q0, 
                                       mode, frame_type,
                                       &d_tot, &valid,
                                       c_1, c_2); 

         if (valid)
           {
             /* sum(r_i) + lambda*[sum(d_i+d_diff_i)]: */
             /* c_tot = r_sum + d_tot_lambda;*/
             c_tot = d_tot;
             
             if (c_tot < c_min)
               {
                 c_min = c_tot;
                 
                 /* save qp's: */
                 curr_vo = vo_list;
                 while(curr_vo != NULL)
                   {
                     curr_vol = GetVOConfigLayers(curr_vo);
                     while(curr_vol != NULL)
                       {
			 next_coding_time = GetVolConfigNextCodingTime(curr_vol);
			 end_time = GetVolConfigEndTime(curr_vol);
			 if(next_coding_time <= end_time) /* VO in process */
			 {
			   vo_i = GetVOConfigId(curr_vo);
			   vol_i = GetVolConfigId(curr_vol);
			   qp[vo_i][vol_i] = qp0[vo_i][vol_i];
			 }
                         curr_vol = GetVolConfigNext(curr_vol);
                       }
                     curr_vo = GetVOConfigNext(curr_vo);
                   }
               }
             printf("----> c_min= %f \n",c_min);
           }
         else 
           {
             if (q0 == max_qp_0 || set_break){
               /* save qp's: (should all be maximal) */
               curr_vo = vo_list;
               while(curr_vo != NULL)
                 {
                   curr_vol = GetVOConfigLayers(curr_vo);
                   while(curr_vol != NULL)
                     {
                       vo_i = GetVOConfigId(curr_vo);
                       vol_i = GetVolConfigId(curr_vol);
                       qp[vo_i][vol_i] = qp0[vo_i][vol_i];
                       curr_vol = GetVolConfigNext(curr_vol);
                     }
                   curr_vo = GetVOConfigNext(curr_vo);
                 }
             }
             /* else  continue;*/
           }
         
         printf("Best qp set: %d,%d,%d,%d \n",qp[0][0],qp[1][0],qp[2][0],qp[3][0]);
         if (set_break) break;
         
       } /* rof */
     }
   
#ifdef _RC_DEBUG_

   if (vo_id == last_vo_id && vol_id == last_vol_id)
     WRITE_INT("rate_script",r_sum);

#endif

  /* Store results of current VO in history: */

   rch_store_qp_before(rch, qp[vo_id][vol_id]);

#ifdef _RC_DEBUG_
   /*** JIR ***/printf("JIR: qp= %d\n", qp[vo_id][vol_id]);
#endif

   rch_set_process_flag(rch, 0);  /* VOL[vo_id,vol_id] has been finished */

   return qp[vo_id][vol_id];
}

/***********************************************************CommentBegin******
 *
 * -- get_rch_upm3 --
 *
 * Author : 
 *      Martina Eckert, UPM-GTI     
 *
 * Created :            
 *      28.04.98
 *
 * Purpose :
 *      Access to history data from other files
 *
 * Arguments in :       
 *      Int    vo_id   -  VO Id
 *      Int    vol_id  -  VOL Id
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      RC_HIST  *rch  - History data array  
 *
 *
 * Modified : 
 *
 *
 ***********************************************************CommentEnd********/

RC_HIST *get_rch_upm3(
                      Int  vo_id,
                      Int  vol_id,
                      Int  frame_type
                      )
{

return &rc_hist_upm[vo_id][vol_id][frame_type];

}

/***********************************************************CommentBegin******
 *
 * -- get_rc_cfg_upm3 --
 *
 * Author : 
 *      Martina Eckert, UPM-GTI     
 *
 * Created :            
 *      28.04.98
 *
 * Purpose :
 *      Access to history data from other files
 *
 * Arguments in :       
 *      Int    vo_id   -  VO Id
 *      Int    vol_id  -  VOL Id
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      RC_CFG_PARAM  *rc_cfg  - Configuration data (model parameters)
 *
 *
 * Modified : 
 *
 *
 ***********************************************************CommentEnd********/

RC_CFG_PARAM *get_rc_cfg_upm3(
                              Int  vo_id,
                              Int  vol_id
                              )
{

   return &rc_cfg_upm[vo_id][vol_id];

}

/*********************************************************** End of file ***/
