/******************************************************************************
 *                                                                          
 * This software module was originally developed by 
 *
 *   Oki Electric Industry Co., Ltd. (contact: Shigeru Fukunaga)
 *
 * in the course of development of the MPEG-4 Video (ISO/IEC 14496-2) standard.
 * This software module is an implementation of a part of one or more MPEG-4 
 * Video (ISO/IEC 14496-2) tools as specified by the MPEG-4 Video (ISO/IEC 
 * 14496-2) standard. 
 *
 * ISO/IEC gives users of the MPEG-4 Video (ISO/IEC 14496-2) standard free 
 * license to this software module or modifications thereof for use in hardware
 * or software products claiming conformance to the MPEG-4 Video (ISO/IEC 
 * 14496-2) standard. 
 *
 * Those intending to use this software module in hardware or software products
 * are advised that its use may infringe existing patents. The original 
 * developer of this software module and his/her company, the subsequent 
 * editors and their companies, and ISO/IEC have no liability for use of this 
 * software module or modifications thereof. No license to this software
 * module is granted for non MPEG-4 Video (ISO/IEC 14496-2) standard 
 * conforming products.
 *
 * Oki Electric Industry Co., Ltd. retains full right to use the software 
 * module for his/her own purpose, assign or donate the software module to a 
 * third party and to inhibit third parties from using the code for non MPEG-4 
 * Video (ISO/IEC 14496-2) standard conforming products. This copyright notice 
 * must be included in all copies or derivative works of the software module. 
 *
 * Copyright (c) 1999  Oki Electric Industry Co., Ltd.  All rights reserved.
 *
 *****************************************************************************/


/***********************************************************HeaderBegin*******
 *
 * File:        newpred_d.c
 *
 * Author:      Oki - Shigeru Fukunaga
 * Created:     16-AUG-1999
 *
 * Description: modules for NEWPRED mode (decoder)
 *
 * Notes:
 *
 * Modified:    
 *
 ***********************************************************HeaderEnd*********/


/************************    INCLUDE FILES    ********************************/

#include "momusys.h"
#include "vm_config.h"
#include "newpred.h"
#include "newpred_d.h"
#include "io_generic.h"  /** 31-AUG-1999 **/
#include "mot_padding.h"

#include "vm_compos.h"
#include "post_filter.h"

/***********************************************************CommentBegin******
 *
 * -- InitNewpred_d --
 *
 * Author :
 *      Oki - Shigeru Fukunaga
 *
 * Created :
 *      10-NOV-1998
 *
 * Purpose :
 *	NEWPRED - Initialize NEWPRED mode
 *
 * Arguments in :
 *	RefVop ***ref_vop - reference VOPs
 *	RefList **ref_list - reference list
 *	Int num_vos - number of VOs
 *	Int num_segments - number of NEWPRED segments per VOP
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Void
InitNewpred_d(NpConfig	***np_cfg,	/* config structure for NEWPRED */
	      Int	num_vos,	/* number of VOs */
	      Int	width,		/* width */
	      Int	height)		/* height */
{
  int i, j;
  NpConfig **tmp_np_cfg;	/* config structure */

  /* memory allocation for config structure */
  tmp_np_cfg = (NpConfig **)calloc(num_vos, sizeof(NpConfig *));
  for(i = 0; i < num_vos; i++){
    tmp_np_cfg[i] = (NpConfig *)calloc(1, sizeof(NpConfig));

    /* memory allocation for reference pictures */
    for(j = 0; j < MAXREF; j++){
      tmp_np_cfg[i]->ref_vop[j] = (RefVopD *)calloc(1, sizeof(RefVopD));
      tmp_np_cfg[i]->ref_vop[j]->vop = AllocVop(width + 32, height + 32, 0);
    }

    /* memory allocation for reference pictures list */
    for(j = 0; j < MAXNPSEG; j++){
      tmp_np_cfg[i]->ref_list[j] = (RefList *)calloc(1, sizeof(RefList));
    }
  }

  *np_cfg = tmp_np_cfg;
}
/* end of InitNewpred_d() */


/***********************************************************CommentBegin******
 *
 * -- SetRefVopNewpred_d --
 *
 * Author :
 *      Oki - Shigeru Fukunaga
 *
 * Created :
 *      16-AUG-1999
 *
 * Purpose :
 *	NEWPRED - set of reference VOP
 *
 * Arguments in :
 *	Vop *prev_rec_vop - previous reconstructed VOP
 *	NpConfig *cfg - config structure of NEWPRED
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *	The reference NEWPRED segment is replaced with the old one in the memories, 
 *	only when vop_id_for_prediction does not indicate the immediately 
 *	preceeding VOP.
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Void
SetRefVopNewpred_d(Vop *curr_vop, Vop *prev_vop, NpConfig *np_cfg)
{
  Int i, flag;
/*  Int vop_id = GetVopNewpredVopId(curr_vop);*/
  Int vop_id_pred;
  Int mem_id;		/* reference memory id */

  for(i = 0; i < np_cfg->num_segments; i++){
    vop_id_pred = GetVopNewpredVopIdPred(curr_vop, i);
#ifdef _NP_DEBUG_
#if 0
    fprintf(stdout, "SetRefVop %d %d\n", i, vop_id_pred);
#endif
#endif

    if(vop_id_pred != 0){
      flag = 0;
      for(mem_id = 0; mem_id < MAXREF; mem_id++){
	if(np_cfg->ref_vop[mem_id]->vop_id[i] == vop_id_pred){
	  if(np_cfg->ref_vop[mem_id]->vop != NULL){
	    /* update reconstructed refecence VOP */
	    CopyNPSeg(np_cfg->ref_vop[mem_id]->vop,/* from */
		      prev_vop,			/* to */
		      np_cfg->mba_segment[i],	/* begining MBA */
		      np_cfg->mba_segment[i+1],	/* ending MBA */
		      16);			/* padding size */
	    flag = 1;
	  }
	  else{
	    fprintf(stderr,"NEWPERD : reference memory error (NULL reference)\n");
	    exit(1);
	  }
	}
	if(flag)
	  break;
      }
      if(flag == 0){
	fprintf(stderr,"NEWPERD : reference memory error (memory overflow)\n");
	for(mem_id = 0; mem_id < MAXREF; mem_id++)
	  fprintf(stderr, "(%d) ", np_cfg->ref_vop[mem_id]->vop_id[i]);
	fprintf(stderr, "\n");
	exit(1);
      }
    }
  }
}
/* end of SetRefVopNewpred_d() */


/***********************************************************CommentBegin******
 *
 * -- StoreRefVopNewpred_d --
 *
 * Author :
 *      Oki - Shigeru Fukunaga
 *
 * Created :
 *      16-AUG-1999
 *
 * Purpose :
 *	NEWPRED - store reference VOP
 *
 * Arguments in :
 *	Vop *curr_rec_vop - current reconstructed VOP
 *	NpConfig *cfg - config structure of NEWPRED
 *	Int *erred_segment - error flag of each NEWPRED segment
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Void
StoreRefVopNewpred_d(Vop *curr_rec_vop, NpConfig *np_cfg, Int *erred_segment)
{
  Int i;
  Int mem_id;	/* reference memory ID */
  Vop *tmp_vop;

  /* sotre VOP after padding */
  tmp_vop = CloneVop(curr_rec_vop);
  VopPadding(tmp_vop);

  for(i = 0; i < np_cfg->num_segments; i++){
    /* Erred NEWPRED segment is not stored. */
    if(erred_segment[i] == 0){
      /* detect the oldest VOP memory for each NEWPRED segment */
      mem_id = DetectOldVop_d(np_cfg->ref_vop, i);
      
      /* add current VOP to the reference memory */
      np_cfg->ref_vop[mem_id]->vop_id[i] = GetVopNewpredVopId(curr_rec_vop);
      np_cfg->ref_vop[mem_id]->vop_id_pred[i] = GetVopNewpredVopIdPred(curr_rec_vop, i);
      CopyNPSeg(tmp_vop,			/* from */
		np_cfg->ref_vop[mem_id]->vop, 	/* to */
		np_cfg->mba_segment[i],		/* begining MBA */
		np_cfg->mba_segment[i+1],	/* ending MBA */
		16);				/* padding size */

      /* renew list of reference memory pointer */
      np_cfg->ref_list[i]->vop_id = GetVopNewpredVopId(curr_rec_vop);
      np_cfg->ref_list[i]->mem_pointer = mem_id;

#ifdef _NP_DEBUG_
#if 1
      fprintf(stdout,"[%d %d] %3d(%3d) %3d(%3d) %3d(%3d) %3d(%3d) %3d(%3d)\n", 
	      GetVopNewpredVopId(curr_rec_vop), i,
	      np_cfg->ref_vop[0]->vop_id[i], np_cfg->ref_vop[0]->vop_id_pred[i],
	      np_cfg->ref_vop[1]->vop_id[i], np_cfg->ref_vop[1]->vop_id_pred[i],
	      np_cfg->ref_vop[2]->vop_id[i], np_cfg->ref_vop[2]->vop_id_pred[i],
	      np_cfg->ref_vop[3]->vop_id[i], np_cfg->ref_vop[3]->vop_id_pred[i],
	      np_cfg->ref_vop[4]->vop_id[i], np_cfg->ref_vop[4]->vop_id_pred[i]);
#endif 
#endif
    }
  }
  FreeVop(tmp_vop);
}
/* end of StoreRefVopNewpred_d() */


/***********************************************************CommentBegin******
 *
 * -- DetectOldVop_d --
 *
 * Author :
 *      Oki - Shigeru Fukunaga
 *
 * Created :
 *      16-AUG-1999
 *
 * Purpose :
 *	NEWPRED - detect oldest reference VOP
 *
 * Arguments in :
 *	RefVop **ref_vop - reference VOPs
 *	Int np_seg_id - NEWPRED segment ID
 *
 * Arguments in/out :
 *
 * Arguments out :
 *	Int oldest_mem_id - reference memory ID
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Int
DetectOldVop_d(RefVopD **ref_vop, Int np_seg_id)
{
  int mem_id;
  Int oldest_vop_id = 1000000;
  Int oldest_mem_id = -1;

  for(mem_id = 0; mem_id < MAXREF; mem_id++){
    /* in case of vacant memory */
    if(ref_vop[mem_id]->vop_id[np_seg_id] == 0)
      return(mem_id);
    /* renewal memory ID */
    if(ref_vop[mem_id]->vop_id[np_seg_id] < oldest_vop_id){
      oldest_vop_id = ref_vop[mem_id]->vop_id[np_seg_id];
      oldest_mem_id = mem_id;
    }
  }

  return(oldest_mem_id);
}
/* end of DetectOldVop_d() */


/***********************************************************CommentBegin******
 *
 * -- WriteOutputImageNewpred_d -- 
 *
 * Author :		
 *      Oki - Shigeru Fukunaga
 *
  * Created :		
 *	31-AUG-1999
 *
 * Purpose :
 *	Writes the VOP to an output image just after the decoding.
 *
 * 
 * Arguments in : 	
 *	Vop *local_vop[] - VOP to be displayed
 *	Char output_filename[][] - Name of the output file for the composed
 *				  image sequence
 *      Int post_filter_type - status of deblock filtering
 *
 *
 * Arguments in/out :	
 *
 *
 * Arguments out :	
 *	
 *
 * Return values :	
 *	
 *
 * Side effects :	
 *	
 *
 * Description :	
 *	
 *
 * See also :
 *	
 *
 * Modified : 
 *
 ***********************************************************CommentEnd********/

Void WriteOutputImageNewpred_d(Vop *local_vop, 
			       Char output_filename[][300],
			       Int post_filter_type)
{
  Vop *display_vop, *filter_vop;
  static Int image_counter=0;

  /*
   *	Allocate memory for the display_vop which is used for composition
   *	purposes
   */
  display_vop = AllocVop(local_vop->y_chan->x, local_vop->y_chan->y, 0);
  PutVopBitsPerPixel(8, display_vop);

  /*
   *	Set the display VOP to grey (A=0, Y=0, U=V=128)
   */
  SetConstantImage(GetVopA(display_vop),0);
  SetConstantImage(GetVopY(display_vop),0);
  SetConstantImage(GetVopU(display_vop),128);
  SetConstantImage(GetVopV(display_vop),128);
  

  if(post_filter_type==0)
    BlendVops(local_vop, display_vop);
  else {
    if (GetImageSizeX(GetVopQP(local_vop)) == GetVopWidth(local_vop)/16 ||
	GetVopPredictionType(local_vop) == B_VOP)
      filter_vop = CloneVop(local_vop);
    else
      filter_vop = CloneVop_TMP(local_vop); 
    PostFilter(filter_vop,post_filter_type);
    BlendVops(filter_vop, display_vop);
    FreeVop(filter_vop);
  }

  /*
   *    Write composited display_vop to disk
   */
  if(image_counter == 0)
    WriteVopGeneric(display_vop,output_filename[0],output_filename[1],
		    output_filename[2],output_filename[3],NULL,0,image_counter,
		    IO_FORMAT,IO_OVERWRITE,TRUE);
  else
    WriteVopGeneric(display_vop,output_filename[0],output_filename[1],
		    output_filename[2],output_filename[3],NULL,0,image_counter,
		    IO_FORMAT,IO_APPEND,TRUE);

  image_counter++;
  return;
}
/* end of WriteOutputImageNewpred_d() */


