/* $Id: ztscan_enc.c,v 1.89 1998/11/10 16:42:43 hatrack Exp $ */
/****************************************************************************/
/*   MPEG4 Visual Texture Coding (VTC) Mode Software                        */
/*                                                                          */
/*   This software was jointly developed by the following participants:     */
/*                                                                          */
/*   Single-quant,  multi-quant and flow control                            */
/*   are provided by  Sarnoff Corporation                                   */
/*     Iraj Sodagar   (iraj@sarnoff.com)                                    */
/*     Hung-Ju Lee    (hjlee@sarnoff.com)                                   */
/*     Paul Hatrack   (hatrack@sarnoff.com)                                 */
/*     Shipeng Li     (shipeng@sarnoff.com)                                 */
/*     Bing-Bing Chai (bchai@sarnoff.com)                                   */
/*     B.S. Srinivas  (bsrinivas@sarnoff.com)                               */
/*                                                                          */
/*   Bi-level is provided by Texas Instruments                              */
/*     Jie Liang      (liang@ti.com)                                        */
/*                                                                          */
/*   Shape Coding is provided by  OKI Electric Industry Co., Ltd.           */
/*     Zhixiong Wu    (sgo@hlabs.oki.co.jp)                                 */
/*     Yoshihiro Ueda (yueda@hlabs.oki.co.jp)                               */
/*     Toshifumi Kanamaru (kanamaru@hlabs.oki.co.jp)                        */
/*                                                                          */
/*   OKI, Sharp, Sarnoff, TI and Microsoft contributed to bitstream         */
/*   exchange and bug fixing.                                               */
/*                                                                          */
/*                                                                          */
/* In the course of development of the MPEG-4 standard, this software       */
/* module is an implementation of a part of one or more MPEG-4 tools as     */
/* specified by the MPEG-4 standard.                                        */
/*                                                                          */
/* The copyright of this software belongs to ISO/IEC. ISO/IEC gives use     */
/* of the MPEG-4 standard free license to use this  software module or      */
/* modifications thereof for hardware or software products claiming         */
/* conformance to the MPEG-4 standard.                                      */
/*                                                                          */
/* Those intending to use this software module in hardware or software      */
/* products are advised that use may infringe existing  patents. The        */
/* original developers of this software module and their companies, the     */
/* subsequent editors and their companies, and ISO/IEC have no liability    */
/* and ISO/IEC have no liability for use of this software module or         */
/* modification thereof in an implementation.                               */
/*                                                                          */
/* Permission is granted to MPEG members to use, copy, modify,              */
/* and distribute the software modules ( or portions thereof )              */
/* for standardization activity within ISO/IEC JTC1/SC29/WG11.              */
/*                                                                          */
/* Copyright 1995, 1996, 1997, 1998 ISO/IEC                                 */
/****************************************************************************/

/************************************************************/
/*     Sarnoff Very Low Bit Rate Still Image Coder          */
/*     Copyright 1995, 1996, 1997, 1998 Sarnoff Corporation */
/************************************************************/

/************************************************************/
/*  Filename: ztscan_enc.c                                  */
/*  Author: Bing-Bing Chai                                  */
/*  Date: Dec. 4, 1997                                      */
/*                                                          */
/*  Descriptions:                                           */
/*    This file contains the routines that performs         */
/*    zero tree scanning and entropy encoding.              */
/*                                                          */
/************************************************************/

#include <stdio.h>
#include <stdlib.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <ctype.h>
#include <string.h>
#include <math.h>
#include "startcode.h"
#include "dataStruct.h"
#include "states.h"
#include "globals.h"
#include "errorHandler.h"
#include "ac.h"
#include "bitpack.h"
#include "msg.h"
#include "ztscan_common.h"
#include "ztscanUtil.h"
#include "ztscan_enc.h"
#include "Utils.h"
#include "encQM.h"

/* local global variables */

static ac_encoder ace;

static Int bit_stream_length;
static DATA **dc_coeff;
/* added by Z. Wu @ OKI for SA-prediction */
static Char **dc_mask;


/* functions defined in this file */
static Void cacll_encode();

static Void cachb_encode_SQ_band(SNR_IMAGE *snr_img);
static Void cachb_encode_SQ_tree();
static Void encode_pixel_SQ(Int h,Int w);
static Void mag_sign_encode_SQ(Int h,Int w);

static Void cachb_encode_MQ_band();
static Void cachb_encode_MQ_tree();
static Void encode_pixel_MQ(Int h,Int w);
static Void mag_sign_encode_MQ(Int h,Int w);
static Void mark_ZTR_D(Int h,Int w);

/* begin: added by Rockwell (99/3/3) */

/* IM: VTC error res */
/* error resilience fn, bbc, 11/6/98 */
static Void init_arith_encoder_model(Int color);
static Void close_arith_encoder_model(Int color, Int mode);
static Void check_end_of_packet(Int color);
static Void force_end_of_packet();

/* end: added by Rockwell (99/3/3) */

#ifdef _CHECK_HI_LO_
extern File *acfp;
#endif

/******************************************************************/
/****************************  DC  ********************************/
/******************************************************************/

/*******************************************************/
/**************  Forward DC Prediction  ****************/
/*******************************************************/

/********************************************************
  Function Name
  -------------
  static DATA DC_pred_pix(Int i, Int j)

  Arguments
  ---------
  Int i, Int j: Index of wavelet coefficient (row, col)
  
  Description
  -----------
  DPCM prediction for a DC coefficient, refer to syntax
  for algorithm. 

  Functions Called
  ----------------
  None.

  Return Value
  ------------
    prediction for coeffinfo[i][j].quantized_value
********************************************************/ 
static DATA DC_pred_pix(Int i, Int j)
{
  /*  modified by Z. Wu @ OKI */
  Int pred_i, pred_j, pred_d;

  if ( i==0 || dc_mask[i-1][j] == 0 ) 
    pred_i = 0;
  else	
    pred_i = dc_coeff[i-1][j];

  if ( j==0 || dc_mask[i][j-1] == 0 ) 
    pred_j = 0;
  else 	
    pred_j = dc_coeff[i][j-1];

  if ( i==0 || j== 0 || dc_mask[i-1][j-1] == 0 )
    pred_d = 0;
  else
    pred_d = dc_coeff[i-1][j-1];
  
  if ( abs(pred_d-pred_j) < abs(pred_d-pred_i))	
    return(pred_i);
  else
    return(pred_j);

}



/*****************************************************
  Function Name
  -------------
  Void DC_predict()

  Arguments
  ---------
  None
  
  Description
  -----------
  control program for DC prediction

  Functions Called
  ----------------
  DC_pred_pix(i,j).

  Return Value
  ------------
  none
******************************************************/

static Void DC_predict(Int color)
{
  Int i,j,dc_h,dc_w,offset_dc,max_dc;

  dc_h=mzte_codec.dcHeight;
  dc_w=mzte_codec.dcWidth;

  dc_coeff=(DATA **)calloc(dc_h,sizeof(DATA *));
  for(i=0;i<dc_h;i++)
    dc_coeff[i]=(DATA *)calloc(dc_w,sizeof(DATA));

  dc_mask=(Char **)calloc(dc_h,sizeof(Char *));
  for(i=0;i<dc_h;i++)
    dc_mask[i]=(Char *)calloc(dc_w,sizeof(Char));


  coeffinfo=mzte_codec.SPlayer[color].coeffinfo;

  for(i=0;i<dc_h;i++){
    for(j=0;j<dc_w;j++) {
      dc_coeff[i][j]=coeffinfo[i][j].quantized_value;
      dc_mask [i][j]=coeffinfo[i][j].mask;
    }
	}

  /* prediction */    
  offset_dc=0;
	
  for(i=0;i<dc_h;i++){
    for(j=0;j<dc_w;j++){
      if ( dc_mask[i][j] != 0 ) {
	if(offset_dc>(coeffinfo[i][j].quantized_value-=DC_pred_pix(i,j)))
	  offset_dc=coeffinfo[i][j].quantized_value;
      }
    }
	}

  if(offset_dc>0)
    offset_dc=0;

  /* adjust coeff's by offset_dc */
  max_dc=0;
  for(i=0;i<dc_h;i++){
    for(j=0;j<dc_w;j++){
      if ( dc_mask[i][j] != 0 ) {
	coeffinfo[i][j].quantized_value -=offset_dc;
	/* find max_dc */
	if (max_dc<coeffinfo[i][j].quantized_value)
	  max_dc=coeffinfo[i][j].quantized_value;
      }

    }
	}
    
  mzte_codec.offset_dc=offset_dc;
  mzte_codec.max_dc=max_dc;  /* hjlee */
  noteDebug("DC pred: offset=%d, max_dc=%d",
	    mzte_codec.offset_dc,mzte_codec.max_dc);  

  for(i=0;i<dc_h;i++) {
    free(dc_coeff[i]);
    free(dc_mask[i]);
  }
  free(dc_coeff);
  free(dc_mask);
}




/********************************************************
  Function Name
  -------------
  Void wavelet_dc_encode(Int c)


  Arguments
  ---------
  Int c - color component.
  
  Description
  -----------
  Control program for encode DC information for one 
  color component.

  Functions Called
  ----------------
  None.
  DC_predict()
  put_param()
  cacll_encode()
  
  Return Value
  ------------
  None.

********************************************************/ 
Void wavelet_dc_encode(Int c)
{

  noteDetail("Encoding DC (wavelet_dc_encode)....");
  color=c;

  emit_bits((UShort)mzte_codec.mean[color], 8);
  put_param((UShort)mzte_codec.Qdc[color], 7);
  /* emit_bits(mzte_codec.Qdc[color], 8); */

  DC_predict(color);
  put_param(-mzte_codec.offset_dc,7);
  put_param(mzte_codec.max_dc,7); 
  /* put_param(mzte_codec.max_dc+mzte_codec.offset_dc,7); */  /* hjlee */
  
#ifdef _CHECK_HI_LO_
  acfp=fopen("encode_hl.dat","w");
#endif

  cacll_encode();
  noteDetail("Completed encoding DC.");

#ifdef _CHECK_HI_LO_
  fclose(acfp);
#endif

}



/********************************************************
  Function Name
  -------------
  static Void cacll_encode()

  Arguments
  ---------
  None.

  
  Description
  -----------
  Encode DC information for one color component.

  Functions Called
  ----------------
  mzte_ac_encoder_init()
  mzte_ac_model_init()
  mzte_ac_encode_symbol()
  mzte_ac_model_done()
  mzte_ac_encoder_done()

  Return Value
  ------------
  None.

********************************************************/ 
static Void cacll_encode()
{
  Int dc_h, dc_w,i,j;
  Int numBP, bp;

  dc_w=mzte_codec.dcWidth;
  dc_h=mzte_codec.dcHeight;


  /* init arithmetic coder */
  numBP = ceilLog2(mzte_codec.max_dc+1); /* modified by Sharp (99/2/16) */
  mzte_ac_encoder_init(&ace);
  if ((acm_bpdc=(ac_model *)calloc(numBP,sizeof(ac_model)))==NULL)
    errorHandler("Can't allocate memory for prob model.");
  
  for (i=0; i<numBP; i++) {
    acm_bpdc[i].Max_frequency = Bitplane_Max_frequency;
    mzte_ac_model_init(&(acm_bpdc[i]),2,NULL,ADAPT,1);
  }
  coeffinfo=mzte_codec.SPlayer[color].coeffinfo;
  for (bp=numBP-1; bp>=0; bp--) {
    for(i=0;i<dc_h;i++)
      for(j=0;j<dc_w;j++){
	if( coeffinfo[i][j].mask == 1) {
	  mzte_ac_encode_symbol(&ace, &(acm_bpdc[bp]),
				((coeffinfo[i][j].quantized_value)>>bp)&1);
				}
      }
  }
  /* close arithmetic coder */
  for (i=0; i<numBP; i++) 
    mzte_ac_model_done(&(acm_bpdc[i]));
  free(acm_bpdc);

  bit_stream_length=mzte_ac_encoder_done(&ace);
}


/*********************************************************************/
/*****************************  AC  **********************************/
/*********************  Single and Multi Quant  **********************/
/*********************************************************************/

static Void bitplane_encode(Int val,Int l,Int max_bplane)
{
  register Int i,k=0;

  for(i=max_bplane-1;i>=0;i--,k++)
    mzte_ac_encode_symbol(&ace,&acm_bpmag[l][k],(val>>i)&1);
}


/*********************************************************************/
/*****************************  AC  **********************************/
/*************************  Single quant  ****************************/
/*********************************************************************/

/*******************************************************
  The following single quant routines are for band by
  band scan order.
*******************************************************/
/********************************************************
  Function Name
  -------------
  Void wavelet_higher_bands_encode_SQ_band(Int col)

  Arguments
  ---------
  None.

  Description
  -----------
  Control program for encoding AC information for one 
  color component. Single quant mode.

  Functions Called
  ----------------
  cachb_encode_SQ_band()
  mzte_ac_encoder_init()
  mzte_ac_model_init()
  mzte_ac_model_done()
  mzte_ac_encoder_done()

  Return Value
  ------------
  None.

********************************************************/ 
Void wavelet_higher_bands_encode_SQ_band(Int col)
{
  SNR_IMAGE *snr_image;
    
  noteDetail("Encoding AC (wavelet_higher_bands_encode_SQ)....");

  color=col;
  snr_image=&(mzte_codec.SPlayer[color].SNRlayer.snr_image);
  
/* begin: added by Rockwell (99/3/3) */
  /* IM: VTC error res */ 
  if (mzte_codec.error_res_flag)
    {
  /* init arithmetic coder */
/*  mzte_ac_encoder_init(&ace);*/

/*  probModelInitSQ(color);*/
  /* init arithmetic coder */
  init_arith_encoder_model(color);  /* ph, 11/13/98 */
  /*mzte_ac_encoder_init(&ace);
    probModelInitSQ(color);*/
    }
  else
    {
  /* init arithmetic coder */
  mzte_ac_encoder_init(&ace);

  probModelInitSQ(color);
    }

  cachb_encode_SQ_band(snr_image);

  /* IM: VTC error res */ 
  if (mzte_codec.error_res_flag)
    {
  /* close arithmetic coder */
  /* write last packet to file, ph, 11/13/98 */
  if(packet_size+ace.bitCount>0)
  {
    /* if color is V then write packet header */
    TU_last--;
    close_arith_encoder_model(color,mzte_codec.curSpatialLev==0 || color==2);
    if (mzte_codec.curSpatialLev==0 || color==2)
      force_end_of_packet();
    else
      TU_last++;
  }
    }
  else 
    {
  /* close arithmetic coder */
  probModelFreeSQ(color);

  bit_stream_length=mzte_ac_encoder_done(&ace);
    }

  /* close arithmetic coder */
/*  probModelFreeSQ(color);*/

/*  bit_stream_length=mzte_ac_encoder_done(&ace);*/

  noteDetail("Completed encoding AC.");
/* end: added by Rockwell (99/3/3) */
}


/********************************************************
  Function Name
  -------------
  static Void cachb_encode_SQ_band()

  Arguments
  ---------
  None.

  Description
  -----------
  Encode AC information for single quant mode, tree-depth scan.

  Functions Called
  ----------------
  codeBlocks();
  encode_pixel_SQ_band()

  Return Value
  ------------
  None.

********************************************************/ 
static Void cachb_encode_SQ_band(SNR_IMAGE *snr_image)
{
  Int h,w,ac_h,ac_w,ac_h2,ac_w2;
  Int n; /* layer index - for codeBlocks function */
  Int k; /* block jump for the layer */

  /* ac_h, ac_w init */
  ac_h2=mzte_codec.SPlayer[color].height;
  ac_w2=mzte_codec.SPlayer[color].width;
  ac_h=ac_h2>>1;
  ac_w=ac_w2>>1;

  height=mzte_codec.Image[color].height;
  width=mzte_codec.Image[color].width;

  /* Get layer index - for codeBlocks function */
  n = -1;
  for (w=mzte_codec.dcWidth; w < ac_w2; w<<=1)
    n++;

  setProbModelsSQ(color);

  coeffinfo=mzte_codec.SPlayer[color].coeffinfo;

  /* begin: added by Rockwell (99/3/3) */
  /* IM: VTC error res */	
  if (mzte_codec.error_res_flag)
    {
  /* scan each coefficients in the spatial layer */  
  k = 1<<n;
  for(h=0;h<ac_h;h+=k)
		{
    for(w=ac_w;w<ac_w2;w+=k)
    {
      /* LH */
      codeBlocks(h,w,n,&encode_pixel_SQ,color);
			if(n>0 && n<5) /* checking end of segment, ph, 11/13/98 */
				check_segment_size(color);

      /* HL */
      h += ac_h;
      w -= ac_w;
      codeBlocks(h,w,n,&encode_pixel_SQ,color);
			if(n>0 && n<5) /* checking end of segment, ph, 11/13/98 */
				check_segment_size(color);

      /* HH */
      w += ac_w;
      codeBlocks(h,w,n,&encode_pixel_SQ,color);
		 if(n>0 && n<5) /* checking end of segment, ph, 11/13/98 */
			 check_segment_size(color);

      /* Set h back to where it started. w is already there */
      h -= ac_h;
    }
		check_end_of_packet(color);
	}
    }
  else 
    {
  /* scan each coefficients in the spatial layer */  
  k = 1<<n;
  for(h=0;h<ac_h;h+=k)
    for(w=ac_w;w<ac_w2;w+=k)
    {
      /* LH */
      codeBlocks(h,w,n,&encode_pixel_SQ,color);

      /* HL */
      h += ac_h;
      w -= ac_w;
      codeBlocks(h,w,n,&encode_pixel_SQ,color);

      /* HH */
      w += ac_w;
      codeBlocks(h,w,n,&encode_pixel_SQ,color);

      /* Set h back to where it started. w is already there */
      h -= ac_h;
    }
    }
/* end: added by Rockwell (99/3/3) */
}


/*******************************************************
  The following single quant routines are for tree
  depth scan order.
*******************************************************/
/********************************************************
  Function Name
  -------------
  Void wavelet_higher_bands_encode_SQ_tree()

  Arguments
  ---------
  None.

  Description
  -----------
  Control program for encoding AC information for one 
  color component. Single quant mode.

  Functions Called
  ----------------
  cachb_encode_SQ_tree()
  mzte_ac_encoder_init()
  mzte_ac_model_init()
  mzte_ac_model_done()
  mzte_ac_encoder_done()

  Return Value
  ------------
  None.

********************************************************/ 
Void wavelet_higher_bands_encode_SQ_tree()
{

  noteDetail("Encoding AC (wavelet_higher_bands_encode_SQ)....");

/* begin: added by Rockwell (99/3/3) */
  /* IM: VTC error res */
  if (mzte_codec.error_res_flag)
    {
  /* init arithmetic coder */
/*  mzte_ac_encoder_init(&ace);*/

/*  for (color=0; color<mzte_codec.colors; color++) */
/*    probModelInitSQ(color);*/

  init_arith_encoder_model(-1);  /* bbc, 11/6/98 */
    }
  else
    {
  /* init arithmetic coder */
  mzte_ac_encoder_init(&ace);

  for (color=0; color<mzte_codec.colors; color++) 
    probModelInitSQ(color);
    }

  cachb_encode_SQ_tree();

  if (mzte_codec.error_res_flag)
    {
  /* write last packet to file, bbc, 11/6/98 */
	if(packet_size+ace.bitCount>0){
		TU_last--;
		close_arith_encoder_model(-1,1); /* write packet header */
	}
  
/*  for (color=0; color<mzte_codec.colors; color++) */
    /* close arithmetic coder */
/*    probModelFreeSQ(color);*/

/*  bit_stream_length=mzte_ac_encoder_done(&ace);*/
    }
  else
    {
  for (color=0; color<mzte_codec.colors; color++) 
    /* close arithmetic coder */
    probModelFreeSQ(color);

  bit_stream_length=mzte_ac_encoder_done(&ace);
    }

  noteDetail("Completed encoding AC.");

/* end: added by Rockwell (99/3/3) */
}


/********************************************************
  Function Name
  -------------
  static Void cachb_encode_SQ_tree()

  Arguments
  ---------
  None.

  Description
  -----------
  Encode AC information for single quant mode, tree-depth scan.

  Functions Called
  ----------------
  encode_pixel_SQ_tree()

  Return Value
  ------------
  None.

********************************************************/ 
static Void cachb_encode_SQ_tree()
{
  Int h,w,dc_h,dc_w,dc_h2,dc_w2;

  dc_h=mzte_codec.dcHeight;
  dc_w=mzte_codec.dcWidth;
  dc_h2=dc_h<<1;
  dc_w2=dc_w<<1;

  for(h=0;h<dc_h;h++)
    for(w=0;w<dc_w;w++)
      for (color=0; color<mzte_codec.colors; color++) 
      {  	
	SNR_IMAGE *snr_image;
	Int tw,sw,sh,n;

	snr_image=&(mzte_codec.SPlayer[color].SNRlayer.snr_image);
	
	height=mzte_codec.Image[color].height;
	width=mzte_codec.Image[color].width;
	
	setProbModelsSQ(color);

	coeffinfo=mzte_codec.SPlayer[color].coeffinfo;

/* begin: added by Rockwell (99/3/3) */

	/* IM: VTC error res */
	if (mzte_codec.error_res_flag)
	  {
	/* LH */
	n = 0;
	for (tw=mzte_codec.dcWidth; tw < width; tw<<=1)
	{
	  sh = h << n;
	  sw = (w+dc_w) << n;
	  codeBlocks(sh,sw,n,&encode_pixel_SQ,color);
		if(n>0 && n<5)   /* checking end of segment, bbc, 11/10/98 */
			check_segment_size(color);   
	  n++;
	}
	check_end_of_packet(-1);  /* error resilience code, bbc, 11/6/98 */

	/* HL */
	n = 0;
	for (tw=mzte_codec.dcWidth; tw < width; tw<<=1)
	{
	  sh = (h+dc_h) << n;
	  sw = w << n;
	  codeBlocks(sh,sw,n,&encode_pixel_SQ,color);
		if(n>0 && n<5)   /* checking end of segment, bbc, 11/10/98 */
			check_segment_size(color);   
	  n++;
	}
	check_end_of_packet(-1);  /* error resilience code, bbc, 11/6/98 */

	/* HH */
	n = 0;
	for (tw=mzte_codec.dcWidth; tw < width; tw<<=1)
	{
	  sh = (h+dc_h) << n;
	  sw = (w+dc_w) << n;
	  codeBlocks(sh,sw,n,&encode_pixel_SQ,color);
		 if(n>0 && n<5)   /* checking end of segment, bbc, 11/10/98 */
			 check_segment_size(color);   
						  
	  n++;
	}
	check_end_of_packet(-1);  /* error resilience code, bbc, 11/6/98 */
	  }
	else
	  {

	/* LH */
	n = 0;
	for (tw=mzte_codec.dcWidth; tw < width; tw<<=1)
	{
	  sh = h << n;
	  sw = (w+dc_w) << n;
	  codeBlocks(sh,sw,n,&encode_pixel_SQ,color);
	  n++;
	}
	/* HL */
	n = 0;
	for (tw=mzte_codec.dcWidth; tw < width; tw<<=1)
	{
	  sh = (h+dc_h) << n;
	  sw = w << n;
	  codeBlocks(sh,sw,n,&encode_pixel_SQ,color);
	  n++;
	}
	/* HH */
	n = 0;
	for (tw=mzte_codec.dcWidth; tw < width; tw<<=1)
	{
	  sh = (h+dc_h) << n;
	  sw = (w+dc_w) << n;
	  codeBlocks(sh,sw,n,&encode_pixel_SQ,color);
	  n++;
	}

	  }
      }

/* end: added by Rockwell (99/3/3) */
}

/*************************************************************
  The following single quant routines are for both scan orders.
  ************************************************************/

/********************************************************
  Function Name
  -------------
  static Void encode_pixel_SQ(Int h,Int w)

  Arguments
  ---------
  Int h,Int w - position of a pixel in height and width
  
  Description
  -----------
  Encoding the type and/or value of a coefficient, a
  recursive function.

  Functions Called
  ----------------
  mag_sign_encode_SQ()
  mzte_ac_encode_symbol()

  Return Value
  ------------
  None.

********************************************************/ 
static Void encode_pixel_SQ(Int h,Int w)
{
  UChar zt_type;
  Int l;

  if(coeffinfo[h][w].type == ZTR_D)
    return;

  l=xy2wvtDecompLev(w,h);

  /* code leave coefficients, value only, no type */
  if(IS_STATE_LEAF(coeffinfo[h][w].state)){

      /* Map type to leaf code word ZTR->0, VZTR->1 */
      zt_type = (coeffinfo[h][w].type!=ZTR);
      mzte_ac_encode_symbol(&ace,acm_type[l][CONTEXT_LINIT],zt_type);
      if (coeffinfo[h][w].type==VZTR)

	mag_sign_encode_SQ(h,w);


    return;
  }
 
  /* code zerotree symbol */


    mzte_ac_encode_symbol(&ace,acm_type[l][CONTEXT_INIT],
			  zt_type=coeffinfo[h][w].type);



  /* code magnitude and sign */
  /* For Arbitrary-Shape, out-node will always has zero coefficient,
     so only IZ or ZTR may be the zt_type -- SL*/

  switch(zt_type){
    case IZ : 
      break; /* will code the four children */
    case VZTR:
      mag_sign_encode_SQ(h,w);
    case ZTR:
	mark_ZTR_D(h,w);  /* necessary, for bandwise scan */
      break;
    case VAL:
      mag_sign_encode_SQ(h,w);
      break;
    default: 
      errorHandler("invalid zerotree symbol in single quant encode");
  }
}

/********************************************************
  Function Name
  -------------
  static Void  mag_sign_encode_SQ(Int h,Int w)

  Arguments
  ---------
  Int h,Int w - position of a pixel

  Description
  -----------
  Encode the value of a coefficient.

  Functions Called
  ----------------
  mzte_ac_encode_symbol()

  Return Value
  ------------
  None.

********************************************************/ 



static Void mag_sign_encode_SQ(Int h,Int w)
{
  Int val,v_sign;
  Int l;

  if((val=coeffinfo[h][w].quantized_value)>0)
    v_sign=0;
  else
  {
    val=-val;
    v_sign=1;
  }
    
  l=xy2wvtDecompLev(w,h);


  bitplane_encode(val-1,l,WVTDECOMP_NUMBITPLANES(color,l));
  mzte_ac_encode_symbol(&ace,acm_sign[l],v_sign);


}



/*********************************************************************/
/******************************  AC  *********************************/
/**************************  Multi quant  ****************************/
/*********************************************************************/
static Void bitplane_res_encode(Int val,Int l,Int max_bplane)
{
  register Int i,k=0;

  for(i=max_bplane-1;i>=0;i--,k++)
    mzte_ac_encode_symbol(&ace,&acm_bpres[l][k],(val>>i)&1);
}


/********************************************************
  Function Name
  -------------
  Void wavelet_higher_bands_encode_MQ(Int scanDirection)

  Arguments
  ---------
  Int scanDirection - 0 <=> tree, 1 <=> band

  Description
  -----------
  Control program for encoding AC information for one 
  color component. Multi quant mode.

  Functions Called
  ----------------
  cachb_encode_MQ_band()
  mzte_ac_encoder_init()
  mzte_ac_model_init()
  mzte_ac_model_done()
  mzte_ac_encoder_done()

  Return Value
  ------------
  None.

********************************************************/ 
Void wavelet_higher_bands_encode_MQ(Int scanDirection)
{
  noteDetail("Encoding AC (wavelet_higher_bands_encode_MQ)....");

  /* init arithmetic coder */
  mzte_ac_encoder_init(&ace);

  if (scanDirection==0)
    cachb_encode_MQ_tree();
  else
    cachb_encode_MQ_band();

  /* close arithmetic coder */
  bit_stream_length=mzte_ac_encoder_done(&ace);
}



/********************************************************
  Function Name
  -------------
  static Void mark_ZTR_D(Int h,Int w)

  Arguments
  ---------
  Int h,Int w - position of a pixel

  
  Description
  -----------
  Mark the coefficient at (h,w) and its descendents as
  zerotree descendents. 

  Functions Called
  ----------------
  mark_ZTR_D()


  Return Value
  ------------
  None.

********************************************************/ 
static Void mark_ZTR_D(Int h,Int w)
{
  Int i,j;

  i=h<<1; j=w<<1;

  if(i<height && j<width){
    coeffinfo[i][j].type     = ZTR_D;
    coeffinfo[i+1][j].type   = ZTR_D; 
    coeffinfo[i][j+1].type   = ZTR_D; 
    coeffinfo[i+1][j+1].type = ZTR_D; 
    mark_ZTR_D(i,j);
    mark_ZTR_D(i+1,j);
    mark_ZTR_D(i,j+1);
    mark_ZTR_D(i+1,j+1);
  }
}



/**********************************************************************/
/***************       MQ BAND         ********************************/
/**********************************************************************/


/********************************************************
  Function Name
  -------------
  static Void cachb_encode_MQ_band()

  Arguments
  ---------
  None.

  Description
  -----------
  Encode AC information for all color components for spatial level. 
  Multiple quant, bandwise scan.

  Functions Called
  ----------------
  clear_ZTR_D();
  codeBlocks();
  encode_pixel_MQ_band()

  Return Value
  ------------
  None.

********************************************************/ 
static Void cachb_encode_MQ_band()
{
  Int h,w;
  Int ac_h,ac_w,ac_h2,ac_w2;
  Int acH,acW,acH2,acW2;
  Int layer, nCol;
  Int n; /* layer index - for codeBlocks function */
  Int k; /* block jump for the layer */
 
  /* clear the ZTR_D type from the previous pass */
  for (color=0; color<NCOL; ++color)
  {      
    coeffinfo=mzte_codec.SPlayer[color].coeffinfo;
    height=mzte_codec.SPlayer[color].height;
    width=mzte_codec.SPlayer[color].width;

    clear_ZTR_D(coeffinfo, width, height);
  }

  for (color=0; color<NCOL; ++color)
    probModelInitMQ(color);

  acH=mzte_codec.dcHeight;
  acW=mzte_codec.dcWidth;
  acH2=acH<<1;
  acW2=acW<<1;

  /* scan each coefficients in the spatial layer */
  /* assume luma dimensions are >= chroma dimensions */
  layer=0;
  while(acH2<=mzte_codec.SPlayer[0].height 
	&& acW2<=mzte_codec.SPlayer[0].width)
  {
    nCol = (layer==0) ? 1 : NCOL;
    for (color=0; color < nCol; ++color)
    {      
      SNR_IMAGE *snr_image;

      noteProgress("  Coding Layer %d, Color %d", layer - (color!=0), color);

      ac_h2=acH2;
      ac_w2=acW2;
      ac_h=acH;
      ac_w=acW;

      if (color)
      {
	ac_h2>>=1;
	ac_w2>>=1;
	ac_h>>=1;
	ac_w>>=1;
      }
    
      snr_image=&(mzte_codec.SPlayer[color].SNRlayer.snr_image);

      coeffinfo=mzte_codec.SPlayer[color].coeffinfo;
      height=mzte_codec.SPlayer[color].height;
      width=mzte_codec.SPlayer[color].width;

      probModelInitMQ(color);

      setProbModelsMQ(color);
         
      /* Go through bands */
      n = layer - (color>0);
      k = 1<<n;
      for(h=0;h<ac_h;h+=k)
	for(w=ac_w;w<ac_w2;w+=k)
	{
	  /* LH */
	  /* begin: added by Rockwell (99/3/3) */
	  /* codeBlocks(h,w,n,&encode_pixel_MQ); */
	  codeBlocks(h,w,n,&encode_pixel_MQ,color);
	  /* end: added by Rockwell (99/3/3) */
	  
	  /* HL */
	  h += ac_h;
	  w -= ac_w;
	  /* begin: added by Rockwell (99/3/3) */
	  /* codeBlocks(h,w,n,&encode_pixel_MQ); */
	  codeBlocks(h,w,n,&encode_pixel_MQ,color);
	  /* end: added by Rockwell (99/3/3) */
	  
	  /* HH */
	  w += ac_w;
	  /* begin: added by Rockwell (99/3/3) */ 
	  /* codeBlocks(h,w,n,&encode_pixel_MQ); */
	  codeBlocks(h,w,n,&encode_pixel_MQ,color);
	  /* end: added by Rockwell (99/3/3) */

	  /* Set h back to where it started. w is already there */
	  h -= ac_h;
	}
    

    }

    /* update ranges */
    acH=acH2;
    acW=acW2;
    acW2<<=1;
    acH2<<=1;

    ++layer;
  }

  for (color=0; color<NCOL; ++color)
    probModelFreeMQ(color);

}

/**********************************************************************/
/***************       MQ TREE         ********************************/
/**********************************************************************/


/********************************************************
  Function Name
  -------------
  static Void cachb_encode_MQ_tree()

  Arguments
  ---------
  None.

  Description
  -----------
  Encode AC information for all color components for spatial level. 
  Multiple quant, bandwise scan.

  Functions Called
  ----------------
  clear_ZTR_D();
  encode_pixel_MQ_tree()

  Return Value
  ------------
  None.

********************************************************/ 
static Void cachb_encode_MQ_tree()
{
  Int h,w, dc_h, dc_w;

  /* clear the ZTR_D type from the previous pass */
  for (color=0; color<NCOL; ++color)
  {      
    coeffinfo=mzte_codec.SPlayer[color].coeffinfo;
    height=mzte_codec.SPlayer[color].height;
    width=mzte_codec.SPlayer[color].width;

    clear_ZTR_D(coeffinfo, width, height);
  }


  for (color=0; color<NCOL; ++color)
    probModelInitMQ(color);

  /* ac_h, ac_w init */
  dc_h=mzte_codec.dcHeight;
  dc_w=mzte_codec.dcWidth;

  for (h=0; h<dc_h; ++h)
    for (w=0; w<dc_w; ++w)
    {
      for (color=0; color<NCOL; ++color)
      {      
	SNR_IMAGE *snr_image;
	Int tw,sw,sh,n;

	snr_image=&(mzte_codec.SPlayer[color].SNRlayer.snr_image);
	
	coeffinfo=mzte_codec.SPlayer[color].coeffinfo;
	height=mzte_codec.SPlayer[color].height;
	width=mzte_codec.SPlayer[color].width;
      
	setProbModelsMQ(color);

	/* LH */
	n = 0;
	for (tw=mzte_codec.dcWidth; tw < width; tw<<=1)
	{
	  sh = h << n;
	  sw = (w+dc_w) << n;
	  /* begin: added by Rockwell (99/3/3) */
	  /* codeBlocks(sh,sw,n,&encode_pixel_MQ); */
	  codeBlocks(sh,sw,n,&encode_pixel_MQ,color);
	  /* end: added by Rockwell (99/3/3) */
	  n++;
	}
	/* HL */
	n = 0;
	for (tw=mzte_codec.dcWidth; tw < width; tw<<=1)
	{
	  sh = (h+dc_h) << n;
	  sw = w << n;
	  /* begin: added by Rockwell (99/3/3) */
	  /* codeBlocks(sh,sw,n,&encode_pixel_MQ); */
	  codeBlocks(sh,sw,n,&encode_pixel_MQ,color);
	  /* end: added by Rockwell (99/3/3) */
	  n++;
	}
	/* HH */
	n = 0;
	for (tw=mzte_codec.dcWidth; tw < width; tw<<=1)
	{
	  sh = (h+dc_h) << n;
	  sw = (w+dc_w) << n;
	  /* begin: added by Rockwell (99/3/3) */
	  /* codeBlocks(sh,sw,n,&encode_pixel_MQ); */
	  codeBlocks(sh,sw,n,&encode_pixel_MQ,color);
	  /* end: added by Rockwell (99/3/3) */
	  n++;
	}
      }
    }

  for (color=0; color<NCOL; ++color)
    probModelFreeMQ(color);
}


/**********************************************************************/
/***************       MQ TREE & BAND     *****************************/
/**********************************************************************/

/********************************************************
  Function Name
  -------------
  static Void encode_pixel_MQ(Int h,Int w)

  Arguments
  ---------
  Int h,Int w - position of a pixel in height and width
  
  Description
  -----------
  Encoding the type and/or value of a coefficient, a
  recursive function, multi quant mode.

  Functions Called
  ----------------
  mzte_ac_encode_symbol()
  mark_ZTR_D()
  mag_sign_encode_MQ()
 
  Return Value
  ------------
  None.

********************************************************/

static Void encode_pixel_MQ(Int h,Int w)
{
  Int zt_type;

  /*~~~~~~~~~~~~~~~~~ zerotree descendent or skip  ~~~~~~~~~~~~~~~~~~~*/
  if(coeffinfo[h][w].type==ZTR_D)
    return;

  /*~~~~~~~~~~~~~~ encode zero tree symbol ~~~~~~~~~~~~~~~~~~*/
  if (IS_RESID(w,h,color))
  {
    zt_type = VAL;
  }
  else
  {
    Int czt_type; /* what to put on bitstream */
    Int l;

    l=xy2wvtDecompLev(w,h);  

    zt_type = coeffinfo[h][w].type;
    if(coeffinfo[h][w].mask==1)		/* added by D.-S.Cho, Samsung AIT (99/04/13) */
      switch(coeffinfo[h][w].state)
      {
	case S_INIT:
	  czt_type=zt_type;
	  mzte_ac_encode_symbol(&ace,acm_type[l][CONTEXT_INIT],czt_type);
	  break;
	case S_ZTR:
	  czt_type=zt_type;
	  mzte_ac_encode_symbol(&ace,acm_type[l][CONTEXT_ZTR],czt_type);
	  break;
	case S_ZTR_D:
	  czt_type=zt_type;
	  mzte_ac_encode_symbol(&ace,acm_type[l][CONTEXT_ZTR_D],czt_type);
	  break;
	case S_IZ:
	  czt_type = (zt_type!=IZ);
	  mzte_ac_encode_symbol(&ace,acm_type[l][CONTEXT_IZ],czt_type);
	  break;
	case S_LINIT: 
	  czt_type = (zt_type!=ZTR);
	  mzte_ac_encode_symbol(&ace,acm_type[l][CONTEXT_LINIT],czt_type);
	  break;
	case S_LZTR:
	  czt_type = (zt_type!=ZTR);
	  mzte_ac_encode_symbol(&ace,acm_type[l][CONTEXT_LZTR],czt_type);
	  break;
	case S_LZTR_D:
	  czt_type = (zt_type!=ZTR);
	  mzte_ac_encode_symbol(&ace,acm_type[l][CONTEXT_LZTR_D],czt_type);
	  break;
	default:
	  errorHandler("Invalid state (%d) in multi-quant encoding.", 
		       coeffinfo[h][w].state);
      }
  }


  /*~~~~~~~~~~~~~~~~ mark ztr_d and encode magnitudes ~~~~~~~~~~~~~~~~~*/
  switch(zt_type)
  {
    case ZTR:
    case ZTR_D:
      if(coeffinfo[h][w].mask==1)	/* added by D.-S.Cho, Samsung AIT (99/04/13) */
	mark_ZTR_D(h,w);
    case IZ:
      return;
    case VZTR:
      mark_ZTR_D(h,w);
    case VAL:
      if(coeffinfo[h][w].mask==1)       /* added by D.-S.Cho, Samsung AIT (99/04/13) */
	mag_sign_encode_MQ(h,w);
      break;
    default:
      errorHandler("Invalid type (%d) in multi-quant encoding.", zt_type);     
  }
}




/********************************************************
  Function Name
  -------------
  static Void  mag_sign_encode_MQ(Int h,Int w)

  Arguments
  ---------
  Int h,Int w - position of a pixel

  Description
  -----------
  Encode the value of a coefficient.

  Functions Called
  ----------------
  mzte_ac_encode_symbol()

  Return Value
  ------------
  None.

********************************************************/ 


static Void mag_sign_encode_MQ(Int h,Int w)
{
  Int val,v_sign;
  Int l;

  if(coeffinfo[h][w].skip)
    return;
    
  l=xy2wvtDecompLev(w,h);  

  if((val=coeffinfo[h][w].quantized_value)>=0)
    v_sign=0;
  else
  {
    val=-val;
    v_sign=1;
  }
  
  /* code magnitude */

  if (IS_RESID(w,h,color))
  {
    bitplane_res_encode(val,l,WVTDECOMP_RES_NUMBITPLANES(color));
  }
  else
  {
    bitplane_encode(val-1,l,WVTDECOMP_NUMBITPLANES(color,l));
    mzte_ac_encode_symbol(&ace,acm_sign[l],v_sign);
  }


}

/* begin: added by Rockwell (99/3/3) */

/* ----------------------------------------------------------------- */
/* ----------------- Error resilience related routines ------------- */
/* ----------------------------------------------------------------- */

/* IM: VTC error res */
/* bbc, 11/5/98 */
/* ph, 11/13/98 - added color argument for band-by-band */
static void init_arith_encoder_model(Int color)
{
  /* init arithmetic coder */
  mzte_ac_encoder_init(&ace);

  if(mzte_codec.scan_direction ==0 ){ /* tree depth */
    for (color=0; color<mzte_codec.colors; color++) 
      probModelInitSQ(color);
  }
  else{  /* band-by-band */
      probModelInitSQ(color); /* ph - 11/13/98 */
  }
  /* mzte_ac_model_init(&acmSGMK,2,ifreq,!ADAPT,1); bbc, 11/20/98 */
}

/* IM: VTC error res */
/* bbc, 11/6/98 */
/* ph, 11/13/98 - added color argument for band-by-band */
static Void close_arith_encoder_model(Int color, Int mode)
{
#if ST_ERROR_RES_DEBUG
  noteProgress("\t--- ST Error Res: Enc packet [TU_%d,TU_%d], l=%d bits ---",TU_first,TU_last,
	       packet_size+ace.bitCount+ace.followBits);
#endif

  if(mzte_codec.scan_direction == 0)
  {
    /* tree depth */
    for (color=0; color<mzte_codec.colors; color++) 
      /* close arithmetic coder */
      probModelFreeSQ(color);
  }
  else
  {
      probModelFreeSQ(color); /* ph - 11/13/98 */
  }
    
  bit_stream_length=mzte_ac_encoder_done(&ace);

  if(mode==1)
    write_packet_header_to_file();

  ace.bitCount=ace.followBits=0;   
}



/* IM: VTC error res */
/****************************************************/
/* to check if a segment in a packet has exceeded a */
/* threshold. Add a marker if so. bbc, 11/6/98      */
/****************************************************/
void check_segment_size(Int col)
{
  /* segment not long enough, bbc, 11/16/98 */
  if(packet_size+ace.bitCount+ace.followBits-prev_segs_size<
     (Int)mzte_codec.segment_thresh)
    return;

#if ST_ERROR_RES_DEBUG
  noteProgress("--- ST Error Res: adding segment ---");
#endif

  prev_segs_size=packet_size+ace.bitCount+ace.followBits;

  /* add marker, use ZTR, bbc, 11/10/98  */
  mzte_ac_encode_symbol(&ace,&acmType[col][0][CONTEXT_INIT],ZTR);

  /* add marker, use ZTR, bbc, 11/20/98
  mzte_ac_encode_symbol(&ace,&acmSGMK,0);
  mzte_ac_encode_symbol(&ace,&acmSGMK,1);
  mzte_ac_encode_symbol(&ace,&acmSGMK,1);
  mzte_ac_encode_symbol(&ace,&acmSGMK,1);  */
}


/* IM: VTC error res */
/* check if end of packet is reached, bbc, 11/6/98 */
static Void check_end_of_packet(Int color)
{
  if(packet_size+ace.bitCount+ace.followBits>=targetPacketLength){
    /* write full packet to file, assume that output_buffer */
    /* is large enough to hold the arith part without writing to file */
    close_arith_encoder_model(color,1);

    flush_bits();
    flush_bytes();

    prev_segs_size=0;  /* reset segment size */

    /* start new packet */
    emit_bits((UShort)0,2); /* first bit dummy, second bit HEC=0 */
    packet_size=0;

    if (mzte_codec.scan_direction==0)
      init_arith_encoder_model(color);
    else
    {
      /* don't reinitialize if color change */
      if ((TU_last-TU_max_dc+1) % mzte_codec.dcHeight != 0)
	  init_arith_encoder_model(color);
    }

    TU_first=TU_last+1;
  }
  TU_last++;
}

/* IM: VTC error res */
/* force end of packet is reached, ph, 11/19/98 */
static Void force_end_of_packet()
{
  flush_bits();
  flush_bytes();
  
  prev_segs_size=0;  /* reset segment size */
  
  /* start new packet */
  emit_bits((UShort)0,2); /* first bit dummy, second bit HEC=0 */
  packet_size=0;
  TU_first=TU_last+1;

  TU_last++;
}

/* end: added by Rockwell (99/3/3) */
