/*****************************************************************************
 *
 * This software module was originally developed by
 *
 *   Ulrike Pestel (TUH / ACTS-MoMuSyS)
 *
 * and edited by
 * 
 *   Fernando Jaureguizar (UPM / ACTS-MoMuSyS)
 *   Dae-Sung Cho (Samsung AIT)
 *   Takefumi Nagumo (Sony Corporation)
 *
 * in the course of development of the MPEG-4 Video (ISO/IEC 14496-2) standard.
 * This software module is an implementation of a part of one or more MPEG-4
 * Video (ISO/IEC 14496-2) tools as specified by the MPEG-4 Video (ISO/IEC
 * 14496-2) standard.
 *
 * ISO/IEC gives users of the MPEG-4 Video (ISO/IEC 14496-2) standard free
 * license to this software module or modifications thereof for use in hardware
 * or software products claiming conformance to the MPEG-4 Video (ISO/IEC
 * 14496-2) standard.
 *
 * Those intending to use this software module in hardware or software products
 * are advised that its use may infringe existing patents. The original
 * developer of this software module and his/her company, the subsequent
 * editors and their companies, and ISO/IEC have no liability for use of this
 * software module or modifications thereof in an implementation. Copyright is
 * not released for non MPEG-4 Video (ISO/IEC 14496-2) Standard conforming
 * products.
 *
 * ACTS-MoMuSys partners retain full right to use the code for his/her own
 * purpose, assign or donate the code to a third party and to inhibit third
 * parties from using the code for non MPEG-4 Video (ISO/IEC 14496-2) Standard
 * conforming products. This copyright notice must be included in all copies or
 * derivative works.
 *
 * Copyright (c) 1996
 *
 *****************************************************************************/

/***********************************************************HeaderBegin*******
 *
 * File:        vm_midproc.c
 *
 * Author:      Ulrike Pestel-Schiller
 * Created:     21.05.97
 *
 * Description: Midprocessor
 *
 * Notes:       
 *
 * Modified:    
 *      09.03.98 Fernando Jaureguizar: New formating. Deleted error_flag
 *               parameter in UpsampleTexture()
 *	09.29.99 Dae-Sung Cho (Samsung AIT): added support for OBSS
 *      02.02.00 Takefumi Nagumo (SONY): fixed OBSS bug 
 *
 ***********************************************************HeaderEnd*********/

/************************    INCLUDE FILES    ********************************/

#include <stdio.h>
#include <assert.h>

#include "momusys.h"
#include "mom_access.h"
#include "mom_structs.h"
#include "vm_common_defs.h"
#include "mom_image.h"
#include "vm_midproc.h"
#include "mot_util.h" /* added for OBSS by Samsung AIT (1999-09-29) */
#include "mot_padding.h" /* added for OBSS by Samsung AIT (1999-09-29) */

#include "vm_compos.h"

/***********************************************************CommentBegin******
 *
 * -- Midproc -- Midprocessor in the case of scalability
 *
 * Author :
 *      pestel@tnt.uni-hannover.de
 *
 * Created :
 *      21.05.97    
 *
 * Purpose :
 *      Builds the block Midprocessor of VM
 *
 * Arguments in :
 *      Vop   *rec_vop,    pointer to Vop of reconstructed image BASE Layer
 *      
 * Arguments in/out :
 *
 *
 * Arguments out :
 *      Vop   *rec_vop_int, pointer to Vop of reconstructed interpolated image
 *
 * Return values :
 *      none
 *
 * Side effects :
 *      Allocates memory to interpolated image
 *
 * Description :
 *
 *
 * See also :
 *
 *
 * Modified :
 *     30.07.97 F. Jaureguizar: added the all_255 alpha_uv mask for the
 *              upsampled vop. (SpSc)
 *	09.29.99 Dae-Sung Cho (Samsung AIT): added support for OBSS
 *	10.15.99 Dae-Sung Cho (Samsung AIT): added support for OBSS
 *
 ***********************************************************CommentEnd********/

Vop*
Midproc(
   Vop   *vop,
	Vop   *curr_vop,        /* added for OBSS by Samsung AIT (1999-09-29) */
   Vol   *layer
   )
{
  Vop   *up_vop=NULL;
/* begin : added for OBSS by Samsung AIT (1999-10-15) */
#if 0
  Vop *ref_vop=NULL;
  Int hoffset=0, voffset=0; /* Displacement of spatial reference between the VOP of the */
                            /* current layer and the upsampled reference VOP from lower layer  */
#endif
/* end : added for OBSS by Samsung AIT (1999-10-15) */
  Int width_up;
  Int height_up;
  Float tmp; /* modified (float->Float) for OBSS by Samsung AIT (1999-10-15) */
  
/* begin : added for OBSS by Samsung AIT (1999-09-29) */
  Int hor_spat_ref_in;
  Int ver_spat_ref_in;
  Int hor_spat_ref_out;
  Int ver_spat_ref_out;
/* end : added for OBSS by Samsung AIT (1999-09-29) */

/* begin : added for OBSS by Samsung AIT (1999-09-29) */
  if(!GetVopArbitraryShape(vop))
  {
/* end : added for OBSS by Samsung AIT (1999-09-29) */
  /* VOP for upsampled reconstructed base layer allocated*/
  tmp = (float)GetVopWidth(vop)*((float) GetVolHorSampN(layer)/GetVolHorSampM(layer)  );
  width_up = (Int) tmp;

  tmp = (float)GetVopHeight(vop)*((float) GetVolVerSampN(layer)/GetVolVerSampM(layer));
  height_up = (Int) tmp;

  up_vop=AllocVop(width_up, height_up, GetVopAuxCompCount(vop));  

  /* VOP for upsampled reconstructed base layer allocated*/     
  /*
     up_vop=AllocVop(GetVopWidth(vop)*2, GetVopHeight(vop)*2);
*/
  UpsampleTexture(GetVopY(vop), GetVopY(up_vop), layer);
  UpsampleTexture(GetVopU(vop), GetVopU(up_vop), layer);
  UpsampleTexture(GetVopV(vop), GetVopV(up_vop), layer);
  SetConstantImage(GetVopA(up_vop), 255); /* only rectangular VOPS */
  SetConstantImage(GetVopAuv(up_vop), 255); /* only rectangular VOPS *//*SpSc*/
/* begin : added for OBSS by Samsung AIT (1999-09-29) */
  } else {
  	up_vop  =AllocVop(GetVopWidth(curr_vop), GetVopHeight(curr_vop), GetVopAuxCompCount(vop));
  	PutVopHorSpatRef(GetVopHorSpatRef(curr_vop), up_vop);
  	PutVopVerSpatRef(GetVopVerSpatRef(curr_vop), up_vop);

	PutVopShapeHorSampN(GetVolShapeHorSampN(layer),up_vop);
	PutVopShapeHorSampM(GetVolShapeHorSampM(layer),up_vop);
	PutVopShapeVerSampN(GetVolShapeVerSampN(layer),up_vop);
	PutVopShapeVerSampM(GetVolShapeVerSampM(layer),up_vop);

	hor_spat_ref_in = GetVopHorSpatRef(vop);
	hor_spat_ref_out = GetVopHorSpatRef(up_vop);
	ver_spat_ref_in = GetVopVerSpatRef(vop);
	ver_spat_ref_out = GetVopVerSpatRef(up_vop);

	/* Texture upsampling */
	if(GetVopShape(vop)!=BINARY_SHAPE_ONLY){
  		UpsampleTexture_AR(GetVopY(vop), GetVopY(up_vop), 
					hor_spat_ref_in, ver_spat_ref_in,
					hor_spat_ref_out, ver_spat_ref_out, layer);
  		UpsampleTexture_AR(GetVopU(vop), GetVopU(up_vop), 
					hor_spat_ref_in/2, ver_spat_ref_in/2,
					hor_spat_ref_out/2, ver_spat_ref_out/2, layer);
  		UpsampleTexture_AR(GetVopV(vop), GetVopV(up_vop), 
					hor_spat_ref_in/2, ver_spat_ref_in/2,
					hor_spat_ref_out/2, ver_spat_ref_out/2, layer);
	}

	/* Alpha upsampling */
  	UpsampleAlpha_AR(GetVopA(vop), GetVopA(up_vop), GetVopASamplePos(up_vop),
			hor_spat_ref_in, ver_spat_ref_in, hor_spat_ref_out, ver_spat_ref_out,
			layer);

	/* Alpha mode upsampling */
	UpsampleAlphaMode_AR(GetVopShapeMode(vop), GetVopShapeMode(up_vop),
			hor_spat_ref_in, ver_spat_ref_in, hor_spat_ref_out, ver_spat_ref_out,
			layer);

	/* Alpha motion upsampling */
	UpsampleAlphaMotX_AR(GetVopAMotX(vop), GetVopAMotX(up_vop), 
			hor_spat_ref_in, ver_spat_ref_in, hor_spat_ref_out, ver_spat_ref_out,
			layer);

	UpsampleAlphaMotY_AR(GetVopAMotY(vop), GetVopAMotY(up_vop), 
			hor_spat_ref_in, ver_spat_ref_in, hor_spat_ref_out, ver_spat_ref_out,
			layer);
  } 
/* end : added for OBSS by Samsung AIT (1999-09-29) */

return(up_vop);
}

/* begin : added for OBSS by Samsung AIT (1999-10-15) */
/***********************************************************CommentBegin******
 *
 * -- do_bgc_OBSS -- Background composition for partial enhanceing 
 *		(enhancement type==1 && background_composition==1) in OBSS
 *
 * Author :
 *      Dae-Sung Cho (Samsung AIT)
 *
 * Created :
 *      15-Oct-1999    
 *
 * Purpose :
 *      Background composition for enhancement type == 1
 *
 * Arguments in :
 *      Vop   *rec_vop,    pointer to Vop of reconstructed image BASE Layer
 *      
 * Arguments in/out :
 *
 *
 * Arguments out :
 *      Vop   *rec_vop_int, pointer to Vop of reconstructed interpolated image
 *
 * Return values :
 *      none
 *
 * Side effects :
 *      Allocates memory to interpolated image
 *
 * Description :
 *
 *
 * See also :
 *
 *
 * Modified :
 *      Takefumi Nagumo: Fixed OBSS Bugs
 *
 ***********************************************************CommentEnd********/

Vop*
do_bgc_OBSS(
   Vop   *base_vop,
   Vop   *curr_vop,
   Int   bg_composed_width, 
   Int   bg_composed_height,
   Vol   *layer
   )
{
  Vop *up_vop=NULL;
  Vop *bg_composed_vop=NULL;

  Int hor_spat_ref_in;
  Int ver_spat_ref_in;
  Int hor_spat_ref_out;
  Int ver_spat_ref_out;

  Int width_up, height_up;

  Image *sblk_y, *sblk_uv;
  
/*OBSSFIX_MODE3-start-*/
  if( !GetVopArbitraryShape(base_vop) &&
      !(!GetVopArbitraryShape(base_vop) && GetVopArbitraryShape(curr_vop)) ) {
        fprintf(stderr,"background composition will not be performed !!!\n");
        exit(1);
  }

/*  if(!GetVopArbitraryShape(base_vop)) {*/
/*	fprintf(stderr,"background composition will not be performed !!!\n");*/
/*	exit(1);*/
/*  }*/
/*OBSSFIX_MODE3-end-*/
  
/* Upsampled base image obtained */
  width_up = bg_composed_width; 
  height_up = bg_composed_height; 
  up_vop=AllocVop(width_up, height_up, GetVopAuxCompCount(curr_vop));  
  PutVopHorSpatRef(0,up_vop);
  PutVopVerSpatRef(0,up_vop);
  PutVopShape(GetVopShape(base_vop),up_vop);

  hor_spat_ref_in = GetVopHorSpatRef(base_vop);
  ver_spat_ref_in = GetVopVerSpatRef(base_vop);
  hor_spat_ref_out = GetVopHorSpatRef(up_vop);
  ver_spat_ref_out = GetVopVerSpatRef(up_vop);

  /* Texture upsampling */
  if(GetVopShape(base_vop)!=BINARY_SHAPE_ONLY){
  	UpsampleTexture_AR(GetVopY(base_vop), GetVopY(up_vop), 
					hor_spat_ref_in, ver_spat_ref_in,
					hor_spat_ref_out, ver_spat_ref_out, layer);
  	UpsampleTexture_AR(GetVopU(base_vop), GetVopU(up_vop), 
					hor_spat_ref_in/2, ver_spat_ref_in/2,
					hor_spat_ref_out/2, ver_spat_ref_out/2, layer);
  	UpsampleTexture_AR(GetVopV(base_vop), GetVopV(up_vop), 
					hor_spat_ref_in/2, ver_spat_ref_in/2,
					hor_spat_ref_out/2, ver_spat_ref_out/2, layer);
  }

  /* Alpha upsampling */
  UpsampleAlpha_AR(GetVopA(base_vop), GetVopA(up_vop), GetVopASamplePos(up_vop),
			hor_spat_ref_in, ver_spat_ref_in, hor_spat_ref_out, ver_spat_ref_out,
			layer);

  bg_composed_vop=AllocVop(bg_composed_width, bg_composed_height, GetVopAuxCompCount(curr_vop));  
  PutVopHorSpatRef(0,bg_composed_vop);
  PutVopVerSpatRef(0,bg_composed_vop);

  SetConstantImage(GetVopY(bg_composed_vop),0);
  SetConstantImage(GetVopA(bg_composed_vop),0);
  SetConstantImage(GetVopU(bg_composed_vop),128);
  SetConstantImage(GetVopV(bg_composed_vop),128);

/* The upsamled background image obtained */
  SubsampleAlphaMap(GetVopA(up_vop), GetVopAuv(up_vop), 1, 0);
  BlendVops(up_vop, bg_composed_vop);

/* begin: modified by DSCHO for bug fixing related background composition  ********/
#if 0
/*OBSSFIX_MODE3-start-*/
  if(GetVopShape(bg_composed_vop) == RECTANGULAR)
    SetConstantImage(GetVopA(bg_composed_vop),255);
/*OBSSFIX_MODE3-end-*/  
/* The current image is overlaid over the upsampled background image */
  sblk_y = AllocImage(GetVopWidth(curr_vop), GetVopHeight(curr_vop),SHORT_TYPE);
  sblk_uv = AllocImage((GetVopWidth(curr_vop)+1)/2, (GetVopHeight(curr_vop)+1)/2,SHORT_TYPE);

  /* Enhanced region initialized */
  /*OBSSFIX_MODE3-start-*/
/*Delete the following function by T.Nagumo(SONY)*/  
/*  SetConstantImage(sblk_y,0);*/
/*  SetConstantImage(sblk_uv,128);*/
/*  PutSubImage(GetVopY(bg_composed_vop),sblk_y,*/
/*		GetVopHorSpatRef(curr_vop),GetVopVerSpatRef(curr_vop));*/
/*  PutSubImage(GetVopA(bg_composed_vop),sblk_y,*/
/*		GetVopHorSpatRef(curr_vop),GetVopVerSpatRef(curr_vop));*/
/*  PutSubImage(GetVopU(bg_composed_vop),sblk_uv,*/
/*		GetVopHorSpatRef(curr_vop)/2,GetVopVerSpatRef(curr_vop)/2);*/
/*  PutSubImage(GetVopV(bg_composed_vop),sblk_uv,*/
/*		GetVopHorSpatRef(curr_vop)/2,GetVopVerSpatRef(curr_vop)/2);*/
  /*OBSSFIX_MODE3-end-*/
#endif

  PutVopShape(GetVopShape(base_vop),bg_composed_vop);
  if(GetVopShape(bg_composed_vop) == RECTANGULAR) {
    SetConstantImage(GetVopA(bg_composed_vop),255);
  } else {
    /* The current image is overlaid over the upsampled background image */
    sblk_y = AllocImage(GetVopWidth(curr_vop), GetVopHeight(curr_vop),SHORT_TYPE);
    sblk_uv = AllocImage((GetVopWidth(curr_vop)+1)/2, (GetVopHeight(curr_vop)+1)/2,SHORT_TYPE);

    /* Enhanced region initialized */
    SetConstantImage(sblk_y,0);
    SetConstantImage(sblk_uv,128);
    PutSubImage(GetVopY(bg_composed_vop),sblk_y,
		GetVopHorSpatRef(curr_vop),GetVopVerSpatRef(curr_vop));
    PutSubImage(GetVopA(bg_composed_vop),sblk_y,
		GetVopHorSpatRef(curr_vop),GetVopVerSpatRef(curr_vop));
    PutSubImage(GetVopU(bg_composed_vop),sblk_uv,
		GetVopHorSpatRef(curr_vop)/2,GetVopVerSpatRef(curr_vop)/2);
    PutSubImage(GetVopV(bg_composed_vop),sblk_uv,
		GetVopHorSpatRef(curr_vop)/2,GetVopVerSpatRef(curr_vop)/2);
    FreeImage(sblk_y);
    FreeImage(sblk_uv);
  }
/* end: modified by DSCHO for bug fixing related background composition ***********/

  BlendVops(curr_vop, bg_composed_vop);
  SubsampleAlphaMap(GetVopA(bg_composed_vop), GetVopAuv(bg_composed_vop), 1, 0);
  PutVopShape(GetVopShape(curr_vop),bg_composed_vop);

/* begin: deleted by DSCHO for bug fixing related background composition **********/
/* FreeImage(sblk_y);
  FreeImage(sblk_uv); */
/* end: deleted by DSCHO for bug fixing related background composition ************/
  FreeVop(up_vop);

  return(bg_composed_vop);
}
/* end : added for OBSS by Samsung AIT (1999-10-15) */

/********************************CommentBegin**************************
 *
 * -- UpsampleTexture -- Upsamples an image to double size
 *
 * Author :
 *      Per Thorell (ETX)
 *
 * Created :
 *      25.07.96
 *
 * Description :
 *      Upsamples an image to double size
 *
 * Arguments in :
 *      Image  *small_image,  
 *
 * Arguments in/out :    
 *
 *
 * Arguments out :
 *      Image  *big_image
 *
 * Return values :
 *      none 
 *
 * Side effects :
 *      Modifies output arguments
 *
 * See also : 
 *
 * Modified :
 *      30.11.97 Takefumi Nagumo(SONY): add Vol *layer 
 *
 *********************************CommentEnd***************************/

/* Upsampling Filter Module for MoMuSys CD software */

Void
UpsampleTexture(
   Image   *small_image,  
   Image   *big_image,
   Vol     *layer
   )

{
  Int    x, y, y1, y2, x1, x2;

  SInt   *pict;
  SInt   *pict_dist;

  Int    width, height;
  Int    width_dist, height_dist;

/*1197-3 SONY */
/*  
  Int vertical_sampling_factor_m   = GetVolHorSampM(layer);
  Int vertical_sampling_factor_n   = GetVolHorSampN(layer);
  Int horizontal_sampling_factor_m = GetVolVerSampM(layer);
  Int horizontal_sampling_factor_n = GetVolVerSampN(layer);
*/

  Int vertical_sampling_factor_m   = GetVolVerSampM(layer);
  Int vertical_sampling_factor_n   = GetVolVerSampN(layer);  
  Int horizontal_sampling_factor_m = GetVolHorSampM(layer);
  Int horizontal_sampling_factor_n = GetVolHorSampN(layer);

  /*1197-3 SONY */  
  Int phase = 0;
  
  Int *pict_vert;

  width  = (Int)GetImageSizeX (small_image);
  height = (Int)GetImageSizeY (small_image);
  width_dist  = (Int)GetImageSizeX (big_image);
  height_dist = (Int)GetImageSizeY (big_image); 
/*
  assert(height_dist == 2*height);
  assert(width_dist == 2*width);
*/
  pict = (SInt *) GetImageData(small_image); 
  pict_dist = (SInt *) GetImageData(big_image); 

  pict_vert = (Int *) malloc (sizeof(int)*width*height_dist);

  for(x=0;x< width; x++)
    for(y=0;y< height_dist; y++)
      {
      y1 = (y*vertical_sampling_factor_m )/vertical_sampling_factor_n;
      if(y1 < height - 1)
        y2 = y1+1;
      else
        y2 = y1;
      phase = (16*((y*vertical_sampling_factor_m)%
                   vertical_sampling_factor_n) + vertical_sampling_factor_n/2
                   /*for rounding*/)
               /vertical_sampling_factor_n;
      *(pict_vert+y*width+x) = (int)(16-phase)*(*(pict+y1*width+x))
                                 + phase *(*(pict+y2*width+x));
      }
  
  for(y=0;y< height_dist; y++)
    for(x=0;x< width_dist; x++)
      {
      x1 = (x*horizontal_sampling_factor_m )/horizontal_sampling_factor_n;
      if(x1 < width - 1)
        x2 = x1+1;
      else
        x2 = x1;

      /*Warning*/
      phase = (16*((x*horizontal_sampling_factor_m)%
               horizontal_sampling_factor_n) + horizontal_sampling_factor_n/2
               /*for rounding*/)
              /horizontal_sampling_factor_n;
      *(pict_dist+y*width_dist+x)
        =(unsigned char)(((16-phase)*(*(pict_vert+y*width+x1))
          +phase *(*(pict_vert+y*width+x2)) +128 )>>8 );
      }
  free(pict_vert);
}


/* begin : added for OBSS by Samsung AIT (1999-09-29) */

/********************************CommentBegin**************************
 *
 * -- UpsampleTexture_AR -- Upsamples a texture image to arbitrary size
 *			    for object-based spatial scalability
 *
 * Author :    Dae-Sung Cho (Samsung AIT)
 *
 * Created :   21-June-1999
 *
 * Description :         Upsamples a texture image to arbitrary size
 *				for object-based spatial scalability
 *
 * Arguments in :        Image  *in_image,  
 *			 Int hor_spat_ref_in,
 *			 Int ver_spat_ref_in,
 *			 Int hor_spat_ref_out,
 *			 Int ver_spat_ref_out,
 *			 Vol *layer,
 *
 * Arguments in/out :    
 *
 * Arguments out :       Image  *out_image
 *			 Image  *a_sample_pos,
 *
 * Return values :       none 
 *
 * Side effects :        Modifies output arguments
 *
 * See also : 
 *
 * Modified : 
 *
 *********************************CommentEnd***************************/
Void
UpsampleTexture_AR (
   Image   *in_image,  
   Image   *out_image,
		Int hor_spat_ref_in,
		Int ver_spat_ref_in,
		Int hor_spat_ref_out,
		Int ver_spat_ref_out,
   Vol     *layer
   )
{
  Int    x, y, y1, y2, x1, x2;

  SInt   *pict_in;
  SInt   *pict_out;

  Int    width_in, height_in;
  Int    width_out, height_out;

  Int vertical_sampling_factor_m = GetVolVerSampM(layer);
  Int vertical_sampling_factor_n = GetVolVerSampN(layer);
  Int horizontal_sampling_factor_m = GetVolHorSampM(layer);
  Int horizontal_sampling_factor_n = GetVolHorSampN(layer);

  Int phase = 0;
  
  Int *pict_vert;

  width_in  = (Int)GetImageSizeX (in_image);
  height_in = (Int)GetImageSizeY (in_image); 

  width_out  = (Int)GetImageSizeX (out_image);
  height_out = (Int)GetImageSizeY (out_image); 

  pict_in = (SInt *) GetImageData(in_image); 
  pict_out = (SInt *) GetImageData(out_image); 

  pict_vert = (Int *) malloc (sizeof(int)*width_in*height_out);

  for(x=0;x< width_in; x++) {
    for(y=0;y< height_out + ver_spat_ref_out; y++) {
      if(y >= ver_spat_ref_out) {
         y1 = (y*vertical_sampling_factor_m )/vertical_sampling_factor_n;

	 if(y1 >= ver_spat_ref_in && y1 < ver_spat_ref_in + height_in - 1) {
	     y2 = y1 + 1;
	 } else if(y1 < ver_spat_ref_in) {
	     y2 = y1 = ver_spat_ref_in;
	 } else {
	     y2 = y1 = ver_spat_ref_in + height_in - 1;
	 }

         phase = (16*((y*vertical_sampling_factor_m)%
                   vertical_sampling_factor_n) + vertical_sampling_factor_n/2)
               /vertical_sampling_factor_n;
         *(pict_vert+(y-ver_spat_ref_out)*width_in+x) 
	       = (Int)(16-phase)*(*(pict_in+(y1-ver_spat_ref_in)*width_in+x))
                      	+ phase *(*(pict_in+(y2-ver_spat_ref_in)*width_in+x));
      }
    }
  }
  
  for(y=0;y< height_out; y++) {
    for(x=0;x< width_out + hor_spat_ref_out; x++) {
      if(x >= hor_spat_ref_out) {
        x1 = (x*horizontal_sampling_factor_m )/horizontal_sampling_factor_n;

	if(x1 >= hor_spat_ref_in && x1 < hor_spat_ref_in + width_in - 1) {
            x2 = x1 + 1;
	} else if (x1 < hor_spat_ref_in) {
	    x2 = x1 = hor_spat_ref_in;
	} else {
	    x2 = x1 = hor_spat_ref_in + width_in - 1;
	}

        phase = (16*((x*horizontal_sampling_factor_m)%
               horizontal_sampling_factor_n) + horizontal_sampling_factor_n/2)
              /horizontal_sampling_factor_n;
        *(pict_out+y*width_out+(x-hor_spat_ref_out))
          =(unsigned char)(((16-phase)*(*(pict_vert+y*width_in+x1-hor_spat_ref_in))
            +phase *(*(pict_vert+y*width_in+x2-hor_spat_ref_in)) +128 )>>8 );
      }
    }
  }
  free(pict_vert);
}

/********************************CommentBegin**************************
 *
 * -- UpsampleAlpha_AR -- Upsamples an alpha image to arbitrary size
 *			    for object-based spatial scalability
 *
 * Author :    Dae-Sung Cho (Samsung AIT)
 *
 * Created :   21-June-1999
 *
 * Description :         Upsamples an alpha image to arbitrary size
 *				for object-based spatial scalability
 *
 * Arguments in :        Image  *in_image,  
 *			 Int hor_spat_ref_in,
 *			 Int ver_spat_ref_in,
 *			 Int hor_spat_ref_out,
 *			 Int ver_spat_ref_out,
 *			 Vol *layer,
 *
 * Arguments in/out :    
 *
 * Arguments out :       Image  *out_image,
 *			 Image  *a_sample_pos,
 *
 * Return values :       none 
 *
 * Side effects :        Modifies output arguments
 *
 * See also : 
 *
 * Modified : 
 *
 *********************************CommentEnd***************************/
Void UpsampleAlpha_AR( Image *in_image,  
			Image *out_image, 
				Image *a_sample_pos,
				Int hor_spat_ref_in,
				Int ver_spat_ref_in,
				Int hor_spat_ref_out,
				Int ver_spat_ref_out,
			Vol *layer)
{
  Int i, x, y, x1, y1, x2, y2;

  UInt height_in, width_in;
  UInt height_out, width_out;

  Int skip_pixel, skip_line;

  Int shape_ver_sampling_factor_m = GetVolShapeVerSampM(layer);
  Int shape_ver_sampling_factor_n = GetVolShapeVerSampN(layer);
  Int shape_hor_sampling_factor_m = GetVolShapeHorSampM(layer);
  Int shape_hor_sampling_factor_n = GetVolShapeHorSampN(layer);

  Double hor_samp_factor;
  Double v_factor;
  Int v_factor_int;
  Double ver_samp_factor;
  Double h_factor;
  Int h_factor_int;

  SInt *alpha_in, *alpha_hor, *alpha_out;
  SInt *alpha_spos;

  UChar *buff_hpos = NULL;
  UChar *buff_vpos = NULL;
  UChar *alpha_spos_h = NULL;
  UChar *alpha_spos_v = NULL;

  assert(shape_hor_sampling_factor_m<=shape_hor_sampling_factor_n && 
		shape_ver_sampling_factor_m<=shape_ver_sampling_factor_n);	

  height_in = GetImageSizeY (in_image); 
  width_in  = GetImageSizeX (in_image);
  height_out = GetImageSizeY (out_image); 
  width_out  = GetImageSizeX (out_image);

  alpha_in = (SInt *) GetImageData(in_image); 
  alpha_out = (SInt *) GetImageData(out_image); 
  alpha_spos = (SInt *) GetImageData(a_sample_pos); 

  assert(width_out == GetImageSizeX (a_sample_pos) &&
	   height_out == GetImageSizeY (a_sample_pos));
 
  alpha_hor = (SInt *)calloc(width_out*height_in, sizeof(SInt));

  alpha_spos_h = (UChar *)calloc(width_out+hor_spat_ref_out, sizeof(UChar));
  alpha_spos_v = (UChar *)calloc(height_out+ver_spat_ref_out, sizeof(UChar));

  /* deteriming the horizontal residual sampling position */
  h_factor = log((Double)shape_hor_sampling_factor_n/shape_hor_sampling_factor_m)/log(2);
  h_factor_int = (Int)floor(h_factor+0.000001);
  if(h_factor - h_factor_int > 0.000001) {
     buff_hpos = (UChar *) calloc((shape_hor_sampling_factor_n-shape_hor_sampling_factor_m*(1<<h_factor_int)), sizeof(UChar));

     hor_samp_factor = (Double)shape_hor_sampling_factor_n/(shape_hor_sampling_factor_n-shape_hor_sampling_factor_m*(1<<h_factor_int));
     for (i=1;i<=(shape_hor_sampling_factor_n-shape_hor_sampling_factor_m*(1<<h_factor_int));i++){
        buff_hpos[i-1] =
           ((Int)(shape_hor_sampling_factor_n - hor_samp_factor*i))*(1<<h_factor_int)+(1<<h_factor_int)-1;
     }
  }
 
  /* deteriming the vertical residual sampling position */
  v_factor = log((Double)shape_ver_sampling_factor_n/shape_ver_sampling_factor_m)/log(2);
  v_factor_int = (Int)floor(v_factor+0.000001);
  if(v_factor - v_factor_int > 0.000001) {
     buff_vpos = (UChar *) calloc((shape_ver_sampling_factor_n-shape_ver_sampling_factor_m*(1<<v_factor_int)), sizeof(UChar));

     ver_samp_factor = (Double)shape_ver_sampling_factor_n/(shape_ver_sampling_factor_n-shape_ver_sampling_factor_m*(1<<v_factor_int));
     for (i=1;i<=(shape_ver_sampling_factor_n-shape_ver_sampling_factor_m*(1<<v_factor_int));i++){
        buff_vpos[i-1] =
           ((Int)(shape_ver_sampling_factor_n - ver_samp_factor*i))*(1<<v_factor_int)+(1<<v_factor_int)-1;
     }
  }

  /* deteriming the horizontal sampling position */
  for (x = 0; x < width_out + hor_spat_ref_out; x++) {
     skip_pixel = 0;
     if(h_factor - h_factor_int > 0.000001) {
	for(i=0;i<(shape_hor_sampling_factor_n-shape_hor_sampling_factor_m*(1<<h_factor_int));i++){
	   if (x%((1<<h_factor_int)*shape_hor_sampling_factor_n) == buff_hpos[i])
	   	skip_pixel = 1;
	   if(skip_pixel) break; 
	}
     }
     if( (x%(1<<h_factor_int) != (1<<h_factor_int)-1) || skip_pixel == 1 ){
	alpha_spos_h[x] = 0;
     } else {
	alpha_spos_h[x] = 1;
     }
  }

  /* deteriming the vertical sampling position */
  for (y = 0; y < height_out + ver_spat_ref_out; y++) {
     skip_line = 0;
     if(v_factor - v_factor_int > 0.000001) {
	for(i=0;i<(shape_ver_sampling_factor_n-shape_ver_sampling_factor_m*(1<<v_factor_int));i++){
	   if (y%((1<<v_factor_int)*shape_ver_sampling_factor_n) == buff_vpos[i])
	   	skip_line = 1;
	   if(skip_line) break;
	}
     }
     if( (y%(1<<v_factor_int) != (1<<v_factor_int)-1) || skip_line == 1 ){
	alpha_spos_v[y] = 0;
     } else {
	alpha_spos_v[y] = 1;
     }
  }

  /* making the map indicating alpha sampling position */
  for (y=0; y<height_out; y++) {
     if(alpha_spos_v[y+ver_spat_ref_out]==1) {
        for(x=0; x<width_out; x++) {
	   if(alpha_spos_h[x+hor_spat_ref_out]==1) {
	      *(alpha_spos+y*width_out+x) = 1;
	   } else {
	      *(alpha_spos+y*width_out+x) = 0;
           }
        }
     } else {
        for(x=0; x<width_out; x++) 
	   *(alpha_spos+y*width_out+x) = 0;
     }
  }

  /* do alpha up-samplng */
  /* Horizontal resampling  */
  for (y = 0; y < height_in; y++) { 
     x1 = x2 = 0;
     for (x = 0; x < width_out + hor_spat_ref_out; x++) {
	if( x>=hor_spat_ref_out ) {
	   if( x1>=hor_spat_ref_in && x1<hor_spat_ref_in+width_in ) {
		*(alpha_hor+y*width_out+x2) = *(alpha_in+y*width_in+x1-hor_spat_ref_in);
	   } else {
		*(alpha_hor+y*width_out+x2) = 0;
	   }
	   x2 ++;
	}
	if (alpha_spos_h[x] == 1) x1 ++;
     }
  }

  /* Vertical resampling  */
  for (x = 0; x < width_out; x++) {
     y1 = y2 = 0;
     for (y = 0; y < height_out + ver_spat_ref_out ; y++) { 
        if( y>=ver_spat_ref_out ) {
	   if( y1>=ver_spat_ref_in && y1<ver_spat_ref_in+height_in) {
	   	*(alpha_out+y2*width_out+x) = *(alpha_hor+(y1-ver_spat_ref_in)*width_out+x);
	   } else {
	   	*(alpha_out+y2*width_out+x) = 0;
	   }
	   y2++;
	}
        if (alpha_spos_v[y] == 1) y1 ++;
     }
  }
		
  if(h_factor - h_factor_int > 0.000001)  free(buff_hpos);
  if(v_factor - v_factor_int > 0.000001)  free(buff_vpos);

  free(alpha_hor);
  free(alpha_spos_h);
  free(alpha_spos_v);
}

/********************************CommentBegin**************************
 *
 * -- UpsampleAlphaMode_AR -- Upsamples an alpha mode data to arbitrary size
 *				for object-based spatial scalability
 *
 * Author :    Dae-Sung Cho (Samsung AIT)
 *
 * Created :   21-Jun-1999
 *
 * Description :         Upsamples an alpha mode data to arbitrary size
 *				for object-based spatial scalability
 *
 * Arguments in :        Image  *small_image,  
 *
 * Arguments in/out :    
 *
 * Arguments out :       Image  *big_image
 *
 * Return values :       none 
 *
 * Side effects :        Modifies output arguments
 *
 * See also : 
 *
 * Modified : 
 *
 *********************************CommentEnd***************************/

Void UpsampleAlphaMode_AR ( Image *amode_in,  
			    Image *amode_out, 
			    Int hor_spat_ref_in,
			    Int ver_spat_ref_in,
			    Int hor_spat_ref_out,
			    Int ver_spat_ref_out,
			    Vol *layer
			  )
{
  Int x,y;
  Int xoffset, yoffset, xindex, yindex;
  SInt *shape_mode_in;
  SInt *shape_mode_out;

  UInt height_out, width_out;
  UInt height_in, width_in;

  Int shape_ver_sampling_factor_m = GetVolShapeVerSampM(layer);
  Int shape_ver_sampling_factor_n = GetVolShapeVerSampN(layer);
  Int shape_hor_sampling_factor_m = GetVolShapeHorSampM(layer);
  Int shape_hor_sampling_factor_n = GetVolShapeHorSampN(layer);

  height_in = GetImageSizeY (amode_in); 
  width_in  = GetImageSizeX (amode_in);

  height_out = GetImageSizeY (amode_out); 
  width_out  = GetImageSizeX (amode_out);

  shape_mode_out = (SInt *) GetImageData(amode_out); 
  shape_mode_in = (SInt *) GetImageData(amode_in); 

  xoffset= hor_spat_ref_out-hor_spat_ref_in*shape_hor_sampling_factor_n/shape_hor_sampling_factor_m;
  yoffset= ver_spat_ref_out-ver_spat_ref_in*shape_ver_sampling_factor_n/shape_ver_sampling_factor_m;

  xoffset = xoffset / MB_SIZE;
  yoffset = yoffset / MB_SIZE;
  for (y=0;y<height_out;y++) {
     for (x=0;x<width_out;x++) {
	
	yindex=MIN(MAX(0,(y+yoffset)*shape_ver_sampling_factor_m/shape_ver_sampling_factor_n),height_in-1);
	xindex=MIN(MAX(0,(x+xoffset)*shape_hor_sampling_factor_m/shape_hor_sampling_factor_n),width_in-1);
	shape_mode_out[y*width_out+x]=shape_mode_in[yindex*width_in+xindex];
     }
  }
}

/********************************CommentBegin**************************
 *
 * -- UpsampleAlphaMotX_AR -- Upsamples a horizontal alpha motion data 
 *				to arbitrary size for object-based spatial scalability
 *
 * Author :    Dae-Sung Cho (Samsung AIT)
 *
 * Created :   21-Jun-1999
 *
 * Description :         Upsamples a horizontal alpha motion data 
 *				to arbitrary size for object-based spatial scalability
 *
 * Arguments in :        Image  *in_image,  
 *
 * Arguments in/out :    
 *
 * Arguments out :       Image  *out_image
 *
 * Return values :       none 
 *
 * Side effects :        Modifies output arguments
 *
 * See also : 
 *
 * Modified : 
 *
 *********************************CommentEnd***************************/

Void UpsampleAlphaMotX_AR ( Image *a_mot_x_in,  
			  Image *a_mot_x_out,
			  Int hor_spat_ref_in,
			  Int ver_spat_ref_in,
			  Int hor_spat_ref_out,
			  Int ver_spat_ref_out,
			  Vol *layer
			)
{
  Int x,y,val;
  Int xoffset, yoffset;
  Int xindex, yindex;
  SInt *shape_mot_x_in;
  SInt *shape_mot_x_out;

  UInt height_out, width_out;
  UInt height_in, width_in;

  Int shape_ver_sampling_factor_m = GetVolShapeVerSampM(layer);
  Int shape_ver_sampling_factor_n = GetVolShapeVerSampN(layer);
  Int shape_hor_sampling_factor_m = GetVolShapeHorSampM(layer);
  Int shape_hor_sampling_factor_n = GetVolShapeHorSampN(layer);

  height_in = GetImageSizeY (a_mot_x_in); 
  width_in  = GetImageSizeX (a_mot_x_in);

  height_out = GetImageSizeY (a_mot_x_out); 
  width_out  = GetImageSizeX (a_mot_x_out);

  shape_mot_x_in = (SInt *) GetImageData(a_mot_x_in); 
  shape_mot_x_out = (SInt *) GetImageData(a_mot_x_out); 

  xoffset= hor_spat_ref_out-hor_spat_ref_in*shape_hor_sampling_factor_n/shape_hor_sampling_factor_m;
  yoffset= ver_spat_ref_out-ver_spat_ref_in*shape_ver_sampling_factor_n/shape_ver_sampling_factor_m;

  xoffset = xoffset / MB_SIZE;
  yoffset = yoffset / MB_SIZE;

  for (y=0;y<height_out;y++) {
     for (x=0;x<width_out;x++) {
	
	yindex=MIN(MAX(0,(y+yoffset)*shape_ver_sampling_factor_m/shape_ver_sampling_factor_n),height_in-1);
	xindex=MIN(MAX(0,(x+xoffset)*shape_hor_sampling_factor_m/shape_hor_sampling_factor_n),width_in-1);
/*
	shape_mot_x_out[y*width_out+x]
		=shape_mot_x_in[yindex*width_in+xindex]*shape_hor_sampling_factor_n/shape_hor_sampling_factor_m;
*/
	val = shape_mot_x_in[yindex*width_in+xindex];
	shape_mot_x_out[y*width_out+x] = (val*shape_hor_sampling_factor_n+SIGN(val)*shape_hor_sampling_factor_m/2)/shape_hor_sampling_factor_m;
     }
  }
}

/********************************CommentBegin**************************
 *
 * -- UpsampleAlphaMotY_AR -- Upsamples a vertical alpha motion data 
 *				to arbitrary size for object-based spatial scalability
 *
 * Author :    Dae-Sung Cho (Samsung AIT)
 *
 * Created :   21-Jun-1999
 *
 * Description :         Upsamples a vertical alpha motion data 
 *				to arbitrary size for object-based spatial scalability
 *
 * Arguments in :        Image  *in_image,  
 *
 * Arguments in/out :    
 *
 * Arguments out :       Image  *out_image
 *
 * Return values :       none 
 *
 * Side effects :        Modifies output arguments
 *
 * See also : 
 *
 * Modified : 
 *
 *********************************CommentEnd***************************/

Void UpsampleAlphaMotY_AR ( Image *a_mot_y_in,
			  Image *a_mot_y_out,	
			  Int hor_spat_ref_in,
			  Int ver_spat_ref_in,
			  Int hor_spat_ref_out,
			  Int ver_spat_ref_out,
			  Vol *layer
			)
{
  Int x,y,val;
  Int xoffset, yoffset;
  Int xindex, yindex;
  SInt *shape_mot_y_in;
  SInt *shape_mot_y_out;

  UInt height_out, width_out;
  UInt height_in, width_in;

  Int shape_ver_sampling_factor_m = GetVolShapeVerSampM(layer);
  Int shape_ver_sampling_factor_n = GetVolShapeVerSampN(layer);
  Int shape_hor_sampling_factor_m = GetVolShapeHorSampM(layer);
  Int shape_hor_sampling_factor_n = GetVolShapeHorSampN(layer);

  height_in = GetImageSizeY (a_mot_y_in); 
  width_in  = GetImageSizeX (a_mot_y_in);

  height_out = GetImageSizeY (a_mot_y_out); 
  width_out  = GetImageSizeX (a_mot_y_out);

  shape_mot_y_in = (SInt *) GetImageData(a_mot_y_in); 
  shape_mot_y_out = (SInt *) GetImageData(a_mot_y_out); 

  xoffset= hor_spat_ref_out-hor_spat_ref_in*shape_hor_sampling_factor_n/shape_hor_sampling_factor_m;
  yoffset= ver_spat_ref_out-ver_spat_ref_in*shape_ver_sampling_factor_n/shape_ver_sampling_factor_m;

  xoffset = xoffset / MB_SIZE;
  yoffset = yoffset / MB_SIZE;

  for (y=0;y<height_out;y++) {
     for (x=0;x<width_out;x++) {
	
	yindex=MIN(MAX(0,(y+yoffset)*shape_ver_sampling_factor_m/shape_ver_sampling_factor_n),height_in-1);
	xindex=MIN(MAX(0,(x+xoffset)*shape_hor_sampling_factor_m/shape_hor_sampling_factor_n),width_in-1);
	/*
	shape_mot_y_out[y*width_out+x]
		=shape_mot_y_in[yindex*width_in+xindex]*shape_ver_sampling_factor_n/shape_ver_sampling_factor_m;
	*/
	val = shape_mot_y_in[yindex*width_in+xindex];
	shape_mot_y_out[y*width_out+x] = (val*shape_ver_sampling_factor_n+SIGN(val)*shape_ver_sampling_factor_m/2)/shape_ver_sampling_factor_m;
     }
  }
}

/***********************************************************CommentBegin******
 *
 * -- VopExtendByPadding --
 *
 * Author :
 *      Dae-Sung Cho (Samsung AIT)
 *
 * Created :
 *      23.11.98
 *
 * Purpose :
 *      padding a VOP
 *
 * Arguments in :
 *
 *
 * Arguments in/out :
 *      Vop *curr_vop
 *
 * Arguments out :
 *
 *
 * Return values :
 *
 *
 * Side effects :
 *
 *
 * Description : Repetitive Padding of Extended Area for padded VOP.
 *     	The routine (RecRepetitivePadding()) is applied both in rectangular and in arbitrary VOP
 *      since the input VOP has been previously padded.
 *
 * See also :
 *
 *
 * Modified :
 *
 *
 ***********************************************************CommentEnd********/

Void
VopExtendByPadding(
   Vop   *curr_vop
   )
{
  Int   n,edge = 16;
  Vop   *curr_vop_padded;

  curr_vop_padded = AllocVop (GetVopWidth(curr_vop) + 2 * edge,
                              GetVopHeight(curr_vop) + 2 * edge,
                              GetVopAuxCompCount(curr_vop));
  curr_vop_padded->hor_spat_ref=curr_vop->hor_spat_ref-edge;
  curr_vop_padded->ver_spat_ref=curr_vop->ver_spat_ref-edge;

  /* Store width and height as integer multiples of 16 */
  PutVopWidth(GetVopWidth(curr_vop)+2*edge,curr_vop_padded);
  PutVopHeight(GetVopHeight(curr_vop)+2*edge,curr_vop_padded);

  MakeImageEdge(GetVopY(curr_vop), edge, GetVopY(curr_vop_padded));
  for(n=0;n<GetVopAuxCompCount(curr_vop);n++) /* MAC (SB) 12-Nov-99 */
    MakeImageEdge(GetVopG(n,curr_vop), edge, GetVopG(n,curr_vop_padded)); 
  MakeImageEdge(GetVopU(curr_vop), edge/2, GetVopU(curr_vop_padded));
  MakeImageEdge(GetVopV(curr_vop), edge/2, GetVopV(curr_vop_padded));
  MakeImageEdge(GetVopA(curr_vop), edge, GetVopA(curr_vop_padded));
  MakeImageEdge(GetVopAuv(curr_vop), edge/2, GetVopAuv(curr_vop_padded));

  curr_vop_padded->arbitrary_shape=GetVopShape(curr_vop);
  PutVopBitsPerPixel(GetVopBitsPerPixel(curr_vop),curr_vop_padded);  

  /* Applied in both ractangular and arbitrarily shaped VOP
     since input Vop is already padded */
  RecRepetitivePadding (edge, curr_vop_padded);

  PutVopA(GetVopA(curr_vop_padded),curr_vop);
  PutVopAuv(GetVopAuv(curr_vop_padded),curr_vop);
  PutVopY(GetVopY(curr_vop_padded),curr_vop);
  for(n=0;n<GetVopAuxCompCount(curr_vop);n++) /* MAC (SB) 12-Nov-99 */
    PutVopG(GetVopG(n,curr_vop_padded),n,curr_vop); 
  PutVopU(GetVopU(curr_vop_padded),curr_vop);
  PutVopV(GetVopV(curr_vop_padded),curr_vop);
  FreeImage(GetVopQP(curr_vop_padded));
  for(n=0;n<GetVopAuxCompCount(curr_vop);n++) /* MAC (SB) 12-Nov-99 */
    FreeImage(GetVopGLQP(n,curr_vop_padded));
  FreeImage(GetVopShapeMode(curr_vop_padded));		

  FreeImage(GetVopAMotX(curr_vop_padded));		
  FreeImage(GetVopAMotY(curr_vop_padded));		

  FreeImage(GetVopASamplePos(curr_vop_padded));		

  SfreeVop(curr_vop_padded);
  curr_vop->hor_spat_ref-=edge;
  curr_vop->ver_spat_ref-=edge;
  PutVopWidth(GetVopWidth(curr_vop)+2*edge,curr_vop);
  PutVopHeight(GetVopHeight(curr_vop)+2*edge,curr_vop);
}

/* end : added for OBSS by Samsung AIT (1999-09-29) */
