/************************************************************************/
/*                                                     			*/
/* This software module was originally developed by              	*/
/*                                                               	*/
/* Stefan Rauthenberg (HHI / ACTS-MoMuSys).     	              	*/
/*                                                               	*/
/* and edited by                                                	*/
/*                                                               	*/
/* Jan De Lameillieure (HHI / ACTS-MoMuSys).     	              	*/
/*                                                               	*/
/* in the course of development of the MPEG-4 Video (ISO/IEC 14496-2).	*/
/* This software module is an implementation of a part of one or 	*/
/* more MPEG-4 Video (ISO/IEC 14496-2) tools as specified by the        */
/* MPEG-4 Video (ISO/IEC 14496-2). ISO/IEC gives users of the MPEG-4    */
/* Video free license to this software module or modifications thereof 	*/
/* for use in hardware or software products claiming conformance to the */
/* MPEG-4 Video (ISO/IEC 14496-2). Those intending to use this software */
/* module in hardware or software products are advised that its use may */
/* infringe existing patents. The original developer of this software  	*/
/* module and his/her company, the subsequent editors and their     	*/
/* companies, and ISO/IEC have no liability for use of this software    */
/* module or modifications thereof in an implementation. Copyright is   */
/* not released for non MPEG-4 Video (ISO/IEC 14496-2) conforming 	*/
/* products. ACTS-MoMuSys partners retain full right to use  the code   */
/* for their own purposes, assign or donate the code to a third party   */
/* and to inhibit third parties from using the code for non MPEG-4    	*/
/* Video (ISO/IEC 14496-2) conforming products. This copyright notice 	*/
/* must be included in all copies or derivative works.                  */
/* Copyright (c)1997                                            	*/
/*                                                               	*/
/************************************************************************/

/***********************************************************HeaderBegin*******
 *                                                                         
 * File: sadct_blk.c 
 * 
 * Author: Stefan Rauthenberg (HHI)
 *
 *	Heinrich-Hertz-Institut fuer Nachrichtentechnik GmbH
 *	Image Processing Department
 *	Einsteinufer 37
 *	D-10587 Berlin 
 *	Federal Republic of Germany
 *	Phone: 	+49-30-31002-615
 *	Fax:	+49-30-3927200
 *	email:	rauthenberg@HHI.DE
 *
 * Created:  21/02/95
 *                                                                         
 * Description: 
 *	SADCT (shape adaptive DCT) transformation  of a single block.
 *
 *	User callable functions:
 *
 *		sadct_init()
 *		sadct_free()
 *		
 *		sadct_blk()
 *		saidct_blk()
 *
 * Notes:  
 *
 * Modified: 
 *      20-MAR-97 Jan De Lameillieure (HHI) : adaptation to MoMuSys common
 *			rules
 *      23-APR-97 Jan De Lameillieure (HHI) : free_dmatrix() is replaced by
 *                      free_dmatrix_sadct() and dmatrix() by dmatrix_sadct()
 *                      to prevent confusion with similar functions defined
 *                      in the sprite software
 *     16-JUN-97 Jan De Lameillieure : renaming some include files to sadct_*
 *
 ***********************************************************HeaderEnd*********/

/************************    INCLUDE FILES    ********************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "sadct_bsnrmem.h"
#include "sadct.h"

/* Indexing:	dct_matrix[n] is a pointer to a transformation matrix of
 *              blocksize `n'.  The sensible range of the first index is 
 *		1..N AND not 0 .. N-1 as one might presume.
 */
static Double ***dct_matrix;
static Double ***idct_matrix;

/* N = number of (1D) transformation matrices of different block size.
   For a given N the matrix consists of N x N elements. */
static Int N = 0;

/* buffer to hold temporary results */
static Double ***reorder_h;
static Double ***reorder_v;
static Double **mat_tmp1;
static Double *row_buf;
static Int *l_y, *l_x;

/*
 *	Allocates memory and sets up internal tables to transform
 *	blocks of at most `bksize_max' by `bksize_max' elements.
 *	The allocated memory may be freed by calling `sadct_free()'.
 *	The client code is expected to call `sadct_init()' before 
 *	trying to transform a block for the very first time.
 *	Failure to do so is likely to cause a coredump.
 *
 *	Parameter:	
 *		`bksize_max'	maximum blocksize
 *
 */
Void sadct_init(Int bksize_max, Double scale_dct, Double scale_idct)
{

  if ( bksize_max == N )
    return;	/* already initialized, check prevents a client to 
		   free something which is going to be initialized 
		   exactly the same way. */

  sadct_malloc(bksize_max);

  sadct_trfmat_init(dct_matrix, bksize_max, scale_dct);
  saidct_trfmat_init(idct_matrix, bksize_max, scale_idct);

}

/*
 *	free's any memory allocated by `sadct_init'.  
 */
Void sadct_free(Void)
{

  if ( ! N )
    return;	/* nothing to free */

  sadct_free_internal(dct_matrix, N);
  dct_matrix = 0;
  sadct_free_internal(idct_matrix, N);
  idct_matrix = 0;

  sadct_free_reorder(reorder_v, N);
  reorder_v = 0;

  sadct_free_reorder(reorder_h, N);
  reorder_h = 0;

  free_dmatrix_sadct(mat_tmp1, 0, N-1, 0, N-1);
  mat_tmp1 = 0;

  free_dvector(row_buf, 0, N-1);
  row_buf = 0;
  free_ivector((int *)l_x, 0, N-1);
  l_x = 0;
  free_ivector((int *)l_y, 0, N-1);
  l_y = 0;

  N = 0;

  return;
}

/*
 *	transforms the set of pels of a `bkx' * `bky' block `in' which 
 *	are marked by 1 in the `mask'. 
 *	The function returns in lx[0] - lx[bky-1] the
 *	number of dct coefficient per line and the coefficients are saved
 *	in the upper left corner of `out'; e.g.
 *	lx[0] = 3, lx[1] = 2, lx[2] = 1, lx[3] = 0 ...
 *	
 *		       
 *	X  X  X  - - - - - - - - -	    `out'
 *	X  X  -  - - - - - - - - -
 *      X  -  -  - - - - - - - - -
 *      -  -  -  - ...
 */
Void
sadct_blk(Double **out, Int *lx, Double **in, UChar **mask, Int bky, Int bkx)
{
  Int i, j, jmax, k;
  Double **trf_mat, *row;
  Double c;

  sadct_shiftup_transpose(mat_tmp1, l_y, in, mask, bky, bkx);

#if SADCT_DEBUG > 1
  fprintf_smat(stdout, "sadct_blk:in", in, 0, 0, bky-1, bkx-1, "%4d");
  fprintf_dmat(stdout, "mat transposed", mat_tmp1, 0, 0, bkx-1, bky-1, 
	       "%6.3f ");  
#endif
  memset(lx, 0, sizeof(Int)*bky);

  for (i=0; i<bkx && l_y[i]; i++) {
    jmax = l_y[i];
    trf_mat = dct_matrix[jmax];
    row = mat_tmp1[i];
    for (k=0; k<jmax; k++) {
      for (c=0,j=0; j<jmax; j++) 
	c += trf_mat[k][j] * row[j];
      out[k][lx[k]] = c;
      lx[k]++;
    }
  }
#if SADCT_DEBUG > 1
  fprintf_dmat(stdout, "v transformed matrix", out, 0, 0, bky-1, bkx-1, 
	       "%6.3f ");  
#endif

  /* and finally the horizontal transformation */
  for (i=0; i<bky && lx[i]; i++) {
    jmax = lx[i];
    trf_mat = dct_matrix[jmax];
    memcpy(row_buf, out[i], jmax*sizeof(Double));
    row = out[i];
    for (k=0; k<jmax; k++) {
      for (c=0,j=0; j<jmax; j++) 
	c += trf_mat[k][j] * row_buf[j];
      *row++ = c;
    }    
  }

}

/*
 *	inverse sadct transformation of block `in'.  The spatial positions
 *	of valid pels are marked in `mask' by 1.  Please note that the
 *	dct coefficients encoding those pels are expected to be found in 
 *	the upper left corner of block `in'.
 *	  
 *
 *	The following drawing explains the relation between `in', `out'
 *	and `mask':
 *
 *	in ->     I I I - - - - - 
 *	 	  I I - - - - - -
 *		  I - - - - - - -
 *		  - - ...
 *				        out ->    - - - - O - - -         
 *	mask ->   - - - - 1 - - -         	  - - O O - - - -
 *		  - - 1 1 - - - -		  - - O O - - - -
 *		  - - 1 1 - - - -		  - - - O - - - -
 *		  - - - 1 - - - -		  - - - - - - - -
 *		  - - - - - - - -		  - - ...
 *		  - - ...
 *
 */
Void
saidct_blk(Double **out, Double **in, UChar **mask, Int bky, Int bkx)
{
  Int i, j, k, jmax;
  Double **trf_mat, **dest;
  Double c;
  Double *in_ptr;

  build_v_reorder_tbl(reorder_v, l_y, out, mask, bky, bkx);
#if 0
  fprintf_dmataddr(stdout, "reorder_v (init)", reorder_v, 0, 0, bky-1, bkx-1, 
		   "%6d ", out[0]);
  fprintf_ivec(stdout, "l_y", l_y, 0, bkx-1, "%4d\n");
#endif
  build_h_reorder_tbl(reorder_h, l_x, l_y, mat_tmp1, bky, bkx);

#if 0
  fprintf_dmataddr(stdout, "reorder_h", reorder_h, 0, 0, bky-1, bkx-1, 
		   "%6d ", mat_tmp1[0]);
  fprintf_ivec(stdout, "l_x", l_x, 0, bky-1, "%4d\n");
#endif

  /* inverse horizontal transformation */
  for (i=0; i<bky && l_x[i]; i++) {
    jmax = l_x[i];
    trf_mat = idct_matrix[jmax];
    in_ptr = in[i];

    dest = reorder_h[i];
    for (k=0; k<jmax; k++) {
      for (c=0,j=0; j<jmax; j++) 
	c += trf_mat[k][j] * in_ptr[j];
      *dest[k] = c;
    }    
  }
 
  /* inverse vertical transformation */
  for (i=0; i<bkx && l_y[i]; i++) {
    jmax = l_y[i];
    trf_mat = idct_matrix[jmax];
    in_ptr = mat_tmp1[i];

    dest = reorder_v[i];
    for (k=0; k<jmax; k++) {
      for (c=0,j=0; j<jmax; j++) 
	c += trf_mat[k][j] * in_ptr[j];
      *dest[k] = c;
    }    
  }

}

/*
 *	returns in `lx' the number of active pels per line after shifting
 *	the pels towards the upper left corner.  This function may be
 *	called by a decoder in order to find out how the pels of a partial
 *	block are arranged for transmission.  
 */
Void
sadct_rowlength(Int *lx, UChar **mask, Int bky, Int bkx)
{
  Int iy_out = 0;
  Int iy, ix, l;

  for (ix=0; ix<bkx; ix++) {
    l = 0;
    for (iy=0; iy<bky; iy++) {
      if ( mask[iy][ix] ) 
	l++;
    }
    if ( l ) 
      l_y[iy_out++] = l;
  }

  for (ix=iy_out; ix<bkx; ix++) 
    l_y[ix] = 0;

  for (iy=0; iy<bky; iy++) {
    l = 0;
    for (ix=0; ix<bkx; ix++) {
      if ( l_y[ix] > iy )
	l++;
    }
    lx[iy] = l;
  }

}


/*
 *	`sadct_shiftup_transpose' shifts upwards pels marked in `mask' towards
 *	the top margin of the rectangular block.  If a column consists of more
 *	than one stretch of pels, the gaps between these stretches will be
 *	eleminated.  The resulting columns are transposed and type converted
 *	into `out' and are occupying the upper part of the block without
 *	any gaps in vertical direction.
 *
 */
Void 
sadct_shiftup_transpose(Double **out, Int *l_y, Double **in, UChar **mask, 
			Int bky, Int bkx)
{
  Int iy_out = 0, ix_out;
  Int iy, ix, l;

  for (ix=0; ix<bkx; ix++) {
    ix_out = l = 0;
    for (iy=0; iy<bky; iy++) {
      if ( mask[iy][ix] ) {
	out[iy_out][ix_out++] = in[iy][ix];
	l++;
      }
    }
    if ( l ) 
      l_y[iy_out++] = l;
  }

  /* initialize the length of the unoccupied columns to zero. The term 
     column refers to the pel positions in `in'.  In `out' columns are
     saved as rows (transposition) to speed up calculation. */
  for (ix=iy_out; ix<bkx; ix++) 
    l_y[ix] = 0;

}

Void 
build_v_reorder_tbl(Double ***tbl, Int *l_y, Double **in, UChar **mask, 
		    Int bky, Int bkx)
{
  Int iy_out = 0, ix_out;
  Int iy, ix, l;

  for (ix=0; ix<bkx; ix++) {
    ix_out = l = 0;
    for (iy=0; iy<bky; iy++) {
      if ( mask[iy][ix] ) {
	tbl[iy_out][ix_out++] = in[iy]+ix;
	l++;
      }
    }
    if ( l ) 
      l_y[iy_out++] = l;
  }

  /* initialize the length of the unoccupied columns resp. rows to zero. */
  for (ix=iy_out; ix<bkx; ix++) 
    l_y[ix] = 0;

}
 
Void
build_h_reorder_tbl(Double ***tbl, Int *l_x, Int *l_y, Double **in, Int bky,
		    Int bkx)
{
  Int i, k, jmax;
  Int col_ind;
  Double *row;

  memset(l_x, 0, sizeof(Int)*bky);
  for (i=0; i<bkx && l_y[i]; i++) {
    jmax = l_y[i];
    row = in[i];
    for (k=0; k<jmax; k++) {
      col_ind = l_x[k];
      tbl[k][col_ind] = row + k;
      l_x[k]++;
    }
  }

}
				
Void sadct_malloc(Int bksize_max)
{

  sadct_free();

  dct_matrix = sadct_malloc_internal(bksize_max);
  idct_matrix = sadct_malloc_internal(bksize_max);

  reorder_h = sadct_malloc_reorder(bksize_max);
  reorder_v = sadct_malloc_reorder(bksize_max);

  mat_tmp1 = dmatrix_sadct(0, bksize_max-1, 0, bksize_max-1);

  l_y = (Int *) ivector(0, bksize_max-1);
  l_x = (Int *) ivector(0, bksize_max-1);
  row_buf = dvector(0, bksize_max-1);

  N = bksize_max;
}


Double ***sadct_malloc_internal(Int bksize)
{
  Double ***mat_vec;
  Int i;

  mat_vec = (Double***)malloc((bksize+1)*sizeof(Double **));
  mat_vec[0] = 0;	/* precaution to cause a coredump if somebody
			   tries to use a zero length transformation */

  for (i=1; i<=bksize; i++) 
    mat_vec[i] = dmatrix_sadct(0, i-1, 0, i-1);

  return mat_vec;
}

Double ***sadct_malloc_reorder(Int bksize)
{
  Double ***reorder_tbl = (Double***)malloc(bksize*sizeof(Double**));
  Int i;

  for (i=0; i<bksize; i++) {
    reorder_tbl[i] = (Double**)calloc(bksize,sizeof(Double*));
  }
  return reorder_tbl;
}

Void sadct_free_reorder(Double ***tbl, Int n)
{
  Int i;
  for (i=0; i<n; i++) 
    free(tbl[i]);
  free(tbl);
}

Void sadct_free_internal(Double ***trfmat, Int n)
{
  Int i;

  for (i=1; i<=n; i++) 
    free_dmatrix_sadct(trfmat[i], 0, i-1, 0, i-1);

  free(trfmat);
}

Void sadct_trfmat_init(Double ***matrices, Int bksize, Double scale)
{
  Double **mat, a, factcos;
  Int u, x;
  Int n;

  for (n=1; n<=bksize; n++) {
    mat = matrices[n];
    factcos = M_PI/(2*n);
    a = (2.0 * scale)/n;
    for (u=0; u<n; u++) {
      for (x=0; x<n; x++) {
	mat[u][x] = a * cos(factcos*u*(2*x+1));
	if ( u == 0 )
	  mat[u][x] /= M_SQRT2;
      }
    }
  }
}

Void saidct_trfmat_init(Double ***matrices, Int bksize, Double scale)
{
  Double **mat, factcos;
  Int u, x;
  Int n;

  for (n=1; n<=bksize; n++) {
    mat = matrices[n];
    factcos = M_PI/(2*n);
    for (x=0; x<n; x++) {
      for (u=0; u<n; u++) {
	mat[x][u] = scale * cos(factcos*u*(2*x+1));
	if ( u == 0 )
	  mat[x][u] /= M_SQRT2;
      }
    }
  }
}


#if SADCT_DEBUG > 1
Void sadct_print_dctmat(File *fp, C_Char *s, Int n, C_Char *fmt)
{
  fprintf_dmat(fp, s, dct_matrix[n], 0, 0, n-1, n-1, fmt);
}

Void sadct_print_idctmat(File *fp, C_Char *s, Int n, C_Char *fmt)
{
  fprintf_dmat(fp, s, idct_matrix[n], 0, 0, n-1, n-1, fmt);
}
#endif

