 /***************************************************************
  COPYRIGHT (1996)  DAVID SARNOFF RESEARCH CENTER, INC.
  ALL RIGHTS RESERVED
  SARNOFF PROPRIETARY INFORMATION
  PERMISSION IS GRANTED FOR USE WITHIN MPEG4 STANDARDIZATION PROCESS
 ****************************************************************/

#include <math.h>
#include <malloc.h>
#include <stdio.h>
#include "nr_util.h"

/*****/
Void nrerror(Char *error_text)
{
    fprintf(stderr, "Numerical Recipes run-time error...\n");
    fprintf(stderr, "%s\n", error_text);
    fprintf(stderr, "...now exiting to system...\n");
    exit(1);
}

/*****/
Double *dvector(nl,nh)
Int nl,nh;
{
	Double *v;

	v=(Double *)malloc((U_Int) (nh-nl+1)*sizeof(Double));
	if (!v) nrerror("allocation failure in dvector()");
	return v-nl;
}

/*****/
Int *ivector(nl,nh)
Int nl,nh;
{
	Int *v;

	v=(Int *)malloc((U_Int) (nh-nl+1)*sizeof(Int));
	if (!v) nrerror("allocation failure in ivector()");
	return v-nl;
}

/*****/
Void lubksb(a,n,indx,b)
Double **a,b[];
Int n,*indx;
{
	Int i,ii=0,ip,j;
	Double sum;

	for (i=1;i<=n;i++) {
		ip=indx[i];
		sum=b[ip];
		b[ip]=b[i];
		if (ii)
			for (j=ii;j<=i-1;j++) sum -= a[i][j]*b[j];
		else if (sum) ii=i;
		b[i]=sum;
	}
	for (i=n;i>=1;i--) {
		sum=b[i];
		for (j=i+1;j<=n;j++) sum -= a[i][j]*b[j];
		b[i]=sum/a[i][i];
	}
}

/*****/
Void free_dvector(v,nl,nh)
Double *v;
Int nl,nh;
{
	free((Char*) (v+nl));
}

Void free_ivector(v,nl,nh)
Int *v,nl,nh;
{
	free((Char*) (v+nl));
}


#define TINY 1.0e-20;

/*****/
Void ludcmp(a,n,indx,d)
Int n,*indx;
Double **a,*d;
{
	Int i,imax=0,j,k;
	Double big,dum,sum,temp;
	Double *vv,*dvector();
	Void nrerror(),free_dvector();

	
	vv=dvector(1,n);
	*d=1.0;
	for (i=1;i<=n;i++)
		{
		big=0.0;
		for (j=1;j<=n;j++)
			if ((temp=fabs(a[i][j])) > big) big=temp; 
		if (big == 0.0) nrerror("Singular matrix in routine LUDCMP");
		vv[i]=1.0/big;
		}
		
	for (j=1;j<=n;j++) 
		{
		for (i=1;i<j;i++) 
			{
			sum=a[i][j];
			for (k=1;k<i;k++) sum -= a[i][k]*a[k][j];
			a[i][j]=sum;
			}
		big=0.0;
		for (i=j;i<=n;i++) 
			{
			sum=a[i][j];
			for (k=1;k<j;k++)
				{
				sum -= a[i][k]*a[k][j];
				}
			a[i][j]=sum;
			if ( (dum=vv[i]*fabs(sum)) >= big) 
				{
				big=dum;
				imax=i;
				}
			}
		if (j != imax) 
			{
			for (k=1;k<=n;k++) 
				{
				dum=a[imax][k];
				a[imax][k]=a[j][k];
				a[j][k]=dum;
				}
			*d = -(*d);
			vv[imax]=vv[j];
			}
		indx[j]=imax;
		if (a[j][j] == 0.0) a[j][j]=TINY; 
		if (j != n) 
			{
			dum=1.0/(a[j][j]);
			for (i=j+1;i<=n;i++) 
				{
				a[i][j] *= dum;
				}
			}
		}
	free_dvector(vv,1,n);

}

#undef TINY


/*****/

Void
invert_mat (Double **m, Double **m_inv, Int  n)
{
	Int	i,j,*indx;
	Double	det, d, *col;

	indx = ivector(0,n);
	col  =  dvector(0,n);

	ludcmp (m, n, indx, &d); 
	
	det = 1.;
	for ( i=1; i<=n; i++) det = det * m[i][i];
	if (det == 0.) printf("\n Warning : Determinant is zero \n");

	for ( j=1; j<=n; j++) {
		for ( i=1; i<=n; i++) col[i] = 0.0;
		col[j] = 1.0;
		lubksb ( m, n, indx, col);
		for ( i=1; i<=n; i++) 
			{
			m_inv[i][j]=col[i];
			}
	}

	free_ivector( indx, 0, n);
	free_dvector (  col, 0, n);
	
}

