/****************************************************************************/
/*   MPEG4 Visual Texture Coding (VTC) Mode Software                        */
/*                                                                          */
/*   This software was jointly developed by the following participants:     */
/*                                                                          */
/*   Single-quant,  multi-quant and flow control                            */
/*   are provided by  Sarnoff Corporation                                   */
/*     Iraj Sodagar   (iraj@sarnoff.com)                                    */
/*     Hung-Ju Lee    (hjlee@sarnoff.com)                                   */
/*     Paul Hatrack   (hatrack@sarnoff.com)                                 */
/*     Shipeng Li     (shipeng@sarnoff.com)                                 */
/*     Bing-Bing Chai (bchai@sarnoff.com)                                   */
/*     B.S. Srinivas  (bsrinivas@sarnoff.com)                               */
/*                                                                          */
/*   Bi-level is provided by Texas Instruments                              */
/*     Jie Liang      (liang@ti.com)                                        */
/*                                                                          */
/*   Shape Coding is provided by  OKI Electric Industry Co., Ltd.           */
/*     Zhixiong Wu    (sgo@hlabs.oki.co.jp)                                 */
/*     Yoshihiro Ueda (yueda@hlabs.oki.co.jp)                               */
/*     Toshifumi Kanamaru (kanamaru@hlabs.oki.co.jp)                        */
/*                                                                          */
/*   Scalable Shape Coding is provided by Samsung AIT			    */
/*     Dae-Sung Cho (dscho@sait.samsung.co.kr)				    */
/*     Se-Hoon Son (shson@sait.samsung.co.kr)                               */
/*									    */
/*   OKI, Sharp, Sarnoff, TI and Microsoft contributed to bitstream         */
/*   exchange and bug fixing.                                               */
/*                                                                          */
/*                                                                          */
/* In the course of development of the MPEG-4 standard, this software       */
/* module is an implementation of a part of one or more MPEG-4 tools as     */
/* specified by the MPEG-4 standard.                                        */
/*                                                                          */
/* The copyright of this software belongs to ISO/IEC. ISO/IEC gives use     */
/* of the MPEG-4 standard free license to use this  software module or      */
/* modifications thereof for hardware or software products claiming         */
/* conformance to the MPEG-4 standard.                                      */
/*                                                                          */
/* Those intending to use this software module in hardware or software      */
/* products are advised that use may infringe existing  patents. The        */
/* original developers of this software module and their companies, the     */
/* subsequent editors and their companies, and ISO/IEC have no liability    */
/* and ISO/IEC have no liability for use of this software module or         */
/* modification thereof in an implementation.                               */
/*                                                                          */
/* Permission is granted to MPEG members to use, copy, modify,              */
/* and distribute the software modules ( or portions thereof )              */
/* for standardization activity within ISO/IEC JTC1/SC29/WG11.              */
/*                                                                          */
/* Copyright 1995, 1996, 1997, 1998 ISO/IEC                                 */
/****************************************************************************/
#include <stdio.h>
#include "typedef.h"

#include "ShapeCodec.h"
#include "ShapeCommon.h"
#include "shape_def.h"
#include "ShapeUtil.h"
#include "bitpack.h"

#include "BinArCodec.h"
#include "errorHandler.h"
#include "dwt.h"

#include "idwtmask.h"

#define Error -1

/* begin : added by D.-S.Cho, Samsung AIT (99/04/13) */
Int ShapeDeCoding_V1(UChar *outmask, 
		  Int object_width,
		  Int object_height,
		  Int *constAlpha,
		  UChar *constAlphaValue)
{
  Int	change_CR_disable;
  Int	cols		= object_width;
  Int	rows		= object_height;
  Int	blkn		= 16;
  Int	blkx		= (cols+15)/16;
  Int	blky		= (rows+15)/16;
  Int	**shape_mode    = malloc_2d_Int ( blky, blkx );

  UChar **BAB		= malloc_2d_Char ( blkn, blkn );
  UChar **BABdown	= malloc_2d_Char ( blkn, blkn );
  UChar **BAB_ext	= malloc_2d_Char ( blkn+4, blkn+4 );
  UChar **shape	= malloc_2d_Char ( object_height, object_width );
  UChar **ip;

  Int	i, j, k, l, ret;
  Int	smode;
  Int	bsize;
  Int	CR, ST;

  fprintf(stderr,"Decoding Shape Header...\n");
  change_CR_disable = GetBitsFromStream(1);
  *constAlpha = GetBitsFromStream(1);
  if(*constAlpha) 
      *constAlphaValue = (UChar) GetBitsFromStream(8);

  fprintf(stderr,"Decoding Shape Layer ...\n");
  for ( j=0; j<blky; j++ ) {
      for ( i=0; i<blkx; i++ ) {

	ret = ShapeBaseLayerModeDecode(shape_mode,&CR,&ST,i,j,change_CR_disable,blkx);

        if ( ret == Error ) {
            fprintf(stderr,"\n CAE arithmetic decoding Error !\n");
            return  Error;
        }
        smode = shape_mode[j][i];

        if ( smode == ALL0 || smode == ALL255 ) {
            for ( l=0; l<blkn; l++ ) {
              for ( k=0; k<blkn; k++ ) {
	        if(j*blkn+l < object_height && i*blkn+k < object_width)
                  shape[j*blkn+l][i*blkn+k] = (Char)( smode == ALL0 ? 0 : 1 );
              }
            }
        } else {

            bsize = blkn / CR;

            for ( l=0; l<bsize; l++ ) 
              for ( k=0; k<bsize; k++ ) 
                BABdown [l][k] = (Char)0;

	    /* Add surrounding pixels for CAE. */
            AddBorderToBAB ( i, j, blkn, object_width, object_height,
				CR, blkx, BABdown, BAB_ext, shape, 2 );

	    ret = ShapeBaseLayerContentDecode(i,j,bsize,ST,BAB_ext);
            if ( ret == Error ) {
                fprintf ( stderr, "\n CAE arithmetic decoding Error !\n" );
                return  Error;
            }

	    /* make reconstructed 16x16 shape by BAB */
            for ( l=0; l<bsize; l++ ) 
              for ( k=0; k<bsize; k++ ) 
                BABdown [l][k] = BAB_ext [l+2][k+2];

            if ( CR != 1 ) {
		/* Reconstruct shape from BAB */
                UpSampling_Still  ( i, j, blkn, 
					object_width,object_height,
					CR, blkx, BABdown, BAB, shape );
            } else {
                ip = BAB;
                BAB = BABdown;
                BABdown = ip;
                ip = (UChar **)NULL;
            }

	    /* store result BAB.  */
            for ( l=0; l<blkn; l++ ) {
              for ( k=0; k<blkn; k++ ) {
	        if(j*blkn+l < object_height && i*blkn+k < object_width)
                  shape[j*blkn+l][i*blkn+k] = ( BAB[l][k] == 0 ? 0 : 1 );
              }
            }
        }
      }
  }
  ByteAlignmentDec();

  for(i=0;i<object_height;i++)
      for(j=0;j<object_width;j++)
                outmask[i*object_width+j] = shape[i][j];

  free_2d_Char ( BAB, blkn );
  free_2d_Char ( BABdown, blkn );
  free_2d_Char ( BAB_ext, blkn+4 );

  free_2d_Int  ( shape_mode, blky ); 
  free_2d_Char ( shape, object_height );

  return(0);
}
/* end : added by D.-S.Cho, Samsung AIT */

Int ShapeDeCoding(UChar *mask, 
		  Int width, 
		  Int height, 
		  Int levels,  
		  Int *targetLevel, 
		  Int *constAlpha,
		  UChar *constAlphaValue,
		  Int startCodeEnable,
		  Int fullSizeOut,
		  FILTER **filter)
{
  UChar *outmask, *recmask;
  Int k, ret;
  Int i,j;
  Int object_height, object_width;
  Int minbsize;
  Int change_CR_disable;
/*  Int shapeScalable;*/	/* FPDAM : deleted by SAIT (99/09/03)*/
  Int codedLevels;

  fprintf(stderr,"Decoding Shape Header...\n");
/* FPDAM begin: modified by SAIT (99/09/03)*/
/*  DecodeShapeHeader(&change_CR_disable, constAlpha, constAlphaValue, &shapeScalable);*/
  DecodeShapeHeader(&change_CR_disable, constAlpha, constAlphaValue);
/* FPDAM end: modified by SAIT (99/09/03)*/

/*  if(!shapeScalable) levels = 0;*/	/* FPDAM: deleted by SAIT (99/09/03)*/
  minbsize = 1<< levels;
  object_width = (width+minbsize-1)/minbsize*minbsize;
  object_height = (height+minbsize-1)/minbsize*minbsize; 
  if(object_width!=width || object_height!=height) {
    printf("Object width or height is not multiples of 2^levels\n");
    exit(1);
  }

  outmask = (UChar *)
    malloc(object_width*object_height*sizeof(UChar));
  recmask = (UChar *)
    malloc(object_width*object_height*sizeof(UChar));

  if(outmask == NULL || recmask == NULL) {
    errorHandler("Memory allocation failed\n");
  }

  fprintf(stderr,"Decoding Shape Base Layer ...\n");
  DecodeShapeBaseLayer(outmask, change_CR_disable, object_width, object_height, levels);

/*  if(shapeScalable) {	*/	/* FPDAM: deleted by SAIT (99/09/03)*/
    ret = 0;
    if(!startCodeEnable) {
      codedLevels = GetBitsFromStream(4);
      if(GetBitsFromStream(1)!=MARKER_BIT)
	errorHandler("Incorrect Marker bit in shape enhanced layer decoding.\n");
        *targetLevel = levels-codedLevels;
    }
    if(*targetLevel <0) *targetLevel =0;

    for(k=levels;k>*targetLevel;k--) {
      fprintf(stderr,"Decoding Shape Enhanced Layer %d...\n", k);
      ret=DecodeShapeEnhancedLayer(outmask, object_width, object_height, k,
                        	startCodeEnable, filter[k-1]);
      if(startCodeEnable && ret) break; /* end of enhanced layer */
    }
    if(startCodeEnable) {
      Int code32=0;
      *targetLevel = k;
      if(!ret) ByteAlignmentDec();
      /* search for texture layer start code */
      code32 = GetBitsFromStream(32);
      
      while(code32!=TEXTURE_SPATIAL_START_CODE) {
	code32 = (code32 <<8)|  GetBitsFromStream(8);
      }
      GetBitsFromStream(5);
      if(GetBitsFromStream(1)!= MARKER_BIT) 
	errorHandler("Incorrect Marker bit in the end of Shape Bitstream\n");
    }
/* FPDAM begin : deleted by SAIT (99/09/03)*/
/* }
  else {
    *targetLevel = 0;
  } */
/* FPDAM end : deleted by SAIT (99/09/03)*/

  if(fullSizeOut) { /* interpolate shape to full size */
    for(i=0;i<(height>>*targetLevel);i++)
      for(j=0;j<(width>>*targetLevel);j++)
        recmask[i*object_width+j]=outmask[i*(width>>*targetLevel)+j];
    do_iDWTMask(recmask, mask, width, height, *targetLevel,
	 *targetLevel, filter, 0, 1);
  } else {
    for(i=0;i<(height>>*targetLevel);i++)
      for(j=0;j<(width>>*targetLevel);j++)
        mask[i*(width>>*targetLevel)+j] = outmask[i*(width>>*targetLevel)+j];
  }    
  free(outmask);
  free(recmask);

  return(0);
}

/* FPDAM begin: modified by SAIT (99/09/03)*/
/* Int DecodeShapeHeader(Int *change_CR_disable,
			Int *constAlpha, 
			UChar *constAlphaValue, 
		      	Int *shapeScalable)*/
Int DecodeShapeHeader(Int *change_CR_disable,
			Int *constAlpha, 
			UChar *constAlphaValue)
/* FPDAM end: modified by SAIT (99/09/03)*/
{   
    *change_CR_disable = GetBitsFromStream(1);
    *constAlpha = GetBitsFromStream(1);
    if(*constAlpha) 
      *constAlphaValue = (UChar) GetBitsFromStream(8);
/*    *shapeScalable = GetBitsFromStream(1);*/	/* FPDAM : deleted by SAIT (99/09/03)*/
    if(GetBitsFromStream(1)!=MARKER_BIT)
      errorHandler("Incorrect Marker bit in header decoding.\n");
    return(0);
}

Int
DecodeShapeBaseLayer(UChar *outmask, 
			Int change_CR_disable,
			Int object_width, 
			Int object_height,
			Int levels)
{
    Int coded_width     = object_width >> levels;
    Int coded_height    = object_height >> levels;

    Int	cols		= coded_width;
    Int	rows		= coded_height;
    Int	blkn		= 16;
    Int	blkx		= (cols+15)/16;
    Int	blky		= (rows+15)/16;
    Int **shape_mode    = malloc_2d_Int ( blky, blkx );

    UChar **BAB		= malloc_2d_Char ( blkn, blkn );
    UChar **BABdown	= malloc_2d_Char ( blkn, blkn );
    UChar **BAB_ext	= malloc_2d_Char ( blkn+4, blkn+4 );
    UChar **shape	= malloc_2d_Char ( coded_height, coded_width );
    UChar **ip;

    Int	i, j, k, l, ret;
    Int	smode;
    Int	bsize;
    Int	CR, ST;

    for ( j=0; j<blky; j++ ) {
      for ( i=0; i<blkx; i++ ) {

	ret = ShapeBaseLayerModeDecode(shape_mode,&CR,&ST,i,j,change_CR_disable,blkx);

        if ( ret == Error ) {
            fprintf(stderr,"\n CAE arithmetic decoding Error !\n");
            return  Error;
        }
        smode = shape_mode[j][i];

        if ( smode == ALL0 || smode == ALL255 ) {
            for ( l=0; l<blkn; l++ ) {
              for ( k=0; k<blkn; k++ ) {
	        if(j*blkn+l < coded_height && i*blkn+k < coded_width)
                  shape[j*blkn+l][i*blkn+k] = (Char)( smode == ALL0 ? 0 : 1 );
              }
            }
        } else {

            bsize = blkn / CR;

            for ( l=0; l<bsize; l++ ) 
              for ( k=0; k<bsize; k++ ) 
                BABdown [l][k] = (Char)0;

	    /* Add surrounding pixels for CAE. */
            AddBorderToBAB ( i, j, blkn, coded_width, coded_height,
				CR, blkx, BABdown, BAB_ext, shape, 2 );

	    ret = ShapeBaseLayerContentDecode(i,j,bsize,ST,BAB_ext);
            if ( ret == Error ) {
                fprintf ( stderr, "\n CAE arithmetic decoding Error !\n" );
                return  Error;
            }

	    /* make reconstructed 16x16 shape by BAB */
            for ( l=0; l<bsize; l++ ) 
              for ( k=0; k<bsize; k++ ) 
                BABdown [l][k] = BAB_ext [l+2][k+2];

            if ( CR != 1 ) {
		/* Reconstruct shape from BAB */
                UpSampling_Still  ( i, j, blkn, 
					coded_width,coded_height,
					CR, blkx, BABdown, BAB, shape );
            } else {
                ip = BAB;
                BAB = BABdown;
                BABdown = ip;
                ip = (UChar **)NULL;
            }

	    /* store result BAB.  */
            for ( l=0; l<blkn; l++ ) {
              for ( k=0; k<blkn; k++ ) {
	        if(j*blkn+l < coded_height && i*blkn+k < coded_width)
                  shape[j*blkn+l][i*blkn+k] = ( BAB[l][k] == 0 ? 0 : 1 );
              }
            }
        }
      }
    }
    for(i=0;i<coded_height;i++)
        for(j=0;j<coded_width;j++)
                outmask[i*coded_width+j] = shape[i][j];

    if(GetBitsFromStream(1)!=MARKER_BIT)
        errorHandler("Incorrect Marker bit in shape base layer decoding.\n");

    free_2d_Char ( BAB, blkn );
    free_2d_Char ( BABdown, blkn );
    free_2d_Char ( BAB_ext, blkn+4 );

    free_2d_Int  ( shape_mode, blky ); 
    free_2d_Char ( shape, coded_height );

/*    free(shape);*/
    return ( 0 );
}

Int
ShapeBaseLayerModeDecode(Int **shape_mode,
			Int *CR_store,
			Int *ST_store,
			Int i,Int j,
			Int change_CR_disable,
			Int blkx)
{
    Int	dis_CR = change_CR_disable;
    Int	UL= ( (i == 0      || j == 0) ? 0 : shape_mode[j-1][i-1] );
    Int	UR= ( (i == blkx-1 || j == 0) ? 0 : shape_mode[j-1][i+1] );
    Int	U = ( (j == 0) ? 0 : shape_mode[j-1][i] );
    Int	L = ( (i == 0) ? 0 : shape_mode[j][i-1] );
    Int	index = ( 27 * UL + 9 * U + 3 * UR + L );

    Int	mode = 0;
    Int	CR = 1;
    U_Int code = 0;
    Int	bits;

    do {
        bits = LMMR_first_shape_code_I [index][mode];
        code = LookBitsFromStream ( bits);
    } while ( code != CMMR_first_shape_code_I [index][mode] && (++mode) <= 2 );
	
    if ( code == CMMR_first_shape_code_I [index][mode] ) {

        BitstreamFlushBits_Still( bits);

        if ( mode == BORDER ) {
            if ( dis_CR == 0 ) {
                do {
                    bits = LCR [CR];
                    code = LookBitsFromStream ( bits );

                } while ( code != CCR[CR] && (CR=(CR<<1)) <= 4 );
	
                if ( code == CCR [CR] ) {
                    BitstreamFlushBits_Still( bits);
                } else {
                    fprintf(stderr,"\n Decode shape information CR Error at [%d, %d] \n", i, j );
                    return  Error;
                }
            }
            bits = LST;
            code = GetBitsFromStream ( bits);
        }
        shape_mode[j][i] = mode;
	*CR_store	= CR;
	*ST_store	= (Int)code;
    } else {
        fprintf ( stderr, "\n Decode shape mode Error at [%d, %d] \n", i, j );
        return  Error;
    }

    return ( 0 );
}

Int
ShapeBaseLayerContentDecode(Int i,Int j,Int bsize,Int ST,UChar **BAB)
{
    Int	k, l;
    Int	dir = ST;
    Int	context;
    Int	ctx_max = 1023;
    ArDecoder decoder;

    StartArDecoder_Still( &decoder);

    if ( dir == 0 ) {

        for ( l=0; l<bsize; l++ ) {
            for ( k=0; k<bsize; k++ ) {
                context = ( BAB [l+2+(+0)][k+2+(-1)] << 0 ) +
                          ( BAB [l+2+(+0)][k+2+(-2)] << 1 ) +
                          ( BAB [l+2+(-1)][k+2+(+2)] << 2 ) +
                          ( BAB [l+2+(-1)][k+2+(+1)] << 3 ) +
                          ( BAB [l+2+(-1)][k+2+(+0)] << 4 ) +
                          ( BAB [l+2+(-1)][k+2+(-1)] << 5 ) +
                          ( BAB [l+2+(-1)][k+2+(-2)] << 6 ) +
                          ( BAB [l+2+(-2)][k+2+(+1)] << 7 ) +
                          ( BAB [l+2+(-2)][k+2+(+0)] << 8 ) +
                          ( BAB [l+2+(-2)][k+2+(-1)] << 9 );

                if ( context > ctx_max ) {
                    fprintf ( stderr, "\n Shape context Error !\n" );
                    return  Error;
                }
                BAB [l+2][k+2] = ArDecodeSymbol_Still( &decoder, intra_prob[context] );
            }
            BAB [l+2][bsize+2] = BAB [l+2][bsize+3] = BAB [l+2][bsize+1];
        }

    } else {

        for ( k=0; k<bsize; k++ ) {
            for ( l=0; l<bsize; l++ ) {
                context = ( BAB [l+2+(-1)][k+2+(+0)] << 0 ) +
                          ( BAB [l+2+(-2)][k+2+(+0)] << 1 ) +
                          ( BAB [l+2+(+2)][k+2+(-1)] << 2 ) +
                          ( BAB [l+2+(+1)][k+2+(-1)] << 3 ) +
                          ( BAB [l+2+(+0)][k+2+(-1)] << 4 ) +
                          ( BAB [l+2+(-1)][k+2+(-1)] << 5 ) +
                          ( BAB [l+2+(-2)][k+2+(-1)] << 6 ) +
                          ( BAB [l+2+(+1)][k+2+(-2)] << 7 ) +
                          ( BAB [l+2+(+0)][k+2+(-2)] << 8 ) +
                          ( BAB [l+2+(-1)][k+2+(-2)] << 9 );

                if ( context > ctx_max ) {
                    fprintf ( stderr, "\n Shape context Error !\n" );
                    return  Error;
                }
                BAB [l+2][k+2] = ArDecodeSymbol_Still( &decoder, intra_prob[context] );
            }
            BAB [bsize+2][k+2] = BAB [bsize+3][k+2] = BAB [bsize+1][k+2];
        }

    }

    StopArDecoder_Still ( &decoder);

    return ( 0 );
}

/***********************************************************CommentBegin******
 *
 * -- DecodeShapeEnhancedLayer -- Decodes a binary shape in the enhancement layer.
 *
 * Author :
 *      Dae-Sung Cho (Samsung AIT)
 *
 * Created :
 *      02-Dec-98
 *
 * Arguments:
 *
 *
 * Return values :
 *
 * Side effects :
 *      -
 *
 * Description :        Alpha blocks in the enhancement layer will be coded by bac.
 *                      For the bab coding there are two kinds of coding modes
 *                      for the coded bab, i.e. transitional bab and exceptional bab coding.
 *                      - The transitional bab coding uses scan interleaving (SI) method
 *                        proposed by Samsung AIT. 
 *                      - The exceptional bab coding is performed by the method proposed 
 *                        by Sarnoff Corp.
 *
 * See also :
 *
 ***********************************************************CommentEnd********/

Int DecodeShapeEnhancedLayer(UChar *outmask,	/* shape mask in the lower layer (k) */
                        Int object_width,
                        Int object_height,
			Int levels,
			Int startCodeEnable,
			FILTER *filter)
{
    Int i, j, k, l, p, q, x, y, x2, y2;

    Int bab_type, ret;

    Int coded_width     = object_width >> (levels-1),
	coded_height    = object_height >> (levels-1);

    Int width2          = coded_width,
	height2         = coded_height,
	width           = width2 >> 1,
	height          = height2 >> 1;

    Int NB		= (coded_width>=1024 || coded_height>=1024) ? 2 :
                          (coded_width>=512 || coded_height>=512) ? 1: 0;

    Int mborder		= MBORDER,
    	mblks           = 4+NB,
	mbsize          = 1<<mblks,		/* bab size in the current layer : 16 */
	mbsize_ext      = mbsize+(mborder<<1);  /* bordered bab size in the current layer: 20 */
 
    Int border		= BBORDER,
    	blks            = mblks-1,
	bsize           = 1<<blks,		/* bab size in the lower layer : 8 */
	bsize_ext       = bsize+(border<<1);	/* bordered bab size in the lower layer: 8 */
 
    Int blkx            = (coded_width+mbsize-1)/mbsize,
	blky            = (coded_height+mbsize-1)/mbsize;

    UChar *low_mask;
    UChar *half_mask;                    /* shape mask at the half-higher layer */
    UChar *cur_mask;

    UChar *lower_bab;			/* alpha block in the lower layer */
    UChar *bordered_lower_bab;		/* bordered alpha block in the lower layer */
    UChar *half_bab;			/* alpha block in the half-higher layer */
    UChar *bordered_half_bab;		/* bordered alpha block in the half-higher layer */
    UChar *curr_bab;			/* alpha mb in the current layer */
    UChar *bordered_curr_bab;		/* bordered alpha mb in the current layer */

    ArDecoder	decoder;

    /* Memory allocation */ 
    low_mask = (UChar *) calloc(width*height, sizeof(UChar));
    half_mask = (UChar *) calloc(width*height2, sizeof(UChar));
    cur_mask = (UChar *) calloc(width2*height2, sizeof(UChar));

    lower_bab = (UChar *) calloc(bsize*bsize, sizeof(UChar));
    bordered_lower_bab = (UChar *) calloc(bsize_ext*bsize_ext, sizeof(UChar));
    half_bab = (UChar *) calloc(bsize*mbsize, sizeof(UChar));
    bordered_half_bab = (UChar *) calloc(bsize_ext*mbsize_ext, sizeof(UChar));
    curr_bab = (UChar *) calloc(mbsize*mbsize, sizeof(UChar));
    bordered_curr_bab = (UChar *) calloc(mbsize_ext*mbsize_ext, sizeof(UChar));

    for(j=0;j<height;j++)
	for(i=0;i<width;i++) *(low_mask+j*width+i)= *(outmask+j*width+i);

    /* Decode the Enhancement Layer */
    if(startCodeEnable) {
	ByteAlignmentDec();
 
    	if(LookBitsFromStream(32)!=TEXTURE_SHAPE_START_CODE) return(1);
    	GetBitsFromStream(32);
    	GetBitsFromStream(5);
    	if(GetBitsFromStream(1)!=MARKER_BIT)
      		errorHandler("Incorrect Marker bit in shape enhanced layer decoding.\n");
    }

    StartArDecoder_Still(&decoder);

    for ( j=y=y2=0; j<blky; j++, y+=bsize, y2+=mbsize ) {
      for ( i=x=x2=0; i<blkx; i++, x+=bsize, x2+=mbsize ) {
        /* Initialize BABs */ 
	q = y*width;
	for ( l=p=0; l<bsize; l++, q+=width ) {
          for ( k=0; k<bsize; k++, p++ ) {
	      if(  y+l < height && x+k < width )
		lower_bab[p] = (low_mask[ q+x+k ] != 0);
	      else
		lower_bab[p] = 0;
          }
	}
	for ( l=p=0; l<mbsize; l++) 
          for ( k=0; k<bsize; k++, p++ ) half_bab[p]=0;

	for ( l=p=0; l<mbsize; l++ ) 
          for ( k=0; k<mbsize; k++, p++ ) curr_bab[p]=0;

        /* Add surrounding pixels for BAB in the lower layer */
	AddBorderToLowerBAB(low_mask,
				lower_bab, bordered_lower_bab,
				width, height,
				i, j, bsize, blkx);
        /* Add surrounding pixels for BAB in the current layer */
	AddBorderToEnhBAB(low_mask, cur_mask,
				curr_bab, bordered_curr_bab,
				coded_width, coded_height,
				i, j, mbsize, blkx);

	/* Decode Bab Mode in the enhanced layer */
	bab_type = DecodeEnhLayerBabType(&decoder, filter);

	if(bab_type==EXCEPTIONAL) {
        	/* Add surrounding pixels for BAB in the half layer */
		AddBorderToHalfHigherBAB(low_mask, half_mask,
				half_bab, bordered_half_bab,
				coded_width, coded_height,
				i, j, mbsize, blkx);
	}

	/* Decode mask pixel values in the enhanced layer */
	ret = DecodeEnhancedLayerBAB(&decoder,
					bordered_lower_bab, 
					bordered_half_bab, 
					bordered_curr_bab, 
					bab_type,
					mbsize, 
					filter);
	if (ret == Error) {
          fprintf(stderr,"\n SI arithmetic coding Error !\n");
          return  Error;
        }

	q = mborder*bsize_ext;
	for ( l=p=0; l<mbsize; l++, q+=bsize_ext ) 
          for ( k=0; k<bsize; k++, p++ ) 
	      	half_bab[p]=bordered_half_bab[q+k+border];

	q = mborder*mbsize_ext;
	for ( l=p=0; l<mbsize; l++, q+=mbsize_ext ) 
          for ( k=0; k<mbsize; k++, p++ ) 
	      	curr_bab[p]=bordered_curr_bab[q+k+mborder];

        /* put half-higher BABs to half_mask */
	q = y2*width;
	for ( l=p=0; l<mbsize; l++, q+=width ) 
          for ( k=0; k<bsize; k++, p++ ) 
	     if(  y2+l < height2 && x+k < width )
		half_mask[ q+x+k ] = half_bab[p];

        /* put BABs to cur_mask */
	q = y2*width2;
	for ( l=p=0; l<mbsize; l++, q+=width2 ) 
          for ( k=0; k<mbsize; k++, p++ ) 
	      if(  y2+l < height2 && x2+k < width2 )
		cur_mask[ q+x2+k ] = curr_bab[p];
      }
    }

    for(j=0;j<height2;j++)
      for(i=0; i<width2;i++) outmask[j*width2+i]= cur_mask[j*width2+i];

    StopArDecoder_Still (&decoder);

    if(GetBitsFromStream(1)!=MARKER_BIT)
    	errorHandler("Incorrect Marker bit in shape enhanced layer decoding.\n");

    /* Memory free */
    free(low_mask);
    free(half_mask);
    free(cur_mask);

    free(lower_bab);
    free(bordered_lower_bab);
    free(half_bab);
    free(bordered_half_bab);
    free(curr_bab);
    free(bordered_curr_bab);

    return ( 0 );
}

Int DecodeEnhLayerBabType(ArDecoder *decoder, FILTER *filter)
{
	Int prob, bab_type;

	if(filter->Class==ODD_SYMMETRIC) {
		prob=scalable_bab_type_prob[0];
	} else if(filter->Class==EVEN_SYMMETRIC) {
		prob=scalable_bab_type_prob[1];
	} else {
		fprintf(stderr,"Error: filter type in BAB_TYPE decoding()!\n");
		exit(0);
	}
	bab_type = ArDecodeSymbol_Still(decoder, prob);

	return bab_type;
}

/* Modified by shson */
Int DecodeEnhancedLayerBAB(ArDecoder *decoder,
				UChar *bordered_lower_bab, 
				UChar *bordered_half_bab,
				UChar *bordered_curr_bab, 
				Int bab_type,
				Int mbsize,
				FILTER *filter)
{
	Int scan_order;

	if (bab_type==TRANSITIONAL) {		/* Transitional BAB decoding */
                scan_order = DecideScanOrder(bordered_lower_bab, mbsize);
		DecodeTransitionalBAB (decoder,
					bordered_lower_bab,
					bordered_half_bab,
					bordered_curr_bab,
					mbsize,
					scan_order);
        
	} else if (bab_type==EXCEPTIONAL) {	/* Exceptional BAB decoding */
		DecodeExceptionalBAB (decoder,
					bordered_lower_bab,
					bordered_half_bab,
					bordered_curr_bab,
					mbsize,
					filter);
	} else {
		fprintf(stderr,"BAB type ERROR !\n");
	}
		
	return ( 0 );
}

/* Transitional BAB decoding: Proposed by Samsung AIT */
Void DecodeTransitionalBAB (ArDecoder *decoder,
				UChar *bordered_lower_bab,
				UChar *bordered_half_bab,
				UChar *bordered_curr_bab,
				Int mbsize,
				Int scan_order)
{
	Int		i,j,i2,j2,k,l,prev,next;

	Int		mborder = MBORDER;
	Int		mbsize_ext = mbsize+(mborder<<1);
	Int		border = BBORDER;
	Int		bsize = mbsize >> 1;
	Int		bsize_ext = bsize+(border<<1);

	Int		context, prob=0;

	UChar		*lower_bab_data,
			*half_bab_data,
			*curr_bab_data;
        UChar           *bordered_curr_bab_tr;

	lower_bab_data = bordered_lower_bab + border * bsize_ext + border;
	half_bab_data= bordered_half_bab + mborder * bsize_ext + border;
	curr_bab_data= bordered_curr_bab + mborder * mbsize_ext + mborder;

	/* P0 pixels reconstructing */
        for(j=j2=0,k=l=0; j<bsize; j++, j2+=2, k+=(mbsize_ext<<1), l+=bsize_ext)
	   for(i=i2=0; i<bsize; i++, i2+=2)
		curr_bab_data[k+i2]=lower_bab_data[l+i];

        /*  Current BAB transposing */
        if (scan_order == 1) {
                bordered_curr_bab_tr = (UChar*)calloc(mbsize_ext*mbsize_ext, sizeof(UChar));
                for(j=0; j<mbsize_ext; j++)
                   for(i=0; i<mbsize_ext; i++)
                        bordered_curr_bab_tr[j*mbsize_ext+i]=bordered_curr_bab[i*mbsize_ext+j];
                curr_bab_data= bordered_curr_bab_tr + mborder * mbsize_ext + mborder;
        }

	/* P1 pixels decoding: vertical scanning of the pixel */
	for(i2=1; i2<mbsize; i2+=2) {
	   for(j2=k=0; j2<mbsize; j2+=2, k+=(mbsize_ext<<1) ) {
		prev= curr_bab_data[k+i2-1];
		next= curr_bab_data[k+i2+1];
                       
		if(prev!=next) {
		   context = GetTransitionalBabContext(curr_bab_data,
							i2, 
							j2, 
							mbsize_ext, 
							0); /* pixel type : 0-P1, 1-P2/P3 */

		   prob=scalable_xor_prob_1[context];
		   curr_bab_data[k+i2] = ArDecodeSymbol_Still(decoder, prob);

		} else {
		   curr_bab_data[k+i2] = prev;
		}
	   }
	}

	/* P2/P3 pixels decoding: horizontal scanning of the pixels */
        for(j2=1,k=mbsize_ext; j2<mbsize; j2+=2, k+=(mbsize_ext<<1) ) {
	   for(i2=0; i2<mbsize; i2++) {
		prev= curr_bab_data[k-mbsize_ext+i2];
		next= curr_bab_data[k+mbsize_ext+i2];
		   
		if(prev!=next) {
		   context = GetTransitionalBabContext(curr_bab_data,
							i2, 
							j2, 
							mbsize_ext, 
							1); /* pixel type : 0-P1, 1-P2/P3 */
		   	
		   prob=scalable_xor_prob_23[context];
		   curr_bab_data[k+i2] = ArDecodeSymbol_Still(decoder, prob);
		} else {
		   curr_bab_data[k+i2] = prev;
		}
	   }
	}

        if (scan_order == 1) {
                for(j=0; j<mbsize_ext; j++)
                   for(i=0; i<mbsize_ext; i++)
                        bordered_curr_bab[j*mbsize_ext+i]=bordered_curr_bab_tr[i*mbsize_ext+j];
                free (bordered_curr_bab_tr);
        }

        /* get the half resolution needed */
	if (scan_order==1) curr_bab_data= bordered_curr_bab + mborder * mbsize_ext + mborder;
	for(j2=0,k=0; j2<mbsize; j2++, k+=mbsize_ext ) {
            for(i=0; i<bsize; i++) {
              i2 = i<<1; /* horizontal subsampling */
              half_bab_data[j2*bsize_ext+i] = curr_bab_data[k+i2];
            }
	}
}

/* Exceptional BAB decoding: Proposed by Sarnoff Corp. */
Void DecodeExceptionalBAB (ArDecoder *decoder,
				UChar *bordered_lower_bab,
				UChar *bordered_half_bab,
				UChar *bordered_curr_bab,
				Int mbsize,
				FILTER *filter)
{
	Int		i,j,i2,j2,k,m;

	Int		mborder = MBORDER;
	Int		mbsize_ext = mbsize+(mborder<<1);
	Int		border = BBORDER;
	Int		bsize = mbsize >> 1;
	Int		bsize_ext = bsize+(border<<1);
        Int             half_value, curr_value;
	Int		context, prob=0;
	UChar		*lower_bab_data,
			*half_bab_data,           /* half higher bab data */
			*curr_bab_data;
 
	lower_bab_data = bordered_lower_bab + border * bsize_ext + border;
        half_bab_data = bordered_half_bab + mborder * bsize_ext + border;
	curr_bab_data= bordered_curr_bab + mborder * mbsize_ext + mborder;

	/* First pass: horizontal scanning of 1x2 block */
	for(i2=k=0; i2<mbsize; i2+=2, k+=(bsize_ext<<1)) {
	   for(j=0; j<bsize; j++) {
		for(m=0; m<2; m++) {
			context = GetExceptionalBabContext_HalfHigher(lower_bab_data,
							half_bab_data,
							j,
							i2+m,
							bsize_ext,
							bsize_ext,
							m);
 
			if(filter->Class==ODD_SYMMETRIC) {
			   prob= (m==0) ? sto_enh_odd_prob0[context]
					   : sto_enh_odd_prob1[context];
			} else if(filter->Class==EVEN_SYMMETRIC) {
			   prob= (m==0) ? sto_enh_even_prob0[context]
					   : sto_enh_even_prob1[context];
			} else {
			   fprintf(stderr,"Error: filter type in RasterScanLineEncode() !\n");
			   exit(0);
			}

			if(prob!=0 && prob!=65536 && prob!=65537) {
			    half_value = ArDecodeSymbol_Still(decoder, prob);
			} else {
                            if(prob==0)		 half_value = 1;
			    else if(prob==65536) half_value = 0;
			    else if(prob==65537) {
                                fprintf(stderr,"Error: Not occured probability(=65537) for coded pixels !\n");
                                exit(0);
                            }
                        }

			if(m==0) half_bab_data[k+j]=half_value;		/* T1 pixel */
		        else     half_bab_data[k+bsize_ext+j]=half_value;	/* T0 pixel */
		}
	   }
	}

	/* Second pass: vertical scanning of 2x1 block */
	for(j2=0; j2<mbsize; j2+=2) {
	   for(i=k=0; i<mbsize; i++, k+=mbsize_ext) {
		for(m=0; m<2; m++) {
			context = GetExceptionalBabContext(half_bab_data,
                                			curr_bab_data,
                                			j2+m,
                                			i,
                                			bsize_ext,
							mbsize_ext,
                                			m); 
		   
			if(filter->Class==ODD_SYMMETRIC) {
		   	   prob= (m==0) ? sto_enh_odd_prob0[context] 
					   : sto_enh_odd_prob1[context];
			} else if(filter->Class==EVEN_SYMMETRIC) {
		   	   prob= (m==0) ? sto_enh_even_prob0[context]
					   : sto_enh_even_prob1[context];
			} else {
		   	   fprintf(stderr,"Error: filter type in RasterScanLineEncode() !\n");
		   	   exit(0);
			}

			if(prob!=0 && prob!=65536 && prob!=65537) {
			    curr_value = ArDecodeSymbol_Still(decoder, prob);
			} else {
                            if(prob==0)		 curr_value = 1;
			    else if(prob==65536) curr_value = 0;
			    else if(prob==65537) {
                                fprintf(stderr,"Error: Not occured probability(=65537) for coded pixels !\n");
                                exit(0);
                            }
			}
			curr_bab_data[k+j2+m]=curr_value;
		}
	   }
	}
}

