#include <stdio.h>
#include <stdlib.h>
#include <math.h>

typedef unsigned char	UChar;

#define pxlcObjColor	255
#define opaqueValue	255
#define transpValue	0

#define	DOWN_SAMPLING	0
#define	UP_SAMPLING	1
#define	NO_CONV		2

void copyright_statement();

void print_usage (char *name);

void get_parameter (char *pfile,
                char *srcFile,
                char *dstFile,
                int *widthSrc,
                int *heightSrc,
		int *widthDst,
		int *heightDst,
		int *shape_ver_sampling_factor_m,
		int *shape_ver_sampling_factor_n,
		int *shape_hor_sampling_factor_m,
		int *shape_hor_sampling_factor_n,
                int *frame_no);

void up_down_sample_alpha( UChar *alpha_src,  
				UChar *alpha_dst, 
				int shape_ver_sampling_factor_m,
				int shape_ver_sampling_factor_n,
				int shape_hor_sampling_factor_m,
				int shape_hor_sampling_factor_n,
				int widthSrc,
				int heightSrc,
				int widthDst,
				int heightDst);

void main(int argc, char* argv[])
{
   int t,frame_no=0;
   int image_format;
   int heightSrc=0,widthSrc=0,heightDst=0,widthDst=0;
   int shape_ver_sampling_factor_m=0, 
	shape_ver_sampling_factor_n=0, 
	shape_hor_sampling_factor_m=0, 
	shape_hor_sampling_factor_n=0; 
   char  srcFile[80], fnameA1[80], fnameU1[80], fnameV1[80];
   char  dstFile[80], fnameA2[80], fnameU2[80], fnameV2[80];
   UChar *alpha_src;
   UChar *alpha_dst;
   FILE  *fpSrc, *fpU1, *fpV1, *fpYUV1;
   FILE  *fpDst, *fpU2, *fpV2, *fpYUV2;

   if(argc!=2) {
      print_usage(argv[0]);
      exit(1);
   }

   get_parameter (argv[1], 
		srcFile, 
		dstFile, 
                &widthSrc,
                &heightSrc,
		&widthDst,
		&heightDst,
		&shape_ver_sampling_factor_m,
		&shape_ver_sampling_factor_n,
		&shape_hor_sampling_factor_m,
		&shape_hor_sampling_factor_n,
                &frame_no);
   
   /* Memory allocation */
   alpha_src = (UChar *) calloc(widthSrc*heightSrc, sizeof(UChar));
   alpha_dst = (UChar *) calloc(widthDst*heightDst, sizeof(UChar));
   if (alpha_src==NULL || alpha_dst==NULL) {
        fprintf(stderr,"Error: Memoray allocation failure !!!\n");
        exit(1);
   }
 
   /* File open */
   fprintf(stderr," Source alpha : %s\n", srcFile);
   fprintf(stderr," Destination alpha : %s\n\n", dstFile);
   fpSrc = fopen(srcFile, "r");
   fpDst = fopen(dstFile, "w");
   if (fpSrc==NULL || fpDst==NULL) {
        fprintf(stderr,"Error: input files (%s or %s) does not exist !!!\n", srcFile, dstFile);
        exit(1);
   }

   for(t=0;t<frame_no;t++) {
     fprintf(stderr,"Frame:%d\r",t);

     /* alpha reading */
     fread((char *)alpha_src, widthSrc*heightSrc, sizeof(UChar), fpSrc);
		
     up_down_sample_alpha (alpha_src, alpha_dst,
			shape_ver_sampling_factor_m, 
			shape_ver_sampling_factor_n, 
			shape_hor_sampling_factor_m, 
			shape_hor_sampling_factor_n,
			widthSrc,
			heightSrc,
			widthDst,
			heightDst
			);
	
     /* alpha writing */
     fwrite((char *)alpha_dst, widthDst*heightDst, sizeof(UChar), fpDst);
   }

   /* Memory free */
   free(alpha_src);
   free(alpha_dst);
 
   /* File close */
   fclose(fpSrc);
   fclose(fpDst);

   fprintf(stderr,"Processed Up-Down Sampling !\n");
}

void copyright_statement()
{
   fprintf(stderr," Copyright 1999 Samsung Advanced Institute of Technology (SAIT).\n");
   fprintf(stderr," All Rights Reserved.\n");
   fprintf(stderr," Originally Developed by Dae-Sung Cho (dscho@sait.samsung.co.kr).\n");
   fprintf(stderr,"\n");
}
 
void print_usage (char *name)
{
   copyright_statement();
 
   fprintf(stderr," %s <parameter file>\n\n", name);
   fprintf(stderr,"<input alpha file>\n");
   fprintf(stderr,"<output alpha file>\n");
   fprintf(stderr,"<width of input image>\n");
   fprintf(stderr,"<height of input image>\n");
   fprintf(stderr,"<width of output image>\n");
   fprintf(stderr,"<height of output image>\n");
   fprintf(stderr,"<frame number>\n");
   fprintf(stderr,"<Hor. sampling factor n (numerator of hor. sampling ratio)>\n");
   fprintf(stderr,"<Hor. sampling factor m (denominator of hor. sampling ratio)>\n");
   fprintf(stderr,"<Ver. sampling factor n (numerator of ver. sampling ratio)>\n");
   fprintf(stderr,"<Ver. sampling factor m (denominator of ver. sampling ratio)>\n");
   
   fprintf(stderr,"\n");
   return;
}

void get_parameter (char *pfile,
                char *srcFile,
                char *dstFile,
                int *widthSrc,
                int *heightSrc,
		int *widthDst,
		int *heightDst,
		int *shape_ver_sampling_factor_m,
		int *shape_ver_sampling_factor_n,
		int *shape_hor_sampling_factor_m,
		int *shape_hor_sampling_factor_n,
                int *frame_no)
{
  
   char garbage[201];
   FILE *fp;
 
   fp = fopen(pfile,"r");
   if(fp==NULL) {
        fprintf(stderr,"Error: %s file does not exist !!!\n", pfile);
        exit(1);
   }
  
   fscanf(fp, "%s", srcFile);
   fgets(garbage,201,fp);
   fscanf(fp, "%s", dstFile);
   fgets(garbage,201,fp);
   fscanf(fp, "%d", widthSrc);
   fgets(garbage,201,fp);
   fscanf(fp, "%d", heightSrc);
   fgets(garbage,201,fp);
   fscanf(fp, "%d", widthDst);
   fgets(garbage,201,fp);
   fscanf(fp, "%d", heightDst);
   fgets(garbage,201,fp);
   fscanf(fp,"%d", frame_no);
   fgets(garbage,201,fp);
   fscanf(fp,"%d", shape_hor_sampling_factor_n);
   fgets(garbage,201,fp);
   fscanf(fp,"%d", shape_hor_sampling_factor_m);
   fgets(garbage,201,fp);
   fscanf(fp,"%d", shape_ver_sampling_factor_n);
   fgets(garbage,201,fp);
   fscanf(fp,"%d", shape_ver_sampling_factor_m);
   fgets(garbage,201,fp);

   if(shape_hor_sampling_factor_n>shape_hor_sampling_factor_m) {
	if(widthSrc > widthDst) {
		fprintf(stderr,"Error: Incorrect Hor. sampling ratio !");
		exit(1);
	}
   } else if (shape_hor_sampling_factor_n<shape_hor_sampling_factor_m) {
	if(widthSrc < widthDst) {
		fprintf(stderr,"Error: Incorrect Hor. sampling ratio !");
		exit(1);
	}
   }

   if(shape_ver_sampling_factor_n>shape_ver_sampling_factor_m) {
	if(heightSrc > heightDst) {
		fprintf(stderr,"Error: Incorrect Ver. sampling ratio !");
		exit(1);
	}
   } else if(shape_ver_sampling_factor_n<shape_ver_sampling_factor_m) {
	if(heightSrc < heightDst) {
		fprintf(stderr,"Error: Incorrect Ver. sampling ratio !");
		exit(1);
	}
   }

   fprintf(stderr,"input alpha file : %s\n", srcFile);
   fprintf(stderr,"output alpha file : %s\n", dstFile);
   fprintf(stderr,"input width : %d\n", *widthSrc);
   fprintf(stderr,"input height: %d\n", *heightSrc);
   fprintf(stderr,"output width: %d\n", *widthDst);
   fprintf(stderr,"output height: %d\n", *heightDst);
   fprintf(stderr,"frame number : %d\n", *frame_no);
   fprintf(stderr,"Hor. sampling factor n: %d\n", *shape_hor_sampling_factor_n);
   fprintf(stderr,"Hor. sampling factor m: %d\n", *shape_hor_sampling_factor_m);
   fprintf(stderr,"Ver. sampling factor n: %d\n", *shape_ver_sampling_factor_n);
   fprintf(stderr,"Ver. sampling factor m: %d\n", *shape_ver_sampling_factor_m);
  
   fprintf(stderr,"\n");
}

void up_down_sample_alpha( UChar *alpha_src,  
				UChar *alpha_dst, 
				int shape_ver_sampling_factor_m,
				int shape_ver_sampling_factor_n,
				int shape_hor_sampling_factor_m,
				int shape_hor_sampling_factor_n,
				int widthSrc,
				int heightSrc,
				int widthDst,
				int heightDst)
{
  int i, x, y, x1, y1, x2, y2;
  int skip_pixel, skip_line;
  int widthMax, heightMax;

  double hor_samp_factor;
  double v_factor;
  int v_factor_int;
  double ver_samp_factor;
  double h_factor;
  int h_factor_int;

  int shape_ver_s, shape_ver_b, shape_hor_s, shape_hor_b;

  int hor_filter_type;	/* 0 - downsampling, 1 - upsampling, 2 - no conversion */
  int ver_filter_type;	/* 0 - downsampling, 1 - upsampling, 2 - no conversion */

  UChar *alpha_hor;
  UChar *buff_hpos = NULL;
  UChar *buff_vpos = NULL;
  UChar *alpha_spos_h = NULL;
  UChar *alpha_spos_v = NULL;

  if (shape_hor_sampling_factor_n>shape_hor_sampling_factor_m) {
	hor_filter_type = UP_SAMPLING;
	shape_hor_b = shape_hor_sampling_factor_n;
	shape_hor_s = shape_hor_sampling_factor_m;

	widthMax = widthDst;
	if(widthMax < widthSrc) {
		fprintf(stderr,"Error: Incorrect Hor. sampling ratio !\n");
		exit(0);
	}	
  } else if (shape_hor_sampling_factor_n<shape_hor_sampling_factor_m) {
	hor_filter_type = DOWN_SAMPLING;
	shape_hor_b = shape_hor_sampling_factor_m;
	shape_hor_s = shape_hor_sampling_factor_n;

	widthMax = widthSrc;
	if(widthMax < widthDst) {
		fprintf(stderr,"Error: Incorrect Hor. sampling ratio !\n");
		exit(0);
	}	
  } else {
	hor_filter_type = NO_CONV;
	widthMax = widthSrc;
  }

  if (shape_ver_sampling_factor_n>shape_ver_sampling_factor_m) {
	ver_filter_type = UP_SAMPLING;
	shape_ver_b = shape_ver_sampling_factor_n;
	shape_ver_s = shape_ver_sampling_factor_m;

	heightMax = heightDst;
	if(heightMax < heightSrc) {
		fprintf(stderr,"Error: Incorrect Ver. sampling ratio !\n");
		exit(0);
	}	
  } else if (shape_ver_sampling_factor_n<shape_ver_sampling_factor_m) {
	ver_filter_type = DOWN_SAMPLING;
	shape_ver_b = shape_ver_sampling_factor_m;
	shape_ver_s = shape_ver_sampling_factor_n;

	heightMax = heightSrc;
	if(heightMax < heightDst) {
		fprintf(stderr,"Error: Incorrect Ver. sampling ratio !\n");
		exit(0);
	}	
  } else {
	ver_filter_type = NO_CONV;
	heightMax = heightSrc;
  }

  /* Horizontal filtering */
  alpha_hor = (UChar *)calloc(widthMax*heightSrc, sizeof(UChar));

  if (hor_filter_type==NO_CONV) {
     for (y = 0; y < heightSrc; y++) { 
        for (x = 0; x < widthMax; x++) {
	   if( x>=0 && x<widthSrc ) {
		*(alpha_hor+y*widthMax+x) = *(alpha_src+y*widthSrc+x);
	   } else {
		*(alpha_hor+y*widthMax+x) = 0;
	   }
        }
     }
  } else {
     alpha_spos_h = (UChar *)calloc(widthMax, sizeof(UChar));

     /* deteriming the horizontal residual sampling position */
     h_factor = log((double)shape_hor_b/shape_hor_s)/log(2);
     h_factor_int = (int)floor(h_factor+0.000001);
     if(h_factor - h_factor_int > 0.000001) {

        buff_hpos = (UChar *) calloc((shape_hor_b-shape_hor_s*(1<<h_factor_int)), sizeof(UChar));
   
        hor_samp_factor = (double)shape_hor_b/(shape_hor_b-shape_hor_s*(1<<h_factor_int));
        for (i=1;i<=(shape_hor_b-shape_hor_s*(1<<h_factor_int));i++){
           buff_hpos[i-1] =
              ((int)(shape_hor_b - hor_samp_factor*i))*(1<<h_factor_int)+(1<<h_factor_int)-1;
        }
     }

     /* deteriming the horizontal sampling position */
     for (x = 0; x < widthMax; x++) {
 	skip_pixel = 0;
	if(h_factor - h_factor_int > 0.000001) {
	   for(i=0; i<(shape_hor_b-shape_hor_s*(1<<h_factor_int));i++){
		if (x%((1<<h_factor_int)*shape_hor_b) == buff_hpos[i])
		   skip_pixel = 1;
		if(skip_pixel) break; 
	   }
	}
	if( (x%(1<<h_factor_int) != (1<<h_factor_int)-1) || skip_pixel == 1 ){
	   alpha_spos_h[x] = 0;
	} else {
	   alpha_spos_h[x] = 1;
	}
     }

     if (hor_filter_type==DOWN_SAMPLING) {
        /* Horizontal down-sampling  */
        for (y = 0; y < heightSrc; y++) { 
           x1 = 0;
           for (x = 0; x < widthSrc; x++) {
	      if ( alpha_spos_h[x]==1 ) {
		if( x1 < widthDst ) {
		   *(alpha_hor+y*widthMax+x1) = *(alpha_src+y*widthSrc+x);
		} else {
		   *(alpha_hor+y*widthMax+x1) = 0;
		}
		x1++;
	      }
	   }
        }
     } else if (hor_filter_type==UP_SAMPLING) {
        /* Horizontal up-sampling  */
        for (y = 0; y < heightSrc; y++) { 
           x1 = 0;
           for (x = 0; x < widthDst; x++) {
	      if( x1 < widthSrc ) {
		   *(alpha_hor+y*widthMax+x) = *(alpha_src+y*widthSrc+x1);
	      } else {
		   *(alpha_hor+y*widthMax+x) = 0;
	      }
	      if (alpha_spos_h[x] == 1) x1 ++;
           }
        }
     } else {
	fprintf(stderr,"Error: Incorrect horizontal filter type !\n");
	exit(1);
     }

     if(h_factor - h_factor_int > 0.000001)  free(buff_hpos);
     free(alpha_spos_h);
  }

  /* Vertical filtering */
  if (ver_filter_type==NO_CONV) {
     for (x = 0; x < widthDst; x++) {
        for (y = 0; y < heightDst; y++) { 
	   if( y < heightSrc ) {
		*(alpha_dst+y*widthDst+x) = *(alpha_hor+y*widthMax+x);
	   } else {
		*(alpha_dst+y*widthDst+x) = 0;
	   }
        }
     }
  } else {

     alpha_spos_v = (UChar *)calloc(heightMax, sizeof(UChar));

     /* deteriming the vertical residual sampling position */
     v_factor = log((double)shape_ver_b/shape_ver_s)/log(2);
     v_factor_int = (int)floor(v_factor+0.000001);
     if(v_factor - v_factor_int > 0.000001) {

        buff_vpos = (UChar *) calloc((shape_ver_b-shape_ver_s*(1<<v_factor_int)), sizeof(UChar));

        ver_samp_factor = (double)shape_ver_b/(shape_ver_b-shape_ver_s*(1<<v_factor_int));
        for (i=1;i<=(shape_ver_b-shape_ver_s*(1<<v_factor_int));i++){
           buff_vpos[i-1] =
              ((int)(shape_ver_b - ver_samp_factor*i))*(1<<v_factor_int)+(1<<v_factor_int)-1;
        }
     }

     /* deteriming the vertical sampling position */
     for (y = 0; y < heightMax; y++) {
 	skip_line = 0;
	if(v_factor - v_factor_int > 0.000001) {
	   for(i=0; i<(shape_ver_b-shape_ver_s*(1<<v_factor_int));i++){
		if (y%((1<<v_factor_int)*shape_ver_b) == buff_vpos[i])
		   skip_line = 1;
		if(skip_line) break;
	   }
	}
	if( (y%(1<<v_factor_int) != (1<<v_factor_int)-1) || skip_line == 1 ){
	      alpha_spos_v[y] = 0;
	} else {
	      alpha_spos_v[y] = 1;
	}
     }

     if (ver_filter_type==DOWN_SAMPLING) {
        /* Vertical down-sampling  */
        for (x = 0; x < widthDst; x++) {
           y1 = 0;
           for (y = 0; y < heightSrc; y++) { 
              if (alpha_spos_v[y] == 1) {
	         if( y1<heightDst) {
	   	    *(alpha_dst+y1*widthDst+x) = *(alpha_hor+y*widthMax+x);
	         } else {
	   	    *(alpha_dst+y1*widthDst+x) = 0;
	         }
		 y1++;
	      }
           }
        }
     } else if (ver_filter_type==UP_SAMPLING) {
        /* Vertical up-sampling  */
        for (x = 0; x < widthDst; x++) {
           y1 = 0;
           for (y = 0; y < heightDst; y++) { 
	      if( y1<heightSrc) {
	   	   *(alpha_dst+y*widthDst+x) = *(alpha_hor+y1*widthMax+x);
	      } else {
	   	   *(alpha_dst+y*widthDst+x) = 0;
	      }
              if (alpha_spos_v[y] == 1) y1 ++;
           }
        }
     } else {
	fprintf(stderr,"Error: Incorrect vertical filter type !\n");
	exit(1);
     }

     if(v_factor - v_factor_int > 0.000001)  free(buff_vpos);
     free(alpha_spos_v);
  }

  free(alpha_hor);
}
