/******************************************************************************
 *                                                                          
 * This software module was originally developed by 
 *
 * Noel O'Connor (TELTEC IRELAND / ACTS-MoMuSys).
 *
 * and edited by 
 *
 * Bob Eifrig (NextLevel Systems)
 * Seishi TAKAMURA (NTT)
 * Oki Electric Industry Co., Ltd. (contact: Shigeru Fukunaga)
 *
 * in the course of development of the MPEG-4 Video (ISO/IEC 14496-2) standard.
 * This software module is an implementation of a part of one or more MPEG-4 
 * Video (ISO/IEC 14496-2) tools as specified by the MPEG-4 Video (ISO/IEC 
 * 14496-2) standard. 
 *
 * ISO/IEC gives users of the MPEG-4 Video (ISO/IEC 14496-2) standard free 
 * license to this software module or modifications thereof for use in hardware
 * or software products claiming conformance to the MPEG-4 Video (ISO/IEC 
 * 14496-2) standard. 
 *
 * Those intending to use this software module in hardware or software products
 * are advised that its use may infringe existing patents. The original 
 * developer of this software module and his/her company, the subsequent 
 * editors and their companies, and ISO/IEC have no liability for use of this 
 * software module or modifications thereof in an implementation. Copyright is 
 * not released for non MPEG-4 Video (ISO/IEC 14496-2) standard conforming 
 * products. 
 *
 * ACTS-MoMuSys partners retain full right to use the code for his/her own 
 * purpose, assign or donate the code to a third party and to inhibit third 
 * parties from using the code for non MPEG-4 Video (ISO/IEC 14496-2) standard
 * conforming products. This copyright notice must be included in all copies or
 * derivative works. 
 *
 * Copyright (c) 1997
 *
 *****************************************************************************/


/***********************************************************HeaderBegin*******
 *                                                                         
 * File:        vm_stats.c
 *
 * Author:      Noel O'Connor Teltec Irl.
 * Created:     11-03-96
 *                                                                         
 * Description: All functions in this file deal with the printing of coding
 *              statistics to a specified disk file      
 *
 * Notes:       
 *
 * Modified:    21.04.96 Robert Danielsen: Reformatted. New headers.
 *      01.08.96 Noel O'Connor: All new functions for printing stats to DISPLAY
 *      or to a file for VM 3.0.
 *      23.09.96 Noel O'Connor: New functions to cope with the fact that the
 *              encoder is now coding multiple VOs.
 *      29.11.96 Noel O'Connor: New printing function for the beginning of a
 *              time instant (time values are now Floats).
 *      04.02.97 Noel O'Connor: mods. for non unique VOL Ids
 *	03.03.99 Seishi TAKAMURA (NTT): added GMC coding
 *      16.08.99 Shigeru Fukunaga (Oki): added newpred.h for NEWPRED
 *
 ***********************************************************HeaderEnd*********/

/************************    INCLUDE FILES    ********************************/

#include <stdio.h>
#include "vm_enc_defs.h"
#include "vm_stats.h"

#include "newpred.h"	/** NEWPRED (Oki) 16-AUG-1999 **/

/* inserted by SAMSUNG AIT */
#include "rc.h"
extern Float VOP_QP;
Int PreviousBits[MAX_NUM_VOS][MAX_NUM_VOLS];
/* inserted by SAMSUNG AIT */

/***********************************************************CommentBegin******
 *
 * -- DoPrintSessionStartStat -- Prints VOL syntax stats
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      31-07-96
 *
 * Purpose :            
 *      Record the number of bits expended in coding the VOL syntax which is the
 *      first thing written to the bitstream.  
 * 
 * Arguments in :       
 *      BitCount num_bits[] - array of structures holding the statistics
 *      VO *vo_list - linked list of VOs (incl. VOL Ids which are needed for
 *                                                              accessing the correct BitCount structure).
 *      File *file - where to print
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      
 *
 * Description :        
 * -
 *
 * See also :
 *      PrintSesisonStartStat()
 *
 * Modified :           
 *      
 *
 ***********************************************************CommentEnd********/
Void
DoPrintSessionStartStat(BitCount num_bits[MAX_NUM_VOS][MAX_NUM_VOLS],VO *vo_list,File *file)
{
        VO      *vo;

        Vol     *vol;

        Int     vo_id,
                        vol_id;

        fprintf(file,"********************\n");
        fprintf(file,"*                  *\n");
        fprintf(file,"* Start of Session *\n");
        fprintf(file,"*                  *\n");
        fprintf(file,"********************\n");

        vo = vo_list;

        while(vo != NULL)
        {
                vo_id = GetVOId(vo);

                fprintf(file,"\nVideo Object - Id : %d\n", (int)vo_id);

                vol = GetVOLayers(vo);

                while(vol != NULL)
                {
                        vol_id = GetVolId(vol);

                        fprintf(file,"\n\tVideo Object Layer - Id : %d\n",
                                (int)vol_id);

                        fprintf(file,"\n\t#Bits for VOL syntax : %d\n\n",
                                (int)num_bits[vo_id][vol_id].vol);

                        vol = GetVolNext(vol);
                }

                vo = GetVONext(vo);
        }

        return;
} 
/***********************************************************CommentBegin******
 *
 * -- PrintSessionStartStat -- 
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      23-09-96
 *
 * Purpose :            
 *      Record the number of bits expended in coding the VOL syntax which is the
 *      first thing written to the bitstream. This function sets up a file pointer
 *      indicating where to print this information.  
 * 
 * Arguments in :       
 *      BitCount num_bits[] - array of structures holding the statistics
 *      VO *vo_list - linked list of VOs (incl. VOL Ids which are needed for
 *                                                              accessing the correct BitCount structure).
 *      Int mode - where to print (DISPLAY or STATS_FILE)
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      
 *
 * Description :        
 * -
 *
 * See also :
 *      PrintSesisonStartStat()
 *
 * Modified :           
 *      
 *
 ***********************************************************CommentEnd********/

Void
PrintSessionStartStat(BitCount num_bits[MAX_NUM_VOS][MAX_NUM_VOLS],VO *vo_list,Int mode)
{
        File *file;

        Int             index1,
                                index2;

        if(mode == DISPLAY)
                DoPrintSessionStartStat(num_bits,vo_list,stdout);
        else
        {
                index1 = GetVOId(vo_list);
                index2 = GetVolId(GetVOLayers(vo_list));
                if((file = fopen(num_bits[index1][index2].stats_file,"a")) == NULL)
                {
                        fprintf(stdout,"ERROR(PrintSessionStartStat) : could not open stats file\n");
                        exit(1);
                }

                DoPrintSessionStartStat(num_bits,vo_list,file);

                fclose(file);
        }

        return;
}

/***********************************************************CommentBegin******
 *
 * -- DoPrintTimeStartStat -- Prints information for user at the start of each
 *      time instant
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      31-07-96
 *
 * Purpose :            
 *      Low level function for printing user information at the start of each
 *      time instant.  
 * 
 * Arguments in 
 *      Float time - current time instant
 *      File *file - where to print
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      
 *
 * Description :        
 *      This function actually does the printing
 *
 * See also :
 *      PrintTimeStartStat()
 *
 * Modified :           
 *      
 *
 ***********************************************************CommentEnd********/
Void
DoPrintTimeStartStat(Float time,File *file)
{
                fprintf(file,"\n\n==============================================\n");
                fprintf(file,"Time : %.3f\n", (Float)time);
                fprintf(file,"==============================================\n\n");
}


/***********************************************************CommentBegin******
 *
 * -- PrintTimeStartStat -- Prints information for user at the start of each
 *      time instant
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      31-07-96
 *
 * Purpose :            
 *      Top level function for printing user information at the start of each
 *      time instant.  
 * 
 * Arguments in :       
 *      Float time - current time instant
 *      Char *stats_file - name of statistics file
 *      Int mode - flag (DISPLAY / STATS_FILE)
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      
 *
 * Description :        
 *      This function really only sets up the correct file pointer for printing
 *
 * See also :
 *      DoPrintTimeStartStat()
 *
 * Modified :           
 *      
 *
 ***********************************************************CommentEnd********/
Void
PrintTimeStartStat(Float time, Char *stats_file, Int mode)
{
        File *file;

        if(mode == DISPLAY)
                DoPrintTimeStartStat(time,stdout);
        else
        { 
                if((file = fopen(stats_file,"a")) == NULL)
                {
                        fprintf(stdout,"ERROR(PrintTimeStartStat) : could not open stats file\n");
                        exit(1);
                }
                
                DoPrintTimeStartStat(time,file);

                fclose(file);
        }

        return;
}

/***********************************************************CommentBegin******
 *
 * -- DoPrintTimeEndStat -- Prints information for user at the start of each
 *      time instant
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      31-07-96
 *
 * Purpose :            
 *      Low level function for printing bit statistics for each VOL at the
 *      end of each time instant.
 * 
 * Arguments in 
 *      BitCount        num_bits[] - array of structures holding bit statistics for 
 *                      each VOL.
 *      Int id - VOL id (needed to access correct structure).
 *      Int type - type of VOP (I or P).
 *      FILE *file - where to print (DISPLAY or STATS_FILE).
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      
 *
 * Description :        
 *      This function actually does the printing
 *
 * See also :
 *      PrintVolTimeEndStat()
 *
 * Modified :           
 *      04.02.97 Noel O'Connor: mods. for non unique VOL Ids
 *      21.05.98 Ji Heon Kweon (HYUNDAI) : support for grayscale coding 
 *
 ***********************************************************CommentEnd********/
Void
DoPrintVolTimeEndStat(BitCount num_bits[MAX_NUM_VOS][MAX_NUM_VOLS],Int vo_id,Int vol_id,Int type,File *file,Int aux_comp_count)
{
  Int aux;

  if(type == I_VOP)
  {
    if(num_bits[vo_id][vol_id].sep_mot_shape_text)
    {       
      fprintf(file,"\t#Bits for shape : %d\n",
              (int)num_bits[vo_id][vol_id].shape);
      fprintf(file,"\t#Bits for texture : %d\n",
              (int)num_bits[vo_id][vol_id].texture);
    }
    else {
      
      fprintf(file,"\t#Bits for shape : %d\n",
              (int)num_bits[vo_id][vol_id].shape);
      
      fprintf(file,"\t#Bits for texture_shape : %d\n",
              (int)num_bits[vo_id][vol_id].texture);
    }
  }
  
  else if(num_bits[vo_id][vol_id].sep_mot_shape_text)
  {       
    fprintf(file,"\t#Bits for shape : %d\n",
            (int)num_bits[vo_id][vol_id].shape);
    fprintf(file,"\t#Bits for motion : %d\n",
            (int)num_bits[vo_id][vol_id].motion);
    fprintf(file,"\t#Bits for texture : %d\n",
            (int)num_bits[vo_id][vol_id].texture);
  }
  else  {
    fprintf(file,"\t#Bits for shape : %d\n",
            (int)num_bits[vo_id][vol_id].shape);
    fprintf(file,"\t#Bits for combined motion, shape & texture : %d\n", (int)num_bits[vo_id][vol_id].mot_shape_text);
  }
  fprintf(file,"\t#Bits for vop syntax : %d\n\n",
          (int)num_bits[vo_id][vol_id].syntax);
  fprintf(file,"\tTotal number of bits for this vop : %d\n\n",
          (int)num_bits[vo_id][vol_id].vop);
  
  fprintf(file,"\tPSNR_Y    : %f\n",num_bits[vo_id][vol_id].psnr_y);
  for(aux=0;aux<aux_comp_count;aux++)	/* MAC (SB) 17-Nov-99 */
    fprintf(file,"\tPSNR_G[%d] : %f\n",aux,num_bits[vo_id][vol_id].psnr_g[aux]); /* HYUNDAI (Grayscale) */
  fprintf(file,"\tPSNR_U    : %f\n",num_bits[vo_id][vol_id].psnr_u);
  fprintf(file,"\tPSNR_V    : %f\n\n",num_bits[vo_id][vol_id].psnr_v);
  
  return;
}

/***********************************************************CommentBegin******
 *
 * -- PrintTimeEndStat -- Prints information for user at the start of each
 *      time instant
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      31-07-96
 *
 * Purpose :            
 *      High level function for printing bit statistics for each VOL at the
 *      end of each time instant.
 * 
 * Arguments in 
 *      BitCount        num_bits[] - array of structures holding bit statistics for 
 *                      each VOL.
 * Int vo_id - VO id (needed to access correct structure).
 *      Int vol_id - VOL id (needed to access correct structure).
 *      Int type - type of VOP (I or P).
 *      Int mode - where to print (DISPLAY or STATS_FILE).
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      
 *
 * Description :        
 *      This function really only sets up the correct file pointer for printing
 *
 * See also :
 *      DoPrintVolTimeEndStat()
 *
 * Modified :           
 *       04.02.97 Noel O'Connor: mods. for non unique VOL Ids
 *
 ***********************************************************CommentEnd********/

Void
PrintVolTimeEndStat(BitCount num_bits[MAX_NUM_VOS][MAX_NUM_VOLS],Int vo_id,Int vol_id,Int type,Int mode,Int aux_comp_count)
{
        File *file;

        if(mode == DISPLAY)
                DoPrintVolTimeEndStat(num_bits,vo_id,vol_id,type,stdout,aux_comp_count);
        else
        {
                if((file = fopen(num_bits[vo_id][vol_id].stats_file,"a")) == NULL)
                {
                        fprintf(stdout,"ERROR(PrintTimeEndStat) : could not open stats file\n");
                        exit(1);
                }
                
                DoPrintVolTimeEndStat(num_bits,vo_id,vol_id,type,file,aux_comp_count);

                fclose(file);
        }

        return;
}

/***********************************************************CommentBegin******
 *
 * -- DoPrintSessionEndStat -- Prints bit statistics for user at the end of a
 *      session (i.e. complete run of the encoder).
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      31-07-96
 *
 * Purpose :            
 *      Low level function for printing bit statistics for user at the end of a
 *      session (i.e. complete run of the encoder).  
 * 
 * Arguments in :       
 *      BitCount num_bits[] - structures holding the statistics & name of stats file
 *      VO *vo_list - linked list of VO structures (incl. VOL Ids required to
 *                      correctly access array of BitCounts)
 *      File *file - where to print
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      
 *
 * Description :        
 *      This function actually does the printing
 *
 * See also :
 *      PrintSessionEndStat()
 *
 * Modified :           
 *      21.05.98 Ji Heon Kweon (HYUNDAI) : support for grayscale coding 
 *      28.01.99 M. Eckert: Add output of results for RC UPM3 algorithm,
 *                          only active in case of debugging.
 *      
 *
 ***********************************************************CommentEnd********/
Void
DoPrintSessionEndStat(BitCount num_bits[MAX_NUM_VOS][MAX_NUM_VOLS],VO *vo_list,File *file)
{
        VO      *vo;

        Vol     *vol;

        Int     vo_id,vol_id, aux;

        fprintf(file,"\n\n");
        fprintf(file,"********************\n");
        fprintf(file,"*                  *\n");
        fprintf(file,"*  End of Session  *\n");
        fprintf(file,"*                  *\n");
        fprintf(file,"********************\n");

        vo = vo_list;

        while(vo != NULL)
        {
                vo_id = GetVOId(vo);

                fprintf(file,"\n\nVideo Object - Id : %d\n", (int)vo_id);

                vol = GetVOLayers(vo);
        
                while(vol != NULL)
                {
                        vol_id = GetVolId(vol);

#ifdef _RC_DEBUG_
                        /* RC: Print results for RC algorithm UPM3 */
                        if (dist_vo0 == 0.0) dist_vo0 = num_bits[vo_id][vol_id].psnr_y_ave;
                        beta = (num_bits[vo_id][vol_id].psnr_y_ave - dist_vo0)/20.;
                        beta = 1./pow(10,beta);
#endif

                        fprintf(file,"**********************************************\n");
                        fprintf(file,"Video Object Layer - Id : %d\n\n",
                                (int)vol_id);
                        fprintf(file,"#Bits used in total : %d\n",
                                (int)num_bits[vo_id][vol_id].vol);
                        fprintf(file,"Average #Bits per VOP : %f\n",num_bits[vo_id][vol_id].average);
                        fprintf(file,"Average #Bits_shape per VOP : %f\n",num_bits[vo_id][vol_id].average_shape);
                        fprintf(file,"Average PSNR_Y    : %f\n",num_bits[vo_id][vol_id].psnr_y_ave);
#ifdef _RC_DEBUG_
                        fprintf(file,"beta = %f\n", beta);
#endif
                        for(aux=0;aux<GetVolAuxCompCount(vol);aux++)
                          fprintf(file,"Average PSNR_G[%d] : %f\n",aux,num_bits[vo_id][vol_id].psnr_g_ave[aux]); /* HYUNDAI (Grayscale) */
                        fprintf(file,"Average PSNR_U    : %f\n",num_bits[vo_id][vol_id].psnr_u_ave);
                        fprintf(file,"Average PSNR_V    : %f\n",num_bits[vo_id][vol_id].psnr_v_ave);
                        fprintf(file,"**********************************************\n\n");

                        vol = GetVolNext(vol);
                }

                vo = GetVONext(vo);
        }
        
        return;
}

/***********************************************************CommentBegin******
 *
 * -- PrintSessionEndStat -- Prints bits statistics for user at the end of a
 *      session (i.e. complete run of the encoder).
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      31-07-96
 *
 * Purpose :            
 *      Top level function for printing bit statistics at the end of a
 *      session (i.e. complete run of the encoder).  
 * 
 * Arguments in :       
 *      BitCount num_bits[] - structures holding the statistics & name of stats file
 *      VO *vo_list - linked list of VO structures (incl. VOL Ids required to
 *                      correctly access array of BitCounts)
 *      Int mode - flag (DISPLAY / STATS_FILE)
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      
 *
 * Description :        
 *      This function really only sets up the correct file pointer for printing
 *
 * See also :
 *      DoPrintSessionEndStat()
 *
 * Modified :           
 *      
 *
 ***********************************************************CommentEnd********/ 
Void
PrintSessionEndStat(BitCount num_bits[MAX_NUM_VOS][MAX_NUM_VOLS], VO *vo_list,Int mode)
{
        File *file;

        Int     index1,index2;

        if(mode == DISPLAY)
                DoPrintSessionEndStat(num_bits,vo_list,stdout);
        else
        {
                index1 = GetVOId(vo_list);
                index2 = GetVolId(GetVOLayers(vo_list));

                if((file = fopen(num_bits[index1][index2].stats_file,"a")) == NULL)
                {
                        fprintf(stdout,"ERROR(PrintSessionEndStat) : could not open stats file '%s'\n", num_bits[index1][index2].stats_file);
                        exit(1);
                }

                DoPrintSessionEndStat(num_bits, vo_list,file);

                fclose(file);
        }

        return;
}

/***********************************************************CommentBegin******
 *
 * -- DoPrintVolTimeStartStat -- Print info. for uaser for each VOL at each
 *              time instant
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      23-09-96
 *
 * Purpose :            
 *      Low level function for printing bit statistics for user   
 * 
 * Arguments in :       
 *      Int vol_id - VOL Id
 *      Int frame - current frame being coded
 *      File *file - where to print
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      
 *
 * Description :        
 *      This function actually does the printing
 *
 * See also :
 *      PrintVolTimeStartStat()
 *
 * Modified :           
 *      
 *
 ***********************************************************CommentEnd********/
Void
DoPrintVolTimeStartStat(Int vol_id, Int frame, File *file)
{
        fprintf(file,"\tVideo Object Layer - Id : %d\n", (int)vol_id);
        fprintf(file,"\n\tCoding VOP of frame %d\n", (int)frame);
        fprintf(file,"\n\n");
}

/***********************************************************CommentBegin******
 *
 * -- PrintVolTimeStartStat -- Print info for use for each VOL at the start
 *      of each time instant
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      31-07-96
 *
 * Purpose :            
 *      High level function for printing info. for user for each VOL at the
 *      start of each time instant  
 * 
 * Arguments in :       
 *      BitCount num_bits[] - structures holding the statistics & name of stats file
 * Int vo_id - VO Id
 *      Int vol_id - VOL Id
 *      Int frame - current frame being coded
 *      Int mode - where to print (DISPLAY or STATS_FILE)
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      
 *
 * Description :        
 *      This function really only sets up a file pointer for printing
 *
 * See also :
 *      DoPrintVolTimeStartStat()
 *
 * Modified :           
 *      04.02.97 Noel O'Connor : mods. for non unique VOL Ids
 *
 ***********************************************************CommentEnd********/
Void
PrintVolTimeStartStat(BitCount num_bits[MAX_NUM_VOS][MAX_NUM_VOLS],Int vo_id,Int vol_id,Int frame,Int mode)
{
        File    *file;


        if(mode == DISPLAY)
                DoPrintVolTimeStartStat(vol_id,frame,stdout);
        else
        {

                if((file = fopen(num_bits[vo_id][vol_id].stats_file,"a")) == NULL)
                {
                        fprintf(stdout,"ERROR(PrintSessionEndStat) : could not open stats file '%s'\n", num_bits[vo_id][vol_id].stats_file);
                        exit(1);
                }

                DoPrintVolTimeStartStat(vol_id,frame,file);

                fclose(file);
        }

        return;
}

/***********************************************************CommentBegin******
 *
 * -- DoPrintVOTimeStartStat -- Prints data for user for each VO at the
 *              start of a time instant
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      23-09-96
 *
 * Purpose :            
 *      Low level function for printing info. for the user for each VO at the
 *      start of each time instant.
 * 
 * Arguments in :       
 *      Int id - VO ID
 *      File *file - where to print
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      
 *
 * Description :        
 *      This function actually does the printing
 *
 * See also :
 *      PrintSessionVOTimeStartStat()
 *
 * Modified :           
 *      
 *
 ***********************************************************CommentEnd********/
Void
DoPrintVOTimeStartStat(Int id, File *file)
{
        fprintf(file,"\nVideo Object - Id : %d\n", (int)id);
        fprintf(file,"---------------------\n\n");

        return;
}

/***********************************************************CommentBegin******
 *
 * -- PrintVOTimeStartStat -- Print info. for user for each VO at each
 *              time instant.
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      23-09-96
 *
 * Purpose :            
 *      High level function for printing  info. for user for each VO at each
 *              time instant. 
 * 
 * Arguments in :       
 *      BitCount num_bits[] - structures holding the statistics & name of stats file
 *      VO *vo_list - linked list of VO structures (incl. VOL Ids required to
 *                      correctly access array of BitCounts)
 *      Int mode - where to print (DISPLAY or STATS_FILE)
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      
 *
 * Description :        
 *      This function actually does the printing
 *
 * See also :
 *      DoPrintVOTimeStartStat()
 *
 * Modified :           
 *      
 *
 ***********************************************************CommentEnd********/
Void
PrintVOTimeStartStat(BitCount num_bits[MAX_NUM_VOS][MAX_NUM_VOLS],VO *vo,Int mode)
{
        FILE    *file;

/*      Vol             *vol; */

        Int             vo_id,vol_id;

        if(mode == DISPLAY)
                DoPrintVOTimeStartStat(GetVOId(vo),stdout);
        else
        {
                vo_id = GetVOId(vo);
                vol_id = GetVolId(GetVOLayers(vo));

                if((file = fopen(num_bits[vo_id][vol_id].stats_file,"a")) == NULL)
                {
                        fprintf(stdout,"ERROR(PrintVOTiimeStartStat) : could not open stats file\n");
                        exit(1);
                }

                DoPrintVOTimeStartStat(GetVOId(vo),file);

                fclose(file);
        }

        return;
}

/* Removed cond. def. to allow generation of correct prototypes */
/* MW 14-MAY-1998 thanks to Uli */
/* #ifdef NLSSTATS */
Void
NLSPrintVolTimeEndStat(BitCount *bc, Vop *vop, Float time, File *fp)
{
    static char vop_letter[] = "IPBS";

    fprintf(fp, "%c[%3d]      : time=%.3f video_obj_id=%d layer_id=%d quant=",
        vop_letter[vop->prediction_type], vop->frame, time, vop->vo_id, vop->ident);
    switch (vop->prediction_type) {
 
    case I_VOP:
        fprintf(fp, "%d\nI[%3d] flds : Y=%d C=%d cbpy=%d mcbpc=%d ilace=%d\n",
            vop->intra_quantizer, 
            vop->frame, bc->text_bits.Y, bc->text_bits.C, bc->text_bits.CBPY,
            bc->text_bits.MCBPC, bc->text_bits.interlaced);
        fprintf(fp,
        "I[%3d] mblk : inter=%d inter4v=%d intra=%d field=%d skip=%d fldDCT=%d\n",
            vop->frame, bc->text_bits.no_inter, bc->text_bits.no_inter4v,
            bc->text_bits.no_intra, bc->text_bits.no_field, bc->text_bits.no_skipped,
            bc->text_bits.fieldDCT);
        break;
 
    case P_VOP:
        fprintf(fp, "%d\nP[%3d] flds : Y=%d C=%d MVs=%d cbpy=%d mcbpc=%d ilace=%d\n",
            vop->quantizer, vop->frame, bc->text_bits.Y, bc->text_bits.C, bc->text_bits.vec,
            bc->text_bits.CBPY, bc->text_bits.MCBPC, bc->text_bits.interlaced);
        fprintf(fp,
        "P[%3d] mblk : inter=%d inter4v=%d intra=%d field=%d skip=%d fldDCT=%d\n",
            vop->frame, bc->text_bits.no_inter, bc->text_bits.no_inter4v, bc->text_bits.no_intra,
            bc->text_bits.no_field, bc->text_bits.no_skipped, bc->text_bits.fieldDCT);
        break;
 
/* modified by NTT for GMC coding : start */
      case SPRITE_VOP:
	fprintf(fp, "%d\nS[%3d] flds : Y=%d C=%d MVs=%d cbpy=%d mcbpc=%d ilace=%d\n",
		vop->quantizer, vop->frame, bc->text_bits.Y, bc->text_bits.C, bc->text_bits.vec,
		bc->text_bits.CBPY, bc->text_bits.MCBPC, bc->text_bits.interlaced);
	fprintf(fp,
	"S[%3d] mblk : inter=%d inter4v=%d intra=%d field=%d skip=%d fldDCT=%d GMC=%d\n",
	vop->frame, bc->text_bits.no_inter, bc->text_bits.no_inter4v, bc->text_bits.no_intra,
	bc->text_bits.no_field, bc->text_bits.no_skipped, bc->text_bits.fieldDCT, bc->text_bits.no_GMC);
	break;
/* modified by NTT for GMC coding : end */
    case B_VOP:
        fprintf(fp, "%d\nB[%3d] flds : Y=%d C=%d MVs=%d hdr=%d %d %d %d ilace=%d fldDCT=%d\n",
            vop->B_quantizer, vop->frame, bc->text_bits.Y, bc->text_bits.C, bc->text_bits.vec,
            bc->text_bits.MODB, bc->text_bits.MBTYPE, bc->text_bits.CBPB, bc->text_bits.DQUANT,
            bc->text_bits.interlaced, bc->text_bits.fieldDCT);
        fprintf(fp,
        "B[%3d] mblk : bmt=%d %d %d %d %d %d %d nmv=%d %d %d field=%d skip=%d %d %d\n",
            vop->frame, bc->text_bits.Btype[0], bc->text_bits.Btype[1], bc->text_bits.Btype[2],
            bc->text_bits.Btype[3], bc->text_bits.Btype[4], bc->text_bits.Btype[5],
            bc->text_bits.Btype[6], bc->text_bits.Nmvs[0], bc->text_bits.Nmvs[1],
            bc->text_bits.Nmvs[2], bc->text_bits.no_field, bc->text_bits.no_Pskip,
            bc->text_bits.no_skipped, bc->text_bits.no_noDCT);
        break;
    }

    if(vop->prediction_type == I_VOP)
    {
        if(bc->shape)
        {
            fprintf(fp,"%c[%3d] bits : shape=%d texture=%d ",
                vop_letter[vop->prediction_type], vop->frame, bc->shape, bc->texture);
        }
        else
            fprintf(fp,"%c[%3d] bits : shape+texture=%d ",
                vop_letter[vop->prediction_type], vop->frame, bc->texture);
    }
 
    else if(bc->shape + bc->motion + bc->texture)
    {
        fprintf(fp,"%c[%3d] bits : shape=%d motion=%d texture=%d ",
                vop_letter[vop->prediction_type], vop->frame, bc->shape, bc->motion, bc->texture);
    }
    else
        fprintf(fp,"%c[%3d] bits : shape+motion+texture=%d ",
                vop_letter[vop->prediction_type], vop->frame, bc->mot_shape_text);
 
    fprintf(fp,"vop=%d total=%d\n",
                bc->syntax, bc->vop);
 
    fprintf(fp,"%c[%3d] PSNR : Y=%f U=%f V=%f\n\n",
        vop_letter[vop->prediction_type], vop->frame, bc->psnr_y, bc->psnr_u, bc->psnr_v);
}
/* #endif */
