/******************************************************************************
 *                                                                          
 * This software module was originally developed by 
 *
 * Noel O'Connor (TELTEC IRELAND / ACTS-MoMuSys).                       
 *
 * and edited by 
 *
 * Aasmund Sandvand (Telenor / ACTS-MoMuSys).                           
 * J. Ignacio Ronda (UPM / ACTS-MoMuSys).                               
 * Cecile Dufour (LEP / ACTS-MoMuSys).                                  
 * Minhua Zhou (HHI / ACTS-MoMuSys).                                    
 * Fernando Jaureguizar (UPM / ACTS-MoMuSys).                           
 * Luis Ducla-Soares (IST / ACTS-MoMuSys).                              
 * Martina Eckert (UPM / ACTS-MoMuSys).                                 
 * Noel Brady (TELTEC IRELAND / ACTS-MoMuSys).                          
 * Michael Wollborn (TUH / ACTS-MoMuSys).   
 * Bob Eifrig (NextLevel Systems)
 * Michael Frater (UNSW)
 * Ji Heon Kweon (HYUNDAI)
 * Marc Mongenet (EPFL)
 * Sven Brandau (HHI)
 * Seishi TAKAMURA (NTT)
 * Oki Electric Industry Co., Ltd. (contact: Shigeru Fukunaga)
 * Fujitsu Laboratories Ltd. (contact: Eishi Morimatsu)
 * Dae-Sung Cho (Samsung AIT)
 * Massimo Ravasi (EPFL)
 *
 * in the course of development of the MPEG-4 Video (ISO/IEC 14496-2) standard.
 * This software module is an implementation of a part of one or more MPEG-4 
 * Video (ISO/IEC 14496-2) tools as specified by the MPEG-4 Video (ISO/IEC 
 * 14496-2) standard. 
 *
 * ISO/IEC gives users of the MPEG-4 Video (ISO/IEC 14496-2) standard free 
 * license to this software module or modifications thereof for use in hardware
 * or software products claiming conformance to the MPEG-4 Video (ISO/IEC 
 * 14496-2) standard. 
 *
 * Those intending to use this software module in hardware or software products
 * are advised that its use may infringe existing patents. The original 
 * developer of this software module and his/her company, the subsequent 
 * editors and their companies, and ISO/IEC have no liability for use of this 
 * software module or modifications thereof in an implementation. Copyright is 
 * not released for non MPEG-4 Video (ISO/IEC 14496-2) standard conforming 
 * products. 
 *
 * ACTS-MoMuSys partners retain full right to use the code for his/her own 
 * purpose, assign or donate the code to a third party and to inhibit third 
 * parties from using the code for non MPEG-4 Video (ISO/IEC 14496-2) standard
 * conforming products. This copyright notice must be included in all copies or
 * derivative works. 
 *
 * Copyright (c) 1997
 *
 *****************************************************************************/



/***********************************************************HeaderBegin*******
 *                                                                         
 * File:        vm_config.c
 *
 * Author:      Noel O'Connor Teltec Irl.
 * Created:     08-05-96
 *                                                                         
 * Description: 
 *
 * Notes:       
 *
 * Modified:    04.06.96 - Noel O'Connor : Extra parameters in config file
 *      and associated access functions.
 *                                              30.06.96 - Noel O'Connor : New structures for reading in
 *      control and configuration information (VOControl & VOLConfig).
 *                                              20.09.96 - Noel O'Connor : New structures to facilitate
 *      the coding of multiple VOs. New functions are supplied for reading 
 *      control and configuration information.
 *      29.11.96 - Noel O'Connor : New access function (and related software) for
 *      the new "next_coding_time" field of the VolConfig structure. Also new
 *      function for rounding time values.
 * 31.1.97 - Aasmund Sandvand : Moved control of deblock filter to encoder 
 *          control file
 * 04.02.97 Noel O'Connor : New functions for new .cfg file fields
 * 11.02.97 J. Ignacio Ronda: modified UpdateVolConfigNextCodingTime() to
 *          skip frames due to buffer fullness in the VM5 rate control
 * 20.02.97 Cecile Dufour : added accesses to sprite parameters (from F. Dufaux & R. Crinon)
 * 11.03.97 Minhua Zhou : added M to  UpdateVolConfigNextCodingTime() due to B-VOPs
 * 20.03.97 Fernando Jaureguizar: UpdateVolConfigNextCodingTime() modified
 *          to solve some problems with last skipped frames of the sequence.
 * 23.04.97 Michael Wollborn: Added MPEG quantization stuff completely
 * 26.04.97 Luis Ducla-Soares: Enabled the use of reversible VLCs and added 
 *                             data partitioning stuff
 * 07.05.97 Noel Brady: Enabled use of change_CR_disable.
 * 13.05.97 Minhua Zhou: added obmc_disable and GOV_period
 * 07.07.97 Martina Eckert: 
 *          Introduction of new Get and Put functions for Control File:
 *          - Int GetEncodeControlRCType(EncodeControl *enc_ctl)
 *          - Void PutEncodeControlRCType(Int rc_type, EncodeControl *enc_ctl)
 *          - GetEncodeControlRCAlgorithm(Int rc_algorithm, EncodeControl *enc_ctl)
 *          - Void PutEncodeControlRCAlgorithm(Int rc_algorithm, EncodeControl *enc_ctl)
 *          - Int GetEncodeControlRCRateModel(Int rc_rate_model, EncodeControl *enc_ctl)
 *          - Void PutEncodeControlRCRateModel(Int rc_rate_model, EncodeControl *enc_ctl)
 *          - Int GetEncodeControlRCDistModel(Int rc_dist_model, EncodeControl *enc_ctl)
 *          - Void PutEncodeControlRCDistModel(Int rc_dist_model, EncodeControl *enc_ctl)
 *          - Int GetEncodeControlGlobalBitrate(EncodeControl *enc_ctl)
 *          - Void PutEncodeControlGlobalBitrate(Int R_target, EncodeControl *enc_ctl)
 *          - Int GetEncodeControlGlobalBufferSize(EncodeControl *enc_ctl)
 *          - Void PutEncodeControlGlobalBufferSize(Int glob_buffer_size, EncodeControl *enc_ctl)
 *          > Read new parameters for Global RC in ReadEncoderControlFile:
 *            rc_type, rc_algorithm, rc_rate_model, rc_dist_model, R_target,
 *            glob_buffer_size
 *          Introduction of new Get and Put functions for Config File:
 *          - Void PutVolConfigRCParam1(Float rc_param_1, VolConfig *cfg)
 *          - Void PutVolConfigRCParam2(Float rc_param_2, VolConfig *cfg)
 *          - Float GetVolConfigRCParam1(VolConfig *cfg)
 *          - Float GetVolConfigRCParam2(VolConfig *cfg)
 *          > Read new parameters for Global RC in ReadEncoderControlFile:
 *            rc_param_1, rc_param_2
 * 15.07.97 UPS:added scalability_type
 * 30.07.97 Fernando Jaureguizar: fixing in UpdateVolConfigNextCodingTime()
 * 04.08.97 Minhua Zhou: rename advanced_prediction_disable
 * 04.08.97 Minhua Zhou: modifications for f_codes
 * 29.08.97 Osamu Sunohara(Sony): added functions to select spatial scalable
 *                                enhancement type.
 * 08.09.97 Cecile Dufour(Philips LEP): added functions for static sprites 
*  23.10.97 Minhua Zhou  : changed deblock_filter_disable to post_filter
 *                             _type
 * 04.01.98 Michael Frater: support for non-8-bit video
 * 08.01.98 Martina Eckert: Change of method to read RC parameters from .ctl 
 *          and .cfg files. Now only one parameter string is read from each 
 *          file.
 *          (Change of functions ReadEncoderControlFile and ReadVOLConfigData, 
 *           remove functions Getxxx and Putxxx, introduce functions Get- and 
 *           Put..RCParam)
 *
 * 01210.98 Hung-Ju Lee (Sarnoff) & Jie Liang (TI): 
 *        added the following functions for wavelet texture coding
 *                             PutSOLconfigLayers ();
 *                             ReadSOLConfigData();
 *                             PutSOLconfigNumVOLs();
 *                             AllocSOLconfig();
 *                             FreeSOLconfig();
 *                             GetSOLconfigNext();
 *                             and modified the following functions:
 *                             SallocVOConfig ();
 * 21.05.98 Ji Heon Kweon (HYUNDAI) : support for grayscale coding 
 * 09.06.98 Marc Mongenet (EPFL): added Complexity estimation syntax support
 * 11.08.98 Sven Brandau: changes due to N2339
 * 02.23.99 Dae-Sung Cho (Samsung AIT) : support for scalable shape for STO
 * 15.02.99 U. Benzler : added quarter pel support
 * 03.03.99 Seishi TAKAMURA (NTT): added GMC coding, version2 syntax
 * 16.08.99 Shigeru Fukunaga (Oki): added modules for NEWPRED
 * 18.08.99 Seishi TAKAMURA (NTT): modify GMC coding  (warping type check)
 * 06.09.99 Eishi Morimatsu (Fujitsu Labs.): added DRC support
 * 09.29.99 Dae-Sung Cho (Samsung AIT): added support for OBSS
 * 26.10.99 Massimo Ravasi (EPFL): added Complexity Estimation syntax support
 *                                 Update version 2
 * 11.11.99 Sven Brandau (HHI): added support for MAC
 * 
 ***********************************************************HeaderEnd*********/

/************************    INCLUDE FILES    ********************************/

#include <stdio.h>
#include <limits.h>
#include <string.h>
#include <sys/stat.h>
#include "vm_config.h"
#include "mom_vol.h"
#include "mom_vo.h"
#include "rc.h"
#include "text_quant_mat_def.h"
#include "text_quant_mat_spec.h"
#include "io_generic.h"
#include "vm_vop_bound.h"
#include "sprite_enc_piece.h"
#include "errorHandler.h"
#include "msg.h"



/***********************************************************CommentBegin******
 *
 * -- SallocEncodeControl -- allocates memory for an EncodeControl structure
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      20-09-96
 *
 * Purpose :            
 *      Allocates memory for an EncodeControl structure (but not the file
 *      name fields).
 *
 * Arguments in :       
 *      none
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      EncodeControl *enc_ctl - pointer to allocated structure
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      -
 *
 * See also :
 *      -
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/

EncodeControl *SallocEncodeControl()
{
        EncodeControl *enc_ctl;

        enc_ctl = (EncodeControl *)malloc(sizeof(EncodeControl));

        return(enc_ctl);
}

/***********************************************************CommentBegin******
 *
 * -- AllocEncodeControl -- Allocates memory for an EncodeControl structure
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      20-09-96
 *
 * Purpose :            
 *      Allocates memory for an EncodeControl structure (incl. the file name
 *      fields).
 *
 * Arguments in :       
 *      Int num_vos - number of VOs
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      EncodeControl *enc_ctl - pointer to allocated structure
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      -
 *
 * See also :
 *      -
 *
 * Modified :           
 *      15.01.97 Robert Danielsen: Corrected exit(0) to exit(1) in one case.
 *                      Solution from Paulo Nunes.
 *
 ***********************************************************CommentEnd********/
/* VTC Begin ------------------------------------------ */
EncodeControl *AllocEncodeControl(Int num_vos, Int num_vtcs /* VTC */)
/* VTC End   ------------------------------------------ */

{
        EncodeControl *enc_ctl;

        Int     vo;

/*      Char    *string; */

        if(num_vos > MAX_NUM_VOS)
        {
                fprintf(stderr,"ERROR(AllocEncodeControl): Maximum number of VOs exceeded. See vm_config.h\n");
                exit(1);
        }

        enc_ctl = SallocEncodeControl();

        PutEncodeControlNumVOs(num_vos,enc_ctl);
/* VTC Begin ------------------------------------------ */
        PutEncodeControlNumVTCs(num_vtcs, enc_ctl);   /* VTC */
/* VTC End ------------------------------------------ */

        
        for(vo = 0; vo < num_vos; vo++)
                enc_ctl->vo_cfg_files[vo] = (Char*)malloc(sizeof(Char) * 100);
                
/* VTC Begin ------------------------------------------ */
        for(vo = 0; vo < num_vtcs; vo++)
                enc_ctl->vtc_cfg_files[vo] = (Char*)malloc(sizeof(Char) * 100);
/* VTC End ------------------------------------------ */

        return(enc_ctl);
}

/***********************************************************CommentBegin******
 *
 * -- FreeEncodeControl -- Deallocates all memory associated with a 
 *      EncodeControl structure
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      20-09-96
 *
 * Purpose :            
 *      Deallocates all memory associated with a EncodeControl structure
 *
 * Arguments in :       
 *      EncodeControl *enc_ctl - pointer to EncodeControl structure
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      none
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      -
 *
 * See also :
 *      -
 *
 * Modified :           
 *      -
 *      
 *
 ***********************************************************CommentEnd********/

Void
FreeEncodeControl(EncodeControl *enc_ctl)
{
        Int     num_vos,
                        vo;

/* VTC Begin ------------------------------------------ */
        Int     num_vtcs;
/* VTC End  ------------------------------------------ */

        Char *string;

        num_vos = GetEncodeControlNumVOs(enc_ctl);
        
        for(vo = 0; vo < num_vos; vo++)
        {
                string = GetEncodeControlCfgFile(vo,enc_ctl);
                free((Char*)string);
        }

/* VTC Begin ------------------------------------------ */
        num_vtcs = GetEncodeControlNumVTCs(enc_ctl);
        for(vo = 0; vo < num_vtcs; vo++)
        {
                string = GetEncodeControlVTCCfgFile(vo,enc_ctl);
                free((Char*)string);
        }
/* VTC End  ------------------------------------------ */

        free((Char*)enc_ctl);

        return;
}

/***********************************************************CommentBegin******
 *
 * -- GetEncodeControlXXXX -- Access functions for an EncodeControl structure
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      20-09-96
 *
 * Purpose :            
 *      Access functions for an EncodeControl structure. These functions retrieve
 *      the value of specific fields of the EncodeControl structure.
 *
 * Arguments in :       
 *      EncodeControl *enc_ctl - pointer to structure
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      Value of specified field in structure.
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      -
 *
 * See also :
 *      mom_access.c
 *
 * Modified :           
 *      16.01.98 Martina Eckert: Replace functions Getxxx for RC by one 
 *                               function GetEncodeControlRCParam.
 *
 *      
 *
 ***********************************************************CommentEnd********/
Int
GetEncodeControlNumVOs(EncodeControl *enc_ctl)
{
        return(enc_ctl->num_vos);
}

Char *
GetEncodeControlCfgFile(Int id, EncodeControl *enc_ctl)
{
        return(enc_ctl->vo_cfg_files[id]);
}

Float
GetEncodeControlStartTime(EncodeControl *enc_ctl)
{
        return(enc_ctl->start_time);
}

Float
GetEncodeControlEndTime(EncodeControl *enc_ctl)
{
        return(enc_ctl->end_time);
}

Float
GetEncodeControlTimeInc(EncodeControl *enc_ctl)
{
        return(enc_ctl->time_inc);
}

Char *
GetEncodeControlDispYFile(EncodeControl *enc_ctl)
{
        return(enc_ctl->display_y_file);
}

Char *
GetEncodeControlDispUFile(EncodeControl *enc_ctl)
{
        return(enc_ctl->display_u_file);
}

Char *
GetEncodeControlDispVFile(EncodeControl *enc_ctl)
{
        return(enc_ctl->display_v_file);
}

Int
GetEncodeControlDispSizeX(EncodeControl *enc_ctl)
{
        return(enc_ctl->dim_x);
}

Int
GetEncodeControlDispSizeY(EncodeControl *enc_ctl)
{
        return(enc_ctl->dim_y);
}
Int
GetEncodeControlPostFilterType(EncodeControl *enc_ctl)
{
        return(enc_ctl->post_filter_type);
}

/* UPM Global RC: */


Int
GetEncodeControlRCType(EncodeControl *enc_ctl)
{
        return(enc_ctl->rc_type);
}

Char *
GetEncodeControlRCParam(EncodeControl *enc_ctl)
{
        return(enc_ctl->rc_parameter_string);
}

Int
GetEncodeControlDispBitsPerPixel(EncodeControl *enc_ctl)
{
        return(enc_ctl->bits_per_pixel);
}

/***********************************************************CommentBegin******
 *
 * -- PutEncodeControlXXXX  -- Access functions for an EncodeControl structure
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      20-09-96
 *
 * Purpose :            
 *      Access functions for an EncodeControl structure. These functions set a 
 *      specific field in the structure to a specified value.
 *
 * Arguments in :       
 *      EncodeControl *enc_ctl - pointer to structure
 *      Value fo field to be filled.
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      none
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      -
 *
 * See also :
 *      mom_access.c
 *
 * Modified :           
 *      16.01.98 Martina Eckert: Replace functions Putxxx for RC by one 
 *                               function PutEncodeControlRCParam.
 *
 *      
 *
 ***********************************************************CommentEnd********/

Void
PutEncodeControlNumVOs(Int num_vos,EncodeControl *enc_ctl)
{
        enc_ctl->num_vos = num_vos;
}

Void
PutEncodeControlCfgFile(Char *string,Int id,EncodeControl *enc_ctl)
{
        strcpy(enc_ctl->vo_cfg_files[id],string);
}

Void
PutEncodeControlStartTime(Int time,EncodeControl *enc_ctl)
{
        enc_ctl->start_time = time;
}

Void
PutEncodeControlEndTime(Float time,EncodeControl *enc_ctl)
{
        enc_ctl->end_time = time;
}

Void
PutEncodeControlTimeInc(Float time,EncodeControl *enc_ctl)
{
        enc_ctl->time_inc = time;
}

Void
PutEncodeControlDispYFile(Char *string,EncodeControl *enc_ctl)
{
        strcpy(enc_ctl->display_y_file,string);
}

Void
PutEncodeControlDispUFile(Char *string,EncodeControl *enc_ctl)
{
        strcpy(enc_ctl->display_u_file,string);
}

Void
PutEncodeControlDispVFile(Char *string,EncodeControl *enc_ctl)
{
        strcpy(enc_ctl->display_v_file,string);
}

Void
PutEncodeControlDispSizeX(Int dim,EncodeControl *enc_ctl)
{
        enc_ctl->dim_x = dim;
}

Void
PutEncodeControlDispSizeY(Int dim,EncodeControl *enc_ctl)
{
        enc_ctl->dim_y = dim;
}

Void
PutEncodeControlPostFilterType(Int value,EncodeControl *enc_ctl)
{
        enc_ctl->post_filter_type = value;
}

/* UPM Global RC */

Void
PutEncodeControlRCType(Int rc_type, EncodeControl *enc_ctl)
{
  enc_ctl->rc_type = rc_type;
}

Void
PutEncodeControlRCParam(Char *rc_parameter_string, EncodeControl *enc_ctl)
{
        strcpy(enc_ctl->rc_parameter_string, rc_parameter_string);
}

Void
PutEncodeControlDispBitsPerPixel(Int value,EncodeControl *enc_ctl)
{
        enc_ctl->bits_per_pixel = value;
}

/***********************************************************CommentBegin******
 *
 * -- ReadEncoderControlFile -- Reads the data in the encoder control file
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      20-09-96
 *
 * Purpose :            
 *      Reads the data in the encoder control file and it in an allocated
 *      EncodeControl structure.
 *
 * Arguments in :       
 *      Char *control_file - Name of control file
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      EncodeControl *enc_ctl - pointer to allocated, filled EncodeControl
 *      structure.
 *
 * Side effects :       
 *      Memory is allocated in this function for an EncodeControl structure.
 *
 * Description :        
 *      -
 *
 * See also :
 *      -
 *
 * Modified :           
 *      15.01.97 Robert Danielsen: Corrected exit(0) to exit(1) in one case.
 *                      Solution from Paulo Nunes.
 *      16.01.98 Martina Eckert: Read one parameter string instead of six 
 *                      different parameters for rate control
 *
 ***********************************************************CommentEnd********/
EncodeControl *
ReadEncoderControlFile(Char *control_file)
{
        File    *fpcontrol_file;

        EncodeControl *enc_ctl;

        Int             num_vos,
                                dim_x,
                                dim_y,
                                post_filter_type,
                                bits_per_pixel,
    /* UPM Global RC */         rc_type,
                                vo;

        Char    garbage[201],
                                rc_parameter_string[50],
                                vo_cfg_file[100],
                                display_comp_file[100];

 /* VTC Begin ------------------------------------------ */
        Char    vtc_cfg_file[100];  /* VTC */
        Int     num_vtcs;  /* VTC */
 /* VTC End   ------------------------------------------ */

        if((fpcontrol_file = fopen(control_file,"r")) == NULL)
        {
                fprintf(stderr,"ERROR(ReadEncodeControl): unable to open %s\n",control_file);
                exit(1);
        }

        /* Read from file */

        /* Number of VOs first */
        fscanf(fpcontrol_file,"%d",&num_vos);
        fgets(garbage,201,fpcontrol_file);

 /* VTC Begin ------------------------------------------ */
        /* Number of VTCs first */
        fscanf(fpcontrol_file,"%d",&num_vtcs);
        fgets(garbage,201,fpcontrol_file);

        enc_ctl = AllocEncodeControl(num_vos, num_vtcs);  /* VTC */
 /* VTC End   ------------------------------------------ */

        /* enc_ctl = AllocEncodeControl(num_vos); */

        /* display YUV component file names */
        fscanf(fpcontrol_file,"%s",display_comp_file);
        fgets(garbage,201,fpcontrol_file);
        PutEncodeControlDispYFile(display_comp_file,enc_ctl);

        fscanf(fpcontrol_file,"%s",display_comp_file);
        fgets(garbage,201,fpcontrol_file);
        PutEncodeControlDispUFile(display_comp_file,enc_ctl);

        fscanf(fpcontrol_file,"%s",display_comp_file);
        fgets(garbage,201,fpcontrol_file);
        PutEncodeControlDispVFile(display_comp_file,enc_ctl);

        /* Display size X */
        fscanf(fpcontrol_file,"%d",&dim_x);
        fgets(garbage,201,fpcontrol_file);
        PutEncodeControlDispSizeX(dim_x,enc_ctl);

        /* Display size Y */
        fscanf(fpcontrol_file,"%d",&dim_y);
        fgets(garbage,201,fpcontrol_file);
        PutEncodeControlDispSizeY(dim_y,enc_ctl);

        /* flag to disable deblock filter */
        fscanf(fpcontrol_file,"%d",&post_filter_type);
        fgets(garbage,201,fpcontrol_file);
        PutEncodeControlPostFilterType(post_filter_type,enc_ctl);

        /* pixel depth of display */
        fscanf(fpcontrol_file,"%d",&bits_per_pixel);
        fgets(garbage,80,fpcontrol_file);
        PutEncodeControlDispBitsPerPixel(bits_per_pixel,enc_ctl);

        for(vo = 0; vo < num_vos; vo++)
        {
                /* Config. file names */
                fscanf(fpcontrol_file,"%s",vo_cfg_file);
                fgets(garbage,201,fpcontrol_file);
        
                PutEncodeControlCfgFile(vo_cfg_file,vo,enc_ctl);
        }

 /* VTC Begin   ------------------------------------------ */
        for(vo = 0; vo < num_vtcs; vo++)  /* VTC */
        {
                /* Config. file names */
                fscanf(fpcontrol_file,"%s",vtc_cfg_file);
                fgets(garbage,201,fpcontrol_file);
                fprintf(stderr,"hjlee: vtc_cfg_file: %s\n",vtc_cfg_file);
                PutEncodeControlVTCCfgFile(vtc_cfg_file,vo,enc_ctl);
        }
 /* VTC End   ------------------------------------------ */

        /* UPM Global RC */

        /* Rate Control Type */
        fscanf(fpcontrol_file,"%d",&rc_type);
        fgets(garbage,201,fpcontrol_file);
        PutEncodeControlRCType(rc_type, enc_ctl);
        
        /* Rate Control Parameter String */
        fscanf(fpcontrol_file,"%s",rc_parameter_string);
        fgets(garbage,201,fpcontrol_file);
        PutEncodeControlRCParam(rc_parameter_string, enc_ctl);


       

        return(enc_ctl);
}

/***********************************************************CommentBegin******
 *
 * -- PrintEncodeControl -- Prints the contents of an EncodeControl structure
 *              to a file
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      20-09-96
 *
 * Purpose :            
 *      Prints the contents of an EncodeControl structure to a file. This data
 *      can also be printed to the screen if one of the input arguments is
 *      stdout.
 *
 * Arguments in :       
 *      EncodeControl *enc_ctl - pointer to EncodeControl structure
 *      File *file - file pointer to where to print data
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      none
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      -
 *
 * See also :
 *      -
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/
Void
PrintEncodeControl(EncodeControl *enc_ctl, File *file)
{
        Int             vo;

        fprintf(file,"\n\n\n");
        fprintf(file,"***************************************************\n");
        fprintf(file,"*         MPEG4 Video Verification Model          *\n");
        fprintf(file,"*                                                 *\n");
        fprintf(file,"*             MoMuSys Implementation              *\n");
        fprintf(file,"*                                                 *\n");
        fprintf(file,"***************************************************\n\n\n");

        fprintf(file,"# VOs : %d\n", (Int)GetEncodeControlNumVOs(enc_ctl));
        fprintf(file,"Configuration files :\n");
        for(vo = 0; vo < GetEncodeControlNumVOs(enc_ctl); vo++)
                fprintf(file,"%s\n",GetEncodeControlCfgFile(vo,enc_ctl));
        fprintf(file,"\nSession start time : %f\n",
                (Float)GetEncodeControlStartTime(enc_ctl));
        fprintf(file,"Session end time : %f\n",
                (Float)GetEncodeControlEndTime(enc_ctl));

        fprintf(file,"\nDisplay files : \n");
        fprintf(file,"%s\n",GetEncodeControlDispYFile(enc_ctl));
        fprintf(file,"%s\n",GetEncodeControlDispUFile(enc_ctl));
        fprintf(file,"%s\n",GetEncodeControlDispVFile(enc_ctl));
        fprintf(file,"Display size : %d x %d\n\n",
                (Int)GetEncodeControlDispSizeX(enc_ctl),
                (Int)GetEncodeControlDispSizeY(enc_ctl));
        if(!GetEncodeControlPostFilterType(enc_ctl))
                fprintf(file,"Deblocking Filter is DISABLED\n");
        else
                fprintf(file,"Deblocking Filter for composited image is ENABLED\n");

        /* UPM Global RC */
        if(GetEncodeControlRCType(enc_ctl) == NO_RATE_CONTROL)
          fprintf(file,"No rate control\n");
        else if(GetEncodeControlRCType(enc_ctl) == VM4_RATE_CONTROL)
          fprintf(file,"VM 4 rate control will be used\n");
        else if(GetEncodeControlRCType(enc_ctl) == VM5_RATE_CONTROL)
          fprintf(file,"VM 5 rate control will be used\n");

        fprintf(file,"\n\n");

        return;
}


/***********************************************************CommentBegin******
 *
 * -- SallocVOConfig -- Allocate memory for a VOConfig structure
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      20-09-96
 *
 * Purpose :            
 *      Allocates memory for a VOConfig structure (but not for associated
 *      linked list of VolConfig structures).
 *
 * Arguments in :       
 *      none
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      VOConfig *vo_cfg - pointer to allocated structure.
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      -
 *
 * See also :
 *      -
 *
 * Modified :           
 *
 *     Hung-Ju Lee & Jie Liang 20.01.98: added PutSOLconfigLayers (NULL, vo_cfg)
 *      
 *
 ***********************************************************CommentEnd********/

VOConfig *
SallocVOConfig()
{
        VOConfig *vo_cfg;

        vo_cfg = (VOConfig *)malloc(sizeof(VOConfig));
        
        PutVOConfigLayers(NULL,vo_cfg);
        PutVOConfigNext(NULL,vo_cfg);

        return(vo_cfg);
}

/***********************************************************CommentBegin******
 *
 * -- FreeVOConfigList -- Deallocate all memory associated with a linked
 *      list of VOConfig structures
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      20-09-96
 *
 * Purpose :            
 *      Deallocate all memory associated with a linked list of VOConfig structures.
 *
 * Arguments in :       
 *      VOConfig *vo_cfg_list - pointer to linked list
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      none
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      -
 *
 * See also :
 *      -
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/
Void
FreeVOConfigList(VOConfig *vo_cfg_list)
{
        VOConfig        *curr,
                                                *next;

        curr = vo_cfg_list;
        
        while(curr != NULL)
        {
                next = GetVOConfigNext(curr);

                FreeVolConfigList(GetVOConfigLayers(curr));

                free((Char*)curr);

                curr = next;
        }

        return;
}

/***********************************************************CommentBegin******
 *
 * -- PutVOConfigXXX -- Access functions for VOConfig structure
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      20-09-96
 *
 * Purpose :            
 *      Sets a particular field of a VOConfig structure to a specified value.
 *
 * Arguments in :       
 *      value field is to be set to
 *      VOConfig *vo_cfg - pointer to VOConfig structure.
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      none
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      -
 *
 * See also :
 *      -
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/
Void
PutVOConfigId(Int id,VOConfig *vo_cfg)
{
        vo_cfg->id = id;
}

Void
PutVOConfigNumVOLs(Int num,VOConfig *vo_cfg)
{
        vo_cfg->num_layers = num;
}

Void
PutVOConfigLayers(VolConfig *layers,VOConfig *vo_cfg)
{
        vo_cfg->layers = layers;
}

Void
PutVOConfigNext(VOConfig *next,VOConfig *vo_cfg)
{
        vo_cfg->pnext = next;
}

/***********************************************************CommentBegin******
 *
 * -- GetVOConfigXXX -- access functions for a VOConfig structure
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      20-09-96
 *
 * Purpose :            
 *      Returns the value of a particular field in a VOConfig structure.
 *
 * Arguments in :       
 *      VOConfig *vo_cfg - pointer to VOConfig structure
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      Value of specified field.
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      -
 *
 * See also :
 *      -
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/
Int
GetVOConfigId(VOConfig *vo_cfg)
{
        return(vo_cfg->id);
}

Int
GetVOConfigNumVOLs(VOConfig *vo_cfg)
{
        return(vo_cfg->num_layers);
}

VolConfig *
GetVOConfigLayers(VOConfig *vo_cfg)
{
        return(vo_cfg->layers);
}

VOConfig *
GetVOConfigNext(VOConfig *vo_cfg)
{
        return(vo_cfg->pnext);
}

/***********************************************************CommentBegin******
 *
 * -- AddVOConfigToList -- Adds a VOConfig structure to the end of a linked
 *      list of VOConfigs.
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      20-09-96
 *
 * Purpose :            
 *      Adds a VOConfig structure to the end of a linked list of VOConfigs. This
 *      function assumes that the list is not empty to begin with.
 *
 * Arguments in :       
 *      VOConfig *vo_cfg - pointer to VOConfig to be added
 *
 * Arguments in/out :   
 *      VOConfig *list - pointer to linked list of VOConfigs.
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      none
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      -
 *
 * See also :
 *      -
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/
Void
AddVOConfigToList(VOConfig *vo_cfg,VOConfig *list)
{
        VOConfig        *curr,
                                                *prev;

        prev = list;

        curr = GetVOConfigNext(prev);

        while(curr != NULL)
        {
                prev = curr;

                curr = GetVOConfigNext(prev);
        }

        PutVOConfigNext(vo_cfg,prev);
}

/***********************************************************CommentBegin******
 *
 * -- GetVOConfigId -- Returns a pointer to a VOConfig in a linked list of VOs,
 *                                              identified by it's Id.
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      20-11-96
 *
 * Purpose :            
 *      Returns a pointer to a VOConfig in a linked list of VOConfig,
 *              identified by it's Id.
 *
 * Arguments in :       
 *      VOConfig  *list - linked list of VOConfigs
 * Int vo_cfg_id - VOConfig Id
 *
 * Arguments in/out :   
 *      -
 *
 * Arguments out :      
 *      -
 *
 * Return values :      
 *      -
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      
 *
 * See also :
 *      
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/
VOConfig *
GetVOConfigById(VOConfig *list,Int vo_cfg_id)
{
        VOConfig *curr;

        Int     id;

        curr = list;

        while(curr != NULL)
        {
                id = GetVOConfigId(curr);
                if(id != vo_cfg_id)
                        curr = GetVOConfigNext(curr);
                else
                        break;
        }

        return(curr);
}



/***********************************************************CommentBegin******
 *
 * -- ReadConfigFiles -- Read the data contained in the various configuration
 *      files on disk.
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      20-09-96
 *
 * Purpose :            
 *      Read the data contained in the various configuration
 *      files on disk. This data is placed in a linked list of VOConfig
 *      structures.
 *
 * Arguments in :       
 *      EncodeControl *enc_ctl - pointer to an EncodeControl structure (holds
 *      the name of the configuration files).
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      VOConfig *list - pointer to linked list of VOConfigs
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      -
 *
 * See also :
 *      -
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/
VOConfig *
ReadConfigFiles(EncodeControl *enc_ctl)
{
        Int                             num_vos,
                                                vo;

        VOConfig        *list = NULL,
                                                *curr;

        Char                    *string;

        num_vos = GetEncodeControlNumVOs(enc_ctl);

        for(vo = 0; vo < num_vos; vo++)
        {
                string = GetEncodeControlCfgFile(vo,enc_ctl);

                curr = ReadVOConfigFile(string);

                if(vo == 0)
                        list = curr;
                else
                        AddVOConfigToList(curr,list);
        }

        return(list);
}


/***********************************************************CommentBegin******
 *
 * -- ReadVOLConfigData -- Reads the VOL config information from the VO config
 *      file
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      20-09-96
 *
 * Purpose :            
 *      Reads the VOL config information from the VO config
 *      file and places this in a linked list of VolConfig structures
 *      (one node for each layer, with BASE layer as first node).
 *
 * Arguments in :       
 *      Int num_layer - number of coding layers
 *      File *fpfile - pointer to file to read from
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      VolConfig *vol_config_list - pointer to linked list of VolConfig
 *      structures
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      -
 *
 * See also :
 *      -
 *
 * Modified :           
 *      28.01.97 Robert Danielsen: Added fix from Noel O'Connor: added
 *                      PutVolConfigEnhanceType().
 * 04.02.97 Noel O'Connor: Enabled error res. mode (based on changes sent by
 *                      Luis Ducla-Soares)
 * 26.02.97 Noel O'Connor: Added reverse_vlc and shape_effects fields
 * 10.03.97 Minhua Zhou  : Added M for B-VOPs 
 * 23.04.97 Michael Wollborn: Added MPEG quantization stuff completely
 * 26.04.97 Luis Ducla-Soares: Enabled the use of reversible VLCs and added
 *                             data_partitioning field.
 * 12.08.97 Minhua Zhou : added the calculation of vop_time_increment_resolution 
 * 29.08.97 Osamu Sunohara: added PutVolConfigEnhPredType(). 
 * 08.09.97 Cecile Dufour: added reading parameters for Static Sprites   
 * 19.11.97 Noel Brady: added packet size  
 * 11,12.97 Bob Eifrig: added interlaced, top field first, alternate
 *                      scan, MV file, direct search radius
 * 16.01.98 Martina Eckert: Read one parameter string instead of two
 *                      different parameters for rate control
 * 23.03.98 Michael Wollborn: included user_data() due to N2171, Clause 2.1.9
 * 21.05.98 Ji Heon Kweon : support for grayscale coding 
 * 03.06.98 Marc Mongenet (EPFL): added Complexity Estimation syntax support
 * 31.08.98 Guido Heising: - non version 1 parameters removed, i.e. blending,
 *                           dynamic sprites, sadct, ac/dc pred disable, shape effects.
 *                         - intra_dc_vlc_thr added
 * 28.01.99 Yoshinori Suzuki (Hitachi, Ltd.): added 
 *                                 PutVolConfigRoundingControlDisable() and
 *                                 PutVolConfigInitialRoundingType() 
 * 15.02.99 U. Benzler : added quarter pel support
 * 03.03.99 Seishi TAKAMURA (NTT): added GMC coding, MPEG-4 version2 syntax
 * 16.08.99 Shigeru Fukunaga (Oki): added modules for NEWPRED
 * 18.08.99 Seishi TAKAMURA (NTT): modify GMC coding  (warping type check)
 * 06.09.99 Eishi Morimatsu (Fujitsu Labs.): added DRC support
 * 09.29.99 Dae-Sung Cho (Samsung AIT): added support for OBSS
 * 11.11.99 Sven Brandau (HHI): added support for MAC
 * 11.11.99 U. Benzler (UoH) - added TypeIndication to be able to create conformance bitstreams
 *
  ***********************************************************CommentEnd********/ 
VolConfig *
ReadVOLConfigData(Int num_layers,File *fpfile)
{
  Int                           layer;
  
  Char
    /*    *cfg_file, */
    garbage[201];
  
  VolConfig     *vol_config_list = NULL,
                *curr;

  /* Data to be read from config files */
  
  Int vo_type_ind, /* 11.11.99 U. Benzler (UoH) - added TypeIndication to be able to create conformance bitstreams */
    id,M,time_incre_res,
    disk_seq_x,
    disk_seq_y,
    shape,
    write_coded,
    start_frame,
    end_frame,
    bit_rate,
    alpha_th,
    change_CR_disable,
    obmc_disable,
    quant_precision,
    bits_per_pixel,
    quant_type,
    load_intra_quant_mat, load_nonintra_quant_mat,
    load_gray_intra_quant_mat, load_gray_nonintra_quant_mat,
    intra_quantizer,
    quantizer,
    error_res_disable,
    rounding_control_disable,
    initial_rounding_type,
    /*    intra_acdc_pred_disable, unused SB */
    /* 01.02.99 HHI Schueuer */
    sadct_disable,
    /* end HHI */
    intra_dc_vlc_thr,
    sr_for,
    quarter_pel,		/* MW QPEL 07-JUL-1998 */
    scalability,
    scalability_type,
    enh_pred_type,    /* <-added by Sony 290897 */
    ref_ident,
    ref_samp_dir,
    hor_samp_n,
    hor_samp_m,
    ver_samp_n,
    ver_samp_m,
    enhancement_type,
/* begin: added for OBSS by Samsung AIT (1999-09-29) */
    use_ref_shape,
    use_ref_texture,
    shape_hor_samp_n,
    shape_hor_samp_m,
    shape_ver_samp_n,
    shape_ver_samp_m,
/* end: added for OBSS by Samsung AIT (1999-09-29) */
    intra_period,
    GOV_period,
    sprite_hdim,
    sprite_vdim,
    sprite_left_edge,
    sprite_top_edge,
    /*    write_sprite,    unused SB */
    sprite_usage,
    sprite_shape,
    brightness_change_in_sprite,
    low_latency_sprite_enable,
    no_of_sprite_points,
    warping_accuracy,
    reverse_vlc,
    /*    shape_effects,   unused SB */
    gl_quant,
    disable_gray_quant_update,
    data_partitioning,
                packet_size;
  Int version;	/* MPEG1 / MPEG2 switch  : added NTT */
  
 
  Int                           intra_shape_period;
  Float                         SNR_threshold;
  
  Char coded_y_file[100],
    coded_u_file[100],
    coded_v_file[100],
    coded_a_file[100],
    bitstream[100],
    disk_y_file[100],
    disk_u_file[100],
    disk_v_file[100],
    disk_a_file[100],
    sprite_y_file[100], 
    sprite_u_file[100],
    sprite_v_file[100],
    sprite_a_file[100],
    warp_param_file[100],
    intra_quant_mat_file[100], 
    nonintra_quant_mat_file[100],
    gray_intra_quant_mat_file[100], 
    gray_nonintra_quant_mat_file[100],
    user_data_in_vol_file[100], /* Included due to N2171, Clause 2.1.9 MW 23-MAR-1998 */
    user_data_in_gov_file[100]; /* Included due to N2171, Clause 2.1.9 MW 23-MAR-1998 */

  /* Included due to N2171, Clause 2.1.9 MW 23-MAR-1998 */
  Int is_user_data_in_vol=0, is_user_data_in_gov=0;

  /* Complexity Estimation syntax support - Marc Mongenet (EPFL) - 4 Jun 1998 */
  Int complexity_estimation_disable;
  Char complexity_estimation_file [100];

  Int vol_control_parameters,
      vbv_parameters;
  Char vbv_parameter_string[50];

  Float frame_rate,
    disk_seq_frame_rate,
    start_time;
   Char
    rc_parameter_string[50];   /* UPM Global RC */
  
  /** added for NEWPRED (Oki) 16-AUG-1999 **/
  Int newpred_enable;
  Int newpred_backward_message_type;
  Int newpred_segment_type;
  Char newpred_ctl_file[100];
  /** end of NEWPRED (Oki) 16-AUG-1999 **/

/* >>> added for DRC by Fujitsu (top)    <<< */
  Int	drc_enable, switching_mode;
/* >>> added for DRC by Fujitsu (bottom) <<< */

  /* added for MAC (SB) 11-Nov-99 */
  Int   mac_enable = 0;
  Int   shape_extension, n;
  Char  disk_g_file[100], coded_g_file[100];


  for(layer = 0; layer < num_layers; layer++)
    {
      curr = AllocVolConfig();
      
      /* ReadFromFile */
      
/* modified NTT for version2 syntax : start */
      /* MPEG Version1 / Version2 */
      fscanf(fpfile,"%d",&version);
      fgets(garbage,201,fpfile);
      PutVolConfigVersion(version, curr);
/* modified NTT for version2 syntax : end */

      /* VOL Id */
      fscanf(fpfile,"%d",&id);
      fgets(garbage,201,fpfile);
      PutVolConfigId(id,curr);

      /* VOL TypeIndication *//* 11.11.99 U. Benzler (UoH) - added TypeIndication to be able to create conformance bitstreams */
      fscanf(fpfile,"%d",&id);
      fgets(garbage,201,fpfile);
      PutVolConfigTypeIndication(id,curr);

      
      /* VOL Frame Rate */
      fscanf(fpfile,"%f",&frame_rate);
      fgets(garbage,201,fpfile);
      PutVolConfigFrameRate(frame_rate,curr);

      /* VOL Time Increment Resolution */
      /* 10.08.98 Sven Brandau:  "change time_incre_res to 16 Bit" due to N2339, Clause 2.1.2 */
      for (time_incre_res=1;time_incre_res<65536;time_incre_res++) {
        Double unit;
        unit = (Double)time_incre_res/frame_rate;
        if (fabs(unit-(Double)((Int)(unit+0.0001)))<0.0001)
          break;
         
      }  
      if (time_incre_res==65536) time_incre_res=65535;
      PutVolConfigTimeIncrementResolution(time_incre_res,curr);
      
      /* VOL value of M (>=1)  */
      fscanf(fpfile,"%d",&M);
      fgets(garbage,201,fpfile);
      PutVolConfigM(M,curr);
      
      /* X dim of source sequence */
      fscanf(fpfile,"%d",&disk_seq_x);
      fgets(garbage,201,fpfile);
      PutVolConfigDiskSeqX(disk_seq_x,curr);
      
      /* Y dim of source sequence */
      fscanf(fpfile,"%d",&disk_seq_y);
      fgets(garbage,201,fpfile);
      PutVolConfigDiskSeqY(disk_seq_y,curr);
      
      /* Disk seq. frame rate */
      fscanf(fpfile,"%f",&disk_seq_frame_rate);
      fgets(garbage,201,fpfile);
      PutVolConfigDiskSeqFrameRate(disk_seq_frame_rate,curr);
      
      /* Type of alpha channel */
      fscanf(fpfile,"%d",&shape);
      fgets(garbage,201,fpfile);
 
      /* HYUNDAI (Grayscale) */
      if(shape == 2) shape = 3;
      else if(shape == 3) shape = 2;
  
      PutVolConfigShape(shape,curr);

      /* start MAC enable - K. Suehring 2000-02-11 */
      fscanf(fpfile,"%d",&mac_enable);
      fgets(garbage,201,fpfile); 
      if ((mac_enable!=0)&&(mac_enable!=1)) {
        fprintf(stderr,"mac_enable must be '0' or '1'\n");
        exit(-1);
      }
      if (mac_enable) {
        /* begin MAC extensions - Sven Brandau 11-Nov-99 */
        fscanf(fpfile,"%d",&shape_extension);
        fgets(garbage,201,fpfile); 
        PutVolConfigShapeExtension(shape_extension,curr);
        if (shape==GREY_SCALE) {
          if (version==1)
            PutVolConfigAuxCompCount(1,curr); /* Version 1 = one graylevel shape */
          else
            PutVolConfigAuxCompCount(GetAuxCompCount(shape_extension),curr);
        } else
          PutVolConfigAuxCompCount(0,curr);
        if (GetVolConfigAuxCompCount(curr)>MAX_MAC) {
          fprintf(stderr,"Can't encode more than %d auxillary components! ... exit\n", MAX_MAC );
          exit(-1);
        }
        /* end MAC extensions - Sven Brandau 11-Nov-99 */
       } else {
         /*  set standard values if MAC disabled */
         PutVolConfigShapeExtension(0,curr);
         if (shape==GREY_SCALE) { 
           PutVolConfigAuxCompCount(1,curr);
         }else {
           PutVolConfigAuxCompCount(0,curr);
         }
       }
      /* end MAC enable - K. Suehring 2000-02-11 */
    
      /* obmc_disable */
       fscanf(fpfile,"%d",&obmc_disable);
       fgets(garbage,201,fpfile);
       PutVolConfigOBMCDisable(obmc_disable,curr);
     
      /* quant_precision */
       fscanf(fpfile,"%d",&quant_precision);
       fgets(garbage,80,fpfile);
       PutVolConfigQuantPrecision(quant_precision,curr);
     
      /* bits_per_pixel */
       fscanf(fpfile,"%d",&bits_per_pixel);
       fgets(garbage,80,fpfile);
       PutVolConfigBitsPerPixel(bits_per_pixel,curr);
      
       PutVolConfigShapeEffects(0,curr);
      /* Type of shape effects to be used
      fscanf(fpfile,"%d",&shape_effects);
      fgets(garbage,201,fpfile);
      PutVolConfigShapeEffects(shape_effects,curr);
      Ensure correct default  value for the time being - NOC 
      if(shape_effects != NO_SHAPE_EFFECTS)
        {
          fprintf(stderr,"Shape effects flag in .cfg file must be set to 0\n");
          fprintf(stderr,"Shape effects are not currently supported.\n");
          exit(1);
        }*/

   
      /* Source seq file names */
      fscanf(fpfile,"%s",disk_y_file);
      fgets(garbage,201,fpfile);
      PutVolConfigY(disk_y_file,curr);
      
      fscanf(fpfile,"%s",disk_u_file);
      fgets(garbage,201,fpfile);
      PutVolConfigU(disk_u_file,curr);
      
      fscanf(fpfile,"%s",disk_v_file);
      fgets(garbage,201,fpfile);
      PutVolConfigV(disk_v_file,curr);
      
      fscanf(fpfile,"%s",disk_a_file);
      fgets(garbage,201,fpfile);
      PutVolConfigA(disk_a_file,curr);

      /* start MAC enable - K. Suehring 2000-02-11 */
      if ((mac_enable==0) && (shape==GREY_SCALE)){
        /* if mac diabled an grayscale use binary file for alpha */
        PutVolConfigG(disk_a_file,0,curr);
      } else {
        /* MAC stuff - SB 11-Nov-99 */
        for(n=0;n<GetVolConfigAuxCompCount(curr);n++) {
          fscanf(fpfile,"%s",disk_g_file);
          fgets(garbage,201,fpfile);
          PutVolConfigG(disk_g_file,n,curr);
        }
      }
      /* end MAC enable - K. Suehring 2000-02-11 */

      /* Write/No-write coded flag */
      fscanf(fpfile,"%d",&write_coded);
      fgets(garbage,201,fpfile);
      PutVolConfigWriteCoded(write_coded,curr);
      
      /* Coded seq file names */
      fscanf(fpfile,"%s",coded_y_file);
      fgets(garbage,201,fpfile);
      PutVolConfigCodedY(coded_y_file,curr);
      
      fscanf(fpfile,"%s",coded_u_file);
      fgets(garbage,201,fpfile);
      PutVolConfigCodedU(coded_u_file,curr);
                
      fscanf(fpfile,"%s",coded_v_file);
      fgets(garbage,201,fpfile);
      PutVolConfigCodedV(coded_v_file,curr);
      
      fscanf(fpfile,"%s",coded_a_file);
      fgets(garbage,201,fpfile);
      PutVolConfigCodedA(coded_a_file,curr);

      /* start MAC enable - K. Suehring 2000-02-11 */
      if ((mac_enable==0) && (shape==GREY_SCALE)){
        /* if mac diabled an grayscale use binary file for alpha */
        strncat (coded_a_file,".0.aux",98);
        PutVolConfigCodedG(coded_a_file,0,curr);
      } else {
        /* MAC stuff - SB 11-Nov-99 */
        for(n=0;n<GetVolConfigAuxCompCount(curr);n++) {
          fscanf(fpfile,"%s",coded_g_file);
          fgets(garbage,201,fpfile);
          PutVolConfigCodedG(coded_g_file,n,curr);
        }
      }
      /* end MAC enable - K. Suehring 2000-02-11 */

      /* Bitstream file name */
      fscanf(fpfile,"%s",bitstream);
      fgets(garbage,201,fpfile);
      PutVolConfigBitstream(bitstream,curr);
      
      /* Start frame */
      fscanf(fpfile,"%d",&start_frame);
      fgets(garbage,201,fpfile);
      PutVolConfigStartFrame(start_frame,curr);
      
      /* End frame */
      fscanf(fpfile,"%d",&end_frame);
      fgets(garbage,201,fpfile);
      PutVolConfigEndFrame(end_frame,curr);
      
      /* Start time */
      fscanf(fpfile,"%f",&start_time);
      fgets(garbage,201,fpfile);
      PutVolConfigStartTime(start_time,curr);

      /* Target bitrate */
      fscanf(fpfile,"%d",&bit_rate);
      fgets(garbage,201,fpfile);
      PutVolConfigBitrate(bit_rate,curr);
      
      /* Type of Rate control -> ctl-file! 
      fscanf(fpfile,"%d",&rate_control);
      fgets(garbage,201,fpfile);
      PutVolConfigRateControl(rate_control,curr);
      */

       /* UPM: Global RC parameter string */
      fscanf(fpfile,"%s",rc_parameter_string);
      fgets(garbage,201,fpfile);
      PutVolConfigRCParam(rc_parameter_string, curr);

      
       /* Threshold for shape coding */
      fscanf(fpfile,"%d",&alpha_th);
      fgets(garbage,201,fpfile);
      PutVolConfigAlphaTh(alpha_th,curr);
      
      /* change_CR_disable */
      fscanf(fpfile,"%d",&change_CR_disable);
      fgets(garbage,201,fpfile);
      PutVolConfigChangeCRDisable(change_CR_disable,curr);

      /* Quantizer type */
      fscanf(fpfile,"%d",&quant_type);
      fgets(garbage,201,fpfile);
      PutVolConfigQuantType(quant_type,curr);

      /* Added stuff for MPEG quantization 23-APR-1997 MW */

      /* load_intra_quant_mat flag (0-no, 1-userdef, >1-predef) */
      fscanf(fpfile,"%d",&load_intra_quant_mat);
      fgets(garbage,201,fpfile);
      PutVolConfigLoadIntraQuantMat(load_intra_quant_mat,curr);
      
      /* intra quant. mat. filename (if userdef.) */
      fscanf(fpfile,"%s",intra_quant_mat_file);
      fgets(garbage,201,fpfile);
      PutVolConfigIntraQuantMatFile(intra_quant_mat_file,curr);
      
      /* load_nonintra_quant_mat flag (0-no, 1-userdef, >1-predef) */
      fscanf(fpfile,"%d",&load_nonintra_quant_mat);
      fgets(garbage,201,fpfile);
      PutVolConfigLoadNonintraQuantMat(load_nonintra_quant_mat,curr);
      
      /* nonintra quant. mat. filename (if userdef.) */
      fscanf(fpfile,"%s",nonintra_quant_mat_file);
      fgets(garbage,201,fpfile);
      PutVolConfigNonintraQuantMatFile(nonintra_quant_mat_file,curr);
          
      /* load_gray_intra_quant_mat flag (0-no, 1-userdef, >1-predef) */
      fscanf(fpfile,"%d",&load_gray_intra_quant_mat);
      fgets(garbage,201,fpfile);
      PutVolConfigLoadGrayIntraQuantMat(load_gray_intra_quant_mat,curr);
          
      /* gray intra quant. mat. filename (if userdef.) */
      fscanf(fpfile,"%s",gray_intra_quant_mat_file);
      fgets(garbage,201,fpfile);
      PutVolConfigGrayIntraQuantMatFile(gray_intra_quant_mat_file,curr);

      /* load_gray_nonintra_quant_mat flag (0-no, 1-userdef, >1-predef) */
      fscanf(fpfile,"%d",&load_gray_nonintra_quant_mat);
      fgets(garbage,201,fpfile);
      PutVolConfigLoadGrayNonintraQuantMat(load_gray_nonintra_quant_mat,curr);

      /* gray nonintra quant. mat. filename (if userdef.) */
      fscanf(fpfile,"%s",gray_nonintra_quant_mat_file);
      fgets(garbage,201,fpfile);
      PutVolConfigGrayNonintraQuantMatFile(gray_nonintra_quant_mat_file,curr);
      /* End of MPEG-like quantization stuff */
      
      /* INTRA Quantizer */
      fscanf(fpfile,"%d",&intra_quantizer);
      fgets(garbage,201,fpfile);
      PutVolConfigIntraQuantizer(intra_quantizer,curr);
      
      /* Quantizer */
      fscanf(fpfile,"%d",&quantizer);
      fgets(garbage,201,fpfile);
      PutVolConfigQuantizer(quantizer,curr);

      /* B_Quantizer */
      fscanf(fpfile,"%d",&quantizer);
      fgets(garbage,201,fpfile);
      PutVolConfigBQuantizer(quantizer,curr);


      /* quantizer for GL alpha maps : I_VOPs*/
      fscanf(fpfile,"%d",&gl_quant);
      fgets(garbage,201,fpfile);
      PutVolConfigIntraGLQuant(gl_quant,curr);
      
      /* HYUNDAI (Grayscale) */
      /* quantizer for GL alpha maps : P-VOPs*/
      fscanf(fpfile,"%d",&gl_quant);
      fgets(garbage,201,fpfile);
      PutVolConfigGLQuant(gl_quant,curr);
      
      /* HYUNDAI (Grayscale) */
      /* quantizer for GL alpha maps : B-VOPs*/
      fscanf(fpfile,"%d",&gl_quant);
      fgets(garbage,201,fpfile);
      PutVolConfigBGLQuant(gl_quant,curr);
      
      /* quantizer update flag for GL quantization of alpha maps */
      fscanf(fpfile,"%d",&disable_gray_quant_update);
      fgets(garbage,201,fpfile);
      PutVolConfigDisableGrayQuantUpdate(disable_gray_quant_update,curr);
      
      /* Rounding Control disable flag */
      fscanf(fpfile,"%d",&rounding_control_disable);
      fgets(garbage,201,fpfile);
      PutVolConfigRoundingControlDisable(rounding_control_disable,curr);

      /* Initial Value for Rounding Control */
      fscanf(fpfile,"%d",&initial_rounding_type);
      fgets(garbage,201,fpfile);
      PutVolConfigInitialRoundingType(initial_rounding_type,curr);

      /* Error Resilience disable flag */
      fscanf(fpfile,"%d",&error_res_disable);
      fgets(garbage,201,fpfile);
      PutVolConfigErrorResDisable(error_res_disable,curr);
      
      /* Data Partitioning Enable Flag */
      fscanf(fpfile,"%d",&data_partitioning);
      fgets(garbage,201,fpfile);
      if (error_res_disable) data_partitioning=0;
      PutVolConfigDataPartEnable(data_partitioning,curr);


      /* Flag to indicate whether or not to use reversible VLCs */
    
      fscanf(fpfile,"%d",&reverse_vlc);
      fgets(garbage,201,fpfile);
      if (error_res_disable || (data_partitioning==0)) reverse_vlc=0;
      PutVolConfigReverseVlc(reverse_vlc,curr);

      /* size of video packet in bits */
      fscanf(fpfile,"%d",&packet_size);
      fgets(garbage,201,fpfile);
      PutVolConfigPacketSize(packet_size,curr);

      /* for error resil. mode: specifies the period in frame for INTRA shape */
      fscanf(fpfile,"%d",&intra_shape_period);
      fgets(garbage,201,fpfile);
      PutVolConfigIntraShapePeriod(intra_shape_period,curr);
                                  
      /** added for NEWPRED (Oki) 16-AUG-1999 **/
      if(version != 1) { /* version ID == version 2 */
	/* NEWPRED enable flag */
	fscanf(fpfile,"%d",&newpred_enable);
	fgets(garbage,201,fpfile);
	PutVolConfigNewpredEnable(newpred_enable,curr);

	if(newpred_enable){
	  /* NEWPRED backward message type */
	  fscanf(fpfile,"%d",&newpred_backward_message_type);
	  fgets(garbage,201,fpfile);
	  PutVolConfigNewpredBackwardMessageType(newpred_backward_message_type,curr);
	
	/* NEWPRED segment type */
	  fscanf(fpfile,"%d",&newpred_segment_type);
	  fgets(garbage,201,fpfile);
	  PutVolConfigNewpredSegmentType(newpred_segment_type,curr);
	
	  /* NEWPRED control filename */
	  fscanf(fpfile,"%s",newpred_ctl_file);
	  fgets(garbage,201,fpfile);
	  PutVolConfigNewpredCtlFile(newpred_ctl_file,curr);
	}
	else{  /** Oki 26-AUG-1999 **/
	  PutVolConfigNewpredSegmentType(0,curr);
	}
      }
      else{
	PutVolConfigNewpredEnable(0,curr);
	PutVolConfigNewpredSegmentType(0,curr); /** Oki 26-AUG-1999 **/
      }
      /** end of NEWPRED (Oki) 16-AUG-1999 **/

/* >>> added for DRC by Fujitsu (top)    <<< */
      if(version != 1) { /* version ID == version 2 */
        /* Reading reduced resolution vop enable (0:disabled, 1:enabled) */
        fscanf(fpfile,"%d",&drc_enable);
        fgets(garbage,201,fpfile);
        PutVolConfigDrcEnable(drc_enable,curr);

        /* switching mode */
        fscanf(fpfile,"%d",&switching_mode);
        fgets(garbage,201,fpfile);
        PutVolConfigDrcSwitchingMode(switching_mode,curr);
	
      } else {
        PutVolConfigDrcEnable(0,curr);
        PutVolConfigDrcSwitchingMode(0,curr);
      }
/* >>> added for DRC by Fujitsu (bottom) <<< */

      PutVolConfigACDCPredDisable(0,curr);  

      /* intra_dc_vlc_thr field */  /* 27.08.98 Sven Brandau: added */
      fscanf(fpfile,"%d",&intra_dc_vlc_thr);
      fgets(garbage,201,fpfile);
      PutVolConfigIntraDCVlcThr(intra_dc_vlc_thr,curr);

      /* 01.02.99 HHI Schueuer */
      /* SADCT disable flag */
      fscanf(fpfile,"%d",&sadct_disable);
      fgets(garbage,201,fpfile);
      PutVolConfigSADCTDisable(sadct_disable,curr);
      /* end HHI */

      /* MW QPEL 07-JUL-1998 >> */
      /* Quarter-Pel Flag */
      fscanf(fpfile,"%d",&quarter_pel);
      fgets(garbage,80,fpfile);
      PutVolConfigQuarterPel(quarter_pel,curr);
      /* << MW QPEL 07-JUL-1998 */

/* modified NTT for version2 syntax : start */
      if (GetVolConfigVersion(curr) == 1 && GetVolConfigQuarterPel(curr)) {
	  fprintf(stderr, "error: Qpel is not defiend MPEG-4 version1 syntax, resetting to Qpel off\n");
	  PutVolConfigQuarterPel(0,curr); /* 990324 U. Benzler : changed to simply switch off Version 2 tools in Version 1 mode */
	  /*exit(-1);*/
      }
/* modified NTT for version2 syntax : end */

      /* sr_per_coded_frame */
      fscanf(fpfile,"%d",&sr_for);
      fgets(garbage,201,fpfile);
      PutVolConfigSearchRange(sr_for,curr);
      
        
      /* Scalability flag */
      fscanf(fpfile,"%d",&scalability);
      fgets(garbage,201,fpfile);
      PutVolConfigScalability(scalability,curr);        

      /* Scalability type for ssp (UPS)*/
      fscanf(fpfile,"%d",&scalability_type);
      fgets(garbage,201,fpfile);
      PutVolConfigScalType(scalability_type,curr);

      /* added by Sony 290897 */
      /* put spatial scalable enhancement layer prediction type */
      fscanf(fpfile,"%d",&enh_pred_type);
      fgets(garbage,201,fpfile);
      PutVolConfigEnhPredType(enh_pred_type,curr);              
      /* 290897 */

      /* Ref. layer Id */
      fscanf(fpfile,"%d",&ref_ident);
      fgets(garbage,201,fpfile);
      PutVolConfigRefId(ref_ident,curr);
      
      /* Ref. layer sampling direction */
      fscanf(fpfile,"%d",&ref_samp_dir);
      fgets(garbage,201,fpfile);
      PutVolConfigRefSampDir(ref_samp_dir,curr);
      
      /* Sampling factors (hor and ver n/m) */
      fscanf(fpfile,"%d",&hor_samp_n);
      fgets(garbage,201,fpfile);
      PutVolConfigHorSampN(hor_samp_n,curr);
      
      fscanf(fpfile,"%d",&hor_samp_m);
      fgets(garbage,201,fpfile);
      PutVolConfigHorSampM(hor_samp_m,curr);
      
      fscanf(fpfile,"%d",&ver_samp_n);
      fgets(garbage,201,fpfile);
      PutVolConfigVerSampN(ver_samp_n,curr);
      
      fscanf(fpfile,"%d",&ver_samp_m);
      fgets(garbage,201,fpfile);
      PutVolConfigVerSampM(ver_samp_m,curr);
      
      /* Enhancement type */
      fscanf(fpfile,"%d",&enhancement_type);
      fgets(garbage,201,fpfile);
      PutVolConfigEnhanceType(enhancement_type,curr);
                
/* begin: added for OBSS by Samsung AIT (1999-09-29) */
      /* Usage of shape and texture */
      fscanf(fpfile,"%d",&use_ref_shape);
      fgets(garbage,201,fpfile);
      PutVolConfigUseRefShape(use_ref_shape,curr);

      fscanf(fpfile,"%d",&use_ref_texture);
      fgets(garbage,201,fpfile);
      PutVolConfigUseRefTexture(use_ref_texture,curr);

      /* Shape sampling factors (hor and ver n/m) */
      fscanf(fpfile,"%d",&shape_hor_samp_n);
      fgets(garbage,201,fpfile);
      PutVolConfigShapeHorSampN(shape_hor_samp_n,curr);

      fscanf(fpfile,"%d",&shape_hor_samp_m);
      fgets(garbage,201,fpfile);
      PutVolConfigShapeHorSampM(shape_hor_samp_m,curr);

      fscanf(fpfile,"%d",&shape_ver_samp_n);
      fgets(garbage,201,fpfile);
      PutVolConfigShapeVerSampN(shape_ver_samp_n,curr);

      fscanf(fpfile,"%d",&shape_ver_samp_m);
      fgets(garbage,201,fpfile);
      PutVolConfigShapeVerSampM(shape_ver_samp_m,curr);
/* end: added for OBSS by Samsung AIT (1999-09-29) */
      /* intra_period: Number of periods between two INTRAs */
      fscanf(fpfile,"%d",&intra_period);
      fgets(garbage,201,fpfile);
      PutVolConfigIntraPeriod(intra_period,curr);

      /* GOV_period: Number of VOPs between two GOVs */
      fscanf(fpfile,"%d",&GOV_period);
      fgets(garbage,201,fpfile);
      PutVolConfigGOVPeriod(GOV_period,curr);

      fscanf(fpfile,"%d",&curr->interlaced);
      fgets(garbage,201,fpfile);

      fscanf(fpfile,"%d",&curr->top_field_first);
      fgets(garbage,201,fpfile);

      fscanf(fpfile,"%d",&curr->alternate_scan);
      fgets(garbage,201,fpfile);

      fscanf(fpfile,"%d",&curr->sr_direct);
      fgets(garbage,201,fpfile);

      fscanf(fpfile,"%d",&curr->mvfileusage);
      fgets(garbage,201,fpfile);

      fscanf(fpfile,"%s",curr->mvfilename);
      fgets(garbage,201,fpfile);

      if (curr->mvfileusage &&
          ((curr->mvfile = fopen(curr->mvfilename, (curr->mvfileusage==2) ? "w" : "r")) == NULL)) {
                  fprintf(stderr, "Cannot open %s\n", curr->mvfilename);
                  exit(1);
      }
      curr->mvlineno = 0;

      /* start: Included due to N2171, Clause 2.1.9 MW 23-MAR-1998 */
      /* User data in VOL? */
      fscanf(fpfile,"%d",&is_user_data_in_vol);
      fgets(garbage,201,fpfile);
      PutVolConfigIsUserDataInVol(is_user_data_in_vol,curr);

      fscanf(fpfile,"%s",user_data_in_vol_file);
      fgets(garbage,201,fpfile);
      PutVolConfigUserDataInVolFile(user_data_in_vol_file,curr);


      /* User data in GOV? */
      fscanf(fpfile,"%d",&is_user_data_in_gov);
      fgets(garbage,201,fpfile);
      PutVolConfigIsUserDataInGov(is_user_data_in_gov,curr);

      fscanf(fpfile,"%s",user_data_in_gov_file);
      fgets(garbage,201,fpfile);
      PutVolConfigUserDataInGovFile(user_data_in_gov_file,curr);
      /* end: Included due to N2171, Clause 2.1.9 MW 23-MAR-1998 */

      /* START: Complexity Estimation syntax support - Marc Mongenet (EPFL) - 3 Jun 1998 */
      /* Check if a file with complexity estimation data should be loaded. */
      fscanf(fpfile, "%d", &complexity_estimation_disable);
      fgets(garbage, 201, fpfile);
      PutVolConfigComplexityEstimationDisable(complexity_estimation_disable, curr);

      fscanf(fpfile, "%s", complexity_estimation_file);
      fgets(garbage, 201, fpfile);
      PutVolConfigComplexityEstimationFile(complexity_estimation_file, curr);

      if (!GetVolConfigComplexityEstimationDisable(curr))
        LoadComplexityEstimationFile(curr);
      /* END: Complexity Estimation syntax support */

      /* START VOL Control Parameters - K. Suehring (HHI) */

      /* VOL control parameters */
       fscanf(fpfile,"%d",&vol_control_parameters);
       fgets(garbage,201,fpfile);
       PutVolConfigVolControlParameters(vol_control_parameters,curr);

      /* vbv parameters */
       fscanf(fpfile,"%d",&vbv_parameters);
       fgets(garbage,201,fpfile);
       PutVolConfigVbvParameters(vbv_parameters,curr);

      fscanf(fpfile,"%s",vbv_parameter_string);
      fgets(garbage,201,fpfile);
      PutVolConfigVbvParam(vbv_parameter_string, curr);

      /* END VOL Control Parameters */
     

      /* Reading sprite usage (0:no, 1:static) */
      fscanf(fpfile,"%d",&sprite_usage);
      fgets(garbage,201,fpfile);
      PutVolConfigSpriteUsage(sprite_usage,curr);
      
      if (GetVolConfigGOVPeriod(curr)>0&&
                GetVolConfigSpriteUsage(curr)==STATIC_SPRITE)
                {
                printf("No GOV for STATIC SPRITE\n");
                exit(-1);
                }
       if (GetVolConfigScalability(curr)>0&&
                GetVolConfigSpriteUsage(curr)==STATIC_SPRITE)
                {
                printf("No SCALABILITY for STATIC SPRITE\n");
                exit(-1);
                }
           
/* modified NTT for GMC coding : start
      if (sprite_usage != SPRITE_NOT_USED) 
*/
      if (sprite_usage == STATIC_SPRITE) 
/* modified NTT for GMC coding : end */
        {
          fscanf(fpfile,"%d",&sprite_hdim);
          fgets(garbage,201,fpfile);
          PutVolConfigSpriteHdim(sprite_hdim,curr);

          fscanf(fpfile,"%d",&sprite_vdim);
          fgets(garbage,201,fpfile);
          PutVolConfigSpriteVdim(sprite_vdim,curr);
                        
          fscanf(fpfile,"%d",&sprite_left_edge);
          fgets(garbage,201,fpfile);
          PutVolConfigSpriteLeftEdge(sprite_left_edge,curr);

          fscanf(fpfile,"%d",&sprite_top_edge);
          fgets(garbage,201,fpfile);
          PutVolConfigSpriteTopEdge(sprite_top_edge,curr);

          fscanf(fpfile,"%d",&no_of_sprite_points);
          fgets(garbage,201,fpfile);
          PutVolConfigNoOfSpritePoints(no_of_sprite_points,curr);

          if (no_of_sprite_points>0) 
              {
              fscanf(fpfile,"%s",warp_param_file);
              fgets(garbage,201,fpfile);
              PutVolConfigWarpParamFile(warp_param_file,curr);
              }
          
          fscanf(fpfile,"%d",&warping_accuracy);
          fgets(garbage,201,fpfile);
          PutVolConfigWarpingAccuracy(warping_accuracy,curr);
        
          fscanf(fpfile,"%d",&brightness_change_in_sprite);
          fgets(garbage,201,fpfile);
          PutVolConfigBrightnessChangeInSprite(brightness_change_in_sprite,curr);
          
                                
          /* sprite file names used for STATIC (READ)*/
          fscanf(fpfile,"%s",sprite_y_file);
          fgets(garbage,201,fpfile);
          PutVolConfigSpriteY(sprite_y_file,curr);
          
          fscanf(fpfile,"%s",sprite_u_file);
          fgets(garbage,201,fpfile);
          PutVolConfigSpriteU(sprite_u_file,curr);
          
          fscanf(fpfile,"%s",sprite_v_file);
          fgets(garbage,201,fpfile);
          PutVolConfigSpriteV(sprite_v_file,curr);
          
          fscanf(fpfile,"%s",sprite_a_file);
          fgets(garbage,201,fpfile);
          PutVolConfigSpriteA(sprite_a_file,curr);
          
          fscanf(fpfile,"%d",&sprite_shape);
          fgets(garbage,201,fpfile);

          if(sprite_shape == 2) sprite_shape = 3;
          else if(sprite_shape == 3) sprite_shape = 2;

          PutVolConfigSpriteShape(sprite_shape,curr);

          if (sprite_usage==1) /* STATIC BASIC OR LOW LATENCY SPRITE */
          {
          /* Reading static sprite usage (0:basic, 1:low latency) */
          fscanf(fpfile,"%d",&low_latency_sprite_enable);
          fgets(garbage,201,fpfile);
          PutVolConfigLowLatencySpriteEnable(low_latency_sprite_enable,curr);
          }
                        
          if (sprite_usage == STATIC_SPRITE && low_latency_sprite_enable)
                {
                fscanf(fpfile,"%f",&SNR_threshold);
                fgets(garbage,201,fpfile);
                PutVolConfigSpriteSNRthreshold(SNR_threshold, curr);
                }
        
        InitForEncodeSprite(curr);
        if (GetVolConfigLowLatencySpriteEnable(curr))
                InitForEncodeLowLatencyStaticSprite(curr);
          
        }/* end of case SPRITE_USED */          
/* modified NTT for GMC coding :start */
      else if (sprite_usage == GMC_SPRITE) {
	  if (GetVolConfigVersion(curr) == 1) {
	      fprintf(stderr, "error: GMC is not defiend MPEG-4 version1 syntax, resetting to SPRITE_NOT_USED\n");
	      sprite_usage = SPRITE_NOT_USED;
	      PutVolConfigSpriteUsage(sprite_usage,curr); /* 990324 U. Benzler : changed to simply switch off Version 2 tools in Version 1 mode */
	      /*exit(-1);*/
	  }

	  fscanf(fpfile,"%d",&no_of_sprite_points);
	  fgets(garbage,201,fpfile);
	  PutVolConfigNoOfSpritePoints(no_of_sprite_points,curr);

	  /* warping type check for GMC add NTT (990818) */
	  if (! (no_of_sprite_points == 0 || no_of_sprite_points == 1 ||
		 no_of_sprite_points == 2 || no_of_sprite_points == 3)) {
	      /* perspective warping(=4) is not use. */
	      fprintf(stderr, "error: no_of_sprite_points error: %d\n", no_of_sprite_points);
	      exit(-1);
	  }

	  fscanf(fpfile,"%d",&warping_accuracy);
	  fgets(garbage,201,fpfile);
	  PutVolConfigWarpingAccuracy(warping_accuracy,curr);

	  fscanf(fpfile,"%d",&brightness_change_in_sprite);
	  fgets(garbage,201,fpfile);
	  PutVolConfigBrightnessChangeInSprite(brightness_change_in_sprite,curr);

	  PutVolConfigSpriteHdim(GetVolConfigDiskSeqX(curr),curr);
	  PutVolConfigSpriteVdim(GetVolConfigDiskSeqY(curr),curr);
	  PutVolConfigSpriteLeftEdge(0,curr);
	  PutVolConfigSpriteTopEdge(0,curr);
	  PutVolConfigSpriteShape(0,curr);
	  PutVolConfigLowLatencySpriteEnable(0,curr);

	  InitForEncodeSprite(curr);
      }
/* modified NTT for GMC coding :end */
      
      if(layer == 0)
      vol_config_list = curr;
      else
      AddVolConfigToList(curr,vol_config_list);
      
    }
  
  return(vol_config_list);
}



/***********************************************************CommentBegin******
 *
 * -- ReadVOConfigFile -- Reads the VO config information from the VO config
 *      file
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      20-09-96
 *
 * Purpose :            
 *      Reads the VO config information (not VOL config info.) from the VO config
 *      file and places this in a VOConfig structure. Also calls a function to
 *      read the VOL information in the config file.
 *
 * Arguments in :       
 *      Char *file_name - name of file to read from
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      VOConfig *vo_cfg - pointer to filled VOConfig structure.
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      -
 *
 * See also :
 *      -
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/
VOConfig *
ReadVOConfigFile(Char *file_name)
{
        FILE    *fpfile;

        Int             id,
                                num_layers;
/*                              layer; */

        Char    garbage[201];

        VOConfig                *vo_cfg;

        VolConfig               *vol_list = NULL;
/*                                                      *vol; */

        if((fpfile = fopen(file_name,"r")) == NULL)
        {
                fprintf(stderr,"ERROR(ReadVOConfigFile): Can't open %s\n",file_name);
                exit(1);
        }

        vo_cfg = SallocVOConfig();

        fscanf(fpfile,"%d",(Int *)(&id));
        fgets(garbage,201,fpfile);
        PutVOConfigId(id,vo_cfg);

        fscanf(fpfile,"%d,", (Int *)(&num_layers));
        fgets(garbage,201,fpfile);
        PutVOConfigNumVOLs(num_layers,vo_cfg);

        vol_list = ReadVOLConfigData(num_layers,fpfile);

        fclose(fpfile);

        PutVOConfigLayers(vol_list,vo_cfg);


        return(vo_cfg);
}





/***********************************************************CommentBegin******
 *
 * -- AllocVolConfig -- Allocates memory for VolConfig structure
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      30-06-96
 *
 * Purpose :            
 *      Allocates memory for a VolConfig structure
 *
 * Arguments in :       
 *      none
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      VolConfig *vol_config - pointer to allocated structure
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      
 *
 * See also :
 *      
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/
 

VolConfig *
AllocVolConfig()
{
        VolConfig       *vol_config;

        vol_config = (VolConfig *)malloc(sizeof(VolConfig));
        vol_config->pnext = NULL;
        vol_config->p_orisprite = NULL;
        vol_config->psprite = NULL;
        vol_config->tab_transmit = NULL;
        vol_config->tab_QP_store = NULL;
        vol_config->tab_DQUANT_store = NULL;
        vol_config->prev_warp_param = NULL;

        return(vol_config);
}


/***********************************************************CommentBegin******
 *
 * -- FreeVolConfig -- deallocates memory for a VolConfig structure
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      30-06-96
 *
 * Purpose :            
 *      Deallocates all memory for a VolConfig structure
 *
 * Arguments in :       
 *      VolConfig *vol_cfg - pointer to VolConfig structure to be freed
 *
 * Arguments in/out :   
 *      none    
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      none
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      
 *
 * See also :
 *      
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/
 
Void
FreeVolConfig(VolConfig *vol_cfg)
{
Vop *vop;
Int *tab;
SInt *stab;
Sprite_motion *prev_warp_param;

        if(vol_cfg != NULL)
                {
                vop = GetVolConfigOriSprite(vol_cfg);
                if (vop!=NULL)
                        FreeVop(vop); 
                prev_warp_param = GetVolConfigPrevWarpParam(vol_cfg);
                if (prev_warp_param!=NULL)
                        free((Char*)prev_warp_param); 
                tab = GetVolConfigSpriteTabTransmit(vol_cfg);
                if (tab!=NULL)
                        free((Char*)tab); 
                tab = GetVolConfigSpriteTabDQUANTStore(vol_cfg);
                if (tab!=NULL)
                        free((Char*)tab); 
                stab = GetVolConfigSpriteTabQPStore(vol_cfg);
                if (stab!=NULL)
                        free((Char*)stab); 
                free((Char*)vol_cfg);
                }

        return;
}


/***********************************************************CommentBegin******
 *
 * -- FreeVolConfigList -- deallocates a linked list of VolConfigs 
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      30-06-96
 *
 * Purpose :            
 *      Deallocates all memory for a linked list of VolConfigs
 *
 * Arguments in :       
 *      VolConfig *vol_cfg_list - linked list of VolConfigs
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      none
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      
 *
 * See also :
 *      
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/
 
Void
FreeVolConfigList(VolConfig *vol_cfg_list)
{
        VolConfig       *curr,
                                                *next;

        curr = vol_cfg_list;

        while(curr != NULL)
        {
                next = GetVolConfigNext(curr);

                FreeVolConfig(curr);

                curr = next;
        }

        return;
}


/***********************************************************CommentBegin******
 *
 * -- AddVolConfigToList -- Adds a VolConfig to a linked list of VolConfigs
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      30-06-96
 *
 * Purpose :            
 *      This function adds a VolConfig to the end of a linked list of VolConfigs
 *      **n.b.** the function assumes that the list is not empty (i.e. it has at
 *      least one node).
 *
 * Arguments in :       
 *      VolConfig *vol_cfg - VolConfig to be added
 *
 * Arguments in/out :   
 *      VolConfig *list - linked list of VolConfigs
 *
 * Arguments out :      
 *      
 *
 * Return values :      
 *      none
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      
 *
 * See also :
 *      
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/
 
Void
AddVolConfigToList(VolConfig *vol_cfg, VolConfig *list)
{
        VolConfig       *curr,
                        *prev;

        prev = list;
        curr = GetVolConfigNext(prev);

        while(curr != NULL)
        {
                prev = curr;
                curr = GetVolConfigNext(prev);
        }
        
        PutVolConfigNext(vol_cfg,prev);
}



/***********************************************************CommentBegin******
 *
 * -- PutVolConfigXXXX -- Access functions for VolConfig
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      30-06-96
 *
 * Purpose :            
 *      To set particular fields in a VolConfig strcuture
 *
 * Arguments in :       
 *      Value for field to be set
 *
 * Arguments in/out :   
 *      VolConfig *cfg - pointer to VolConfig structure
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      none
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      
 *
 * See also :
 *      
 *
 * Modified :           
 *      04.02.97 Noel O'Connor : Added 
 *              PutVolConfigRateControl(), PutVolConfigIntraQuantizer()
 *      26.02.97 Noel O'Connor : added PutVolConfigReverseVlc() & 
 *              PutVolConfigShapeEffects() & GL quantizer access functions
 *      10.03.97 Minhua Zhou   : Added PutVolConfigM()
 *      23.04.97 Michael Wollborn: Added MPEG quantization stuff completely
 *      26.04.97 Luis Ducla-Soares: added PutVolConfigDataPartEnable().
 *      07.05.97 Noel Brady: Add PutVolConfigChangeCRDisable().
 *      09.05.97 Minhua Zhou: Modified PutVolConfigModTimeBase
 *      13.05.97 Minhua Zhou: Added PutVolConfigOBMCDisable()
 *      13.05.97 Minhua Zhou: added PutVolConfigGOVPeriod() 
 *      12.08.97 Minhua Zhou: added  PutVolConfigTimeIncrementResolution()
 *      29.08.97 Osamu Sunohara: added PutVolConfigEnhPredType()
 *      08.09.97 Cecile Dufour: added PutVolConfigOriSprite(),
 *                              PutVolConfigSpriteTransmitMode(),
 *                              PutVolConfigCheckStaticSpritePrediction()
 *                              PutVolConfigSpriteTabTransmit()
 *                              PutVolConfigSpriteInitPieceWidth()
 *                              PutVolConfigSpriteInitPieceHeight()
 *                              PutVolConfigSpriteInitPieceHorRef()
 *                              PutVolConfigSpriteInitPieceVerRef()
 *                              PutVolConfigSpriteSNRthreshold()
 *      23.10.97 Cecile Dufour: add PutVolConfigSpriteTabQPStore()
 *                              PutVolConfigSpriteTabDQUANTStore()
 *      14.04.98 Cecile Dufour: add PutVolConfigLowLatencySpriteEnable()
 *      23.03.98 M. Wollborn add PutVolConfigIsUserDataInVol()
 *                               PutVolConfigUserDataInVolFile()
 *                               PutVolConfigIsUserDataInGov()
 *                               PutVolConfigUserDataInGovFile() due to N2171, Cl. 2.1.9
 *      03.06.98 Marc Mongenet (EPFL): added Complexity Estimation syntax support
 *      15.02.99 U. Benzler : added quarter pel support (PutVolConfigQuarterPel)
 *	03.03.99 Seishi TAKAMURA (NTT): added MPEG-4 version2 syntax (PutVolConfigVersion)
 *	04.06.99 Karsten Suehring (HHI): Added VOL Control Parameters
 *	16.08.99 Shigeru Fukunaga (Oki): added modules for NEWPRED
 *	06.09.99 Eishi Morimatsu (Fujitsu Labs.): added DRC support
 *	09.29.99 Dae-Sung Cho (Samsung AIT): added support for OBSS
 *  26.10.99 Massimo Ravasi (EPFL): added Complexity Estimation syntax support
 *                                  Update version 2
 *      11.11.99 U. Benzler (UoH) - added TypeIndication to be able to create conformance bitstreams
 *
 ***********************************************************CommentEnd********/

/* modified NTT for version2 syntax : start */
Void
PutVolConfigVersion(Int		version,
		    VolConfig	*cfg)
{
    cfg->mpeg4_version = version;
}
/* modified NTT for version2 syntax : end */

Void PutVolConfigTimeIncrementResolution(Int value,VolConfig *cfg) {
   cfg->time_increment_resolution=value;
 }

Void
PutVolConfigId(Int id,VolConfig *cfg)
{
        cfg->ident = id;
}

Void
PutVolConfigTypeIndication(Int type_indication,VolConfig *cfg) 
/* 11.11.99 U. Benzler (UoH) - added TypeIndication to be able to create conformance bitstreams */
{
        cfg->type_indication = type_indication;
}

Void
PutVolConfigFrameRate(Float fr, VolConfig *cfg)
{
        cfg->frame_rate = fr;
}

Void
PutVolConfigM(Int M, VolConfig *cfg)
{
        cfg->M = M;
}

Void
PutVolConfigDiskSeqX(Int x, VolConfig *cfg)
{
        cfg->disk_seq_x = x;
}

Void
PutVolConfigDiskSeqY(Int y, VolConfig *cfg)
{
        cfg->disk_seq_y = y;
}

Void
PutVolConfigDiskSeqFrameRate(Float fr, VolConfig *cfg)
{
        cfg->disk_seq_frame_rate = fr;
}

Void
PutVolConfigQuantPrecision(Int QuantPrecision,VolConfig *cfg)
{
        cfg->quant_precision = QuantPrecision;
}
Void
PutVolConfigBitsPerPixel(Int BitsPerPixel,VolConfig *cfg)
{
        cfg->bits_per_pixel = BitsPerPixel;
}

Void
PutVolConfigShape(Int shape, VolConfig *cfg)
{
        cfg->shape = shape;
}

Void
PutVolConfigShapeEffects(Int shape_effects,VolConfig *cfg)
{
        cfg->shape_effects = shape_effects;
}
Void
PutVolConfigOBMCDisable(Int OBMCDisable,VolConfig *cfg)
{
        cfg->obmc_disable = OBMCDisable;
}
Void
PutVolConfigVolControlParameters(Int VolControlParameters,VolConfig *cfg)
{
        cfg->vol_control_parameters = VolControlParameters;
}
Void
PutVolConfigVbvParameters(Int VbvParameters,VolConfig *cfg)
{
        cfg->vbv_parameters = VbvParameters;
}
Void
PutVolConfigVbvParam(Char *vbv_parameter_string, VolConfig *cfg)
{
        strcpy(cfg->vbv_parameter_string, vbv_parameter_string);
}

Void
PutVolConfigY(Char *name, VolConfig *cfg)
{
        strcpy(cfg->disk_y_file,name);
}

Void
PutVolConfigU(Char *name, VolConfig *cfg)
{
        strcpy(cfg->disk_u_file,name);
}

Void
PutVolConfigV(Char *name, VolConfig *cfg)
{
        strcpy(cfg->disk_v_file,name);
}

Void
PutVolConfigA(Char *name, VolConfig *cfg)
{
        strcpy(cfg->disk_a_file,name);
}


Void
PutVolConfigWriteCoded(Int write, VolConfig *cfg)
{
        cfg->write_coded = write;
}

Void
PutVolConfigCodedY(Char *name, VolConfig *cfg)
{
        strcpy(cfg->coded_y_file,name);
}

Void
PutVolConfigCodedU(Char *name, VolConfig *cfg)
{
        strcpy(cfg->coded_u_file,name);
}

Void
PutVolConfigCodedV(Char *name, VolConfig *cfg)
{
        strcpy(cfg->coded_v_file,name);
}

Void
PutVolConfigCodedA(Char *name, VolConfig *cfg)
{
        strcpy(cfg->coded_a_file,name);
}

Void
PutVolConfigBitstream(Char *name, VolConfig *cfg)
{
        strcpy(cfg->bitstream,name);
}

Void
PutVolConfigStartFrame(Int frame, VolConfig *cfg)
{
        cfg->start_frame = frame;
}

Void
PutVolConfigEndFrame(Int frame, VolConfig *cfg)
{
        cfg->end_frame = frame;
}

Void
PutVolConfigBitrate(Int bit_rate,VolConfig *cfg)
{
        cfg->bit_rate = bit_rate;
}

Void
PutVolConfigQuantType(Int type,VolConfig *cfg)
{
        cfg->quant_type = type;
}

/* Added MPEG-like quantization stuff, 23-APR-1997 MW */

Void
PutVolConfigLoadIntraQuantMat(Int load_intra_quant_mat, 
                              VolConfig *cfg)
{
        cfg->load_intra_quant_mat = load_intra_quant_mat;
}

Void
PutVolConfigIntraQuantMatFile(Char *intra_quant_mat_file, 
                              VolConfig *cfg)
{
        strcpy(cfg->intra_quant_mat_file,intra_quant_mat_file);
}

Void
PutVolConfigLoadNonintraQuantMat(Int load_nonintra_quant_mat, 
                                 VolConfig *cfg)
{
        cfg->load_nonintra_quant_mat = load_nonintra_quant_mat;
}

Void
PutVolConfigNonintraQuantMatFile(Char *nonintra_quant_mat_file, 
                                 VolConfig *cfg)
{
        strcpy(cfg->nonintra_quant_mat_file,nonintra_quant_mat_file);
}

Void
PutVolConfigLoadGrayIntraQuantMat(Int load_gray_intra_quant_mat, 
                                  VolConfig *cfg)
{
        cfg->load_gray_intra_quant_mat = load_gray_intra_quant_mat;
}

Void
PutVolConfigGrayIntraQuantMatFile(Char *gray_intra_quant_mat_file, 
                                  VolConfig *cfg)
{
        strcpy(cfg->gray_intra_quant_mat_file,gray_intra_quant_mat_file);
}

Void
PutVolConfigLoadGrayNonintraQuantMat(Int load_gray_nonintra_quant_mat, 
                                     VolConfig *cfg)
{
        cfg->load_gray_nonintra_quant_mat = load_gray_nonintra_quant_mat;
}

Void
PutVolConfigGrayNonintraQuantMatFile(Char *gray_nonintra_quant_mat_file, 
                                     VolConfig *cfg)
{
        strcpy(cfg->gray_nonintra_quant_mat_file,gray_nonintra_quant_mat_file);
}

/* End of MPEG-like quantization stuff, 23-APR-1997 MW */

Void
PutVolConfigRoundingControlDisable(Int disable,VolConfig *cfg)
{
	cfg->rounding_control_disable = disable;
}

Void
PutVolConfigInitialRoundingType(Int initial_value,VolConfig *cfg)
{
	cfg->initial_rounding_type = initial_value;
}


Void
PutVolConfigErrorResDisable(Int disable,VolConfig *cfg)
{
        cfg->error_res_disable = disable;
}

Void
PutVolConfigDataPartEnable(Int enable, VolConfig *cfg)
{
        cfg->data_partitioning = enable;
}

Void
PutVolConfigReverseVlc(Int reverse_vlc,VolConfig *cfg)
{
        cfg->reverse_vlc = reverse_vlc;
}

Void
PutVolConfigPacketSize(Int packet_size,VolConfig *cfg)
{
        cfg->packet_size = packet_size;
}

/** added for NEWPRED (Oki) 16-AUG-1999 **/
Void
PutVolConfigNewpredEnable(Int enable,VolConfig *cfg)
{
	cfg->newpred_enable = enable;
}

Void
PutVolConfigNewpredBackwardMessageType(Int flag,VolConfig *cfg)
{
	cfg->newpred_backward_message_type = flag;
}

Void
PutVolConfigNewpredSegmentType(Int flag,VolConfig *cfg)
{
	cfg->newpred_segment_type = flag;
}

Void
PutVolConfigNewpredCtlFile(Char *newpred_ctl_file, VolConfig *cfg)
{
	strcpy(cfg->newpred_ctl_file,newpred_ctl_file);
}
/** end of NEWPRED (Oki) 16-AUG-1999 **/

Void
PutVolConfigACDCPredDisable(Int disable,VolConfig *cfg)
{
        cfg->intra_acdc_pred_disable = disable;
}

Void
PutVolConfigSearchRange(Int sr,VolConfig *cfg)
{
        cfg->sr_for = sr;
}

/* MW QPEL 07-JUL-1998 >> */
Void
PutVolConfigQuarterPel(Int quarter_pel, VolConfig *cfg)
{
        cfg->quarter_pel = quarter_pel;
}
/* << MW QPEL 07-JUL-1998 */


Void
PutVolConfigAlphaTh(Int alpha_th,VolConfig *cfg)
{
        cfg->alpha_th = alpha_th;
}

Void PutVolConfigChangeCRDisable( Int change_CR_disable, VolConfig *cfg)
{
        cfg->change_CR_disable = change_CR_disable;
}

Void
PutVolConfigScalability(Int scale,VolConfig *cfg)
{
        cfg->scalability = scale;
}

/*for ssp (UPS)*/
Void
PutVolConfigScalType(Int scalability_type,VolConfig *cfg)
{
        cfg->scal_type = scalability_type;
}/*end of ssp (UPS)*/
                
/* added by Sony 290897 */
Void
PutVolConfigEnhPredType(Int enh_pred_type,VolConfig *cfg)
{
        cfg->ss_enh_pred_type = enh_pred_type;
}
/* 290897 */

Void
PutVolConfigRefId(Int id, VolConfig *cfg)
{
        cfg->ref_ident = id;
}

Void
PutVolConfigRefSampDir(Int dir,VolConfig *cfg)
{
        cfg->ref_samp_dir = dir;
}

Void
PutVolConfigHorSampN(Int value, VolConfig *cfg)
{
        cfg->hor_samp_n = value;
}

Void
PutVolConfigHorSampM(Int value, VolConfig *cfg)
{
        cfg->hor_samp_m = value;
}

Void
PutVolConfigVerSampN(Int value, VolConfig *cfg)
{
        cfg->ver_samp_n = value;
}

Void
PutVolConfigVerSampM(Int value, VolConfig *cfg)
{
        cfg->ver_samp_m = value;
}

Void
PutVolConfigEnhanceType(Int type,VolConfig *cfg)
{
        cfg->enhancement_type = type;
}

/* begin: added for OBSS by Samsung AIT (1999-09-29) */
Void
PutVolConfigUseRefShape(Int value, VolConfig *cfg)
{
        cfg->use_ref_shape = value;
}
 
Void
PutVolConfigUseRefTexture(Int value, VolConfig *cfg)
{
        cfg->use_ref_texture = value;
}
 
Void
PutVolConfigShapeHorSampN(Int value, VolConfig *cfg)
{
        cfg->shape_hor_samp_n = value;
}
 
Void
PutVolConfigShapeHorSampM(Int value, VolConfig *cfg)
{
        cfg->shape_hor_samp_m = value;
}

Void
PutVolConfigShapeVerSampN(Int value, VolConfig *cfg)
{
        cfg->shape_ver_samp_n = value;
}

Void
PutVolConfigShapeVerSampM(Int value, VolConfig *cfg)
{
        cfg->shape_ver_samp_m = value;
}
/* end: added for OBSS by Samsung AIT (1999-09-29) */

Void
PutVolConfigIntraPeriod(Int ir,VolConfig *cfg)
{
        cfg->intra_period = ir;
}

Void
PutVolConfigGOVPeriod(Int ir,VolConfig *cfg)
{
        cfg->GOV_period = ir;
}

/* start: Included for user_data() due to N2171, Cl. 2.1.9 MW 23-MAR-1998 */
Void
PutVolConfigIsUserDataInVol(Int value, VolConfig *cfg)
{
        cfg->is_user_data_in_vol = value;
}

Void
PutVolConfigUserDataInVolFile(Char *file, VolConfig *cfg)
{
        strcpy(cfg->user_data_in_vol_file, file);
}

Void
PutVolConfigIsUserDataInGov(Int value, VolConfig *cfg)
{
        cfg->is_user_data_in_gov = value;
}

Void
PutVolConfigUserDataInGovFile(Char *file, VolConfig *cfg)
{
        strcpy(cfg->user_data_in_gov_file, file);
}
/* end: Included for user_data() due to N2171, Cl. 2.1.9 MW 23-MAR-1998 */


/* START: Complexity Estimation syntax support - Marc Mongenet (EPFL) - 3 Jun 1998 */
Void
PutVolConfigComplexityEstimationFile (Char * file, VolConfig * cfg)
{
  strcpy(cfg->complexity_estimation_file, file);
}

Void
PutVolConfigComplexityEstimationDisable (Int value, VolConfig * cfg)
{
  cfg->complexity_estimation_disable = value;
}

Void
PutVolConfigEstimationMethod (Int value, VolConfig * cfg)
{
  cfg->estimation_method = value;
}

Void
PutVolConfigShapeComplexityEstimationDisable (Int value, VolConfig * cfg)
{
  cfg->shape_complexity_estimation_disable = value;
}

Void
PutVolConfigOpaque (Int value, VolConfig * cfg)
{
  cfg->opaque = value;
}

Void
PutVolConfigDCECSOpaque (Int value, VolConfig * cfg)
{
  cfg->dcecs_opaque = value;
}

Void
PutVolConfigTransparent (Int value, VolConfig * cfg)
{
  cfg->transparent = value;
}

Void
PutVolConfigDCECSTransparent (Int value, VolConfig * cfg)
{
  cfg->dcecs_transparent = value;
}

Void
PutVolConfigIntraCAE (Int value, VolConfig * cfg)
{
  cfg->intra_cae = value;
}

Void
PutVolConfigDCECSIntraCAE (Int value, VolConfig * cfg)
{
  cfg->dcecs_intra_cae = value;
}

Void
PutVolConfigInterCAE (Int value, VolConfig * cfg)
{
  cfg->inter_cae = value;
}

Void
PutVolConfigDCECSInterCAE (Int value, VolConfig * cfg)
{
  cfg->dcecs_inter_cae = value;
}

Void
PutVolConfigNoUpdate (Int value, VolConfig * cfg)
{
  cfg->no_update = value;
}

Void
PutVolConfigDCECSNoUpdate (Int value, VolConfig * cfg)
{
  cfg->dcecs_no_update = value;
}

Void
PutVolConfigUpsampling (Int value, VolConfig * cfg)
{
  cfg->upsampling = value;
}

Void
PutVolConfigDCECSUpsampling (Int value, VolConfig * cfg)
{
  cfg->dcecs_upsampling = value;
}

Void
PutVolConfigTextureComplexityEstimationSet1Disable (Int value, VolConfig * cfg)
{
  cfg->texture_complexity_estimation_set_1_disable = value;
}

Void
PutVolConfigIntraBlocks (Int value, VolConfig * cfg)
{
  cfg->intra_blocks = value;
}

Void
PutVolConfigDCECSIntraBlocks (Int value, VolConfig * cfg)
{
  cfg->dcecs_intra_blocks = value;
}

Void
PutVolConfigInterBlocks (Int value, VolConfig * cfg)
{
  cfg->inter_blocks = value;
}

Void
PutVolConfigDCECSInterBlocks (Int value, VolConfig * cfg)
{
  cfg->dcecs_inter_blocks = value;
}

Void
PutVolConfigInter4vBlocks (Int value, VolConfig * cfg)
{
  cfg->inter4v_blocks = value;
}

Void
PutVolConfigDCECSInter4vBlocks (Int value, VolConfig * cfg)
{
  cfg->dcecs_inter4v_blocks = value;
}

Void
PutVolConfigNotCodedBlocks (Int value, VolConfig * cfg)
{
  cfg->not_coded_blocks = value;
}

Void
PutVolConfigDCECSNotCodedBlocks (Int value, VolConfig * cfg)
{
  cfg->dcecs_not_coded_blocks = value;
}

Void
PutVolConfigTextureComplexityEstimationSet2Disable (Int value, VolConfig * cfg)
{
  cfg->texture_complexity_estimation_set_2_disable = value;
}

Void
PutVolConfigDCTCoefs (Int value, VolConfig * cfg)
{
  cfg->dct_coefs = value;
}

Void
PutVolConfigDCECSDCTCoefs (Int value, VolConfig * cfg)
{
  cfg->dcecs_dct_coefs = value;
}

Void
PutVolConfigDCTLines (Int value, VolConfig * cfg)
{
  cfg->dct_lines = value;
}

Void
PutVolConfigDCECSDCTLines (Int value, VolConfig * cfg)
{
  cfg->dcecs_dct_lines = value;
}

Void
PutVolConfigVLCSymbols (Int value, VolConfig * cfg)
{
  cfg->vlc_symbols = value;
}

Void
PutVolConfigDCECSVLCSymbols (Int value, VolConfig * cfg)
{
  cfg->dcecs_vlc_symbols = value;
}

Void
PutVolConfigVLCBits (Int value, VolConfig * cfg)
{
  cfg->vlc_bits = value;
}

Void
PutVolConfigDCECSVLCBits (Int value, VolConfig * cfg)
{
  cfg->dcecs_vlc_bits = value;
}

Void
PutVolConfigMotionCompensationComplexityDisable (Int value, VolConfig * cfg)
{
  cfg->motion_compensation_complexity_disable = value;
}

Void
PutVolConfigAPM (Int value, VolConfig * cfg)
{
  cfg->apm = value;
}

Void
PutVolConfigDCECSAPM (Int value, VolConfig * cfg)
{
  cfg->dcecs_apm = value;
}

Void
PutVolConfigNPM (Int value, VolConfig * cfg)
{
  cfg->npm = value;
}

Void
PutVolConfigDCECSNPM (Int value, VolConfig * cfg)
{
  cfg->dcecs_npm = value;
}

Void
PutVolConfigInterpolateMCQ (Int value, VolConfig * cfg)
{
  cfg->interpolate_mc_q = value;
}

Void
PutVolConfigDCECSInterpolateMCQ (Int value, VolConfig * cfg)
{
  cfg->dcecs_interpolate_mc_q = value;
}

Void
PutVolConfigForwBackMCQ (Int value, VolConfig * cfg)
{
  cfg->forw_back_mc_q = value;
}

Void
PutVolConfigDCECSForwBackMCQ (Int value, VolConfig * cfg)
{
  cfg->dcecs_forw_back_mc_q = value;
}

Void
PutVolConfigHalfpel2 (Int value, VolConfig * cfg)
{
  cfg->halfpel2 = value;
}

Void
PutVolConfigDCECSHalfpel2 (Int value, VolConfig * cfg)
{
  cfg->dcecs_halfpel2 = value;
}

Void
PutVolConfigHalfpel4 (Int value, VolConfig * cfg)
{
  cfg->halfpel4 = value;
}

Void
PutVolConfigDCECSHalfpel4 (Int value, VolConfig * cfg)
{
  cfg->dcecs_halfpel4 = value;
}

/* START: Complexity Estimation syntax support - Update version 2 - Massimo Ravasi (EPFL) - 26 Oct 1999 */

Void
PutVolConfigVersion2ComplexityEstimationDisable (Int value, VolConfig * cfg)
{
  cfg->version_2_complexity_estimation_disable = value;
}

Void
PutVolConfigSadct(Int value, VolConfig * cfg)
{
  cfg->sadct = value;
}

Void
PutVolConfigDCECSSadct(Int value, VolConfig * cfg)
{
  cfg->dcecs_sadct = value;
}

Void
PutVolConfigQuarterpel(Int value, VolConfig * cfg)
{
  cfg->quarterpel = value;
}

Void
PutVolConfigDCECSQuarterpel(Int value, VolConfig * cfg)
{
  cfg->dcecs_quarterpel = value;
}

/* END: Complexity Estimation syntax support - Update Version 2 */
/* END: Complexity Estimation syntax support */


Void 
PutVolConfigSpriteHdim(Int sprite_hdim,VolConfig *cfg)
{
        cfg->sprite_hdim = sprite_hdim;
}

Void 
PutVolConfigSpriteVdim(Int sprite_vdim,VolConfig *cfg)
{
        cfg->sprite_vdim = sprite_vdim;
}

Void 
PutVolConfigSpriteLeftEdge(Int sprite_left_edge,VolConfig *cfg)
{
        cfg->sprite_left_edge = sprite_left_edge;
}

Void 
PutVolConfigSpriteTopEdge(Int sprite_top_edge,VolConfig *cfg)
{
        cfg->sprite_top_edge = sprite_top_edge;
}

Void
PutVolConfigBlendFact(Int blend_fact,VolConfig *cfg)
{
        cfg->blend_fact = blend_fact;
}

Void
PutVolConfigWarpParamFile(Char *name, VolConfig *cfg)
{
        strcpy(cfg->warp_param_file,name);
}

Void
PutVolConfigPrevWarpParam(Sprite_motion *prev_warp_param, VolConfig *cfg)
{
        cfg->prev_warp_param = prev_warp_param;
}

Void
PutVolConfigWarpParamCounter(Int counter, VolConfig *cfg)
{
        cfg->warp_param_counter = counter;
}

Void
PutVolConfigSpriteUsage( Int sprite_usage, VolConfig *cfg)
{
        cfg->sprite_usage = sprite_usage;
}

Void
PutVolConfigWarpingAccuracy( Int warping_accuracy, VolConfig *cfg)
{
        cfg->warping_accuracy = warping_accuracy;
}

Void
PutVolConfigSpriteShape( Int shape, VolConfig *cfg)
{
        cfg->sprite_shape = shape;
}

Void
PutVolConfigNoOfSpritePoints( Int no_of_sprite_points , VolConfig *cfg )
{
        cfg->no_of_sprite_points = no_of_sprite_points;
}

Void
PutVolConfigRefPointCoord( TrajPoint *ref_point_coord , VolConfig *cfg )
{
        cfg->ref_point_coord = ref_point_coord;
}

Void
PutVolConfigBrightnessChangeInSprite( Int brightness_change_in_sprite , VolConfig *cfg )
{
        cfg->brightness_change_in_sprite = brightness_change_in_sprite;
}

Void
PutVolConfigWriteSprite(Int write, VolConfig *cfg)
{
        cfg->write_sprite = write;
}

Void
PutVolConfigSpriteY(Char *name, VolConfig *cfg)
{
        strcpy(cfg->sprite_y_file,name);
}

Void
PutVolConfigSpriteU(Char *name, VolConfig *cfg)
{
        strcpy(cfg->sprite_u_file,name);
}

Void
PutVolConfigSpriteV(Char *name, VolConfig *cfg)
{
        strcpy(cfg->sprite_v_file,name);
}

Void
PutVolConfigSpriteA(Char *name, VolConfig *cfg)
{
        strcpy(cfg->sprite_a_file,name);
}
Void
PutVolConfigSprite(Vop *sprite, VolConfig *cfg)
{
        cfg->psprite = sprite;
}

Void
PutVolConfigOriSprite(Vop *orisprite, VolConfig *cfg)
{
        cfg->p_orisprite = orisprite;
}

Void
PutVolConfigSpriteTransmitMode(Int sprite_transmit_mode, VolConfig *cfg)
{
        cfg->sprite_transmit_mode = sprite_transmit_mode;
}

Void
PutVolConfigCheckStaticSpritePrediction(Int check_staticsprite_prediction, VolConfig *cfg)
{
        cfg->check_staticsprite_prediction = check_staticsprite_prediction; 
}

Void
PutVolConfigSpriteTabTransmit(Int *tab_transmit, VolConfig *cfg)
{
        cfg->tab_transmit = tab_transmit;
}

Void
PutVolConfigSpriteTabQPStore(SInt *tab_QP_store, VolConfig *cfg)
{
        cfg->tab_QP_store = tab_QP_store;
}

Void
PutVolConfigSpriteTabDQUANTStore(Int *tab_DQUANT_store, VolConfig *cfg)
{
        cfg->tab_DQUANT_store = tab_DQUANT_store;
}

Void
PutVolConfigSpriteInitPieceWidth( Int init_piece_width, VolConfig *cfg)
{
        cfg->init_piece_width = init_piece_width;
}

Void
PutVolConfigSpriteInitPieceHeight( Int init_piece_height, VolConfig *cfg)
{
        cfg->init_piece_height = init_piece_height;
}

Void
PutVolConfigSpriteInitPieceHorRef( Int init_piece_hor_ref, VolConfig *cfg)
{
        cfg->init_piece_hor_ref = init_piece_hor_ref;
}

Void
PutVolConfigSpriteInitPieceVerRef( Int init_piece_ver_ref, VolConfig *cfg)
{
        cfg->init_piece_ver_ref = init_piece_ver_ref;
}

Void
PutVolConfigSpriteSNRthreshold( Float SNR_threshold, VolConfig *cfg)
{
        cfg->SNR_threshold = SNR_threshold;
}

Void
PutVolConfigLowLatencySpriteEnable( Int low_latency_sprite_enable, 
                                                        VolConfig *cfg)
{
        cfg->low_latency_sprite_enable = low_latency_sprite_enable;
}

Void
PutVolConfigNext(VolConfig *next, VolConfig *cfg)
{
        cfg->pnext = next;
}

Void
PutVolConfigQuantizer(Int Q,VolConfig *cfg)
{
        cfg->quantizer = Q;
}

Void
PutVolConfigBQuantizer(Int Q,VolConfig *cfg)
{
        cfg->B_quantizer = Q;
}


Void
PutVolConfigIntraQuantizer(Int Q,VolConfig *cfg)
{
        cfg->intra_quantizer = Q;
}

Void
PutVolConfigStartTime(Float time,VolConfig *cfg)
{
        cfg->start_time = time;
}

Void
PutVolConfigFrame(Int frame,VolConfig *cfg)
{
        cfg->frame = frame;
}

Void
PutVolConfigFrameSkip(Int frame_skip,VolConfig *cfg)
{
        cfg->frame_skip = frame_skip;
}

Void
PutVolConfigEndTime(Float time,VolConfig *cfg)
{
        cfg->end_time = time;
}

Void
PutVolConfigTimeInc(Float time,VolConfig *cfg)
{
        cfg->time_inc = time;
}

Void
PutVolConfigModTimeBase(Int time,VolConfig *cfg)
{
        cfg->modulo_time_base[0] = cfg->modulo_time_base[1];
        cfg->modulo_time_base[1] = time;
}

Void
PutVolConfigNextCodingTime(Float time,VolConfig *cfg)
{
        cfg->next_coding_time = time;
}

Void
PutVolConfigGLQuant(Int gl_quant,VolConfig *cfg)
{
        cfg->gl_quant = gl_quant;
}

/* 01.02.99 HHI Schueuer */
Void
PutVolConfigSADCTDisable(Int flag,VolConfig *cfg)
{
	cfg->sadct_disable = flag;
}


/* HYUNDAI (Grayscale) */
Void
PutVolConfigIntraGLQuant(Int intra_gl_quant,VolConfig *cfg)
{
        cfg->intra_gl_quant = intra_gl_quant;
}

/* HYUNDAI (Grayscale) */
Void
PutVolConfigBGLQuant(Int B_gl_quant,VolConfig *cfg)
{
        cfg->B_gl_quant = B_gl_quant;
}

Void
PutVolConfigDisableGrayQuantUpdate(Int disable_gray_quant_update,
                                   VolConfig *cfg)
{
        cfg->disable_gray_quant_update = disable_gray_quant_update;
}

/* UPM: Parameter string for Multi Rate Control */


Void
PutVolConfigRCParam(Char *rc_parameter_string, VolConfig *cfg)
{
        strcpy(cfg->rc_parameter_string, rc_parameter_string);
}



Void
PutVolConfigIntraShapePeriod(Int period, VolConfig *cfg)
{
        cfg->intra_shape_period = period;
}

Void
PutVolConfigIntraDCVlcThr(Int intra_dc_vlc_thr, VolConfig *cfg)
{
        cfg->intra_dc_vlc_thr = intra_dc_vlc_thr;
}

/* >>> added for DRC by Fujitsu (top)    <<< */
Void
PutVolConfigDrcEnable(Int drc_enable,VolConfig *cfg)
{
	cfg->drc_enable = drc_enable;
}

Void
PutVolConfigDrcSwitchingMode(Int switching_mode,VolConfig *cfg)
{
	cfg->drc_switching_mode = switching_mode;
}
/* >>> added for DRC by Fujitsu (bottom) <<< */


/***********************************************************CommentBegin******
 *
 * -- GetVolConfigBase -- Returns the base VolConfig structure from a list
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      30-06-96
 *
 * Purpose :            
 *      Given a linked list of VolConfig structures, this function returns the
 *      BASE Volconfig structure (recognisable by its scalability flag)
 *
 * Arguments in :       
 *      VolConfig *list - linked list of VolConfigs
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      VolConfig *cfg - pointer to BASE VolConfig
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      
 *
 * See also :
 *      
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/

VolConfig *
GetVolConfigBase(VolConfig *list)
{
        VolConfig *cfg;

        cfg = list;

        while(cfg != NULL)
        {
                if(GetVolConfigScalability(cfg) == BASE_LAYER)
                        return(cfg);
                else
                        cfg = GetVolConfigNext(cfg);
        }

        return(cfg);
}


/***********************************************************CommentBegin******
 *
 * -- GetVolConfigById -- returns a particular VolConfig from a list
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      30-06-96
 *
 * Purpose :            
 *      Given a linked list of VolConfigs this function returns a particular
 *      VolConfig identified by its Id.
 *
 * Arguments in :       
 *      VolConfig *list - linked list of VolConfigs
 *      Int vol_cfg_id - Id of require VolConfig        
 *
 * Arguments in/out :   
 *      none    
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      VolConfig *curr - pointer to required VolConfig structure
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      
 *
 * See also :
 *      
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/
VolConfig *
GetVolConfigById(VolConfig *list,Int vol_cfg_id)
{
        VolConfig       *curr;
        Int id;

        curr = list;

        while(curr != NULL)
        {
                id = GetVolConfigId(curr);
                if(id != vol_cfg_id)
                        curr = GetVolConfigNext(curr);
                else
                        break;
        }

        return(curr);
}

/***********************************************************CommentBegin******
 *
 * -- GetVolConfigXXXX -- Access functions for VolConfig
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      30-06-96
 *
 * Purpose :            
 *      To obtain the value of particular fields in a VolConfig structure
 *
 * Arguments in :       
 *      VolConfig *cfg - pointer to VolConfig structure
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      Value of required field
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      
 *
 * See also :
 *      mom_access.c for similar functions for other structures
 *
 * Modified :           
 *      04.02.97 Noel O'Connor : added 
 *                                      GetVolConfigRateControl(), GetVolConfigIntraQuantizer()
 *      11.02.97 J. Ignacio Ronda: modified UpdateVolConfigNextCodingTime() to
 *          skip frames due to buffer fullness in the VM5 rate control
 * 14.02.97 J. Ignacio Ronda: modified UpdateVolConfigNextCodingTime() to
 *          skip frames to read from disk due to buffer fullness.
 *      26.02.97 Noel O'Connor: Added GetVolConfigReverseVlc() &
 *                                      GetVolConfigShapeEffects() and GL quantizer access functions
 *
 *       10.03.97 Minhua Zhou: added GetVolConfigM()
 *       23.04.97 Michael Wollborn: Added MPEG quantization stuff completely
 *       26.04.97 Luis Ducla-Soares: added GetVolConfigDataPartEnable().
 *       09.05.97 Minhua Zhou : modified GetVolConfigModTimeBase
 *       13.05.97 Minhua Zhou : added GetVolConfigOBMCDisable()
 *                              and GetVolConfigGOVPeriod()
 *       12.08.97 Minhua Zhou : added GetVolConfigTimeIncrementResolution()
 *       29.08.97 Osamu Sunohara: added GetVolConfigEnhPredType()
 *       08.09.97 Cecile Dufour: added static sprites func. (see PutVolConfig)
 *       27.11.97 M. Eckert: Changes for independent frame type rate control
 *                           frame_type -> UpdateVolConfigNextCodingTime()
 *      23.03.98 M. Wollborn add GetVolConfigIsUserDataInVol()
 *                               GetVolConfigUserDataInVolFile()
 *                               GetVolConfigIsUserDataInGov()
 *                               GetVolConfigUserDataInGovFile() due to N2171, Cl. 2.1.9
 *      14.04.98 Cecile Dufour: add GetVolConfigLowLatencySpriteEnable
 *      03.06.98 Marc Mongenet (EPFL): added Completely Estimation syntax support
 *	28.01.99 Yoshinori Suzuki (Hitachi, Ltd.) : added 
 *				GetVolConfigRoundingControlDisable() and
 *				GetVolConfigInitialRoundingType().
 *      15.02.99 U. Benzler : added quarter pel support (GetVolConfigQuarterPel)
 *	03.03.99 Seishi TAKAMURA (NTT): added MPEG-4 version2 syntax (GetVolConfigVersion)
 *	16.08.99 Shigeru Fukunaga (Oki): added modules for NEWPRED
 *	06.09.99 Eishi Morimatsu (Fujitsu Labs.): added DRC support
 *	09.29.99 Dae-Sung Cho (Samsung AIT): added support for OBSS
 *  26.10.99 Massimo Ravasi (EPFL): added Complexity Estimation syntax support
 *                                  Update version 2
 *      11.11.99 U. Benzler (UoH) - added TypeIndication to be able to create conformance bitstreams
 *
 ***********************************************************CommentEnd********/

/* modified NTT for version2 syntax : start */
Int
GetVolConfigVersion(VolConfig	*cfg)
{
    return (cfg->mpeg4_version);
}
/* modified NTT for version2 syntax : end */

Int GetVolConfigTimeIncrementResolution(VolConfig *cfg) {
   return cfg->time_increment_resolution;
 }
        
Int
GetVolConfigId(VolConfig *cfg)
{
        return(cfg->ident);
}

Int
GetVolConfigTypeIndication(VolConfig *cfg) 
/* 11.11.99 U. Benzler (UoH) - added TypeIndication to be able to create conformance bitstreams */
{
        return(cfg->type_indication);
}

Float
GetVolConfigFrameRate(VolConfig *cfg)
{
        return(cfg->frame_rate);
}

Int 
GetVolConfigM(VolConfig *cfg)
{
        return(cfg->M);
}


Int
GetVolConfigDiskSeqX(VolConfig *cfg)
{
        return(cfg->disk_seq_x);
}

Int
GetVolConfigDiskSeqY(VolConfig *cfg)
{
        return(cfg->disk_seq_y);
}

Float
GetVolConfigDiskSeqFrameRate(VolConfig *cfg)
{
        return(cfg->disk_seq_frame_rate);
}

Int
GetVolConfigShape(VolConfig *cfg)
{
        return(cfg->shape);
}

Int
GetVolConfigShapeEffects(VolConfig *cfg)
{
        return(cfg->shape_effects);
}

Char *
GetVolConfigY(VolConfig *cfg)
{
        return(cfg->disk_y_file);
}

Char *
GetVolConfigU(VolConfig *cfg)
{
        return(cfg->disk_u_file);
}

Char *
GetVolConfigV(VolConfig *cfg)
{
        return(cfg->disk_v_file);
}

Char *
GetVolConfigA(VolConfig *cfg)
{
        return(cfg->disk_a_file);
}


Int
GetVolConfigWriteCoded(VolConfig *cfg)
{
        return(cfg->write_coded);
}

Char *
GetVolConfigCodedY(VolConfig *cfg)
{
        return(cfg->coded_y_file);
}

Char *
GetVolConfigCodedU(VolConfig *cfg)
{
        return(cfg->coded_u_file);
}

Char *
GetVolConfigCodedV(VolConfig *cfg)
{
        return(cfg->coded_v_file);
}

Char *
GetVolConfigCodedA(VolConfig *cfg)
{
        return(cfg->coded_a_file);
}

Char *
GetVolConfigBitstream(VolConfig *cfg)
{
        return(cfg->bitstream);
}

Int
GetVolConfigStartFrame(VolConfig *cfg)
{
        return(cfg->start_frame);
}

Int
GetVolConfigEndFrame(VolConfig *cfg)
{
        return(cfg->end_frame);
}

Int
GetVolConfigBitrate(VolConfig *cfg)
{
        return(cfg->bit_rate);
}

Int
GetVolConfigQuantType(VolConfig *cfg)
{
        return(cfg->quant_type);
}

/* Added MPEG-like quantization stuff, 23-APR-1997 MW */

Int
GetVolConfigLoadIntraQuantMat(VolConfig *cfg)
{
        return(cfg->load_intra_quant_mat);
}

Int *
GetVolConfigIntraQuantMat(VolConfig *cfg)
{
        return(cfg->intra_quant_mat);
}

Char *
GetVolConfigIntraQuantMatFile(VolConfig *cfg)
{
        return(cfg->intra_quant_mat_file);
}

Int
GetVolConfigLoadNonintraQuantMat(VolConfig *cfg)
{
        return(cfg->load_nonintra_quant_mat);
}

Int *
GetVolConfigNonintraQuantMat(VolConfig *cfg)
{
        return(cfg->nonintra_quant_mat);
}

Char *
GetVolConfigNonintraQuantMatFile(VolConfig *cfg)
{
        return(cfg->nonintra_quant_mat_file);
}

Int
GetVolConfigLoadGrayIntraQuantMat(VolConfig *cfg)
{
        return(cfg->load_gray_intra_quant_mat);
}

Int *
GetVolConfigGrayIntraQuantMat(VolConfig *cfg)
{
        return(cfg->gray_intra_quant_mat);
}

Char *
GetVolConfigGrayIntraQuantMatFile(VolConfig *cfg)
{
        return(cfg->gray_intra_quant_mat_file);
}

Int
GetVolConfigLoadGrayNonintraQuantMat(VolConfig *cfg)
{
        return(cfg->load_gray_nonintra_quant_mat);
}

Int *
GetVolConfigGrayNonintraQuantMat(VolConfig *cfg)
{
        return(cfg->gray_nonintra_quant_mat);
}

Char *
GetVolConfigGrayNonintraQuantMatFile(VolConfig *cfg)
{
        return(cfg->gray_nonintra_quant_mat_file);
}

/* End of MPEG-like quantization stuff, 23-APR-1997 MW */

Int
GetVolConfigRoundingControlDisable(VolConfig *cfg)
{
	return(cfg->rounding_control_disable);
}

Int
GetVolConfigInitialRoundingType(VolConfig *cfg)
{
	return(cfg->initial_rounding_type);
}

Int
GetVolConfigErrorResDisable(VolConfig *cfg)
{
        return(cfg->error_res_disable);
}

Int
GetVolConfigDataPartEnable(VolConfig *cfg)
{
        return(cfg->data_partitioning);
}
Int
GetVolConfigOBMCDisable(VolConfig *cfg)
{
        return(cfg->obmc_disable);
}
Int
GetVolConfigVolControlParameters(VolConfig *cfg)
{
        return(cfg->vol_control_parameters);
}
Int
GetVolConfigVbvParameters(VolConfig *cfg)
{
        return(cfg->vbv_parameters);
}
Char *
GetVolConfigVbvParam(VolConfig *cfg)
{
        return(cfg->vbv_parameter_string);
}
Int
GetVolConfigQuantPrecision(VolConfig *cfg)
{
        return(cfg->quant_precision);
}

Int
GetVolConfigBitsPerPixel(VolConfig *cfg)
{
        return(cfg->bits_per_pixel);
}
Int
GetVolConfigReverseVlc(VolConfig *cfg)
{
        return(cfg->reverse_vlc);
}

Int
GetVolConfigPacketSize(VolConfig *cfg)
{
        return(cfg->packet_size);
}

/** added for NEWPRED (Oki) 16-AUG-1999 **/
Int
GetVolConfigNewpredEnable(VolConfig *cfg)
{
	return(cfg->newpred_enable);
}

Int
GetVolConfigNewpredBackwardMessageType(VolConfig *cfg)
{
	return(cfg->newpred_backward_message_type);
}

Int
GetVolConfigNewpredSegmentType(VolConfig *cfg)
{
	return(cfg->newpred_segment_type);
}

Char *
GetVolConfigNewpredCtlFile(VolConfig *cfg)
{
	return(cfg->newpred_ctl_file);
}
/** end of NEWPRED (Oki) 16-AUG-1999 **/

Int
GetVolConfigACDCPredDisable(VolConfig *cfg)
{
        return(cfg->intra_acdc_pred_disable);
}

Int
GetVolConfigSearchRange(VolConfig *cfg)
{
        return(cfg->sr_for);
}

/* MW QPEL 07-JUL-1998 >> */
Int
GetVolConfigQuarterPel(VolConfig *cfg)
{
        return(cfg->quarter_pel);
}
/* << MW QPEL 07-JUL-1998 */

Int
GetVolConfigAlphaTh(VolConfig *cfg)
{
        return(cfg->alpha_th);
}

Int
GetVolConfigChangeCRDisable(VolConfig *cfg)
{
        return(cfg->change_CR_disable);
}

Int
GetVolConfigScalability(VolConfig *cfg)
{
        return(cfg->scalability);
}
Int
GetVolConfigScalType(VolConfig *cfg)
{
        return(cfg->scal_type);
}

/* added by Sony 290897 */
Int
GetVolConfigEnhPredType(VolConfig *cfg)
{
        return(cfg->ss_enh_pred_type);
}
/* 290897 */

Int
GetVolConfigRefId(VolConfig *cfg)
{
        return(cfg->ref_ident);
}

Int
GetVolConfigRefSampDir(VolConfig *cfg)
{
        return(cfg->ref_samp_dir);
}

Int
GetVolConfigHorSampN(VolConfig *cfg)
{
        return(cfg->hor_samp_n);
}

Int
GetVolConfigHorSampM(VolConfig *cfg)
{
        return(cfg->hor_samp_m);
}

Int
GetVolConfigVerSampN(VolConfig *cfg)
{
        return(cfg->ver_samp_n);
}

Int
GetVolConfigVerSampM(VolConfig *cfg)
{
        return(cfg->ver_samp_m);
}

Int
GetVolConfigEnhanceType(VolConfig *cfg)
{
        return(cfg->enhancement_type);
}

/* begin: added for OBSS by Samsung AIT (1999-09-29) */
Int
GetVolConfigUseRefShape(VolConfig *cfg)
{
        return(cfg->use_ref_shape);
}

Int
GetVolConfigUseRefTexture(VolConfig *cfg)
{
        return(cfg->use_ref_texture);
}

Int
GetVolConfigShapeHorSampN(VolConfig *cfg)
{
        return(cfg->shape_hor_samp_n);
}

Int
GetVolConfigShapeHorSampM(VolConfig *cfg)
{
        return(cfg->shape_hor_samp_m);
}

Int
GetVolConfigShapeVerSampN(VolConfig *cfg)
{
        return(cfg->shape_ver_samp_n);
}

Int
GetVolConfigShapeVerSampM(VolConfig *cfg)
{
        return(cfg->shape_ver_samp_m);
}
/* end: added for OBSS by Samsung AIT (1999-09-29) */

Int
GetVolConfigIntraPeriod(VolConfig *cfg)
{
        return(cfg->intra_period);
}

Int
GetVolConfigGOVPeriod(VolConfig *cfg)
{
        return(cfg->GOV_period);
}

/* start: Included for user_data() due to N2171, Cl. 2.1.9 MW 23-MAR-1998 */
Int
GetVolConfigIsUserDataInVol(VolConfig *cfg)
{
        return(cfg->is_user_data_in_vol);
}

Char *
GetVolConfigUserDataInVolFile(VolConfig *cfg)
{
        return(cfg->user_data_in_vol_file);
}

Int
GetVolConfigIsUserDataInGov(VolConfig *cfg)
{
        return(cfg->is_user_data_in_gov);
}

Char *
GetVolConfigUserDataInGovFile(VolConfig *cfg)
{
        return(cfg->user_data_in_gov_file);
}
/* end: Included for user_data() due to N2171, Cl. 2.1.9 MW 23-MAR-1998 */


/* START: Complexity Estimation syntax support - Marc Mongenet (EPFL) - 3 Jun 1998 */
Char *
GetVolConfigComplexityEstimationFile (const VolConfig * cfg)
{
  return (Char *)cfg->complexity_estimation_file;
}

Int
GetVolConfigComplexityEstimationDisable (const VolConfig * cfg)
{
  return cfg->complexity_estimation_disable;
}

Int
GetVolConfigEstimationMethod (const VolConfig * cfg)
{
  return cfg->estimation_method;
}

Int
GetVolConfigShapeComplexityEstimationDisable (const VolConfig * cfg)
{
  return cfg->shape_complexity_estimation_disable;
}

Int
GetVolConfigOpaque (const VolConfig * cfg)
{
  return cfg->opaque;
}

Int
GetVolConfigDCECSOpaque (const VolConfig * cfg)
{
  return cfg->dcecs_opaque;
}

Int
GetVolConfigTransparent (const VolConfig * cfg)
{
  return cfg->transparent;
}

Int
GetVolConfigDCECSTransparent (const VolConfig * cfg)
{
  return cfg->dcecs_transparent;
}

Int
GetVolConfigIntraCAE (const VolConfig * cfg)
{
  return cfg->intra_cae;
}

Int
GetVolConfigDCECSIntraCAE (const VolConfig * cfg)
{
  return cfg->dcecs_intra_cae;
}

Int
GetVolConfigInterCAE (const VolConfig * cfg)
{
  return cfg->inter_cae;
}

Int
GetVolConfigDCECSInterCAE (const VolConfig * cfg)
{
  return cfg->dcecs_inter_cae;
}

Int
GetVolConfigNoUpdate (const VolConfig * cfg)
{
  return cfg->no_update;
}

Int
GetVolConfigDCECSNoUpdate (const VolConfig * cfg)
{
  return cfg->dcecs_no_update;
}

Int
GetVolConfigUpsampling (const VolConfig * cfg)
{
  return cfg->upsampling;
}

Int
GetVolConfigDCECSUpsampling (const VolConfig * cfg)
{
  return cfg->dcecs_upsampling;
}

Int
GetVolConfigTextureComplexityEstimationSet1Disable (const VolConfig * cfg)
{
  return cfg->texture_complexity_estimation_set_1_disable;
}

Int
GetVolConfigIntraBlocks (const VolConfig * cfg)
{
  return cfg->intra_blocks;
}

Int
GetVolConfigDCECSIntraBlocks (const VolConfig * cfg)
{
  return cfg->dcecs_intra_blocks;
}

Int
GetVolConfigInterBlocks (const VolConfig * cfg)
{
  return cfg->inter_blocks;
}

Int
GetVolConfigDCECSInterBlocks (const VolConfig * cfg)
{
  return cfg->dcecs_inter_blocks;
}

Int
GetVolConfigInter4vBlocks (const VolConfig * cfg)
{
  return cfg->inter4v_blocks;
}

Int
GetVolConfigDCECSInter4vBlocks (const VolConfig * cfg)
{
  return cfg->dcecs_inter4v_blocks;
}

Int
GetVolConfigNotCodedBlocks (const VolConfig * cfg)
{
  return cfg->not_coded_blocks;
}

Int
GetVolConfigDCECSNotCodedBlocks (const VolConfig * cfg)
{
  return cfg->dcecs_not_coded_blocks;
}

Int
GetVolConfigTextureComplexityEstimationSet2Disable (const VolConfig * cfg)
{
  return cfg->texture_complexity_estimation_set_2_disable;
}

Int
GetVolConfigDCTCoefs (const VolConfig * cfg)
{
  return cfg->dct_coefs;
}

Int
GetVolConfigDCECSDCTCoefs (const VolConfig * cfg)
{
  return cfg->dcecs_dct_coefs;
}

Int
GetVolConfigDCTLines (const VolConfig * cfg)
{
  return cfg->dct_lines;
}

Int
GetVolConfigDCECSDCTLines (const VolConfig * cfg)
{
  return cfg->dcecs_dct_lines;
}

Int
GetVolConfigVLCSymbols (const VolConfig * cfg)
{
  return cfg->vlc_symbols;
}

Int
GetVolConfigDCECSVLCSymbols (const VolConfig * cfg)
{
  return cfg->dcecs_vlc_symbols;
}

Int
GetVolConfigVLCBits (const VolConfig * cfg)
{
  return cfg->vlc_bits;
}

Int
GetVolConfigDCECSVLCBits (const VolConfig * cfg)
{
  return cfg->dcecs_vlc_bits;
}

Int
GetVolConfigMotionCompensationComplexityDisable (const VolConfig * cfg)
{
  return cfg->motion_compensation_complexity_disable;
}

Int
GetVolConfigAPM (const VolConfig * cfg)
{
  return cfg->apm;
}

Int
GetVolConfigDCECSAPM (const VolConfig * cfg)
{
  return cfg->dcecs_apm;
}

Int
GetVolConfigNPM (const VolConfig * cfg)
{
  return cfg->npm;
}

Int
GetVolConfigDCECSNPM (const VolConfig * cfg)
{
  return cfg->dcecs_npm;
}

Int
GetVolConfigInterpolateMCQ (const VolConfig * cfg)
{
  return cfg->interpolate_mc_q;
}

Int
GetVolConfigDCECSInterpolateMCQ (const VolConfig * cfg)
{
  return cfg->dcecs_interpolate_mc_q;
}

Int
GetVolConfigForwBackMCQ (const VolConfig * cfg)
{
  return cfg->forw_back_mc_q;
}

Int
GetVolConfigDCECSForwBackMCQ (const VolConfig * cfg)
{
  return cfg->dcecs_forw_back_mc_q;
}

Int
GetVolConfigHalfpel2 (const VolConfig * cfg)
{
  return cfg->halfpel2;
}

Int
GetVolConfigDCECSHalfpel2 (const VolConfig * cfg)
{
  return cfg->dcecs_halfpel2;
}

Int
GetVolConfigHalfpel4 (const VolConfig * cfg)
{
  return cfg->halfpel4;
}

Int
GetVolConfigDCECSHalfpel4 (const VolConfig * cfg)
{
  return cfg->dcecs_halfpel4;
}

/* START: Complexity Estimation syntax support - Update version 2 - Massimo Ravasi (EPFL) - 26 Oct 1999 */

Int
GetVolConfigVersion2ComplexityEstimationDisable (const VolConfig * cfg)
{
  return cfg->version_2_complexity_estimation_disable;
}

Int
GetVolConfigSadct(const VolConfig * cfg)
{
  return cfg->sadct;
}

Int
GetVolConfigDCECSSadct(const VolConfig * cfg)
{
  return cfg->dcecs_sadct;
}

Int
GetVolConfigQuarterpel(const VolConfig * cfg)
{
  return cfg->quarterpel;
}

Int
GetVolConfigDCECSQuarterpel(const VolConfig * cfg)
{
  return cfg->dcecs_quarterpel;
}

/* END: Complexity Estimation syntax support - Update Version 2 */
/* END: Complexity Estimation syntax support */


Int
GetVolConfigSpriteHdim(VolConfig *cfg)
{
        return(cfg->sprite_hdim);
}

Int
GetVolConfigSpriteVdim(VolConfig *cfg)
{
        return(cfg->sprite_vdim);
}

Int
GetVolConfigSpriteLeftEdge(VolConfig *cfg)
{
        return(cfg->sprite_left_edge);
}

Int
GetVolConfigSpriteTopEdge(VolConfig *cfg)
{
        return(cfg->sprite_top_edge);
}

Int
GetVolConfigBlendFact(VolConfig *cfg)
{
        return(cfg->blend_fact);
}

Char *
GetVolConfigWarpParamFile(VolConfig *cfg)
{
        return(cfg->warp_param_file);
}

Sprite_motion *
GetVolConfigPrevWarpParam(VolConfig *cfg)
{
        return(cfg->prev_warp_param);
}

Int
GetVolConfigWarpParamCounter(VolConfig *cfg)
{
        return(cfg->warp_param_counter);
}

Int
GetVolConfigSpriteUsage(VolConfig *cfg)
{
        return( cfg->sprite_usage );
}

Int
GetVolConfigWarpingAccuracy(VolConfig *cfg)
{
        return( cfg->warping_accuracy );
}

Int
GetVolConfigSpriteShape(VolConfig *cfg)
{
        return( cfg->sprite_shape );
}

Int 
GetVolConfigNoOfSpritePoints( VolConfig *cfg )
{
        return( cfg->no_of_sprite_points );
}

TrajPoint *
GetVolConfigRefPointCoord( VolConfig *cfg )
{
        return( cfg->ref_point_coord );
}

Int
GetVolConfigBrightnessChangeInSprite( VolConfig *cfg )
{
        return( cfg->brightness_change_in_sprite );
}

Int
GetVolConfigWriteSprite(VolConfig *cfg)
{
        return(cfg->write_sprite);
}

Char *
GetVolConfigSpriteY(VolConfig *cfg)
{
        return(cfg->sprite_y_file);
}

Char *
GetVolConfigSpriteU(VolConfig *cfg)
{
        return(cfg->sprite_u_file);
}

Char *
GetVolConfigSpriteV(VolConfig *cfg)
{
        return(cfg->sprite_v_file);
}

Char *
GetVolConfigSpriteA(VolConfig *cfg)
{
        return(cfg->sprite_a_file);
}

Vop *
GetVolConfigSprite(VolConfig *cfg)
{
        return(cfg->psprite);
}

Vop *
GetVolConfigOriSprite(VolConfig *cfg)
{
        return(cfg->p_orisprite);
}

Int
GetVolConfigSpriteTransmitMode(VolConfig *cfg)
{
        return(cfg->sprite_transmit_mode);
}

Int
GetVolConfigCheckStaticSpritePrediction(VolConfig *cfg)
{
        return(cfg->check_staticsprite_prediction);
}

Int *
GetVolConfigSpriteTabTransmit(VolConfig *cfg)
{
        return(cfg->tab_transmit);
}

SInt *
GetVolConfigSpriteTabQPStore(VolConfig *cfg)
{
        return(cfg->tab_QP_store);
}

Int *
GetVolConfigSpriteTabDQUANTStore(VolConfig *cfg)
{
        return(cfg->tab_DQUANT_store);
}

Int
GetVolConfigSpriteInitPieceWidth(VolConfig *cfg)
{
        return(cfg->init_piece_width);
}

Int
GetVolConfigSpriteInitPieceHeight(VolConfig *cfg)
{
        return(cfg->init_piece_height);
}

Int
GetVolConfigSpriteInitPieceHorRef(VolConfig *cfg)
{
        return(cfg->init_piece_hor_ref);
}

Int
GetVolConfigSpriteInitPieceVerRef(VolConfig *cfg)
{
        return(cfg->init_piece_ver_ref);
}

Float
GetVolConfigSpriteSNRthreshold(VolConfig *cfg)
{
        return(cfg->SNR_threshold);
}

Int
GetVolConfigLowLatencySpriteEnable(VolConfig *cfg)
{
        return(cfg->low_latency_sprite_enable);
}

VolConfig *
GetVolConfigNext(VolConfig *cfg)
{
        return(cfg->pnext);
}

Int
GetVolConfigQuantizer(VolConfig *cfg)
{
        return(cfg->quantizer);
}

Int
GetVolConfigIntraQuantizer(VolConfig *cfg)
{
        return(cfg->intra_quantizer);
}

Int
GetVolConfigBQuantizer(VolConfig *cfg)
{
        return(cfg->B_quantizer);
}

Float
GetVolConfigStartTime(VolConfig *cfg)
{
        return(cfg->start_time);
}

Int
GetVolConfigFrameSkip(VolConfig *cfg)
{
        return(cfg->frame_skip);
}

Float
GetVolConfigEndTime(VolConfig *cfg)
{
        return(cfg->end_time);
}

Float
GetVolConfigTimeInc(VolConfig *cfg)
{
        return(cfg->time_inc);
}

Int
GetVolConfigModTimeBase(VolConfig *cfg,Int i)
{
        return(cfg->modulo_time_base[i]);
}

Float
GetVolConfigNextCodingTime(VolConfig *cfg)
{
        return(cfg->next_coding_time);
}

/* 01.02.99 HHI Schueuer */
Int
GetVolConfigSADCTDisable(VolConfig *cfg)
{
	return(cfg->sadct_disable);
}
/* end HHI */


/* UPM: New parameters for Multi Rate Control */

/* -> GetEncodeControlRCType
Int
GetVolConfigRateControl(VolConfig *cfg)
{
        return(cfg->rate_control);
}
*/

Char *
GetVolConfigRCParam(VolConfig *cfg)
{
        return(cfg->rc_parameter_string);
}

Int
GetVolConfigGLQuant(VolConfig *cfg)
{
        return(cfg->gl_quant);
}

/* HYUNDAI (Grayscale) */
Int
GetVolConfigIntraGLQuant(VolConfig *cfg)
{
        return(cfg->intra_gl_quant);
}

/* HYUNDAI (Grayscale) */
Int
GetVolConfigBGLQuant(VolConfig *cfg)
{
        return(cfg->B_gl_quant);
}

Int
GetVolConfigDisableGrayQuantUpdate(VolConfig *cfg)
{
        return(cfg->disable_gray_quant_update);
}

Int
GetVolConfigIntraDCVlcThr(VolConfig *cfg)
{
        return(cfg->intra_dc_vlc_thr);
}

Void
UpdateVolConfigNextCodingTime(VolConfig *cfg, 
                              Int vo_id, 
                              Int vol_id, 
                              Int rc_type,
                              Int rc_algorithm,
                              Int frame_type)
{
        Float   time_inc,
                new_time;

        Int     skip_frame = 0;

        /* RC2 */
        Int     frame;

        time_inc = GetVolConfigTimeInc(cfg);
        new_time = cfg->next_coding_time;
      
        do
                {
                new_time += time_inc*(Float)GetVolConfigM(cfg);

                if (new_time > cfg->end_time)
                        break;
#ifdef _RC_DEBUG_
                save_buffer(vo_id, vol_id); /* Formerly done in RC_FrameSkipping 
                                               but this is only RC_DEBUG stuff!*/
#endif
#ifndef _RC_NO_SKIPPING_

              if (rc_type == VM5_RATE_CONTROL && 
                    rc_algorithm == RC_ALGORITHM_Q2)
                        {
                        skip_frame = RC_FrameSkipping(vo_id, vol_id);
                        if (skip_frame)
                                {
                                RC_UpdateAfterSkipping(vo_id, vol_id, frame_type);
                                /* RC2: update the frame number and put in VolConfigFrame */
                                frame = GetVolConfigFrame(cfg) + GetVolConfigFrameSkip(cfg);
                                fprintf(stdout,
                   "....................Frame Skipping: newframe =%d\n",frame);
            PutVolConfigFrame(frame, cfg);

                                /* RC2 */
                                }
                        }
#endif 

                }while(skip_frame);

        new_time = RoundTime(new_time);

        cfg->next_coding_time = new_time;

        return;
}

Int GetVolConfigIntraShapePeriod(VolConfig *cfg)
{
        return cfg->intra_shape_period;
}


/* added by Sven Brandau, 19.08.98 */
Int GetVolConfigInterlaced(VolConfig *cfg)
{
        return cfg->interlaced;
}

/* >>> added for DRC by Fujitsu (top)    <<< */
Int
GetVolConfigDrcEnable(VolConfig *cfg)
{
	return(cfg->drc_enable);
}

Int
GetVolConfigDrcSwitchingMode(VolConfig *cfg)
{
	return(cfg->drc_switching_mode);
}
/* >>> added for DRC by Fujitsu (bottom) <<< */

/***********************************************************CommentBegin******
 *
 * -- GetVolConfigFrame -- Special access function for te "next" frame
 *              to code of a VolConfig.
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      24-09-96
 *
 * Purpose :            
 *      This is a special access function for the "next" P-frame to be coded
 *      field of the VolConfig structure. The current value of the field is 
 *      returned and the value of the field is incremented by the value of 
 *      the frame skip field. THIS FUNCTION SHOULD ONLY BE CALLED ONCE PER
 *      CODING INSTANT.
 *
 * Arguments in :       
 *      VolConfig *cfg - pointer to VolConfig structure
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      next frame to be coded
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      
 *
 * See also :
 *      mom_access.c for similar functions for other structures
 *
 * Modified :           
 *
 * 11.03.97  Minhua Zhou:  added M to frame number      
 *
 ***********************************************************CommentEnd********/ 

Int
GetVolConfigFrame(VolConfig *cfg)
{
        Int     frame;

        frame = cfg->frame;

        /* Inc. value in structure to next frame to be coded */
        cfg->frame  = cfg->frame + GetVolConfigM(cfg)*cfg->frame_skip;

#ifdef _RC_DEBUG_
        fprintf(stdout,"Coding frame =%d\n",frame);
#endif
        return(frame);
}

/***********************************************************CommentBegin******
 *
 * -- PrintVolConfig -- Prints the contents of a VolConfig structure
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      30-06-96
 *
 * Purpose :            
 *      This function prints the contents of a VolConfig structure. This
 *      is printed either to a file or to the display.
 * 
 * Arguments in :       
 *      VolConfig *cfg - pointer to VolConfig structure
 *      File *file - file pointer (where to write to)
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      none
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      
 *
 * See also :
 *      
 *
 * Modified :           
 *
 * 23.04.97 Michael Wollborn: Added MPEG quantization stuff completely
 * 26.04.97 Luis Ducla-Soares: Added Data Partitioning stuff.
 * 04.06.98 Marc Mongenet: Added Complexity Estimation syntax support
 * 18.01.99 Takefumi Nagumo: Added Hierarchy_type for scalable.
 * 15.02.99 U. Benzler : added quarter pel support
 * 03.03.99 Seishi TAKAMURA (NTT): added GMC coding
 * 16.08.99 Shigeru Fukunaga (Oki): added modules for NEWPRED
 * 06.09.99 Eishi Morimatsu (Fujitsu Labs.): added DRC support
 * 09.29.99 Dae-Sung Cho (Samsung AIT): added support for OBSS
 * 11.11.99 U. Benzler (UoH) - added TypeIndication to be able to create conformance bitstreams
 *
 *
 ***********************************************************CommentEnd********/
 
Void
PrintVolConfig(VolConfig *cfg,File *file)
{


        fprintf(file,"\n*** Control Configuration ***\n\n");
        fprintf(file,"Layer Id : %d\n", (Int)GetVolConfigId(cfg));
        fprintf(file,"Type Indication : %d\n", (Int)GetVolConfigTypeIndication(cfg));
        fprintf(file,"Frame rate : %f\n",GetVolConfigFrameRate(cfg));
        fprintf(file,"Source Y seq. : %s\n",GetVolConfigY(cfg));
        fprintf(file,"Source U seq. : %s\n",GetVolConfigU(cfg));
        fprintf(file,"Source V seq. : %s\n",GetVolConfigV(cfg));
        if(GetVolConfigShape(cfg) == RECTANGULAR)
                fprintf(file,"No shape coding\n");
        else if(GetVolConfigShape(cfg) == BINARY)
                fprintf(file,"Binary shape coding - Source seq. : %s\n",GetVolConfigA(cfg));
        else
                fprintf(file,"Grey Level shape coding - Source seq. : %s\n",GetVolConfigA(cfg));
        if(GetVolConfigShapeEffects(cfg) == NO_SHAPE_EFFECTS)
                fprintf(file,"NO_SHAPE_EFFECTS mode selected\n");
        fprintf(file,"Source seq. resolution : %d x %d\n",
                (Int)GetVolConfigDiskSeqX(cfg),
                (Int)GetVolConfigDiskSeqY(cfg));
        fprintf(file,"Source seq. frame rate : %f\n",GetVolConfigDiskSeqFrameRate(cfg));
        fprintf(file,"Start frame : %d\n", (Int)GetVolConfigStartFrame(cfg));
        fprintf(file,"End frame : %d\n", (Int)GetVolConfigEndFrame(cfg));
        fprintf(file,"Start time : %f\n", (Float)GetVolConfigStartTime(cfg));
        fprintf(file,"End time : %f\n", (Float)GetVolConfigEndTime(cfg));
        if(GetVolConfigWriteCoded(cfg))
        {
                fprintf(file,"Coded VOPs will be written to disk\n");
                fprintf(file,"Coded Y seq. : %s\n",GetVolConfigCodedY(cfg));
                fprintf(file,"Coded U seq. : %s\n",GetVolConfigCodedU(cfg));
                fprintf(file,"Coded V seq. : %s\n",GetVolConfigCodedV(cfg));
                if(GetVolConfigShape(cfg) != RECTANGULAR)
                        fprintf(file,"Coded A seq. : %s\n",GetVolConfigCodedA(cfg));
        }
        else
                fprintf(file,"Coded VOPs will NOT be written to disk\n");

        fprintf(file,"\n");
        fprintf(file,"Bitstream file for this layer : %s\n",
                GetVolConfigBitstream(cfg));
        fprintf(file,"Target Bitrate : %d\n", (Int)GetVolConfigBitrate(cfg));
        /* UPM: this is now printed in PrintEncodeControl: 
        if(GetVolConfigRateControl(cfg) == NO_RATE_CONTROL)
                fprintf(file,"No rate control\n");
        else if(GetVolConfigRateControl(cfg) == VM4_RATE_CONTROL)
                fprintf(file,"VM 4 rate control will be used\n");
        else if(GetVolConfigRateControl(cfg) == VM5_RATE_CONTROL)
                fprintf(file,"VM 5 rate control will be used\n");
        */
        fprintf(file,"\n");

        fprintf(file,"\n*** Coding Configuration ***\n\n");
        if(GetVolConfigQuantType(cfg))
          {
          fprintf(file,"MPEG2 Quantization mode selected - \n");
          fprintf(file,"\tinitial INTRA value : %d\n",GetVolConfigIntraQuantizer(cfg));
          fprintf(file,"\tinitial INTER value in P-VOPs: %d\n",GetVolConfigQuantizer(cfg));
          fprintf(file,"\tinitial INTER value in B-VOPs: %d\n",GetVolConfigBQuantizer(cfg));
        
          if(GetVolConfigLoadIntraQuantMat(cfg)==0)
            fprintf(file,"Default INTRA matrix used\n");
          else if(GetVolConfigLoadIntraQuantMat(cfg)==1)
            fprintf(file,"User defined INTRA matrix >>%s<< used\n",
                    GetVolConfigIntraQuantMatFile(cfg));
          else
            fprintf(file,"Predefined special INTRA matrix number %d used\n",
                    GetVolConfigLoadIntraQuantMat(cfg));

          if(GetVolConfigLoadNonintraQuantMat(cfg)==0)
            fprintf(file,"Default NONINTRA matrix used\n");
          else if(GetVolConfigLoadNonintraQuantMat(cfg)==1)
            fprintf(file,"User defined NONINTRA matrix >>%s<< used\n",
                    GetVolConfigNonintraQuantMatFile(cfg));
          else
            fprintf(file,"Predefined special NONINTRA matrix number %d used\n",
                    GetVolConfigLoadNonintraQuantMat(cfg));
          }
        else
          {
          fprintf(file,"H.263 quantization mode selected - \n");
          fprintf(file,"\tinitial INTRA value : %d\n",
                  GetVolConfigIntraQuantizer(cfg));
          fprintf(file,"\tinitial INTER value in P-VOPs: %d\n",GetVolConfigQuantizer(cfg));
          fprintf(file,"\tinitial INTER value in B-VOPs: %d\n",GetVolConfigBQuantizer(cfg));
          }

        if(GetVolConfigShape(cfg) == GREY_SCALE)
          {
            /* HYUNDAI (Grayscale) */
            fprintf(file,"Gray level alpha map quantizer(I-VOPs) : %d\n",
                    GetVolConfigIntraGLQuant(cfg));
            /* HYUNDAI (Grayscale) */
            fprintf(file,"Gray level alpha map quantizer(P-VOPs) : %d\n",
                    GetVolConfigGLQuant(cfg));
            fprintf(file,"Gray level alpha map quantizer(B-VOPs) : %d\n",
                    GetVolConfigBGLQuant(cfg));
            if(GetVolConfigDisableGrayQuantUpdate(cfg))
              fprintf(file,"Gray level alpha map quantizer WILL NOT be updated\n");
            else
              fprintf(file,"Gray level alpha map quantizer WILL be updated\n");

            if(GetVolConfigQuantType(cfg))
              {
                if(GetVolConfigLoadGrayIntraQuantMat(cfg)==0)
                  fprintf(file,"Default gray INTRA matrix used\n");
                else if(GetVolConfigLoadGrayIntraQuantMat(cfg)==1)
                  fprintf(file,"User defined gray INTRA matrix >>%s<< used\n",
                          GetVolConfigGrayIntraQuantMatFile(cfg));
                else
                  fprintf(file,"Predefined special gray INTRA matrix number %d used\n",
                          GetVolConfigLoadGrayIntraQuantMat(cfg));
                
                if(GetVolConfigLoadGrayNonintraQuantMat(cfg)==0)
                  fprintf(file,"Default gray NONINTRA matrix used\n");
                else if(GetVolConfigLoadGrayNonintraQuantMat(cfg)==1)
                  fprintf(file,"User defined gray NONINTRA matrix >>%s<< used\n",
                          GetVolConfigGrayNonintraQuantMatFile(cfg));
                else
                  fprintf(file,"Predefined special gray NONINTRA matrix number %d used\n",
                          GetVolConfigLoadGrayNonintraQuantMat(cfg));
              }
          }

        if(GetVolConfigRoundingControlDisable(cfg))
	  fprintf(file,"Rounding Control is DISABLED\n");
	else
	  fprintf(file,"Rounding Control is ENABLED\n");
	fprintf(file,"Initial value for Rounding Control : %d\n", (Int)GetVolConfigInitialRoundingType(cfg));

        if(GetVolConfigErrorResDisable(cfg))
          fprintf(file,"Error Resilience mode is DISABLED\n");
        else
          {
            fprintf(file,"Error Resilience mode is ENABLED\n");
            if(GetVolConfigReverseVlc(cfg))
              fprintf(file,"Reversible VLCs will be used\n");
            else
              {
                fprintf(file,"Regular VLCs will be used!\n");
                /* fprintf(file,"Reversible VLCs will NOT be used\n");
                   fprintf(file,"\nNormal VLCs temporarily disabled!\n");
                   fprintf(file,"Please use RVLCs when using the error resilient mode.\n");
                   exit(1);*/
              }
            
            if (GetVolConfigDataPartEnable(cfg))
              fprintf(file,"Data Partitioning will be used\n");
            else
              fprintf(file,"NO Data Partitioning will be used\n");
            
          }

	/** added for NEWPRED (Oki) 16-AUG-1999 **/
	if(GetVolConfigNewpredEnable(cfg)){
	  fprintf(file,"NEWPRED will be used ");
	  fprintf(file,"(newpred_backward_message_type : %d)\n",
		  GetVolConfigNewpredBackwardMessageType(cfg));
	  fprintf(file,"(newpred_segment_type : %d)\n",
		  GetVolConfigNewpredSegmentType(cfg));
	  fprintf(file,"(control file : %s)\n",
		  GetVolConfigNewpredCtlFile(cfg));
	}
	else
	  fprintf(file,"NEWPRED will NOT be used\n");
	/** end of NEWPRED (Oki) 16-AUG-1999 **/

/* >>> added for DRC by Fujitsu (top) <<< */
	if(GetVolConfigDrcEnable(cfg)==0)
	  fprintf(file,"Reduced Resolution VOP is DISABLED\n");
	else
	  fprintf(file,"Reduced Resolution VOP is ENABLED\n");
/* >>> added for DRC by Fujitsu (bottom) <<< */

        /* Complexity Estimation syntax support - Marc Mongenet (EPFL) - 4 Jun 1998 */
        if (GetVolConfigComplexityEstimationDisable(cfg))
          fprintf(file, "Complexity Estimation data transmission is DISABLED\n");
        else
          fprintf(file, "Complexity Estimation data transmission is ENABLED\n");


        if(GetVolConfigACDCPredDisable(cfg))
                fprintf(file,"AC/DC prediction of Intra macroblocks is DISABLED\n");
        else
                fprintf(file,"AC/DC prediction of Intra macroblocks is ENABLED\n");
	/* 01.02.99 HHI Schueuer */
	if(GetVolConfigSADCTDisable(cfg))
	  fprintf(file,"SADCT coding is DISABLED\n");
	else
	  fprintf(file,"SADCT coding is ENABLED\n");
	/* end HHI */
        fprintf(file,"COMBINED motion/shape/texture coding will be used\n");
        fprintf(file,"search_range_per_coded_frame : %d\n", (Int)GetVolConfigSearchRange(cfg));
        fprintf(file,"alpha threshold : %d\n", (Int)GetVolConfigAlphaTh(cfg));

        fprintf(file,"\n");

	/* MW QPEL 07-JUL-1998 >> */
        if (GetVolConfigQuarterPel(cfg)) {
          fprintf(file,"Quarter Pel MC ON\n");
        }
        else {
          fprintf(file,"Quarter Pel MC OFF\n");
        }
	/* << MW QPEL 07-JUL-1998 */

/* begin: added for OBSS by Samsung AIT (1999-09-29) */
        if(GetVolConfigShape(cfg) != BINARY_SHAPE_ONLY)
        {
/* end: added for OBSS by Samsung AIT (1999-09-29) */
        if(GetVolConfigScalability(cfg))
        {
                fprintf(file,"Scalability Type : %d\n",
                        (Int)GetVolConfigScalType(cfg));
                if(GetVolConfigScalType(cfg) == 0 ||
                   GetVolConfigScalType(cfg) == 3 ||
                   GetVolConfigScalType(cfg) == 4)
                  fprintf(file,"\tHierarchy type: 1 (Temporal scalability)\n");
                else if (GetVolConfigScalType(cfg) == 1)
                  fprintf(file,"\tHierarchy type: 0 (Spatial scalability)\n");
                fprintf(file,"Reference layer Id : %d\n",
                        (Int)GetVolConfigRefId(cfg));
                fprintf(file,"Reference layer sampling direction : %d\n",
                        (Int)GetVolConfigRefSampDir(cfg));
                fprintf(file,"Hor. sampling factor n : %d\n",
                        (Int)GetVolConfigHorSampN(cfg));
                fprintf(file,"Hor. sampling factor m : %d\n",
                        (Int)GetVolConfigHorSampM(cfg));
                fprintf(file,"Ver. sampling factor n : %d\n",
                        (Int)GetVolConfigVerSampN(cfg));
                fprintf(file,"Ver. sampling factor m : %d\n",
                        (Int)GetVolConfigVerSampM(cfg));
                if(GetVolConfigEnhanceType(cfg))
                        fprintf(file,"PARTIAL region enhancment of base layer\n");
                else
                        fprintf(file,"FULL region enhancement of base layer\n");
/* begin: added for OBSS by Samsung AIT (1999-09-29) */
                if(GetVolConfigShape(cfg)==1 && GetVolConfigScalType(cfg)==1)
                {
                   fprintf(file,"Usage of ref. shape : %d\n",
                        (Int)GetVolConfigUseRefShape(cfg));
                   fprintf(file,"Usage of ref. texture : %d\n",
                        (Int)GetVolConfigUseRefTexture(cfg));
                   fprintf(file,"Shape hor. sampling factor n : %d\n",
                        (Int)GetVolConfigShapeHorSampN(cfg));
                   fprintf(file,"Shape hor. sampling factor m : %d\n",
                        (Int)GetVolConfigShapeHorSampM(cfg));
                   fprintf(file,"Shape ver. sampling factor n : %d\n",
                        (Int)GetVolConfigShapeVerSampN(cfg));
                   fprintf(file,"Shape ver. sampling factor m : %d\n",
                        (Int)GetVolConfigShapeVerSampM(cfg));
                }
/* end: added for OBSS by Samsung AIT (1999-09-29) */
        }
/* begin: added for OBSS by Samsung AIT (1999-09-29) */
        } else
        {
		if( GetVolConfigVersion(cfg) != 1) {
           if(GetVolConfigScalability(cfg))
           {
                fprintf(file,"Shape hor. sampling factor n : %d\n",
                        (Int)GetVolConfigShapeHorSampN(cfg));
                fprintf(file,"Shape hor. sampling factor m : %d\n",
                        (Int)GetVolConfigShapeHorSampM(cfg));
                fprintf(file,"Shape ver. sampling factor n : %d\n",
                        (Int)GetVolConfigShapeVerSampN(cfg));
                fprintf(file,"Shape ver. sampling factor m : %d\n",
                        (Int)GetVolConfigShapeVerSampM(cfg));
           }
		} else {
			fprintf(stderr,"spatial scalable shape coding is not defined in version1\n");
			exit(1);
		}
        }
/* end: added for OBSS by Samsung AIT (1999-09-29) */

        if (GetVolConfigSpriteUsage(cfg) == SPRITE_NOT_USED)
                fprintf(file,"NO SPRITE USAGE\n"); 
/* modified NTT for GMC coding :start */
	else if (GetVolConfigSpriteUsage(cfg) == GMC_SPRITE) {
	    fprintf(file,"GMC coding\n"); 
	    fprintf(file,"No of sprite points : %d\n", GetVolConfigNoOfSpritePoints(cfg));
	    fprintf(file,"Warping Accuracy : %d\n", GetVolConfigWarpingAccuracy(cfg));
	    fprintf(file,"Brightness_change_in_sprite : %d\n", GetVolConfigBrightnessChangeInSprite(cfg));
	}
/* modified NTT for GMC coding :end */
        else
        if (GetVolConfigSpriteUsage(cfg) != SPRITE_NOT_USED) 
                {
                if (GetVolConfigSpriteUsage(cfg) == STATIC_SPRITE) 
                                {
                                fprintf(file,"OFF-LINE STATIC SPRITE ");
                                if (GetVolConfigLowLatencySpriteEnable(cfg)==0)
                                        fprintf(file," (BASIC SPRITE)\n");
                                else
                                        fprintf(file," (LOW LATENCY SPRITE)\n");
                                }
                
                fprintf(file,"Sprite horiz. size : %d\n",GetVolConfigSpriteHdim(cfg));
                fprintf(file,"Sprite vert. size : %d\n",GetVolConfigSpriteVdim(cfg));
                fprintf(file,"Sprite left edge :  %d\n",GetVolConfigSpriteLeftEdge(cfg));
                fprintf(file,"Sprite top  edge :  %d\n",GetVolConfigSpriteTopEdge(cfg));
                fprintf(file,"No of sprite points : %d\n", GetVolConfigNoOfSpritePoints(cfg));
                fprintf(file,"Warping Accuracy : %d\n", GetVolConfigWarpingAccuracy(cfg));
                fprintf(file,"Brightness_change_in_sprite : %d\n", GetVolConfigBrightnessChangeInSprite(cfg));

                
                if (GetVolConfigSpriteUsage(cfg) == STATIC_SPRITE)
                        {
                        fprintf(file,"Sprite will be read from disk\n");
                        fprintf(file,"Sprite Y seq. : %s\n",GetVolConfigSpriteY(cfg));
                        fprintf(file,"Sprite U seq. : %s\n",GetVolConfigSpriteU(cfg));
                        fprintf(file,"Sprite V seq. : %s\n",GetVolConfigSpriteV(cfg));
                        fprintf(file,"Sprite A seq. : %s\n",GetVolConfigSpriteA(cfg));
                        if (GetVolConfigNoOfSpritePoints(cfg))
                                fprintf(file,"Warping parameters file : %s\n",
                                        GetVolConfigWarpParamFile(cfg));
                        else
                                fprintf(file,"Zero global motion parameters\n");
                        if (GetVolConfigLowLatencySpriteEnable(cfg))
                                fprintf(file,"Sprite Threshold in SNR : %f\n",
                                        GetVolConfigSpriteSNRthreshold(cfg));
                        }
        } /* case SPRITE_ENABLE */

        fprintf(file,"\n");


        return;
}

/***********************************************************CommentBegin******
 *
 * -- SetUpVOList -- Initialise a linked list of VOs 
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      23-09-96
 *
 * Purpose :            
 *      Given the configuration information for a set of VOs, this function
 *      sets up and initialises a linked list of VO structures.
 * 
 * Arguments in :       
 *      VOConfig *vo_config_list - pointer to linked list containing configuration
 *              information
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      VO *vo_list - pointer to linked list of VOs.
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      
 *
 * See also :
 *      
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/
VO *
SetUpVOList(VOConfig *vo_config_list)
{
        VO      *vo_list = NULL,
                        *vo;

        VOConfig *vo_config;

        vo_config = vo_config_list;

        while(vo_config != NULL)
        {
                vo = ConfigureVO(vo_config);

                if(vo_list == NULL)
                        vo_list = vo;
                else
                        AddVOToList(vo,vo_list);

                vo_config = GetVOConfigNext(vo_config);
        }

        return(vo_list);
}


/***********************************************************CommentBegin******
 *
 * -- PrintVolConfigList -- Prints the data contained in a linked list of
 *      VolConfig structures
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      30-06-96
 *
 * Purpose :            
 *      This function prints the data contained in each node of a linked list
 *      of VolConfig structures. The BASE layer is printed first followed by
 *      the other layers in the order they are encountered.
 *
 * Arguments in :       
 *      VolConfig *list - linked list 
 *      FILE *file - where to print
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      none
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      
 *
 * See also :
 *      
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/
        
Void
PrintVolConfigList(VolConfig *list,File *file)
{
        VolConfig       *curr;
        
        Int     layer = 0;

                        
        fprintf(file,"Video Object Layer\n");
        fprintf(file,"==================\n\n");


        curr = list;

        while(curr != NULL)
        {
                fprintf(file,"Layer %d\n", (Int)layer);
                fprintf(file,"--------\n\n");

                if(!GetVolConfigScalability(curr))
                        fprintf(file,"*** This is the BASE layer ***\n\n");
                else
                        fprintf(file,"*** This is an ENHANCEMENT layer *** \n\n");

                PrintVolConfig(curr,file);

                curr = GetVolConfigNext(curr);
                layer++;
        }

        return;
}

/***********************************************************CommentBegin******
 *
 * -- PrintVOConfigList -- Prints the data contained in a linked list of
 *      VOConfig structures
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      30-06-96
 *
 * Purpose :            
 *      This function prints the data contained in each node of a linked list
 *      of VOConfig structures. 
 *
 * Arguments in :       
 *      VOConfig *list - linked list 
 *      File *file - where to print
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      none
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      
 *
 * See also :
 *      
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/
Void
PrintVOConfigList(VOConfig *list,File *file)
{
        VOConfig *curr;

        curr = list;

        while(curr != NULL)
        {
                fprintf(file,"********************\n");
                fprintf(file,"*                  *\n");
                fprintf(file,"*    VIDEO OBJECT  *\n");
                fprintf(file,"*                  *\n");
                fprintf(file,"********************\n\n");
                fprintf(file,"VO Id : %d\n", (Int)GetVOConfigId(curr));
                fprintf(file,"# VOLs : %d\n\n", (Int)GetVOConfigNumVOLs(curr));

                PrintVolConfigList(GetVOConfigLayers(curr),file);

                curr = GetVOConfigNext(curr);
        }

        return;
}



/***********************************************************CommentBegin******
 *
 * -- ConfigureVO -- Configures a single VO based on it's configuration 
 *      infromation
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      23-09-96
 *
 * Purpose :            
 *      Given a VOConfig structure this functions sets up a VO structure for
 *      coding based on this config. data.
 * 
 * Arguments in :       
 *      VOConfig *vo_config - pointer to VOConfig structure
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      VO *vo - pointer to allocated and initialised VO structure.
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      
 *
 * See also :
 *      
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/

VO *
ConfigureVO(VOConfig *vo_config)
{
        VO *vo;

        Vol *vol,
                        *vol_list = NULL;
        
        VolConfig *vol_config;

        vo = SallocVO();

        PutVOId(GetVOConfigId(vo_config),vo);

        vol_config = GetVOConfigLayers(vo_config);

        while(vol_config != NULL)
        {
                vol = ConfigVol(vol_config);

                if(vol_list == NULL)
                        vol_list = vol;
                else
                        AddVolToList(vol,vol_list);

                vol_config = GetVolConfigNext(vol_config);
        }

        PutVOLayers(vol_list,vo);

        return(vo);
}



/***********************************************************CommentBegin******
 *
 * -- ConfigVol --  Configure a Vol structure using a VolConfig
 *      structure
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      30-06-96
 *
 * Purpose :            
 *      A VolConfig structure contains all the information for a Vol
 *      read from the encoder configuration file. This function
 *      allocates a Vol structure and initialises it with the relevant
 *      configuration information. The Vop in the Vol structure is also
 *      allocated. It is set up to have the dimensions of the the 
 *      source seq. on disk. Thus this Vol will act as an input
 *      buffer for Vops being read from disk
 *
 * Arguments in :       
 *      VolConfig *cfg - pointer to VOlConfig structure (assumed that the
 *      config has been read already).  
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none    
 *
 * Return values :      
 *      Vol *vol - newly allocated and filled Vol structure
 *
 * Side effects :       
 *      Memory is allocated for the Vol (n.b. and also it's associated
 *      Vop) in this function.
 *
 * Description :        
 *      -
 *
 * See also :
 *      -
 *
 * Modified :           
 *      04.02.97 Noel O'Connor: enabled error resilient mode (based on changes
 *                              by Luis Ducla-Soares - 30.01.97)
 *      26.02.97 Noel O'Connor: added shape_effects * reverse_vlc
 *      23.04.97 Michael Wollborn: Added MPEG quantization stuff completely
 *      26.04.97 Luis Ducla-Soares: Added data partitioning
 *      13.05.97 Minhua Zhou : added OBMCDisable
 *      12.08.97 Minhua Zhou : added time_increment_resolution
 *      04.06.98 Marc Mongenet: added Complexity Estimation syntax support
 *      18.01.99 Takefumi Nagumo: added hierarchy_type for scalable
 *      15.02.99 U. Benzler : added quarter pel support
 *	03.03.99 Seishi TAKAMURA (NTT): added GMC coding, version2 syntax
 *      04.06.99 Karsten Suehring (HHI): VOL Control Parameters 
 *	16.08.99 Shigeru Fukunaga (Oki): added modules for NEWPRED
 *	06.09.99 Eishi Morimatsu (Fujitsu Labs.): added DRC support
 *	09.29.99 Dae-Sung Cho (Samsung AIT): added support for OBSS
 *  26.10.99 Massimo Ravasi (EPFL): added Complexity Estimation syntax support
 *                                  Update version 2
 *      11.11.99 U. Benzler (UoH) - added visual_object_type_indication to be able to create conformance bitstreams
 *
 ***********************************************************CommentEnd********/
 
Vol *
ConfigVol(VolConfig *cfg)
{
        Vol     *vol;

        Vop     *vop;

        Int     dim_x,
                        dim_y;

	LInt	vbv_param[3];

        Int 	i;
	Char *string[3], sep_char[1],
        *vbv_string=NULL;

        /* Allocate Vol structure */
        vol = SallocVol();

/* modified NTT for version2 syntax : start */
	switch (GetVolConfigVersion(cfg)) {
	  case 1:	/* MPEG-4 version 1 */	
	    PutVolIsObjectLayerIdentifier(0, vol);
	    PutVolVisualObjectLayerVerid(1, vol);
	    break;
	  case 2:	/* MPEG-4 version 2 */
	    PutVolIsObjectLayerIdentifier(1, vol);
	    PutVolVisualObjectLayerVerid(2, vol);
	    break;
	  default:
	    fprintf(stderr, "error: mpeg-4 version %d is not defiend\n",
		    GetVolConfigVersion(cfg));
	    exit(-1);
	}
	PutVolVisualObjectLayerPriority(1, vol);
/* modified NTT for version2 syntax : end */

        /* MAC (SB) 11-Nov-99 */
        PutVolShapeExtension(GetVolConfigShapeExtension(cfg),vol);
        PutVolAuxCompCount(GetVolConfigAuxCompCount(cfg),vol);


        /* Do syntax fields first */
        PutVolId(GetVolConfigId(cfg),vol);
        PutVolVisualObjectTypeIndication(GetVolConfigTypeIndication(cfg),vol); /* 11.11.99 U. Benzler (UoH) - added visual_object_type_indication to be able to create conformance bitstreams */


        PutVolShape(GetVolConfigShape(cfg),vol);
        PutVolShapeEffects(GetVolConfigShapeEffects(cfg),vol);
        PutVolInterlaced(GetVolConfigInterlaced(cfg),vol); /* added by Sven brandau, 19.08.98 */
        PutVolOBMCDisable(GetVolConfigOBMCDisable(cfg),vol);  
        PutVolTimeIncrementResolution(GetVolConfigTimeIncrementResolution(cfg),vol);   
        if(GetVolConfigShape(cfg) == RECTANGULAR)
        {
                PutVolWidth(GetVolConfigDiskSeqX(cfg),vol);
                PutVolHeight(GetVolConfigDiskSeqY(cfg),vol);
        }
        PutVolQuantType(GetVolConfigQuantType(cfg),vol);

        /* Added MPEG-like quantization completely, 23-APR-1997 MW */
        if(GetVolConfigQuantType(cfg))
          ConfigVolMPEGQuant(cfg,vol);

        PutVolDisableGrayQuantUpdate(GetVolConfigDisableGrayQuantUpdate(cfg),
                                     vol);
        /* START: VOL Control Parameters - Karsten Suehring (HHI) */
        if (GetVolConfigVolControlParameters(cfg))
        {
		PutVolVolControlParameters(1,vol);
		PutVolChromaFormat (1,vol); /* Chroma Format 4:2:0 only possible */
		if (GetVolConfigM(cfg) > 1) PutVolLowDelay (0,vol);
                else PutVolLowDelay (1,vol);
        } else
	{
		PutVolVolControlParameters(0,vol);
	}
        if (GetVolConfigVbvParameters(cfg))
	{
                PutVolVbvParameters (1,vol);
		sep_char[0]='|';
                vbv_string= GetVolConfigVbvParam(cfg);
   		if ((string[0] = strtok(vbv_string, sep_char)) == NULL)
     			error_exit("ERROR ConfigVol: Error in VBV parameters selection! \n");
		for (i=1;i<3;i++)
   		  if ((string[i] = strtok(NULL, sep_char)) == NULL)
     			error_exit("ERROR ConfigVol: Error in VBV parameters selection! \n");
		
		for (i=0;i<3;i++)
                {  
		  sscanf(string[i],"%ld", &vbv_param[i]);
		  if ((i<2)&&(vbv_param[i]<1))
			error_exit("ERROR ConfigVol: Error in VBV parameters selection! \n");
		}
		if (vbv_param[2]<0) 
                        error_exit("ERROR ConfigVol: Error in VBV parameters selection! \n");
		PutVolFirstHalfBitRate((vbv_param[0]>>15)&32767,vol);
		PutVolLatterHalfBitRate(vbv_param[0]&32767,vol);
		PutVolFirstHalfVbvBufferSize((vbv_param[1]>>3)&32767,vol);
		PutVolLatterHalfVbvBufferSize(vbv_param[1]&7,vol);
		PutVolFirstHalfVbvOccupancy((vbv_param[2]>>15)&2047,vol);
		PutVolLatterHalfVbvOccupancy(vbv_param[2]&32767,vol);

	} else
	{
		PutVolVbvParameters (0,vol);
	}

        /* END: VOL Control Parameters */

        /* START: Complexity Estimation syntax support - Marc Mongenet (EPFL) - 4 Jun 1998 */
        PutVolComplexityEstimationDisable(GetVolConfigComplexityEstimationDisable(cfg), vol);
        PutVolEstimationMethod(GetVolConfigEstimationMethod(cfg), vol);
        PutVolShapeComplexityEstimationDisable(GetVolConfigShapeComplexityEstimationDisable(cfg), vol);
        PutVolOpaque(GetVolConfigOpaque(cfg), vol);
        PutVolTransparent(GetVolConfigTransparent(cfg), vol);
        PutVolIntraCAE(GetVolConfigIntraCAE(cfg), vol);
        PutVolInterCAE(GetVolConfigInterCAE(cfg), vol);
        PutVolNoUpdate(GetVolConfigNoUpdate(cfg), vol);
        PutVolUpsampling(GetVolConfigUpsampling(cfg), vol);
        PutVolTextureComplexityEstimationSet1Disable(GetVolConfigTextureComplexityEstimationSet1Disable(cfg), vol);
        PutVolIntraBlocks(GetVolConfigIntraBlocks(cfg), vol);
        PutVolInterBlocks(GetVolConfigInterBlocks(cfg), vol);
        PutVolInter4vBlocks(GetVolConfigInter4vBlocks(cfg), vol);
        PutVolNotCodedBlocks(GetVolConfigNotCodedBlocks(cfg), vol);
        PutVolTextureComplexityEstimationSet2Disable(GetVolConfigTextureComplexityEstimationSet2Disable(cfg), vol);
        PutVolDCTCoefs(GetVolConfigDCTCoefs(cfg), vol);
        PutVolDCTLines(GetVolConfigDCTLines(cfg), vol);
        PutVolVLCSymbols(GetVolConfigVLCSymbols(cfg), vol);
        PutVolVLCBits(GetVolConfigVLCBits(cfg), vol);
        PutVolMotionCompensationComplexityDisable(GetVolConfigMotionCompensationComplexityDisable(cfg), vol);
        PutVolAPM(GetVolConfigAPM(cfg), vol);
        PutVolNPM(GetVolConfigNPM(cfg), vol);
        PutVolInterpolateMCQ(GetVolConfigInterpolateMCQ(cfg), vol);
        PutVolForwBackMCQ(GetVolConfigForwBackMCQ(cfg), vol);
        PutVolHalfpel2(GetVolConfigHalfpel2(cfg), vol);
        PutVolHalfpel4(GetVolConfigHalfpel4(cfg), vol);

      	/* START: Complexity Estimation syntax support - Update version 2 - Massimo Ravasi (EPFL) - 26 Oct 1999 */
				PutVolVersion2ComplexityEstimationDisable(GetVolConfigVersion2ComplexityEstimationDisable(cfg), vol);
				PutVolSadct(GetVolConfigSadct(cfg), vol);
				PutVolQuarterpel(GetVolConfigQuarterpel(cfg), vol);
        /* END: Complexity Estimation syntax support - Update Version 2 */
        /* END: Complexity Estimation syntax support */


        PutVolErrorResDisable(GetVolConfigErrorResDisable(cfg),vol); /* Error Resilience is now supported */
        PutVolDataPartEnable(GetVolConfigDataPartEnable(cfg),vol);
        PutVolReverseVlc(GetVolConfigReverseVlc(cfg),vol);
	/** added for NEWPRED (Oki) 16-AUG-1999 **/
	PutVolNewpredEnable(GetVolConfigNewpredEnable(cfg),vol);
	PutVolNewpredSegmentType(GetVolConfigNewpredSegmentType(cfg),vol); /** Oki 26-AUG-1999 **/
	if(GetVolConfigNewpredEnable(cfg)){
	  PutVolNewpredBackwardMessageType(GetVolConfigNewpredBackwardMessageType(cfg),vol);
	}
	/** end of NEWPRED (Oki) 16-AUG-1999 **/
/* >>> added for DRC by Fujitsu (top) <<< */
	if(GetVolConfigDrcEnable(cfg)==0)
	    PutVolReducedResolutionVopEnable(0,vol);
	else
	    PutVolReducedResolutionVopEnable(1,vol);
/* >>> added for DRC by Fujitsu (bottom) <<< */
        PutVolACDCPredDisable(GetVolConfigACDCPredDisable(cfg),vol);
	/* 01.02.99 HHI Schueuer */
	PutVolSADCTDisable(GetVolConfigSADCTDisable(cfg),vol);
	/* end HHI */
	PutVolQuarterPel(GetVolConfigQuarterPel(cfg),vol);		/* MW QPEL 07-JUL-1998 */
        PutVolQuantPrecision(GetVolConfigQuantPrecision(cfg),vol);
        PutVolBitsPerPixel(GetVolConfigBitsPerPixel(cfg),vol);
        PutVolScalability(GetVolConfigScalability(cfg),vol);
/* begin: added for OBSS by Samsung AIT (1999-09-29) */
        if(GetVolConfigShape(cfg) != BINARY_SHAPE_ONLY)
        {
/* end: added for OBSS by Samsung AIT (1999-09-29) */
        if(GetVolScalability(vol))
        {
                PutVolScalType(GetVolConfigScalType(cfg),vol);
                if(GetVolConfigScalType (cfg) == 0 || /*Temporal scalability*/
                   GetVolConfigScalType (cfg) == 3 ||  
                   GetVolConfigScalType (cfg) == 4 )
                  PutVolHierarchyType(1,vol);
                else if (GetVolConfigScalType (cfg) == 1) /*Spatial Scalability*/
                  PutVolHierarchyType(0,vol);
                else{
                  fprintf(stderr,"Scalable Type 2 (both temporal & spatial scalability) is not defined in version1\n");
                  exit(1);
                }
                
                PutVolRefId(GetVolConfigRefId(cfg),vol);
                PutVolRefSampDir(GetVolConfigRefSampDir(cfg),vol);
                PutVolHorSampN(GetVolConfigHorSampN(cfg),vol);
                PutVolHorSampM(GetVolConfigHorSampM(cfg),vol);
                PutVolVerSampN(GetVolConfigVerSampN(cfg),vol);
                PutVolVerSampM(GetVolConfigVerSampM(cfg),vol);
                PutVolEnhanceType(GetVolConfigEnhanceType(cfg),vol);

/* begin: added for OBSS by Samsung AIT (1999-09-29) */
                if(GetVolConfigShape(cfg)==1 && GetVolConfigScalType(cfg)==1) {
                        PutVolUseRefShape(GetVolConfigUseRefShape(cfg),vol);
                        PutVolUseRefTexture(GetVolConfigUseRefTexture(cfg),vol);
                        PutVolShapeHorSampN(GetVolConfigShapeHorSampN(cfg),vol);
                        PutVolShapeHorSampM(GetVolConfigShapeHorSampM(cfg),vol);
                        PutVolShapeVerSampN(GetVolConfigShapeVerSampN(cfg),vol);
                        PutVolShapeVerSampM(GetVolConfigShapeVerSampM(cfg),vol);
                }
/* end: added for OBSS by Samsung AIT (1999-09-29) */
        }
/* begin: added for OBSS by Samsung AIT (1999-09-29) */
        } else
        {
		if (GetVolConfigVersion(cfg) !=1) {
           if(GetVolScalability(vol))
           {
                PutVolShapeHorSampN(GetVolConfigShapeHorSampN(cfg),vol);
                PutVolShapeHorSampM(GetVolConfigShapeHorSampM(cfg),vol);
                PutVolShapeVerSampN(GetVolConfigShapeVerSampN(cfg),vol);
                PutVolShapeVerSampM(GetVolConfigShapeVerSampM(cfg),vol);
           }
		} else {
			fprintf(stderr,"scalable shape coding is not defined in version1\n");
			exit(1);	
		}
        }
/* end: added for OBSS by Samsung AIT (1999-09-29) */

        /* Now attach a vop. This is the vop into which the disk
                 frame will be read */
        dim_x = GetVolConfigDiskSeqX(cfg);
        dim_y = GetVolConfigDiskSeqY(cfg);

        PutVolSpriteUsage(GetVolConfigSpriteUsage(cfg),vol);

        if (cfg->interlaced) {
            /*
             * All of the following "unimplemented" interlaced modes make sense
             * but simply have not been coded or tested.  These should all be
             * version 2 work items.                  Bob Eifrig (7-Nov-97).
             */
            if (GetVolConfigScalability(cfg)) {
                printf("Interlaced scalability is not implemented\n");
                exit(1);
            }
/* modified by NTT for GMC coding : start
            if (GetVolSpriteUsage(vol) != SPRITE_NOT_USED) {
*/
	    if (GetVolSpriteUsage(vol) == STATIC_SPRITE) {
/* modified by NTT for GMC coding : end */
                printf("Interlaced sprites are not defined\n");
                exit(1);
            }
            if (GetVolConfigErrorResDisable(cfg) == 0) {
                printf("Interlaced error resilience coding is not implemented\n");
                exit(1);
            }
        } else
            cfg->alternate_scan = 0;            /* Non-zero only in interlaced mode */

        if (GetVolSpriteUsage(vol) != SPRITE_NOT_USED)
                {
                PutVolSpriteHdim(GetVolConfigSpriteHdim(cfg),vol);
                PutVolSpriteVdim(GetVolConfigSpriteVdim(cfg),vol);
                PutVolSpriteLeftEdge(GetVolConfigSpriteLeftEdge(cfg),vol);
                PutVolSpriteTopEdge(GetVolConfigSpriteTopEdge(cfg),vol);
                PutVolNoOfSpritePoints(GetVolConfigNoOfSpritePoints(cfg),vol);
                PutVolWarpingAccuracy(GetVolConfigWarpingAccuracy(cfg),vol);
                PutVolBrightnessChangeInSprite(GetVolConfigBrightnessChangeInSprite(cfg),vol);
                if (GetVolConfigNoOfSpritePoints(cfg)>0)
                        PutVolRefPointCoord(GetVolConfigRefPointCoord(cfg),vol);
                if (GetVolSpriteUsage(vol) == STATIC_SPRITE)
                        PutVolLowLatencySpriteEnable(GetVolConfigLowLatencySpriteEnable(cfg),vol);

                PutVolSprite(GetVolConfigSprite(cfg),vol);
                PutVopId(GetVolId(vol), GetVolSprite(vol));
                }
        
        vop = AllocVop(dim_x,dim_y,GetVolConfigAuxCompCount(cfg));
        PutVopId(GetVolId(vol), vop);

        PutVolVop(vop,vol);
        PutVopBitsPerPixel(GetVolBitsPerPixel(vol),vop);
        PutVopQuantPrecision(GetVolQuantPrecision(vol),vop);

        return(vol);
}       

/***********************************************************CommentBegin******
 *
 * -- InitControlParams -- Initialise certain control parameters
 *
 * Author :             
 *      Noel O'Connor Teltec Irl.
 *
 * Created :            
 *      23-09-96
 *
 * Purpose :            
 *      This function initialises some useful control parameters based on
 *      information read form the configuration files. The parameters:
 *              - initialise the "next" frame to be read
 *              - Calculate frame skip field
 *              - Calculate vol time increment
 *              - Calculate vol start & end times
 *              - Calculate session time increment
 *              - Calculate session start & end times
 * 
 * Arguments in :       
 *      none
 *      
 * Arguments in/out :   
 *      VOConfig *vo_cfg - linked list of VOConfig structures containing the
 *              config data read from the configuration files.
 *      EncodeControl - Pointer to EncodeControl structure containing
 *              control information read from control file.
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      none 
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      -
 *
 * See also :
 *      
 *
 * Modified :   
 *        10.03.97  Minhua Zhou: Added M to the calculation of END time 
 *
 *      
 *
 ***********************************************************CommentEnd********/


Void
InitControlParams(VOConfig *vo_config_list,EncodeControl *enc_ctl)
{
        VOConfig        *vo_cfg;

        VolConfig       *vol_cfg;

        Int                             start_frame, M,
                                                end_frame,
                                                frame_skip;

        Float                   disk_frame_rate,
                                                vol_frame_rate,
                                                start_time,
                                                min_start_time = MAX_FLOAT_VAL,
                                                max_end_time = 0,
                                                min_time_inc = MAX_FLOAT_VAL,
                                                end_time,
                                                time_inc;

        vo_cfg = vo_config_list;
        while(vo_cfg != NULL)
        {
                vol_cfg = GetVOConfigLayers(vo_cfg);
                while(vol_cfg != NULL)
                {
                        /* Init. "next" frame to be coded to start frame */
                        start_frame = GetVolConfigStartFrame(vol_cfg);
                        PutVolConfigFrame(start_frame,vol_cfg);

                        /* Calculate frame skip field */
                        disk_frame_rate = GetVolConfigDiskSeqFrameRate(vol_cfg);
                        vol_frame_rate = GetVolConfigFrameRate(vol_cfg);
                        M     =  GetVolConfigM(vol_cfg);
                        frame_skip = (Int)(disk_frame_rate/vol_frame_rate);
                        PutVolConfigFrameSkip(frame_skip,vol_cfg);

                        /* Calculate time increment */
                        time_inc = (Float)1000/vol_frame_rate;
                        PutVolConfigTimeInc(time_inc,vol_cfg);

                        /* Calculate end time */
                        start_time = GetVolConfigStartTime(vol_cfg);
                        end_frame = GetVolConfigEndFrame(vol_cfg);
/*** modified ***/
                        /* end_time = start_time + (((end_frame - start_frame)/(M*frame_skip)) * M * time_inc); */
                        end_time = start_time + (((Float)(end_frame - start_frame)/(Float)(M*frame_skip)) * M * time_inc); 
/*** modified ***/
                        end_time = RoundTime(end_time);
                        PutVolConfigEndTime(end_time,vol_cfg);

                        /* Calculate minimium time increment */
                        if(time_inc < min_time_inc)
                                min_time_inc = time_inc;

                        /* Calculate minimum start time */
                        if(start_time < min_start_time)
                                min_start_time = start_time;

                        /* Calculate maximum end time */
                        if(end_time > max_end_time)
                                max_end_time = end_time;

                        /* Initialise the modulo time base to zero */
                        /* Minhua Zhou 09.05.97: two times of call
                           to set both time bases to zero */
                        vol_cfg->modulo_time_base[0] =0;
                        vol_cfg->modulo_time_base[1] =0;
                

                        /* Initialise next coding time to start time */
                        PutVolConfigNextCodingTime(start_time,vol_cfg);

                        vol_cfg = GetVolConfigNext(vol_cfg);
                }
                
                vo_cfg = GetVOConfigNext(vo_cfg);
        }

        /* Record the max and min times */
        PutEncodeControlStartTime(min_start_time,enc_ctl);
        PutEncodeControlEndTime(max_end_time,enc_ctl);
        PutEncodeControlTimeInc(min_time_inc,enc_ctl);

        return;
}

/***********************************************************CommentBegin******
 *
 * -- ConfigVolMPEGQuant --  Configure a Vol structure using a VolConfig
 *      structure, here especially the MPEG quantization stuff
 *
 * Author :             
 *      Michael Wollborn, TUH
 *
 * Created :            
 *      23-APR-1997
 *
 * Purpose :            
 *      The parts for the MPEG quantization stuff are configured in
 *      a separate module due to the length of the source code.
 *
 * Arguments in :       
 *      VolConfig *cfg - pointer to VOlConfig structure (assumed that the
 *      config has been read already).  
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      Vol *vol - pointer to VOL to be configured      
 *
 * Return values :      
 *      none
 *
 * Side effects :       
 *
 * Description :        
 *      -
 *
 * See also :
 *      -
 *
 * Modified :           
 *
 ***********************************************************CommentEnd********/
 
Void
ConfigVolMPEGQuant(VolConfig *cfg, Vol *vol)
{
  Int i, type, aux;
  Int *qmat_cfg, *qmat_vol;

  /* IntraQuantMat */
  type = GetVolConfigLoadIntraQuantMat(cfg);

  if(type == 0)         /* default flat matrix */
    {
      PutVolLoadIntraQuantMat(0,vol);
      qmat_vol = GetVolIntraQuantMat(vol);
      qmat_cfg = GetVolConfigIntraQuantMat(cfg);
      
      for(i=0; i<64; i++)
        qmat_cfg[i] =qmat_vol[i] = mpeg_iqmat_def[i];
    }
  else if(type == 1)    /* user defined matrix */
    {
      PutVolLoadIntraQuantMat(1,vol);
      qmat_vol = GetVolIntraQuantMat(vol);
      qmat_cfg = GetVolConfigIntraQuantMat(cfg);
      
      LoadUserQuantMat(GetVolConfigIntraQuantMatFile(cfg),qmat_cfg);
      LoadUserQuantMat(GetVolConfigIntraQuantMatFile(cfg),qmat_vol);
    }
  else if(type < 2+NUM_MPEG_QMAT_SPEC)  /* predefined special matrix */
    {
      PutVolLoadIntraQuantMat(1,vol);
      qmat_vol = GetVolIntraQuantMat(vol);
      qmat_cfg = GetVolConfigIntraQuantMat(cfg);
      
      for(i=0; i<64; i++)
        qmat_cfg[i] = qmat_vol[i] = mpeg_iqmat_spec[type-2][i];
    }
  else
    {
      printf("ERROR: wrong flag for load_intra_quant_mat: %d\n",type);
      exit(1);
    }

  /* NonintraQuantMat */
  type = GetVolConfigLoadNonintraQuantMat(cfg);

  if(type == 0)         /* default flat matrix */
    {
      PutVolLoadNonintraQuantMat(0,vol);
      qmat_vol = GetVolNonintraQuantMat(vol);
      qmat_cfg = GetVolConfigNonintraQuantMat(cfg);
      
      for(i=0; i<64; i++)
        qmat_cfg[i] =qmat_vol[i] = mpeg_nqmat_def[i];
    }
  else if(type == 1)    /* user defined matrix */
    {
      PutVolLoadNonintraQuantMat(1,vol);
      qmat_vol = GetVolNonintraQuantMat(vol);
      qmat_cfg = GetVolConfigNonintraQuantMat(cfg);
      
      LoadUserQuantMat(GetVolConfigNonintraQuantMatFile(cfg),qmat_cfg);
      LoadUserQuantMat(GetVolConfigNonintraQuantMatFile(cfg),qmat_vol);
    }
  else if(type < 2+NUM_MPEG_QMAT_SPEC)  /* predefined special matrix */
    {
      PutVolLoadNonintraQuantMat(1,vol);
      qmat_vol = GetVolNonintraQuantMat(vol);
      qmat_cfg = GetVolConfigNonintraQuantMat(cfg);
      
      for(i=0; i<64; i++)
        qmat_cfg[i] = qmat_vol[i] = mpeg_nqmat_spec[type-2][i];
    }
  else
    {
      printf("ERROR: wrong flag for load_intra_quant_mat: %d\n",type);
      exit(1);
    }

  /* GrayIntraQuantMat */
  type = GetVolConfigLoadGrayIntraQuantMat(cfg);

  if(type == 0)         /* default flat matrix */
    {
      for(aux=0;aux<GetVolConfigAuxCompCount(cfg);aux++) { /* MAC (SB) 17-Nov-99 */
        PutVolLoadGrayIntraQuantMat(0,aux,vol);
        qmat_vol = GetVolGrayIntraQuantMat(aux,vol);
        qmat_cfg = GetVolConfigGrayIntraQuantMat(cfg);
      
        for(i=0; i<64; i++)
          qmat_cfg[i] =qmat_vol[i] = mpeg_giqmat_def[i];
      }
    }
  else if(type == 1)    /* user defined matrix */
    {
      for(aux=0;aux<GetVolConfigAuxCompCount(cfg);aux++) { /* MAC (SB) 17-Nov-99 */
        PutVolLoadGrayIntraQuantMat(1,aux,vol);
        qmat_vol = GetVolGrayIntraQuantMat(aux,vol);
        qmat_cfg = GetVolConfigGrayIntraQuantMat(cfg);
      
        LoadUserQuantMat(GetVolConfigGrayIntraQuantMatFile(cfg),qmat_cfg);
        LoadUserQuantMat(GetVolConfigGrayIntraQuantMatFile(cfg),qmat_vol);
      }
    }
  else if(type < 2+NUM_MPEG_QMAT_SPEC)  /* predefined special matrix */
    {
      for(aux=0;aux<GetVolConfigAuxCompCount(cfg);aux++) { /* MAC (SB) 17-Nov-99 */

        PutVolLoadGrayIntraQuantMat(1,aux,vol);
        qmat_vol = GetVolGrayIntraQuantMat(aux,vol);
        qmat_cfg = GetVolConfigGrayIntraQuantMat(cfg);
      
        for(i=0; i<64; i++)
          qmat_cfg[i] = qmat_vol[i] = mpeg_giqmat_spec[type-2][i];
      }
    }
  else
    {
      printf("ERROR: wrong flag for load_intra_quant_mat: %d\n",type);
      exit(1);
    }

  /* GrayNonintraQuantMat */
  type = GetVolConfigLoadGrayNonintraQuantMat(cfg);

  if(type == 0)         /* default flat matrix */
    {
      for(aux=0;aux<GetVolConfigAuxCompCount(cfg);aux++) { /* MAC (SB) 17-Nov-99 */
        PutVolLoadGrayNonintraQuantMat(0,aux,vol);
        qmat_vol = GetVolGrayNonintraQuantMat(aux,vol);
        qmat_cfg = GetVolConfigGrayNonintraQuantMat(cfg);
      
        for(i=0; i<64; i++)
          qmat_cfg[i] =qmat_vol[i] = mpeg_gnqmat_def[i];
      }
    }
  else if(type == 1)    /* user defined matrix */
    {
      for(aux=0;aux<GetVolConfigAuxCompCount(cfg);aux++) { /* MAC (SB) 17-Nov-99 */
        PutVolLoadGrayNonintraQuantMat(1,aux,vol);
        qmat_vol = GetVolGrayNonintraQuantMat(aux,vol);
        qmat_cfg = GetVolConfigGrayNonintraQuantMat(cfg);
      
        LoadUserQuantMat(GetVolConfigGrayNonintraQuantMatFile(cfg),qmat_cfg);
        LoadUserQuantMat(GetVolConfigGrayNonintraQuantMatFile(cfg),qmat_vol);
      }
    }
  else if(type < 2+NUM_MPEG_QMAT_SPEC)  /* predefined special matrix */
    {
      for(aux=0;aux<GetVolConfigAuxCompCount(cfg);aux++) { /* MAC (SB) 17-Nov-99 */
        PutVolLoadGrayNonintraQuantMat(1,aux,vol);
        qmat_vol = GetVolGrayNonintraQuantMat(aux,vol);
        qmat_cfg = GetVolConfigGrayNonintraQuantMat(cfg);
      
        for(i=0; i<64; i++)
          qmat_cfg[i] = qmat_vol[i] = mpeg_gnqmat_spec[type-2][i];
      }
    }
  else
    {
      printf("ERROR: wrong flag for load_intra_quant_mat: %d\n",type);
      exit(1);
    }

  return;
}





/***********************************************************CommentBegin******
 *
 * -- LoadUserQuantMat --  Load a user defined quantization matrix
 *
 * Author :             
 *      Michael Wollborn, TUH
 *
 * Created :            
 *      23-APR-1997
 *
 * Purpose :            
 *      A user defined quantization matrix is loaded from the disk
 *
 * Arguments in :       
 *      Char *filename - file with the quantization matrix coefficients
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      Vol *vol - pointer to VOL to be configured      
 *
 * Return values :      
 *      none
 *
 * Side effects :       
 *
 * Description :        
 *      -
 *
 * See also :
 *      -
 *
 * Modified :           
 *
 ***********************************************************CommentEnd********/
 
Void
LoadUserQuantMat(Char *qmat_filename, Int *qmat)
{
  FILE  *qmat_file;

  Int i,coef;

  Char  garbage[201];

  /* Check if file is there */
  if((qmat_file = fopen(qmat_filename,"r")) == NULL)
    {
      fprintf(stderr,"ERROR(LoadUserQuantMat): unable to open %s\n",
              qmat_filename);
      exit(1);
    }

  /* Load matrix elements, one element in each line */
  for(i=0; i<64; i++)
    {
      fscanf(qmat_file,"%d",&coef);
      fgets(garbage,201,qmat_file);
      qmat[i] = coef;
    }

  /* Close file and return */
  fclose(qmat_file);

  return;
}




/***********************************************************CommentBegin******
 *
 * -- LoadComplexityEstimationFile -- Init complexity estimation structs from file.
 *
 * Author :
 *      Marc Mongenet, Swiss Federal Institute of Technology, Lausanne
 *
 * Created :
 *      04-06-98
 *
 * Purpose :
 *      This function init the VolConfig structure part for complexity estimation.
 *      Data are loaded from a file. exit() is called if something goes wrong
 *      while reading the file or an invalid data is read.
 *
 * Arguments in :
 *
 * Arguments in/out :
 *      VolConfig *cfg - pointer to VolConfig structure (assumed that the
 *      config has been read already).  
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *      -
 *
 * See also :
 *
 * Modified :
 *    26.10.99 Massimo Ravasi (EPFL): added Complexity Estimation syntax support
 *                                    Update version 2
 *
 ***********************************************************CommentEnd********/

Void LoadComplexityEstimationFile (VolConfig * cfg)
{
  const Char * file_name = GetVolConfigComplexityEstimationFile(cfg);
  FILE * file_ptr;
  Char line [202];
  Char * line_addr = line;
  Int read1, read2;


  /*
   * Open file with define_VOP_complexity_estimation_header() data.
   */
  file_ptr = fopen(file_name, "r");
  if (file_ptr == 0) {
    fprintf(stderr,
            "ERROR(LoadComplexityEstimationFile): "
            "unable to open file '%s'\n",
            file_name);
    exit(1);
  }


  /*
   * Skip the comments section.
   */
  do {
    line_addr = fgets(line, 201, file_ptr);
    if (line_addr != line) {
      fprintf(stderr,
              "ERROR(LoadComplexityEstimationFile): "
              "error while looking for 'estimation_method' in file '%s'\n",
              file_name);
      exit(1);      
    }
  } while (line[0] == '#');


  /*
   * Read and check estimation_method
   */
  PutVolConfigEstimationMethod(atoi(line), cfg);
  if (GetVolConfigEstimationMethod(cfg) != 0 && GetVolConfigEstimationMethod(cfg) != 1) {
    fprintf(stderr,
            "ERROR(LoadComplexityEstimationFile): "
            "'estimation_method' %d not supported in file '%s'\n",
            GetVolConfigEstimationMethod(cfg),
            file_name);
    exit(1);
  }


  /*
   * Read shape_complexity_estimation_disable
   */
  line_addr = fgets(line, 201, file_ptr);
  if (line_addr != line) {
    fprintf(stderr,
            "ERROR(LoadComplexityEstimationFile): "
            "error while reading 'shape_complexity_estimation_disable' in file '%s'\n",
            file_name);
    exit(1);      
  }
  PutVolConfigShapeComplexityEstimationDisable(atoi(line), cfg);

  if (GetVolConfigShapeComplexityEstimationDisable(cfg)) {
    int i;
    for (i = 0; i < 6; ++i) {
      line_addr = fgets(line, 201, file_ptr);
      if (line_addr != line) {
        fprintf(stderr,
                "ERROR(LoadComplexityEstimationFile): "
                "error while skipping 'shape_complexity_estimation' data in file '%s'\n",
                file_name);
        exit(1);
      }
      if (atoi(line) != 0)
        fprintf(stderr,
                "WARNING(LoadComplexityEstimationFile): "
                "ignoring one of 'shape_complexity_estimation' enabled flag in file '%s'\n",
                file_name);
    }

  } else { /* !shape_complexity_estimation_disable */

    /* Read opaque */
    ReadDCECSFileEntry("opaque", &read1, &read2, file_ptr, file_name);
    PutVolConfigOpaque(read1, cfg);
    PutVolConfigDCECSOpaque(read2, cfg);

    /* Read transparent */
    ReadDCECSFileEntry("transparent", &read1, &read2, file_ptr, file_name);
    PutVolConfigTransparent(read1, cfg);
    PutVolConfigDCECSTransparent(read2, cfg);

    /* Read intra_cae */
    ReadDCECSFileEntry("intra_cae", &read1, &read2, file_ptr, file_name);
    PutVolConfigIntraCAE(read1, cfg);
    PutVolConfigDCECSIntraCAE(read2, cfg);

    /* Read inter_cae */
    ReadDCECSFileEntry("inter_cae", &read1, &read2, file_ptr, file_name);
    PutVolConfigInterCAE(read1, cfg);
    PutVolConfigDCECSInterCAE(read2, cfg);

    /* Read no_update */
    ReadDCECSFileEntry("no_update", &read1, &read2, file_ptr, file_name);
    PutVolConfigNoUpdate(read1, cfg);
    PutVolConfigDCECSNoUpdate(read2, cfg);

    /* Read upsampling */
    ReadDCECSFileEntry("upsampling", &read1, &read2, file_ptr, file_name);
    PutVolConfigUpsampling(read1, cfg);
    PutVolConfigDCECSUpsampling(read2, cfg);

    if (GetVolConfigOpaque(cfg) == 0 &&
        GetVolConfigTransparent(cfg) == 0 &&
        GetVolConfigIntraCAE(cfg) == 0 &&
        GetVolConfigInterCAE(cfg) == 0 &&
        GetVolConfigNoUpdate(cfg) == 0 &&
        GetVolConfigUpsampling(cfg) == 0) {
      fprintf(stderr,
              "WARNING(LoadComplexityEstimationFile): "
              "enabling shape_complexity_estimation_disable "
              "because no correponding flag selected.\n");
      PutVolConfigShapeComplexityEstimationDisable(1, cfg);
    }
  }

  /*
   * Read texture_complexity_estimation_set_1_disable
   */
  line_addr = fgets(line, 201, file_ptr);
  if (line_addr != line) {
    fprintf(stderr,
            "ERROR(LoadComplexityEstimationFile):"
            " error while reading 'texture_complexity_estimation_set_1_disable' in file '%s'\n",
            file_name);
    exit(1);      
  }
  PutVolConfigTextureComplexityEstimationSet1Disable(atoi(line), cfg);

  if (GetVolConfigTextureComplexityEstimationSet1Disable(cfg)) {
    int i;
    for (i = 0; i < 4; ++i) {
      line_addr = fgets(line, 201, file_ptr);
      if (line_addr != line) {
        fprintf(stderr,
                "ERROR(LoadComplexityEstimationFile): "
                "error while skipping 'texture_complexity_estimation_set_1' data in file '%s'\n",
                file_name);
        exit(1);
      }      
      if (atoi(line) != 0)
        fprintf(stderr,
                "WARNING(LoadComplexityEstimationFile): "
                "ignoring one of 'texture_complexity_estimation_set_1' enabled flag in file '%s'\n",
                file_name);
    }

  } else { /* !texture_complexity_estimation_set_1_disable */

    /* Read intra_blocks */
    ReadDCECSFileEntry("intra_blocks", &read1, &read2, file_ptr, file_name);
    PutVolConfigIntraBlocks(read1, cfg);
    PutVolConfigDCECSIntraBlocks(read2, cfg);

    /* Read inter_blocks */
    ReadDCECSFileEntry("inter_blocks", &read1, &read2, file_ptr, file_name);
    PutVolConfigInterBlocks(read1, cfg);
    PutVolConfigDCECSInterBlocks(read2, cfg);

    /* Read inter4v_blocks */
    ReadDCECSFileEntry("inter4v_blocks", &read1, &read2, file_ptr, file_name);
    PutVolConfigInter4vBlocks(read1, cfg);
    PutVolConfigDCECSInter4vBlocks(read2, cfg);

    /* Read not_coded_blocks */
    ReadDCECSFileEntry("not_coded_blocks", &read1, &read2, file_ptr, file_name);
    PutVolConfigNotCodedBlocks(read1, cfg);
    PutVolConfigDCECSNotCodedBlocks(read2, cfg);

    if (GetVolConfigIntraBlocks(cfg) == 0 &&
        GetVolConfigInterBlocks(cfg) == 0 &&
        GetVolConfigInter4vBlocks(cfg) == 0 &&
        GetVolConfigNotCodedBlocks(cfg) == 0) {
      fprintf(stderr,
              "WARNING(LoadComplexityEstimationFile): "
              "enabling texture_complexity_estimation_set_1_disable "
              "because no correponding flag selected.\n");
      PutVolConfigTextureComplexityEstimationSet1Disable(1, cfg);
    }
 }

  /*
   * Read texture_complexity_estimation_set_2_disable
   */
  line_addr = fgets(line, 201, file_ptr);
  if (line_addr != line) {
    fprintf(stderr,
            "ERROR(LoadComplexityEstimationFile):"
            " error while reading 'texture_complexity_estimation_set_2_disable' in file '%s'\n",
            file_name);
    exit(1);      
  }
  PutVolConfigTextureComplexityEstimationSet2Disable(atoi(line), cfg);

  if (GetVolConfigTextureComplexityEstimationSet2Disable(cfg)) {
    int i;
    for (i = 0; i < 4; ++i) {
      line_addr = fgets(line, 201, file_ptr);
      if (line_addr != line) {
        fprintf(stderr,
                "ERROR(LoadComplexityEstimationFile): "
                "error while skipping 'texture_complexity_estimation_set_2' data in file '%s'\n",
                file_name);
        exit(1);
      }      
      if (atoi(line) != 0)
        fprintf(stderr,
                "WARNING(LoadComplexityEstimationFile): "
                "ignoring one 'texture_complexity_estimation_set_2' enabled flag in file '%s'\n",
                file_name);
    }

  } else { /* !texture_complexity_estimation_set_2_disable */

    /* Read dct_coefs */
    ReadDCECSFileEntry("dct_coefs", &read1, &read2, file_ptr, file_name);
    PutVolConfigDCTCoefs(read1, cfg);
    PutVolConfigDCECSDCTCoefs(read2, cfg);

    /* Read dct_lines */
    ReadDCECSFileEntry("dct_lines", &read1, &read2, file_ptr, file_name);
    PutVolConfigDCTLines(read1, cfg);
    PutVolConfigDCECSDCTLines(read2, cfg);

    /* Read vlc_symbols */
    ReadDCECSFileEntry("vlc_symbols", &read1, &read2, file_ptr, file_name);
    PutVolConfigVLCSymbols(read1, cfg);
    PutVolConfigDCECSVLCSymbols(read2, cfg);

    /* Read vlc_bits */
    ReadDCECSFileEntry("vlc_bits", &read1, &read2, file_ptr, file_name);
    PutVolConfigVLCBits(read1, cfg);
    PutVolConfigDCECSVLCBits(read2, cfg);

    if (GetVolConfigDCTCoefs(cfg) == 0 &&
        GetVolConfigDCTLines(cfg) == 0 &&
        GetVolConfigVLCSymbols(cfg) == 0 &&
        GetVolConfigVLCBits(cfg) == 0) {
      fprintf(stderr,
              "WARNING(LoadComplexityEstimationFile): "
              "enabling texture_complexity_estimation_set_2_disable "
              "because no correponding flag selected.\n");
      PutVolConfigTextureComplexityEstimationSet2Disable(1, cfg);
    }
  }

  /*
   * Read motion_compensation_complexity_disable
   */
  line_addr = fgets(line, 201, file_ptr);
  if (line_addr != line) {
    fprintf(stderr,
            "ERROR(LoadComplexityEstimationFile): "
            "error while reading 'motion_compensation_complexity_disable' in file '%s'\n",
            file_name);
    exit(1);      
  }
  PutVolConfigMotionCompensationComplexityDisable(atoi(line), cfg);

  if (GetVolConfigMotionCompensationComplexityDisable(cfg)) {
    int i;
    for (i = 0; i < 6; ++i) {
      line_addr = fgets(line, 201, file_ptr);
      if (line_addr != line) {
        fprintf(stderr,
                "ERROR(LoadComplexityEstimationFile): "
                "error while skipping 'motion_compensation_complexity' data in file '%s'\n",
                file_name);
        exit(1);
      }      
      if (atoi(line) != 0)
        fprintf(stderr,
                "WARNING(LoadComplexityEstimationFile): "
                "ignoring one of 'motion_compensation_complexity' enabled flag in file '%s'\n",
                file_name);
    }

  } else { /* !motion_compensation_complexity_disable */

    /* Read apm */
    ReadDCECSFileEntry("apm", &read1, &read2, file_ptr, file_name);
    PutVolConfigAPM(read1, cfg);
    PutVolConfigDCECSAPM(read2, cfg);

    /* Read npm */
    ReadDCECSFileEntry("npm", &read1, &read2, file_ptr, file_name);
    PutVolConfigNPM(read1, cfg);
    PutVolConfigDCECSNPM(read2, cfg);

    /* Read interpolate_mc_q */
    ReadDCECSFileEntry("interpolate_mc_q", &read1, &read2, file_ptr, file_name);
    PutVolConfigInterpolateMCQ(read1, cfg);
    PutVolConfigDCECSInterpolateMCQ(read2, cfg);

    /* Read forw_back_mc_q */
    ReadDCECSFileEntry("forw_back_mc_q", &read1, &read2, file_ptr, file_name);
    PutVolConfigForwBackMCQ(read1, cfg);
    PutVolConfigDCECSForwBackMCQ(read2, cfg);

    /* Read halfpel2 */
    ReadDCECSFileEntry("halfpel2", &read1, &read2, file_ptr, file_name);
    PutVolConfigHalfpel2(read1, cfg);
    PutVolConfigDCECSHalfpel2(read2, cfg);

    /* Read halfpel4 */
    ReadDCECSFileEntry("halfpel4", &read1, &read2, file_ptr, file_name);
    PutVolConfigHalfpel4(read1, cfg);
    PutVolConfigDCECSHalfpel4(read2, cfg);

    if (GetVolConfigAPM(cfg) == 0 &&
        GetVolConfigNPM(cfg) == 0 &&
        GetVolConfigInterpolateMCQ(cfg) == 0 &&
        GetVolConfigForwBackMCQ(cfg) == 0 &&
        GetVolConfigHalfpel2(cfg) == 0 &&
        GetVolConfigHalfpel4(cfg) == 0) {
      fprintf(stderr,
              "WARNING(LoadComplexityEstimationFile): "
              "enabling motion_compensation_complexity_disable"
              " because no correponding flag selected.\n");
      PutVolConfigMotionCompensationComplexityDisable(1, cfg);
    }
  }

	/* START: Complexity Estimation syntax support - Update version 2 - Massimo Ravasi (EPFL) - 26 Oct 1999 */
  /*
   * Read version2_complexity_estimation_disable
   */
  if (GetVolConfigEstimationMethod(cfg) == 1) {
		line_addr = fgets(line, 201, file_ptr);
		if (line_addr != line) {
			fprintf(stderr,
							"ERROR(LoadComplexityEstimationFile): "
							"error while reading 'version2_complexity_estimation_disable' in file '%s'\n",
							file_name);
			exit(1);      
		}
		PutVolConfigVersion2ComplexityEstimationDisable(atoi(line), cfg);
		
		if (GetVolConfigVersion2ComplexityEstimationDisable(cfg)) {
			int i;
			for (i = 0; i < 2; ++i) {
				line_addr = fgets(line, 201, file_ptr);
				if (line_addr != line) {
					fprintf(stderr,
									"ERROR(LoadComplexityEstimationFile): "
									"error while skipping 'version2_complexity_estimation' data in file '%s'\n",
									file_name);
					exit(1);
				}      
				if (atoi(line) != 0)
					fprintf(stderr,
									"WARNING(LoadComplexityEstimationFile): "
									"ignoring one of 'version2_complexity_estimation' enabled flag in file '%s'\n",
									file_name);
			}
			
		} else { /* !version2_complexity_estimation_disable */
			
			/* Read sadct */
			ReadDCECSFileEntry("sadct", &read1, &read2, file_ptr, file_name);
			PutVolConfigSadct(read1, cfg);
			PutVolConfigDCECSSadct(read2, cfg);
			
			/* Read quarterpel */
			ReadDCECSFileEntry("quarterpel", &read1, &read2, file_ptr, file_name);
			PutVolConfigQuarterpel(read1, cfg);
			PutVolConfigDCECSQuarterpel(read2, cfg);
			
			if (GetVolConfigSadct(cfg) == 0 &&
					GetVolConfigQuarterpel(cfg) == 0) {
				fprintf(stderr,
								"WARNING(LoadComplexityEstimationFile): "
								"enabling version2_complexity_estimation_disable"
								" because no correponding flag selected.\n");
				PutVolConfigMotionCompensationComplexityDisable(1, cfg);
			}
		}
  } else { /* GetVolConfigEstimationMethod(cfg) != 1 */
		int i;
		line_addr = fgets(line, 201, file_ptr);
		if (line_addr != line) {
			fprintf(stderr,
							"ERROR(LoadComplexityEstimationFile): "
							"error while skipping 'version2_complexity_estimation_disable' flag in file '%s'\n",
							file_name);
			exit(1);
		}      
		if (atoi(line) == 0)
			fprintf(stderr,
							"WARNING(LoadComplexityEstimationFile): "
							"ignoring the 'version2_complexity_estimation_disable' disabled flag in file '%s'\n",
							file_name);
    for (i = 0; i < 2; ++i) {
      line_addr = fgets(line, 201, file_ptr);
      if (line_addr != line) {
        fprintf(stderr,
                "ERROR(LoadComplexityEstimationFile): "
                "error while skipping 'version2_complexity_estimation' data in file '%s'\n",
                file_name);
        exit(1);
      }      
      if (atoi(line) != 0)
        fprintf(stderr,
                "WARNING(LoadComplexityEstimationFile): "
                "ignoring one of 'version2_complexity_estimation' enabled flag in file '%s'\n",
                file_name);
    }

	}
  /* END: Complexity Estimation syntax support - Update Version 2 */
}


/***********************************************************CommentBegin******
 *
 * -- ReadDCECSFileEntry -- Read a line entry in the complexity data file.
 *
 * Author :
 *      Marc Mongenet, Swiss Federal Institute of Technology, Lausanne
 *
 * Created :
 *      09-06-98
 *
 * Purpose :
 *      This function read a line in the complexity data file and exit with an
 *      error message if something goes wrong. It also gives warnings for
 *      unexpected estimation data.
 *
 * Arguments in :
 *      C_Char * entry_name  - name of the complexity data (for error message)
 *      C_Char * file_name   - file_name pointer
 *
 * Arguments in/out :
 *      Int * r1             - pointer to first read value
 *      Int * r2             - pointer to second read value
 *      FILE * file          - file pointer
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *      A file line has been read.
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/

Void ReadDCECSFileEntry (C_Char * entry_name,
                         Int * r1,
                         Int * r2,
                         FILE * file,
                         C_Char * file_name)
{
  Char line [202];
  Char * line_addr = fgets(line, 201, file);

  if (line_addr != line || sscanf(line, "%d %d", r1, r2) != 2) {
    fprintf(stderr,
            "ERROR(LoadComplexityEstimationFile): "
            "error while reading '%s' entry in file '%s'\n",
            entry_name,
            file_name);
    exit(1);
  }

  if (*r1 && *r2 == 0) {
    fprintf(stderr,
            "WARNING(ReadDCECSFileEntry): %s == %d may not be encoded\n",
            entry_name,
            *r2);
  }
}




/***********************************************************CommentBegin******
 *
 * -- Utilities for manipulation VTCconfig: wavelet texture layer structure
 *
 * Author :             
 *      Hung-Ju Lee  (Sarnoff Corporation)
 *
 * Created :            
 *      02-20-98
 *
 * Purpose :            
 *
 * Arguments in :       
 *
 * Arguments in/out :   
 *      none
 *
 * Arguments out :      
 *      none
 *
 * Return values :      
 *      none
 *
 * Side effects :       
 *      none
 *
 * Description :        
 *      
 *
 * See also :
 *      
 *
 * Modified :           
 *
 *      
 *
 ***********************************************************CommentEnd********/

/****************************************************************************************/
/* The following functions are added by Hung-Ju Lee (Sarnoff) for Visual Texture Coding */
/****************************************************************************************/
/* VTC Begin ------------------------------------------ */

Int
GetEncodeControlNumVTCs(EncodeControl *enc_ctl)
{
        return(enc_ctl->num_vtcs);
}

Char *
GetEncodeControlVTCCfgFile(Int id,EncodeControl *enc_ctl)
{
        return(enc_ctl->vtc_cfg_files[id]);
}

Void
PutEncodeControlNumVTCs(Int num_vtcs,EncodeControl *enc_ctl)  /* VTC */
{
        enc_ctl->num_vtcs = num_vtcs;
}

Void
PutEncodeControlVTCCfgFile(Char *string,Int id,EncodeControl *enc_ctl)  /* VTC */
{
        strcpy(enc_ctl->vtc_cfg_files[id],string);
}

VTCConfig *
SallocVTCConfig()
{
        VTCConfig *vtc_cfg;

        vtc_cfg = (VTCConfig *)malloc(sizeof(VTCConfig));
        PutVTCConfigNext(NULL,vtc_cfg);

        return(vtc_cfg);
}

Void
PutVTCConfigNext(VTCConfig *next,VTCConfig *vtc_cfg)
{
        vtc_cfg->pnext = next;
}

VTCConfig *
ReadVTCConfigFiles(EncodeControl *enc_ctl)
{
        Int                             num_vtcs,
                                                vo;

        VTCConfig       *list = NULL,
                                                *curr;

        Char                    *string;

        num_vtcs = GetEncodeControlNumVTCs(enc_ctl);

        for(vo=0; vo<num_vtcs; vo++)
        {
                string = GetEncodeControlVTCCfgFile(vo,enc_ctl);

                curr = ReadVTCConfigFileData(string);

                if(vo == 0)
                        list = curr;
                else
                        AddVTCConfigToList(curr,list);
        }

        return(list);
}

Void
AddVTCConfigToList(VTCConfig *vtc_cfg, VTCConfig *list)
{
        VTCConfig       *curr,*prev;

        prev = list;
        curr = GetVTCConfigNext(prev);

        while (curr != NULL)
        {
                prev = curr;
                curr = GetVTCConfigNext(prev);
        }
        
        PutVTCConfigNext(vtc_cfg,prev);
}

VTCConfig *
GetVTCConfigNext(VTCConfig *cfg)
{
        return(cfg->pnext);
}

Void
FreeVTCConfig(VTCConfig *vtc_config)
{
        VTCConfig       *curr, *next;

        curr = vtc_config;

        while(curr != NULL)
          {
            next = GetVTCConfigNext(curr);

            free(curr);
            curr = next;
          }
        
        return;
}

VTCConfig *
ReadVTCConfigFileData(Char *string)
{
  Char           garbage[201];
  File           *fpfile;
  VTCConfig      *curr;
  Int            spa_lev, snr_lev, col;
  Int            SNR_scalability_levels;

  if((fpfile = fopen(string,"r")) == NULL)
  {
    fprintf(stderr,"ERROR(ReadVOConfigFile): Can't open %s\n",string);
    exit(1);
  }
                
      /* Data to be read from config files */
      curr = SallocVTCConfig();  

/* begin: added by Sharp (99/11/16) */
      /* VTC version_id */
      fscanf(fpfile,"%d",&curr->version_id);
      fgets(garbage,201,fpfile);

			if ( curr->version_id != 1 && curr->version_id != 2 ){
				fprintf(stderr, "Unsupported .cfg file. Version ID should be put at the top of .cfg.\n");
				exit(1);
			}
/* end: added by Sharp (99/11/16) */

      /* VTC id */
      fscanf(fpfile,"%d",&curr->id);
      fgets(garbage,201,fpfile);
     
      /* bitstream file name */
      fscanf(fpfile,"%s",curr->bitstream);
      fgets(garbage,201,fpfile);
            
      /* horizontal size */
      fscanf(fpfile,"%d",&curr->width);
      fgets(garbage,201,fpfile);

      /* horizontal size */
      fscanf(fpfile,"%d",&curr->height);
      fgets(garbage,201,fpfile);

 /*begin: added by Sharp (99/2/16)*/
       /* tiling */
       fscanf(fpfile, "%d", &curr->tiling_disable);
       fgets(garbage, 201, fpfile);
       /* jump_table */
       fscanf(fpfile, "%d", &curr->tiling_jump_table_enable);
       fgets(garbage, 201, fpfile);
 
       /* tile width */
       fscanf(fpfile, "%d", &curr->tile_width);
       fgets(garbage, 201, fpfile);
 
       /* tile height */
       fscanf(fpfile, "%d", &curr->tile_height);
       fgets(garbage, 201, fpfile);
 
 /*end: added by Sharp (99/2/16)*/

      /* Type of alpha channel */
      fscanf(fpfile,"%d",&curr->sa_dwt);
      fgets(garbage,201,fpfile);

      /* alphaTH */
      fscanf(fpfile,"%d",&curr->alphaTH);
      fgets(garbage,201,fpfile);

      /* change_CR_disable */
      fscanf(fpfile,"%d",&curr->change_CR_disable);
      fgets(garbage,201,fpfile);

 /* SAIT_V2 begin : added by Samsung AIT (99/02/23) */
       /* STO_shape_scalable */
 /* FPDAM begin: deleted by SAIT (99/09/03) */
       /* fscanf(fpfile,"%d",&curr->STO_shape_scalable);
       fgets(garbage,201,fpfile);*/
 /* FPDAM end: deleted by SAIT (99/09/03) */
 /* SAIT_V2 end */

      /* Source VTC file names */
      fscanf(fpfile,"%s",curr->image_file);
      fgets(garbage,201,fpfile);
            
      fscanf(fpfile,"%s",curr->shape_file);
      fgets(garbage,201,fpfile);
      
      /* Coded VTC file names */
      fscanf(fpfile,"%s",curr->rec_file);
      fgets(garbage,201,fpfile);
            
      /* read filter type */
      fscanf(fpfile,"%d",&curr->wvtType);
      fgets(garbage,201,fpfile);
  
      /* read wavelet decomposition levels */
      fscanf(fpfile,"%d",&curr->wvtDecompLev);
      fgets(garbage,201,fpfile);
  
  /*** SL: added to handle non-uniform downloadable filters : BEGIN ****/
  /* read wavelet downloadable or default */
  fscanf(fpfile,"%d",&curr->wvtDownload);
  fgets(garbage,80,fpfile);

  /* read wavelet uniform or not */
  fscanf(fpfile,"%d",&curr->wvtUniform);
  fgets(garbage,80,fpfile);
  /* default wavelet filters are uniform */
  if(!curr->wvtDownload) curr->wvtUniform=1;
  if(curr->wvtUniform)
    curr->wvtFilters = (Int *)malloc(sizeof(Int));
  else
    curr->wvtFilters = (Int *)malloc(sizeof(Int)*curr->wvtDecompLev);
  
  if(curr->wvtFilters==NULL) 
    errorHandler("error allocating memory:  curr->wvtFilters\n");
  /* read in filter numbers if applicable */
  /* for int type filter: 0, 2, 4 */
  /* for float type: 1, 3, 5-10 */
  
  if(!curr->wvtDownload) {
    if(curr->wvtType==0)
       curr->wvtFilters[0]=0;
    else
       curr->wvtFilters[0]=1;
  }
  else  if(curr->wvtUniform) {
    fscanf(fpfile,"%d", curr->wvtFilters);
    /* check to see if the filters match its type */
    if(curr->wvtFilters[0] !=0 
       && curr->wvtFilters[0] !=2
       && curr->wvtFilters[0] !=4) {
      if(curr->wvtType==0)
        errorHandler("Filter %d is not integer filter.\n", 
                     curr->wvtFilters[0]);
    }
    else {
      if(curr->wvtType!=0)
        errorHandler("Filter %d is not float filter.\n", 
                     curr->wvtFilters[0]);
    }
  }
  else {
    for(spa_lev=0; spa_lev<curr->wvtDecompLev; spa_lev++) {
      fscanf(fpfile,"%d", curr->wvtFilters+spa_lev);
      /* check to see if the filters match its type */
      if(curr->wvtFilters[spa_lev] !=0 
         && curr->wvtFilters[spa_lev] !=2
         && curr->wvtFilters[spa_lev] !=4) {
        if(curr->wvtType==0)
          errorHandler("Filter %d is not integer filter.\n", 
                       curr->wvtFilters[spa_lev]);
      }
      else {
        if(curr->wvtType!=0)
          errorHandler("Filter %d is not float filter.\n", 
                       curr->wvtFilters[spa_lev]);
      }
    }
  }
  fgets(garbage,80,fpfile); 
  /*** SL: added to handle non-uniform downloadable filters : END****/
  
      /* read quantization type: 0-Single Q 1- Multiple Q  2- Bi-level Q */
      fscanf(fpfile,"%d",&curr->quantization_type);
      fgets(garbage,201,fpfile);

      /* scan order: 0=tree-depth, 1=band by band   */
      fscanf(fpfile,"%d",&curr->scan_direction);
      fgets(garbage,201,fpfile);
  
      /* read  SNR length enable*/
      fscanf(fpfile,"%d",&curr->SNR_start_code_enable);
      fgets(garbage,201,fpfile);
  
  
      fscanf(fpfile,"%d",&curr->target_bitrate);
      fgets(garbage,201,fpfile);
      
      fscanf(fpfile,"%d",&curr->target_spatial_levels);
      fgets(garbage,201,fpfile);
      
      fscanf(fpfile,"%d",&curr->target_snr_levels);
      fgets(garbage,201,fpfile);
      
/* SAIT_V2 begin : added by Samsung AIT (99/02/23) */
      fscanf(fpfile,"%d %d", &curr->target_shape_levels, &curr->full_size_out);
      fgets(garbage,80,fpfile);
/* SAIT_V2 end */

/*begin: added by Sharp (99/2/16)*/
      fscanf(fpfile, "%d %d", &curr->target_tile_id_from, &curr->target_tile_id_to);
      fgets(garbage, 201, fpfile);
/*end: added by Sharp (99/2/16)*/
		         
      /* Read in deringing parameters. If either is zero deringing is disabled */
      fscanf(fpfile,"%d %d",
             &curr->deringWindowSize,&curr->deringThreshold);
      fgets(garbage,201,fpfile);
    
 /* begin: added by Rockwell (99/3/3) */
 
       /* IM: VTC error res */
       /* add error resilience flags in config file */
 
       /* 1-error res ON, 0-error res OFF          */
       fscanf(fpfile,"%d",&curr->error_res_flag);
       fgets(garbage,201,fpfile);
       /* add error res packet size in config file */
       fscanf(fpfile,"%d",&curr->packet_size);
       fgets(garbage,201,fpfile);
       if(curr->packet_size>BUFFER_SIZE <<3)
 	errorHandler("Allowable maximum packet size is %d bits.",
 		     BUFFER_SIZE <<3);
 
       /* add error res segment size in config file */
       fscanf(fpfile,"%d",&curr->segment_thresh);
       fgets(garbage,201,fpfile);
 
 /* end: added by Rockwell (99/3/3) */

      /* Read in DC Lum Quant */
      fscanf(fpfile,"%d",&curr->QdcY);
      fgets(garbage,201,fpfile);

      fscanf(fpfile,"%d",&curr->QdcUV);
      fgets(garbage,201,fpfile);


      /* only support 4:2:0 color format 8-bits per pixel */
      curr->colors    = 3;
      curr->bit_depth = 8;


  /* Read in the spatial scalability levels */
  fscanf(fpfile,"%d",&curr->spatial_scalability_levels);
  
  /* add for mq spatial layer flexability * - ph 7/16 */
  if (curr->quantization_type == 2)
  {
      if (curr->spatial_scalability_levels != curr->wvtDecompLev)
      {
        /* read in usedefault */
        fscanf(fpfile,"%d",&curr->defaultSpatialScale);
        if (curr->defaultSpatialScale==0)
        {
          /* read in spatial layer info */
          for (spa_lev=0; spa_lev<curr->spatial_scalability_levels-1; 
               ++spa_lev)   
            fscanf(fpfile,"%d",&curr->lastWvtDecompInSpaLayer[spa_lev]);
        }
      }
  }
  fgets(garbage,80,fpfile);

  /* special code to take care of SQ band by band, b.b. Chai, 4/27/98 */
  if(curr->quantization_type == 1 && curr->scan_direction == 1){
    curr->spatial_scalability_levels=curr->wvtDecompLev;
    noteWarning("SQ, band-by-band mode has %d spatial layers, ignore this" \
                " field.",curr->wvtDecompLev);

    for (col=0; col<curr->colors; col++) {
      if ((curr->SPlayer[col] = (SNR_PARAM *)malloc(sizeof(SNR_PARAM)*
                               curr->wvtDecompLev )) == NULL)
        errorHandler("error allocating memory \n");
    }

    fscanf(fpfile,"%d",&SNR_scalability_levels);
    for(spa_lev=0;spa_lev<curr->wvtDecompLev; spa_lev++) {
      for (col=0; col<curr->colors; col++){
        curr->SPlayer[col][spa_lev].SNR_scalability_levels =1;
        if ((curr->SPlayer[col][spa_lev].Quant = 
             (Int *)malloc(sizeof(Int)))==NULL)
          errorHandler("error allocating memory \n");
      }
    }
    fscanf(fpfile,"%d",&curr->SPlayer[0][0].Quant[0]);
    if (curr->colors != MONO) { 
      fscanf(fpfile,"%d",&curr->SPlayer[1][0].Quant[0]);
      curr->SPlayer[2][0].Quant[0] =
        curr->SPlayer[1][0].Quant[0];
    }
  }
  else{/* other cases */
    for (col=0; col<curr->colors; col++) {
      if ((curr->SPlayer[col] = (SNR_PARAM *)malloc(sizeof(SNR_PARAM)*
                            curr->spatial_scalability_levels )) == NULL)
        errorHandler("error allocating memory \n");
    }

    for (spa_lev=0; spa_lev<curr->spatial_scalability_levels; spa_lev++) {
      fscanf(fpfile,"%d",&SNR_scalability_levels);
      
      for (col=0; col<curr->colors; col++) {
        curr->SPlayer[col][spa_lev].SNR_scalability_levels = 
          SNR_scalability_levels;
        if ((curr->SPlayer[col][spa_lev].Quant = 
             (Int *)malloc(sizeof(Int)*SNR_scalability_levels))==NULL)
          errorHandler("error allocating memory \n");
      }
      
      for (snr_lev=0; snr_lev<SNR_scalability_levels; snr_lev++) {
        fscanf(fpfile,"%d",&curr->SPlayer[0][spa_lev].Quant[snr_lev]);
        if (curr->colors != MONO) { 
          fscanf(fpfile,"%d",&curr->SPlayer[1][spa_lev].Quant[snr_lev]);
          curr->SPlayer[2][spa_lev].Quant[snr_lev] =
            curr->SPlayer[1][spa_lev].Quant[snr_lev];
        }
      }
      
    fgets(garbage,80,fpfile);
    }
  }


#if 0

      fscanf(fpfile,"%d",&curr->spatial_scalability_levels);
      fgets(garbage,201,fpfile);
      for (col=0; col<curr->colors; col++) {
        if ((curr->SPlayer[col] = (SNR_PARAM *)malloc(sizeof(SNR_PARAM)*
                                  curr->spatial_scalability_levels )) == NULL)
          fprintf(stderr,"error allocating memory \n");
      }

      for (spa_lev=0; spa_lev<curr->spatial_scalability_levels; spa_lev++) {
        fscanf(fpfile,"%d",&SNR_scalability_levels);

        for (col=0; col<curr->colors; col++) {
          curr->SPlayer[col][spa_lev].SNR_scalability_levels = SNR_scalability_levels;
          if ((curr->SPlayer[col][spa_lev].Quant = 
               (int *)malloc(sizeof(int)*SNR_scalability_levels))==NULL)
            fprintf(stderr,"error allocating memory \n");
        }
        
        for (snr_lev=0; snr_lev<SNR_scalability_levels; snr_lev++) {
          fscanf(fpfile,"%d",&curr->SPlayer[0][spa_lev].Quant[snr_lev]);
          fscanf(fpfile,"%d",&curr->SPlayer[1][spa_lev].Quant[snr_lev]);
          curr->SPlayer[2][spa_lev].Quant[snr_lev] =
            curr->SPlayer[1][spa_lev].Quant[snr_lev];
        }
        
        fgets(garbage,201,fpfile);
      }
#endif



      /* check parms */
      if (curr->spatial_scalability_levels > curr->wvtDecompLev)
        {
          fprintf(stderr, "\n\n Parameter Error: \n");
          exit(-1);
        }      

   fclose(fpfile);

  return curr;
}


/* VTC End ------------------------------------------ */


/*******************************************************/
/* MAC V2 - new functions - Sven Brandau, HHI 10.11.99 */
Void PutVolConfigShapeExtension( Int vol_shape_extension, VolConfig *cfg )
{
  cfg->shape_extension = vol_shape_extension;
}

Int GetVolConfigShapeExtension( VolConfig *cfg )
{
  return cfg->shape_extension;
}

Void PutVolConfigAuxCompCount( Int vol_aux_comp_count, VolConfig *cfg )
{
  cfg->aux_comp_count = vol_aux_comp_count;
}

Int GetVolConfigAuxCompCount( VolConfig *cfg )
{
  return cfg->aux_comp_count;
}

Void PutVolConfigG( char* disk_g_file, Int aux_comp, VolConfig *cfg )
{
  strcpy( cfg->disk_g_file[aux_comp], disk_g_file );
}

Void PutVolConfigCodedG( char* coded_g_file, Int aux_comp, VolConfig *cfg )
{
  strcpy( cfg->coded_g_file[aux_comp], coded_g_file );
}


/* MAC V2 - end ---------------------------------------*/
