/******************************************************************************
 *                                                                          
 * This software module was originally developed by 
 *
 *   Oki Electric Industry Co., Ltd. (contact: Shigeru Fukunaga)
 *
 * in the course of development of the MPEG-4 Video (ISO/IEC 14496-2) standard.
 * This software module is an implementation of a part of one or more MPEG-4 
 * Video (ISO/IEC 14496-2) tools as specified by the MPEG-4 Video (ISO/IEC 
 * 14496-2) standard. 
 *
 * ISO/IEC gives users of the MPEG-4 Video (ISO/IEC 14496-2) standard free 
 * license to this software module or modifications thereof for use in hardware
 * or software products claiming conformance to the MPEG-4 Video (ISO/IEC 
 * 14496-2) standard. 
 *
 * Those intending to use this software module in hardware or software products
 * are advised that its use may infringe existing patents. The original 
 * developer of this software module and his/her company, the subsequent 
 * editors and their companies, and ISO/IEC have no liability for use of this 
 * software module or modifications thereof. No license to this software
 * module is granted for non MPEG-4 Video (ISO/IEC 14496-2) standard 
 * conforming products.
 *
 * Oki Electric Industry Co., Ltd. retains full right to use the software 
 * module for his/her own purpose, assign or donate the software module to a 
 * third party and to inhibit third parties from using the code for non MPEG-4 
 * Video (ISO/IEC 14496-2) standard conforming products. This copyright notice 
 * must be included in all copies or derivative works of the software module. 
 *
 * Copyright (c) 1999  Oki Electric Industry Co., Ltd.  All rights reserved.
 *
 *****************************************************************************/

/***********************************************************HeaderBegin*******
 *
 * File:        newpred.c
 *
 * Author:      Oki - Shigeru Fukunaga
 * Created:     16-AUG-1999
 *
 * Description: modules for NEWPRED mode (encoder)
 *
 * Notes:
 *
 * Modified:    
 *
 ***********************************************************HeaderEnd*********/


/************************    INCLUDE FILES    ********************************/

#include "momusys.h"
#include "vm_config.h"
#include "newpred.h"
#include "mom_bitstream_i.h"

/***********************************************************CommentBegin******
 *
 * -- InitNewpred --
 *
 * Author :
 *      Oki - Shigeru Fukunaga
 *
 * Created :
 *      16-AUG-1999
 *
 * Purpose :
 *	NEWPRED - Initialize NEWPRED mode
 *
 * Arguments in :
 *	VOConfig *vo_config_list - config structure of all VOs
 *	RefVop ***ref_vop - reference VOPs
 *	Int num_vos - number of VOs
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Void
InitNewpred(VOConfig	*vo_config_list,	/* config structure of all VOs */
	    RefVop 	****ref_vop,		/* reference memory for NEWPRED */
	    Int		num_vos)		/* number of VOs */
{
  int i, j;
  VOConfig *curr;		/* config structure of current VO */
  VolConfig *cfg;		/* config structure of base VOL */
  RefVop ***tmp_ref_vop;	/* reference memory */

  /* memory allocation for reference pictures */
  tmp_ref_vop = (RefVop ***)calloc(num_vos, sizeof(RefVop **));
  for(i = 0; i < num_vos; i++){
    tmp_ref_vop[i] = (RefVop **)calloc(MAXREF, sizeof(RefVop *));
    for(j = 0; j < MAXREF; j++){
      tmp_ref_vop[i][j] = (RefVop *)calloc(1, sizeof(RefVop));
    }
  }

  /* set config structure of current VO */
  curr = vo_config_list;

  for(i = 0; i < num_vos; i++){
    /* set config structure of base VOL of current VO */
    /* error resilience and NEWPRED is available only in base layer */
    cfg = GetVOConfigLayers(curr);

    /* read NEWPRED control file */
    ReadNewpredCtlFile(cfg, GetVolConfigNewpredCtlFile(cfg));

    /* memory allocation for reference pictures */
    for(j = 0; j < MAXREF; j++){
      tmp_ref_vop[i][j]->vop_id = (Int *)calloc(cfg->num_segments, sizeof(Int));
      tmp_ref_vop[i][j]->vop_id_pred = (Int *)calloc(cfg->num_segments, sizeof(Int));
      tmp_ref_vop[i][j]->vop = AllocVop(GetVolConfigDiskSeqX(cfg) + 32,
                                        GetVolConfigDiskSeqY(cfg) + 32, 0 );
    }

    /* set next config structure of VO */
    curr = GetVOConfigNext(curr);
  }

  *ref_vop = tmp_ref_vop;
}
/* end of InitNewpred() */


/***********************************************************CommentBegin******
 *
 * -- ReadNewpredCtlFile --
 *
 * Author :
 *      Oki - Shigeru Fukunaga
 *
 * Created :
 *      16-AUG-1999
 *
 * Purpose :
 *	NEWPRED - Read control file for NEWPRED
 *
 * Arguments in :
 *	Char filename - filename of control file
 *
 * Arguments in/out :
 *	VolConfig *cfg - VOL confguration list
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *	In the forward bitstream exchange, vop_id_for_prediction of each 
 *      NEWPRED segment	is decided in advance. And the values are descrived 
 *      in control file.
 *	In the real system, the extra mechanisms for receiving the backward 
 *	messages from the decoder and deciding vop_id_for_prediction, 
 *	that is non normative, are needed.
 *	(These mechanisms, which are described in VM document, are not implemented
 *	in this reference software, because these are out scope of standard.)
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Void
ReadNewpredCtlFile(VolConfig *cfg, Char *filename)
{
  Int	i, j;
  Int	vop_id, np_seg_id, vop_id_pred;
  File	*fp_np;
  Char	garbage[201];

  if((fp_np = fopen(filename, "r")) == NULL){
    fprintf(stderr,"ERROR(ReadNewpredCtlFile): unable to open %s\n", filename);
    exit(1);
  }

  /* number of NEWPRED segments */
  fscanf(fp_np, "%d", &(cfg->num_segments));
  fgets(garbage,201,fp_np);

  /* first MB address of each NEWPRED segment */
  cfg->mba_segment = (Int *)calloc(cfg->num_segments + 1, sizeof(Int));
  for(j = 0; j <= cfg->num_segments; j++){
    fscanf(fp_np, "%d", &(cfg->mba_segment[j]));
    fgets(garbage,201,fp_np);
  }

  if(GetVolConfigNewpredSegmentType(cfg) || GetVolConfigErrorResDisable(cfg)){
    if(cfg->num_segments != 1){
      printf("ERROR(NewpredCtlFile): The number of NEWPRED segments should be 1 in the following cases.\n");
      printf("  1. NEWPRED segment type is VOP,\n");
      printf("  2. resync_marker_disable is 1.\n");
      exit(1);
    }
  }

#ifdef _NP_DEBUG_
#if 0
  fprintf(stderr,"%d : %d %d %d %d\n",
	  cfg->num_segments, 
	  cfg->mba_segment[0], cfg->mba_segment[1], cfg->mba_segment[2], cfg->mba_segment[3]);
  sleep(2);
#endif
#endif

  /* number of VOPs */
  fscanf(fp_np, "%d", &(cfg->num_vops));
  fgets(garbage,201,fp_np);
  
  /* vop_id_for_prediction for each NEWPRED segment */
  cfg->vop_id_pred = (Int **)calloc(cfg->num_vops, sizeof(Int *));
  for(i = 0; i < cfg->num_vops; i++)
    cfg->vop_id_pred[i] = (Int *)calloc(cfg->num_segments, sizeof(Int));
  while(1){
    fscanf(fp_np, "%d %d %d", &vop_id, &np_seg_id, &vop_id_pred);
    fgets(garbage,201,fp_np);
    if(vop_id == 0)
      break;
    cfg->vop_id_pred[vop_id - 1][np_seg_id] = vop_id_pred;
  }
  
#ifdef _NP_DEBUG_
#if 0
  for(i = 0; i < cfg->num_vops; i++)
    fprintf(stderr,"[%d] %d %d %d\n", 
	    i, cfg->vop_id_pred[i][0], cfg->vop_id_pred[i][1], cfg->vop_id_pred[i][2]);
#endif
#endif

  fclose(fp_np);
}
/* end of ReadNewpredCtlFile() */


/***********************************************************CommentBegin******
 *
 * -- SetRefVopNewpred --
 *
 * Author :
 *      Oki - Shigeru Fukunaga
 *
 * Created :
 *      16-AUG-1999
 *
 * Purpose :
 *	NEWPRED - set of reference VOP
 *
 * Arguments in :
 *	Vop *prev_vop - previous original VOP
 *	Vop *prev_rec_vop - previous reconstructed VOP
 *	RefVop ***ref_vop - reference VOPs
 *	VolConfig *cfg - config structure of VOL
 *	Int vo_id - VOL ID
 *	Int vop_id - VOP ID
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *	The reference NEWPRED segment is replaced with the old one in the memories, 
 *	only when vop_id_for_prediction does not indicate the immediately 
 *	preceeding VOP.
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Void
SetRefVopNewpred(Vop *prev_vop, Vop *prev_rec_vop, RefVop ***ref_vop,
		 VolConfig *cfg, Int vo_id, Int vop_id)
{
  Int i, flag;
  Int mem_id;		/* reference memory id */

#if 0
  if(vop_id == 1)
    return;		/* reference VOP is not set in first frame (I-VOP) */
#endif

  for(i = 0; i < cfg->num_segments; i++){
    if(cfg->vop_id_pred[vop_id-1][i] != vop_id - 1){
      flag = 0;
      for(mem_id = 0; mem_id < MAXREF; mem_id++){
	if(ref_vop[vo_id][mem_id]->vop_id[i] == cfg->vop_id_pred[vop_id-1][i]){
	  if(ref_vop[vo_id][mem_id]->vop != NULL){
	    /* update reconstructed refecence VOP */
	    CopyNPSeg(ref_vop[vo_id][mem_id]->vop,	/* from */
		      prev_vop,			        /* to */
		      cfg->mba_segment[i],	        /* begining MBA */
		      cfg->mba_segment[i+1],	        /* ending MBA */
		      16);			        /* padding size */
	    
	    /* update original refecence VOP */
	    /* As all original VOPs except immediately preceeding VOP have already removed,
	       reconsutructed VOP are used instead of original one. */
	    CopyNPSeg(ref_vop[vo_id][mem_id]->vop,	/* from */
		      prev_rec_vop,		        /* to */
		      cfg->mba_segment[i],		/* begining MBA */
		      cfg->mba_segment[i+1],	        /* ending MBA */
		      16);				/* padding size */

	    flag = 1;
	  }
	  else{
	    fprintf(stderr,"NEWPERD : reference memory error (NULL reference)\n");
	    exit(1);
	  }
	}
	if(flag)
	  break;
      }
      if(flag == 0){
	fprintf(stderr,"NEWPERD : reference memory error (memory overflow)\n");
	exit(1);
      }
    }
  }
}
/* end of SetRefVopNewpred() */


/***********************************************************CommentBegin******
 *
 * -- StoreRefVopNewpred --
 *
 * Author :
 *      Oki - Shigeru Fukunaga
 *
 * Created :
 *      16-AUG-1999
 *
 * Purpose :
 *	NEWPRED - store reference VOP
 *
 * Arguments in :
 *	Vop *curr_rec_vop - current reconstructed VOP
 *	RefVop ***ref_vop - reference VOPs
 *	VolConfig *cfg - config structure of VOL
 *	Int vo_id - VOL ID
 *	Int vop_id - VOP ID
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Void
StoreRefVopNewpred(Vop *curr_rec_vop, RefVop ***ref_vop, 
		   VolConfig *cfg, Int vo_id, Int vop_id)
{
  Int i;
  Int mem_id;	/* reference memory ID */
    
  for(i = 0; i < cfg->num_segments; i++){
    /* detect the oldest VOP memory for each NEWPRED segment */
    mem_id = DetectOldVop(ref_vop, vo_id, i);
  
    /* add current VOP to the reference memory */
    ref_vop[vo_id][mem_id]->vop_id[i] = vop_id;
    ref_vop[vo_id][mem_id]->vop_id_pred[i] = cfg->vop_id_pred[vop_id-1][i];
    CopyNPSeg(curr_rec_vop,			/* from */
	      ref_vop[vo_id][mem_id]->vop, 	/* to */
	      cfg->mba_segment[i],		/* begining MBA */
	      cfg->mba_segment[i+1],		/* ending MBA */
	      16);				/* padding size */

#ifdef _NP_DEBUG_
#if 1
    fprintf(stdout,"[%d %d] %3d(%3d) %3d(%3d) %3d(%3d) %3d(%3d) %3d(%3d)\n", 
	    vop_id, i,
	    ref_vop[vo_id][0]->vop_id[i], ref_vop[vo_id][0]->vop_id_pred[i],
	    ref_vop[vo_id][1]->vop_id[i], ref_vop[vo_id][1]->vop_id_pred[i],
	    ref_vop[vo_id][2]->vop_id[i], ref_vop[vo_id][2]->vop_id_pred[i],
	    ref_vop[vo_id][3]->vop_id[i], ref_vop[vo_id][3]->vop_id_pred[i],
	    ref_vop[vo_id][4]->vop_id[i], ref_vop[vo_id][4]->vop_id_pred[i]);
#endif 
#endif
  }
}
/* end of StoreRefVopNewpred() */


/***********************************************************CommentBegin******
 *
 * -- DetectOldVop --
 *
 * Author :
 *      Oki - Shigeru Fukunaga
 *
 * Created :
 *      16-AUG-1999
 *
 * Purpose :
 *	NEWPRED - detect oldest reference VOP
 *
 * Arguments in :
 *	RefVop **ref_vop - reference VOPs
 *	Int np_seg_id - NEWPRED segment ID
 *
 * Arguments in/out :
 *
 * Arguments out :
 *	Int oldest_mem_id - reference memory ID
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Int
DetectOldVop(RefVop ***ref_vop, Int vo_id, Int np_seg_id)
{
  int mem_id;
  Int oldest_vop_id = 1000000;
  Int oldest_mem_id = -1;

  for(mem_id = 0; mem_id < MAXREF; mem_id++){
    /* in case of vacant memory */
    if(ref_vop[vo_id][mem_id]->vop_id[np_seg_id] == 0)
      return(mem_id);
    /* renewal memory ID */
    if(ref_vop[vo_id][mem_id]->vop_id[np_seg_id] < oldest_vop_id){
      oldest_vop_id = ref_vop[vo_id][mem_id]->vop_id[np_seg_id];
      oldest_mem_id = mem_id;
    }
  }

  return(oldest_mem_id);
}
/* end of DetectOldVop() */


/***********************************************************CommentBegin******
 *
 * -- BitstreamPutNEWPRED -- 
 *
 * Author :		
 *	0ki - Shigeru Fukunaga
 *
 * Created :		
 *	16-AUG-1999
 *
 * Purpose :		
 *	NEWPRED - insert NEWPRED syntax
 * 
 * Arguments in : 	
 *      VolConfig * vol_config 	- pointer to the VOL configuration data
 *	Vop *curr              	- pointer to the current VOP
 *	Int np_vop_id		- VOP ID
 *	Int np_seg_id		- NEWPRED segment ID
 *	Int flag		- Intra/Inter flag
 *
 * Arguments in/out :	
 *	Image *bitstream  	- intermediate bitstream for syntax
 *
 * Arguments out :	
 *
 * Return values :	
 *
 * Side effects :	
 *
 * Description :	
 *
 * See also :
 *
 * Modified :	
 *	
 ***********************************************************CommentEnd********/
Void
BitstreamPutNEWPRED(VolConfig *vol_config, Vop *curr, Int np_vop_id, 
		    Int np_seg_id, Image *bitstream, Int flag)
{
  Int bits;
  Int tmp_vop_id = np_vop_id;
  Int tmp_vop_id_pred;
  Int vop_id_modulo;

  if(GetVopNewpredSegmentType(curr)){
    /* In case NEWPRED segment type is VOP, segment ID is always 0 */
    np_seg_id = 0;
  }

  bits = ceil(log((double)GetVopTimeIncrementResolution(curr))/log(2.0));
  if (bits<1) bits = 1;
  bits += 3;
  bits = (bits > 15) ? 15 : bits;

  vop_id_modulo = pow(2, bits);
  while(tmp_vop_id > vop_id_modulo - 1)
    tmp_vop_id -= (vop_id_modulo - 1);

  /* vop_id (4-15 bits) */
  BitstreamPutBits(bitstream, tmp_vop_id, bits);

  /* vop_id_for_prediction */
  if(flag){
    /* Only when reference frame is not immediately preceeding frame,
       vop_id_for_prediction field exists. */
    if(vol_config->vop_id_pred[np_vop_id-1][np_seg_id] == np_vop_id - 1)
      /* vop_id_for_prediction_indication (1 bit) */
      BitstreamPutBits(bitstream, 0, 1);
    else{
      /* vop_id_for_prediction_indication (1 bit) */
      BitstreamPutBits(bitstream, 1, 1);
      /* vop_id_for_prediction (4-15 bits) */
      tmp_vop_id_pred = vol_config->vop_id_pred[np_vop_id-1][np_seg_id];
      while(tmp_vop_id_pred > vop_id_modulo - 1)
	tmp_vop_id_pred -= (vop_id_modulo - 1);
      BitstreamPutBits(bitstream, tmp_vop_id_pred, bits);
#ifdef _NP_DEBUG_
#if 1
      fprintf(stdout, "NEWPRED: [%d](%d) - refered %d\n", 
	      tmp_vop_id, np_seg_id, tmp_vop_id_pred);
#endif
#endif
    }
  }
  else
    /* vop_id_for_prediction_indication (1 bit) */
    BitstreamPutBits(bitstream, 0, 1);

  BitstreamPutBits(bitstream,1,1);	/* marker bit */
}
/* end of BitstreamPutNEWPRED() */
