/******************************************************************************
 *                                                                          
 * This software module was originally developed by 
 *
 *   Fujitsu Laboratories Ltd. (contact: Eishi Morimatsu)
 *
 * in the course of development of the MPEG-4 Video (ISO/IEC 14496-2) standard.
 * This software module is an implementation of a part of one or more MPEG-4 
 * Video (ISO/IEC 14496-2) tools as specified by the MPEG-4 Video (ISO/IEC 
 * 14496-2) standard. 
 *
 * ISO/IEC gives users of the MPEG-4 Video (ISO/IEC 14496-2) standard free 
 * license to this software module or modifications thereof for use in hardware
 * or software products claiming conformance to the MPEG-4 Video (ISO/IEC 
 * 14496-2) standard. 
 *
 * Those intending to use this software module in hardware or software products
 * are advised that its use may infringe existing patents. THE ORIGINAL 
 * DEVELOPER OF THIS SOFTWARE MODULE AND HIS/HER COMPANY, THE SUBSEQUENT 
 * EDITORS AND THEIR COMPANIES, AND ISO/IEC HAVE NO LIABILITY FOR USE OF THIS 
 * SOFTWARE MODULE OR MODIFICATIONS THEREOF. No license to this software
 * module is granted for non MPEG-4 Video (ISO/IEC 14496-2) standard 
 * conforming products.
 *
 * Fujitsu Laboratories Ltd. retains full right to use the software module for 
 * his/her own purpose, assign or donate the software module to a third party 
 * and to inhibit third parties from using the code for non MPEG-4 Video 
 * (ISO/IEC 14496-2) standard conforming products. This copyright notice must 
 * be included in all copies or derivative works of the software module. 
 *
 * Copyright (c) 1999.  
 *
 *****************************************************************************/

/***********************************************************HeaderBegin*******
 *
 * File:        drc_util_enc.c
 *
 * Author:      Fujitsu Laboratories - Eishi Morimatsu
 * Created:     06-SEP-1999
 *
 * Description: modules for DRC
 *
 * Notes:
 *
 * Modified:    
 *	14.12.99 Eishi Morimatsu (Fujitsu Labs.): bug fix for DRC
 *
 ***********************************************************HeaderEnd*********/


/************************    INCLUDE FILES    ********************************/
#include <stdlib.h>
#include <stdio.h>
#include "momusys.h"
#include "vm_common_defs.h"
#include "vm_enc_defs.h"
#include "drc_enc_defs.h"
#include "text_bits.h"
#include "vm_config.h"
#include "drc_util_enc.h"
 
#include "text_bits.h"
#include "vm_config.h"
#include "text_code.h"
#include "drc_util_common.h"

#define Min(x,y) (((x)<(y))?(x):(y))

/***********************************************************CommentBegin******
 *
 * -- DownSamplingTexture --
 *
 * Author :
 *      Fujitsu Laboratories - Eishi Morimatsu
 *
 * Created :
 *      06-SEP-1999
 *
 * Purpose :
 *	DRC - downsampling of normal resolution VOP's texturet data
 *
 * Arguments in :
 *	Vop *normal_vop - normal resolution VOP(input)
 *	Vop *lowres_vop - reduced resolution VOP(output)
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Void 
DownSamplingTexture(Vop *normal_vop, Vop *lowres_vop)
{
  Image *Y_low, *U_low, *V_low, *Y_norm, *U_norm, *V_norm;

  Y_low  = (Image *)GetVopY(lowres_vop);
  U_low  = (Image *)GetVopU(lowres_vop);
  V_low  = (Image *)GetVopV(lowres_vop);

  Y_norm = (Image *)GetVopY(normal_vop);
  U_norm = (Image *)GetVopU(normal_vop);
  V_norm = (Image *)GetVopV(normal_vop);

  DownSamplingImage(Y_norm, Y_low);
  DownSamplingImage(U_norm, U_low);
  DownSamplingImage(V_norm, V_low);
}


/***********************************************************CommentBegin******
 *
 * -- DownSamplingImage --
 *
 * Author :
 *      Fujitsu Laboratories - Eishi Morimatsu
 *
 * Created :
 *      06-SEP-1999
 *
 * Purpose :
 *	DRC - downsampling of normal rsolution VOP's image data
 *
 * Arguments in :
 *	Image *normal_image - image data of normal resolution VOP(input)
 *	Image *lowres_image - image data of reduced resolution VOP(output)
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Void 
DownSamplingImage(Image *normal_image, Image *lowres_image)
{
  Int width = GetImageSizeX(normal_image);
  Int height = GetImageSizeY(normal_image);
  SInt *srcpic, *destpic;
  SInt *tmpptr;
  Int i, j, val;

  srcpic = (SInt *)GetImageData(normal_image);
  destpic = (SInt *)GetImageData(lowres_image);

  tmpptr = destpic;

  for(j = 0; j < height / 2; j++){
    for(i = 0; i < width / 2; i++){
      val  = srcpic[i * 2 + j * 2 * width];
      val += srcpic[i * 2 + 1 + j * 2 * width];
      val += srcpic[i * 2 + (j * 2 + 1)* width];
      val += srcpic[i * 2 + 1 + (j * 2 + 1)* width];
      *tmpptr = ((val + 2) / 4);
      tmpptr ++;
    }
  }
}

/***********************************************************CommentBegin******
 *
 * -- MotionVectorScalingDown --
 *
 * Author :
 *      Fujitsu Laboratories - Eishi Morimatsu
 *
 * Created :
 *      06-SEP-1999
 *
 * Purpose :
 *	DRC - scaling down of motion vector data
 *
 * Arguments in :
 *	Image *mot_x - horizontal motion vector data to be scaled down
 *	Image *mot_y - vertical motion vector data to be scaled down
 *	Image *mot_x_lowres - scaled down horizontal down motion vector data
 *	Image *mot_y_lowres - scaled down vertical motion vector data
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Void 
MotionVectorScalingDown(Image *mot_x, Image *mot_y, 
	Image *mot_x_lowres, Image *mot_y_lowres)
{
  Int sizex = GetImageSizeX(mot_x);
  Int sizey = GetImageSizeY(mot_y);
  float *Vec, *LowVec;
  Int i,j, cnt;
  float vec, lowvec;

  Vec = (float *)GetImageData(mot_x);
  LowVec = (float *)GetImageData(mot_x_lowres);

  for(i = 0, cnt = 0; i < sizey; i++){
    for(j = 0; j < sizex; j++, cnt++){
      vec = Vec[cnt];
      if(vec == 0.0){
	  lowvec = 0.0;
      }
      else if(vec > 0.0){
	lowvec = (vec + 0.5) / 2.0;
      }  
      else{
	lowvec = (vec - 0.5) / 2.0;
      }  
      LowVec[cnt] = lowvec;

    }
  }
    
  Vec = (float *)GetImageData(mot_y);
  LowVec = (float *)GetImageData(mot_y_lowres);

  for(i = 0, cnt = 0; i < sizey; i++){
    for(j = 0; j < sizex; j++, cnt++){
      vec = Vec[cnt];
      if(vec == 0.0){
	  lowvec = 0.0;
      }
      else if(vec > 0.0){
	lowvec = (vec + 0.5) / 2.0;
      }  
      else{
	lowvec = (vec - 0.5) / 2.0;
      }  
      LowVec[cnt] = lowvec;
    }
  }
}  

/***********************************************************CommentBegin******
 *
 * -- MaskDctCoeff --
 *
 * Author :
 *      Fujitsu Laboratories - Eishi Morimatsu
 *
 * Created :
 *      06-SEP-1999
 *
 * Purpose :
 *	DRC - masking of DCT coefficient data
 *
 * Arguments in :
 *	Int *coeff - DCT coefficient data
 *	Int dct_coeff_mask - masking pattern number (0 - 8)
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Void 
MaskDctCoeff(Int *coeff, Int dct_coeff_mask)
{
    int		x, y;
   
    for(y = 0; y < 8; y++){
        if ( y < dct_coeff_mask ) {
	    for(x = 0; x < 8; x++){
                if ( x >= dct_coeff_mask ) coeff[x + 8 * y] = 0;
	    }
	}
        else {
            for(x = 0; x < 8; x++){
                coeff[x + 8 * y] = 0;
	    }
	}
    }
}

/***********************************************************CommentBegin******
 *
 * -- DecideReducedResolution --
 *
 * Author :
 *      Fujitsu Laboratories - Eishi Morimatsu
 *
 * Created :
 *      06-SEP-1999
 *
 * Purpose :
 *	DRC - deciding the resolution of the VOP to be encoded
 *
 * Arguments in :
 *      Vop *prev_vop - previous encoded vop
 *      Vop *curr_vop - current vop to be encoded
 *      Int bitrate - target bitrate
 *      Int switching_mode - resolution switching control mode(0,1 or negative value)
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Int f_count_drc = 0;
Void
DecideReducedResolution (Vop *prev_vop,
                         Vop *curr_vop,
			 Int bitrate,
			 Int switching_mode )
{

    Float switch_para;
    int   interval, sw_flag;

     PutVopSwitchingFlag(0,curr_vop);

    if(switching_mode == 1) {  /* normal switching control */

      if(prev_vop==NULL) { /* first I-VOP */
	  PutVopReducedResolution(0,curr_vop);
          PutVopDctCoefMask(8,curr_vop);
      } else {

#ifdef _DRC_DEBUG_
          printf("DRC DBG: prev_rr=%d, prev_coef=%d\n", GetVopReducedResolution(prev_vop), GetVopDctCoefMask(prev_vop));
          printf("DRC DBG: prev_qp=%3.1f, prev_bits=%d\n", GetVopAverageQp(prev_vop),GetVopBitsUsed(prev_vop));
#endif

          PutVopDctCoefMask(Min( 8, (GetVopDctCoefMask(prev_vop) + 1)),curr_vop);
          PutVopReducedResolution(GetVopReducedResolution(prev_vop),curr_vop);
          switch_para = GetVopAverageQp(prev_vop) * GetVopBitsUsed(prev_vop);

#ifdef _DRC_DEBUG_
          printf("Switching check in Normal Res  : switch_para=%7.2f, th=%7.2f\n", switch_para,(float)bitrate /NORMAL_RES_FR*NORMAL_RES_QP);
          printf("Switching check in Reduced Res : switch_para=%7.2f, th=%7.2f\n", switch_para,(float)bitrate /REDUCED_RES_FR*REDUCED_RES_QP);
#endif

          if (GetVopReducedResolution(prev_vop) == 0 &&
              (int)((switch_para - (float)bitrate / NORMAL_RES_FR * NORMAL_RES_QP) > 0)) { /* switch from Normal Res to Reduced Res */
              PutVopDctCoefMask(8,curr_vop);
              PutVopReducedResolution(1,curr_vop);
              PutVopSwitchingFlag(1,curr_vop);
/*              printf("  --> Switched from NORMAL to REDUCED\n"); */
          } else if (GetVopReducedResolution(prev_vop) == 1 &&
                   (int)((switch_para - (float)bitrate / REDUCED_RES_FR * REDUCED_RES_QP) < 0)) { /* switch from Reduced Res to Normal Res */
              PutVopDctCoefMask(4,curr_vop);
              PutVopReducedResolution(0,curr_vop);
              PutVopSwitchingFlag(2,curr_vop);
/*              printf("  --> Switched from REDUCED to NORMAL\n"); */
          }
      }	

    } else if (switching_mode == 0) {  /* No switching(keep normal resolution) */
      if(prev_vop==NULL) { /* first I-VOP */
        PutVopReducedResolution(0,curr_vop);
        PutVopDctCoefMask(8,curr_vop);
      } else { /* other VOPs */
	  PutVopReducedResolution(0,curr_vop);
          PutVopDctCoefMask(8,curr_vop);
      }

    } else if (switching_mode < 0) {  /* Switching on every nth encoded vop */
      if(prev_vop==NULL) { /* first I-VOP */
        PutVopReducedResolution(0,curr_vop);
        PutVopDctCoefMask(8,curr_vop);
      } else { /* other VOPs */
        interval = -switching_mode;
        sw_flag = f_count_drc % interval;
        PutVopDctCoefMask(Min( 8, (GetVopDctCoefMask(prev_vop) + 1)),curr_vop);
        PutVopReducedResolution(GetVopReducedResolution(prev_vop),curr_vop);
	if(sw_flag == 0) {
	  if(GetVopReducedResolution(prev_vop)) {
	    PutVopReducedResolution(0,curr_vop);
            PutVopDctCoefMask(4,curr_vop);
            PutVopSwitchingFlag(2,curr_vop);
/*              printf("  --> Switched from REDUCED to NORMAL\n"); */
          } else { 
	    PutVopReducedResolution(1,curr_vop);
            PutVopDctCoefMask(8,curr_vop);
            PutVopSwitchingFlag(1,curr_vop);
/*              printf("  --> Switched from NORMAL to REDUCED\n"); */
          }
        }
      }
   }
    
#ifdef _DRC_DEBUG_
   printf("DRC DBG: f_count = %d, curr_rr= %d, switch_flag= %d, curr_coef= %d\n", f_count_drc, GetVopReducedResolution(curr_vop), GetVopSwitchingFlag(curr_vop), GetVopDctCoefMask(curr_vop));
#endif

    f_count_drc++;

    return;

}

/***********************************************************CommentBegin******
 *
 * -- VopCodeShapeTextIntraCom_ReducedRes --
 *
 * Author :
 *      Fujitsu Laboratories - Eishi Morimatsu
 *
 * Created :
 *      06-SEP-1999
 *
 * Purpose :
 *	DRC - encoding module of reduced resolution I-VOP(non-error resilient)
 *
 * Arguments in :
 *      Vop *curr - 
 *      Image **First_stream -
 *      Image **Shape_stream -
 *      Vop *rec_curr, Image *mottext_bitstream -
 *      Bits *bits -
 *      Int rc_type - 
 *      Int vo_id - 
 *	VolConfig *vol_config -
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *	14.12.99 Eishi Morimatsu (Fujitsu Labs.): bug fix for DRC
 *
 ***********************************************************CommentEnd********/
Void 
VopCodeShapeTextIntraCom_ReducedRes(Vop *curr, 
			      Image **First_stream,
			      Image **Shape_stream,
			      Vop *rec_curr, Image *mottext_bitstream,
			      Bits *bits,
			      Int rc_type, 
			      Int vo_id, 
			      VolConfig *vol_config)
{
    Vop *reduced_res_curr=NULL, *reduced_res_rec_curr=NULL;
    Int num_pixels = GetImageSizeX(GetVopY(curr));
    Int num_lines = GetImageSizeY(GetVopY(curr));
    Int *cod_flag;
    Int *vp_num;  /* DRC 1999.12.14 */
    Int i;

	reduced_res_curr = AllocVop(num_pixels/2, num_lines/2, 0);
	CopyVopNonImageField(curr,reduced_res_curr); 
	reduced_res_rec_curr = AllocVop(num_pixels/2, num_lines/2, 0);
	CopyVopNonImageField(rec_curr, reduced_res_rec_curr); 
        cod_flag = (Int*)malloc(num_pixels/32*num_lines/32*sizeof(Int)); 
        vp_num = (Int*)malloc(num_pixels/32*num_lines/32*sizeof(Int));  /* DRC 1999.12.14 */

	SetConstantImage(GetVopA(reduced_res_curr),255);
	SetConstantImage(GetVopAuv(reduced_res_curr),255);
	SetConstantImage(GetVopA(reduced_res_rec_curr),255);
	SetConstantImage(GetVopAuv(reduced_res_rec_curr),255);

	DownSamplingTexture(curr, reduced_res_curr);

        VopCodeShapeTextIntraCom(reduced_res_curr,
				   First_stream,
				   Shape_stream,
				   reduced_res_rec_curr,
				   mottext_bitstream,
				   bits,
				   rc_type,
				   vo_id,
				   vol_config);

        UpSamplingTexture(reduced_res_rec_curr, rec_curr); 
        CopyQPStore(reduced_res_rec_curr, rec_curr); 

        PutVopAverageQp(GetVopAverageQp(reduced_res_rec_curr),curr);

        /*** set all MB's cod flag to coded(=1) ***/
        for(i=0;i<(num_pixels/32*num_lines/32);i++)
        {  /* DRC 1999.12.14 */
          cod_flag[i] = 1;
          vp_num[i] = 1;  /* DRC 1999.12.14 */
        }  /* DRC 1999.12.14 */

        BlockBoundaryFilter(rec_curr, cod_flag, vp_num, 1); /* DRC 1999.12.14 */

        FreeVop(reduced_res_rec_curr);
        FreeVop(reduced_res_curr);
        free((Char*)cod_flag);  /* DRC 1999.12.14 */
        free((Char*)vp_num);  /* DRC 1999.12.14 */
}
