/*****************************************************************************
 "This software module was originally developed by:
 
	Dae-Sung Cho (Samsung AIT)
 
  and edited by

	Dae-Sung Cho (Samsung AIT),
	Se Hoon Son (Samsung AIT)

        in the course of development of the <MPEG-4 Video(ISO/IEC 14496-2)>. This
  software module is an implementation of a part of one or more <MPEG-4 Video
  (ISO/IEC 14496-2)> tools as specified by the <MPEG-4 Video(ISO/IEC 14496-2)
  >. ISO/IEC gives users of the <MPEG-4 Video(ISO/IEC 14496-2)> free license
  to this software module or modifications thereof for use in hardware or
  software products claiming conformance to the <MPEG-4 Video(ISO/IEC 14496-2
  )>. Those intending to use this software module in hardware or software
  products are advised that its use may infringe existing patents. The
  original developer of this software module and his/her company, the
  subsequent editors and their companies, and ISO/IEC have no liability for
  use of this software module or modifications thereof in an implementation.
  Copyright is not released for non <MPEG-4 Video(ISO/IEC 14496-2)>
  conforming products. Samsung AIT (SAIT) retains full right to use the code
  for his/her own purpose, assign or donate the code to a third party and to
  inhibit third parties from using the code for non <MPEG-4 Video(ISO/IEC
  14496-2)> conforming products. This copyright notice must be included in
  all copies or derivative works. Copyright (c) 1997, 1998".
 *****************************************************************************/


#include <stdio.h>
#include <math.h>
#include "momusys.h"
#include "mom_structs.h"
#include "mom_image.h"
#include "mom_vop.h"
#include "mot_util.h"
#include "alp_common_cae.h"
#include "alp_common_si.h"
#include "alp_code_cae.h"
#include "alp_code_si.h"
#include "alp_code_header.h"
#include "alp_common_util.h"
#include "bin_ar_code.h"
#include "alp_common_def.h"
#include "si_intra.h"
#include "shape_def.h"

/***********************************************************HeaderBegin*******
 *
 * File:        alp_code_si.c
 *
 * Author:      Samsung AIT
 * Created:     10-13-97
 *
 * Description: Contains functions used to implement scan interleaving
 *                               coding of spatial scalable binary alpha blocks.
 *
 * Copyright (C) 1997 Samsung AIT All Rights Reserved.                 
 *
 ***********************************************************HeaderEnd*********/

/***********************************************************CommentBegin******
 *
 * -- SI_MB -- Encodes a binary alpha block using SI.
 *
 * Author :		
 *	Dae-Sung Cho	(Samsung AIT)
 *
 * Created :		
 *	11-09-98
 * 
 * Arguments: 	
 *		alpha_smb 		: alpha macroblock to be coded (subsampled)
 *    alpha_mb_comp : prediction for alpha macroblock
 *		x,y						: address of macroblock in pixels
 *		ox,oy				: the differential spatial reference between
 *										current VOP and previous VOP
 *		rec_alpha 		: current reconstructed alpha map
 *		prev_alpha 		: previous reconstructed alpha map
 *    motA_x,motA_y : the motion field for shape
 *		shape_inter 	: flag to tell encoder whether inter coding 
 *										is applicable (always = 1, for P/B-VOPs)
 *		amvbits 			: number of bits used for coding the shape MV  of the MB
 *		send_cr				: =0, CR is assumed equal to 1, 
 *									: =1, CR is transmitted in the bitstream.
 * 		mb_type 			: shape coding mode for the macroblock
 *		shape_stream 	: bitstream produced
 *
 * Return values :	
 *
 * Side effects :	
 *	-
 *
 * Description :	A binary alpha block will be coded using cr,st, and bac.
 *								These bits are added to the <shape_stream>. Decisions
 *								on transposition and INTRA/INTER are made within.
 *
 * See also :
 *
 ***********************************************************CommentEnd********/
Int SI_MB(Vop *ref_vop,			
	  Image *bordered_smb_data,
	  Image *bottom_border_data,
	  Image *right_border_data,
	  UChar *alpha_mb_comp,
	  UChar *alpha_mb_spos_h,
	  UChar *alpha_mb_spos_v,
	  Int x, 
	  Int y, 
	  Int ox, 
	  Int oy,				
	  Image *rec_alpha, 
	  Image *prev_alpha, 
	  Image *motA_x, 
	  Image *motA_y,
	  Int   shape_inter, 
	  Int amvbits, 
	  Int send_cr, 
	  Int *mb_type,
	  UChar *shape_stream, 
	  Vop *rec_curr, 
	  Int error_res_disable)	
{									
	Image   *bordered_smb_v=NULL,
		*alpha_smb_mc=NULL,
		*alpha_smb_mc_v=NULL;

	Image   *pimg;

  	UChar
		alpha_smb_comp[MB_SIZE*MB_SIZE];
  
  	UChar cae_stream[1024], si_stream[1024];
  	UChar h_stream[1024], v_stream[1024];

  	Int
    		ii,jj,
    		mb_s=MB_SIZE,
		h_bits,v_bits,cae_bits,si_bits,
    		offset,
		scan_type,
    		cae_mb_type,si_mb_type,
    		k,
    		best_bits,
		mb_x,mb_y,
		mvx,mvy;

  	Double v_factor;
  	Int v_factor_int;
  	Double h_factor;
  	Int h_factor_int;

	Int hor_res_scan, ver_res_scan;
	Int hor_loop_num, ver_loop_num;
	

/*OBSSFIX_MODE3-start-(MODE1?)-*/
  	Int shape_ver_sampling_factor_m = GetVopShapeVerSampM(rec_curr);
  	Int shape_ver_sampling_factor_n = GetVopShapeVerSampN(rec_curr);
  	Int shape_hor_sampling_factor_m = GetVopShapeHorSampM(rec_curr);
  	Int shape_hor_sampling_factor_n = GetVopShapeHorSampN(rec_curr);
/*  	Int shape_ver_sampling_factor_m = GetVopShapeVerSampM(ref_vop);*/
/*  	Int shape_ver_sampling_factor_n = GetVopShapeVerSampN(ref_vop);*/
/*  	Int shape_hor_sampling_factor_m = GetVopShapeHorSampM(ref_vop);*/
/*  	Int shape_hor_sampling_factor_n = GetVopShapeHorSampN(ref_vop);	*/
/*OBSSFIX_MODE3-end-(MODE1?)-*/
	
	SInt *p;

#ifdef	_DEBUG_OBSS_
	fprintf(stderr,">>>SI_MB()==>\n");
	fprintf(stderr,">>>>> x=%d, y=%d, ox=%d, oy=%d, shape_inter=%d, amvbits=%d, send_cr=%d, mb_type=%d\n", x,y,ox,oy,shape_inter,amvbits,send_cr,*mb_type);
#endif

  	offset = amvbits+1;
  	for (k=0;k<1024;k++) cae_stream[k]=si_stream[k]=v_stream[k]=h_stream[k]=0; 

	if (shape_inter) 
		offset =
			amvbits+1+GetModeCodeLength_Enh(5,x/MB_SIZE,y/MB_SIZE,ref_vop)
				-GetModeCodeLength_Enh(2,x/MB_SIZE,y/MB_SIZE,ref_vop);

	/* TBD: Get the data for the MC bordered block */

	if (shape_inter) {
		mb_x = x/MB_SIZE; mb_y = y/MB_SIZE;
		
		p = (SInt *) GetImageData(motA_x) + mb_y*GetImageSizeX(motA_x) + mb_x;
		mvx = *p;
		p = (SInt *) GetImageData(motA_y) + mb_y*GetImageSizeX(motA_y) + mb_x;
		mvy = *p;
	
		alpha_smb_mc = AllocImage(mb_s+4,mb_s+4,SHORT_TYPE);
		GetBorderedMC(prev_alpha,x+ox,y+oy,mvx,mvy,MB_SIZE/mb_s,alpha_smb_mc); 

		for(ii=0;ii<MB_SIZE*MB_SIZE;ii++) alpha_smb_comp[ii]=alpha_mb_comp[ii];

		p = (SInt *) GetImageData(alpha_smb_mc)+2*GetImageSizeX(alpha_smb_mc)+2;
  		for( ii=0; ii<mb_s; ii++,p+=4) 
			for( jj=0; jj<mb_s; jj++, p++ ) {
    				*p = (SInt) f1bit(alpha_smb_comp[ii*mb_s+jj]);
  			}
	}

  	h_factor = log((Double)shape_hor_sampling_factor_n/shape_hor_sampling_factor_m)/log(2.);
  	v_factor = log((Double)shape_ver_sampling_factor_n/shape_ver_sampling_factor_m)/log(2.);
  	h_factor_int = (Int)floor(h_factor+0.000001);
  	v_factor_int = (Int)floor(v_factor+0.000001);

	hor_res_scan = (h_factor-h_factor_int>0.000001) ? 1 : 0;
	ver_res_scan = (v_factor-v_factor_int>0.000001) ? 1 : 0;
	hor_loop_num = h_factor_int;
	ver_loop_num = v_factor_int;

#ifdef  _DEBUG_OBSS_
        fprintf(stderr,"hor_res_scan=%d\n",hor_res_scan);
        fprintf(stderr,"ver_res_scan=%d\n",ver_res_scan);
        fprintf(stderr,"hor_loop_num=%d\n",hor_loop_num);
        fprintf(stderr,"ver_loop_num=%d\n",ver_loop_num);
#endif


    if( shape_hor_sampling_factor_n==2*shape_hor_sampling_factor_m &&
 		shape_ver_sampling_factor_n==2*shape_ver_sampling_factor_m )
    {
        /* right-bottom bordering from reference VOP */
        PutSubImage(bordered_smb_data,bottom_border_data,0,MB_SIZE+2);
        PutSubImage(bordered_smb_data,right_border_data,MB_SIZE+2,2);

	if (shape_inter) {
		alpha_smb_mc_v = AllocSameImage(alpha_smb_mc);
		TransposeImage(alpha_smb_mc,alpha_smb_mc_v);
	}

	scan_type = DecideScanOrderSI(bordered_smb_data,MB_SIZE);

	if(scan_type)	/* horizontal */
	{
       		si_bits=ShapeCodingSI(bordered_smb_data, 
                                      rec_alpha,
                                      alpha_mb_spos_h,
                                      alpha_mb_spos_v,
                                      hor_loop_num,
                                      ver_loop_num,
                                      hor_res_scan,
                                      ver_res_scan,
                                      si_stream,
                                      error_res_disable);
	} else 	/* vertical */
	{	

		/* TBD: transpose all the data */
		bordered_smb_v = AllocSameImage(bordered_smb_data);
  		TransposeImage(bordered_smb_data,bordered_smb_v);

       		si_bits=ShapeCodingSI(bordered_smb_v, 
					rec_alpha,
						alpha_mb_spos_h,
						alpha_mb_spos_v,
						hor_loop_num,
						ver_loop_num,
						hor_res_scan,
						ver_res_scan,
					si_stream,
					error_res_disable);

		FreeImage(bordered_smb_v);

	}

    } else {
	   if (shape_inter) {
		alpha_smb_mc_v = AllocSameImage(alpha_smb_mc);
		TransposeImage(alpha_smb_mc,alpha_smb_mc_v);
	   }
	   scan_type = 1; 

           /* right-bottom bordering from reference VOP */
           PutSubImage(bordered_smb_data,bottom_border_data,0,MB_SIZE+2);
           PutSubImage(bordered_smb_data,right_border_data,MB_SIZE+2,2);

       	   si_bits=ShapeCodingSI(bordered_smb_data, 
					rec_alpha,
						alpha_mb_spos_h,
						alpha_mb_spos_v,
						hor_loop_num,
						ver_loop_num,
						hor_res_scan,
						ver_res_scan,
					si_stream,
					error_res_disable);
    }

	si_mb_type = *mb_type;

#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>si_scan_type=%d(0-transpose,1-none), si_mb_type=%d\n", scan_type, si_mb_type);
#endif
	
  	if(shape_inter)
	{
		pimg = AllocImage(2,1,SHORT_TYPE);
		GetSubImage(bordered_smb_data,pimg,0,MB_SIZE+1);
		PutSubImage(bordered_smb_data,pimg,0,MB_SIZE+2);
		PutSubImage(bordered_smb_data,pimg,0,MB_SIZE+3);
		FreeImage(pimg);

		bordered_smb_v = AllocSameImage(bordered_smb_data);
  		TransposeImage(bordered_smb_data,bordered_smb_v);

		/* (horizontal) code INTRA and INTER and choose the best one */
      		h_bits = ShapeCodingInterCAE( bordered_smb_data, alpha_smb_mc,
                                    		*mb_type, h_stream, 1 , send_cr,
				    		error_res_disable);

		/* (vertical) code INTRA and INTER and choose the best one */
      		v_bits = ShapeCodingInterCAE( bordered_smb_v, alpha_smb_mc_v, 
						*mb_type, v_stream, 0, send_cr,
						error_res_disable );

      		if( h_bits <= v_bits ) {
			cae_bits = h_bits;
			for( k=0; k<1024; k++ ) cae_stream[k] = h_stream[k];
		} else {
			cae_bits = v_bits;
			for( k=0; k<1024; k++ ) cae_stream[k] = v_stream[k];
		}
		cae_mb_type = 5;

		/* choose the best representation */
		if( (cae_bits+offset) <si_bits)		
		{
			best_bits = cae_bits;
			for( k=0; k<1024; k++ ) shape_stream[k] = cae_stream[k];
			*mb_type = cae_mb_type;
		} else {
			best_bits = si_bits;
			for( k=0; k<1024; k++ ) shape_stream[k] = si_stream[k];
			*mb_type = si_mb_type;
		}

		FreeImage(bordered_smb_v);

	} else {
		best_bits = si_bits;
		for( k=0; k<1024; k++ ) shape_stream[k] = si_stream[k];
		*mb_type = si_mb_type;
	}

#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>mb_type=%d\n", *mb_type);
#endif

	if( best_bits > 1023 ) {
    		fprintf(stderr,"WARNING:best_bits=%d\n",(int)best_bits); 
	}

	if (shape_inter) {	
		FreeImage(alpha_smb_mc);
		FreeImage(alpha_smb_mc_v);
	}
	
  	return best_bits;
}												


/***********************************************************CommentBegin******
 *
 * -- ShapeCodingSI -- Encodes a binary alpha block using SI.
 *
 * Author :		
 *	Dae-Sung Cho (Samsung AIT)
 *
 * Created :		
 *	09-Feb-98
 * 
 * Arguments: 	
 *		alpha_smb 		: alpha macroblock to be coded (subsampled)
 *		rec_alpha 		: current reconstructed alpha map
 *		x,y			: address of macroblock in pixels
 *		bitstream 		: bitstream produced
 *
 * Return values :	
 *
 * Side effects :	
 *	-
 *
 * Description :	A binary alpha block will be coded using bac.
 *			These bits are added to the <shape_stream>. 
 *
 * See also :
 *
 ***********************************************************CommentEnd********/
Int 
ShapeCodingSI(Image *bordered_smb_data,	
	Image *rec_alpha, 
		UChar *alpha_mb_spos_h,
		UChar *alpha_mb_spos_v,
		Int hor_loop_num,
		Int ver_loop_num,
		Int hor_res_scan,
		Int ver_res_scan,
	Char *bitstream,
	Int error_res_disable
	)
{
	Int si_bab_type;

        Int           i,prob,bitnum=0;
        Image         *acstream;

        SInt  *p;

        ArCoder       ar_coder;

        StartArCoder(&ar_coder,error_res_disable);
        acstream = BitstreamInit();

	si_bab_type = DecideBabTypeSI_AR (bordered_smb_data,
				alpha_mb_spos_h,
				alpha_mb_spos_v,
				hor_loop_num,
				ver_loop_num,
				hor_res_scan,
				ver_res_scan);
#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>si_bab_type=%d (0:transitional bab,1:exceptional bab)\n",si_bab_type);
#endif
	/* SI bab type encoding */
	prob=si_babtype_prob[0];
	ArCodeSymbol(si_bab_type,prob,&ar_coder,acstream);	

	if(si_bab_type==0)	/* Transistional BAB */
	{
		/*EncodeTransitionalBab (bordered_smb_data, acstream, &ar_coder);*/
		EncodeTransitionalBab_AR (bordered_smb_data, 
						alpha_mb_spos_h,
						alpha_mb_spos_v,
						hor_loop_num,
						ver_loop_num,
						hor_res_scan,
						ver_res_scan,
					acstream, 
					&ar_coder);
	} else 			/* Exceptional BAB */
	{
		/*EncodeExceptionalBab(bordered_smb_data, acstream, &ar_coder);*/
		EncodeExceptionalBab_AR (bordered_smb_data, 
						alpha_mb_spos_h,
						alpha_mb_spos_v,
						hor_loop_num,
						ver_loop_num,
						hor_res_scan,
						ver_res_scan,
					acstream, 
					&ar_coder);
	}

	StopArCoder(&ar_coder,acstream);

	bitnum=GetImageSizeX(acstream);

        p = (SInt *) GetImageData(acstream); 
	for (i=0;i<bitnum;i++,p++) bitstream[i] = *p;
/*
#ifdef	_DEBUG_OBSS_
fprintf(stderr,"bitstream MB : ");
for (i=0;i<ret_mb;i++) fprintf(stdout,"%d ",bitstream[i]); printf("\n");
#endif
*/
	FreeImage(acstream); 

        return(bitnum);
}

Int DecideBabTypeSI (Image *bordered_smb)
{
	Int i,j,prev,next,curr;
	Int si_bab_type;	
	Int width=GetImageSizeX(bordered_smb),
    	    height=GetImageSizeY(bordered_smb); 
        SInt *smb_data;

	smb_data = (SInt *)GetImageData(bordered_smb) + 2*width + 2;

	/* Vertical scanning */
	si_bab_type = 0;
        for(j=0;j<(width-4);j+=2)
        {
                for(i=1;i<(height-4);i+=2)
                {
                        curr=*(smb_data+i*width+j);
                        prev=*(smb_data+i*width+j-1);
                        next=*(smb_data+i*width+j+1);

			if((prev==next) && (prev!=curr)) 
			{
				si_bab_type = 1;
				break;
			}
		}
		if(si_bab_type==1) break;
	}

	/* Horizontal scanning */
	if(si_bab_type==0)
	{
           for(i=0;i<(height-4);i+=2)
	   {
                for(j=0;j<(width-4);j++) 
		{
                        curr=*(smb_data+i*width+j);
                        prev=*(smb_data+(i-1)*width+j);
                        next=*(smb_data+(i+1)*width+j);

			if((prev==next) && (prev!=curr)) 
			{
				si_bab_type = 1;
				break;
			}
		}
		if(si_bab_type==1) break;
	    }
	}

	return si_bab_type;
	/* 0: transitional bab, 1: exceptional bab */
}

Int DecideBabTypeSI_AR (Image *bordered_smb,
				UChar *alpha_mb_spos_h,
				UChar *alpha_mb_spos_v,
				Int hor_loop_num,
				Int ver_loop_num,
				Int hor_res_scan,
				Int ver_res_scan)
{
	Int i,j,prev,next,curr;
	Int si_bab_type=0;	
	Int width=GetImageSizeX(bordered_smb), /* 20 */
    	    height=GetImageSizeY(bordered_smb);  /* 20 */
	Int start_v=0,start_h=0;
	Int h_scan_freq=1,v_scan_freq=1;	
	Int skip_ver;
	Int tmp;

        SInt *smb_data;

	smb_data = (SInt *)GetImageData(bordered_smb) + 2*width + 2;

	/* Vertical scanning */
	if(hor_res_scan) {
	   i=0;
	   while(alpha_mb_spos_h[i]==1) i++;
	   while(alpha_mb_spos_h[i]==0) i++;
	   tmp = i;

	   start_v = start_h = 0;
	   h_scan_freq = 1 << hor_loop_num;
	   v_scan_freq = 1 << ver_loop_num;

	   if (tmp-(1<<hor_loop_num)>=0)		start_h = tmp - (1<<hor_loop_num);
	   else if (tmp+(1<<hor_loop_num)<MB_SIZE) 	start_h = tmp + (1<<hor_loop_num);
	   else {
		fprintf(stderr,"Error:out of sampling ratio !\n");
		exit(1);
	   }
	
	   i=0;
	   while(alpha_mb_spos_v[i]==0) i++;
	   start_v = i;	
	   	
           for(j=start_h;j<(width-4);j+=h_scan_freq) {
		if ( alpha_mb_spos_h[j] == 0 ) { 
                   for(i=start_v;i<(height-4);i+=v_scan_freq) {
		      if ( alpha_mb_spos_v[i] == 1 ) {
                   	curr=*(smb_data+i*width+j);

		   	if (j-h_scan_freq < -2) prev = *(smb_data+i*width-2);
		   	else	prev = *(smb_data+i*width+j-h_scan_freq);
		   	if (j+h_scan_freq > 17) next = *(smb_data+i*width+17); 
		   	else	next = *(smb_data+i*width+j+h_scan_freq);

		   	if((prev==next) && (prev!=curr)) {
			   si_bab_type = 1;
			   break;
		   	}
		      }
		   }
		} 
		if(si_bab_type==1) break;
	   }
	}

	if(si_bab_type==0) {
	   i=0;
	   while(alpha_mb_spos_h[i]==1) i++;
	   while(alpha_mb_spos_h[i]==0) i++;
	   if(i>start_h && hor_res_scan) tmp=start_h;
	   else	tmp = i;

	   while (hor_loop_num>0) {
	   	start_v = start_h = 0;
		h_scan_freq = 1<<hor_loop_num;
	   	v_scan_freq = 1<<ver_loop_num;

	   	if (tmp-(1<<(hor_loop_num-1))>=0)	tmp = start_h = tmp - (1<<(hor_loop_num-1));
	   	else				start_h = tmp + (1<<(hor_loop_num-1));
		
		i=0;
		while (alpha_mb_spos_v[i]==0)	i++;
		start_v = i;

		for (j=start_h; j<(width-4); j+=h_scan_freq) {
		   for (i=start_v; i<(height-4); i+=v_scan_freq) {
			if (alpha_mb_spos_v[i]==1) {

                   	   curr=*(smb_data+i*width+j);

		   	   if (j-(1<<(hor_loop_num-1))<-2) prev = *(smb_data+i*width-2);
		   	   else	prev = *(smb_data+i*width+j-(1<<(hor_loop_num-1)));
		   	   if (j+(1<<(hor_loop_num-1))>17) next = *(smb_data+i*width+17); 
		   	   else	next = *(smb_data+i*width+j+(1<<(hor_loop_num-1)));
		
		   	   if((prev==next) && (prev!=curr)) {
			      si_bab_type = 1;
			      break;
		   	   }
			} 
		   }
		   if(si_bab_type==1) break;
		}
		if(si_bab_type==1) break;
		hor_loop_num--;
	   }
	}

	/* Horizontal scanning */

	if(si_bab_type==0) {
	   if (ver_res_scan) {
	      i=0;
	      while(alpha_mb_spos_v[i]==1) i++;
	      while(alpha_mb_spos_v[i]==0) i++;
	      tmp = i;

	      start_v = start_h = 0;
	      v_scan_freq = 1 << ver_loop_num;
	      h_scan_freq = 1;
	      skip_ver = v_scan_freq;

	      if (tmp-(1<<ver_loop_num)>=0)		start_v = tmp - (1<<ver_loop_num);
	      else if (tmp+(1<<ver_loop_num)<MB_SIZE) 	start_v = tmp + (1<<ver_loop_num);
	      else {
		fprintf(stderr,"Error:out of sampling ratio !\n");
		exit(1);
	      }
	
              for(i=start_v;i<(height-4);i+=v_scan_freq) {
		if( alpha_mb_spos_v[i]== 0 ) {
                   for(j=start_h;j<(width-4);j+=h_scan_freq) {
		      curr=*(smb_data+i*width+j);
   
		      if (i-v_scan_freq < -2)	prev = *(smb_data-2*width+j);
		      else			prev = *(smb_data+(i-v_scan_freq)*width+j);
		      if (i+v_scan_freq > 17)	next = *(smb_data+17*width+j); 
		      else			next = *(smb_data+(i+v_scan_freq)*width+j);

		      if((prev==next) && (prev!=curr)) {
			si_bab_type = 1;
			break;
		      }
		   }
		}
		if(si_bab_type==1) break;
	      }
	   }
	}

	if(si_bab_type==0) {
	   i=0;
	   while(alpha_mb_spos_v[i]==1) i++;
	   while(alpha_mb_spos_v[i]==0) i++;
	   if(i>start_v && ver_res_scan) tmp=start_v;
	   else	tmp = i;

	   while (ver_loop_num>0) {
	   	start_v = start_h = 0;
	   	v_scan_freq = 1<<ver_loop_num;
		h_scan_freq = 1;

	   	if (tmp-(1<<(ver_loop_num-1))>=0)	tmp = start_v = tmp - (1<<(ver_loop_num-1));
	   	else				start_v = tmp + (1<<(ver_loop_num-1));
		skip_ver = 1<<(ver_loop_num-1);

		for (i=start_v; i<(height-4); i+=v_scan_freq) {
		   for (j=start_h; j<(width-4); j+=h_scan_freq) {

                   	curr=*(smb_data+i*width+j);

		   	if (i-skip_ver<-2) prev = *(smb_data-2*width+j);
		   	else	prev = *(smb_data+(i-skip_ver)*width+j);
		   	if (i+skip_ver>17) next = *(smb_data+17*width+j); 
		   	else	next = *(smb_data+(i+skip_ver)*width+j);

		   	if((prev==next) && (prev!=curr)) {
			   si_bab_type = 1;
			   break;
		   	}
		   }
		   if(si_bab_type==1) break;
		}
		if(si_bab_type==1) break;
		ver_loop_num--;
	   }
	}

	return si_bab_type;
	/* 0: transitional bab, 1: exceptional bab */
}


/*------------------------------------------------------*/
/* Transitional bab data coding.                        */
/*------------------------------------------------------*/
Int EncodeTransitionalBab(Image *bordered_smb,
				Image *acstream, 
				ArCoder *ar_coder)
{
	Int i,j, prev,next,curr;
	Int context,prob;
	Int width=GetImageSizeX(bordered_smb),
    	    height=GetImageSizeY(bordered_smb); 
        SInt *smb_data;

	smb_data = (SInt *)GetImageData(bordered_smb) + 2*width + 2;

/*
#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>EncodeTransitionalBab()==>\n");
fprintf(stderr,">>>>>Vertical scanning()\n");
#endif
*/
	/* Vertical scanning */
        for(j=0;j<(width-4);j+=2) {
                for(i=1;i<(height-4);i+=2) {
                        prev=*(smb_data+i*width+j-1);
                        next=*(smb_data+i*width+j+1);

/*
#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>(%d,%d):prev=%d,next=%d\n",i,j,prev,next);
#endif
*/
			if(prev!=next) {
                        	curr=*(smb_data+i*width+j);
				context = GetContextSI_VS(smb_data, 
						j, 
						i, 
							0,
							width-5,
							0,
							height-5,	
						width); 

                                prob = si_intra_vs_prob[context];
                                ArCodeSymbol(curr,prob,ar_coder,acstream);
/*
#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>curr=%d, context=%d, prob=%d\n",curr,context,prob);
fprintf(stderr,">>>>>ar_coder->L:%d,ar_coder->R:%d\n",ar_coder->L,ar_coder->R);
#endif
*/
			}
		}
	}

/*
#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>Horizontal scanning()\n");
#endif
*/
	/* Horizontal scanning */
        for(i=0;i<(height-4);i+=2) {
                for(j=0;j<(width-4);j++) {
                        prev=*(smb_data+(i-1)*width+j);
                        next=*(smb_data+(i+1)*width+j);

/*
#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>(%d,%d):prev=%d,next=%d\n",i,j,prev,next);
#endif
*/
			if(prev!=next) {
                        	curr=*(smb_data+i*width+j);
				context = GetContextSI_HS(smb_data, 
						j, 
						i, 
							0,
							width-5,
							0,
							height-5,	
						width); 
                                prob = si_intra_hs_prob[context];
                                ArCodeSymbol(curr,prob,ar_coder,acstream);
/*
#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>curr=%d, context=%d, prob=%d\n",curr,context,prob);
fprintf(stderr,">>>>>ar_coder->L:%d,ar_coder->R:%d\n",ar_coder->L,ar_coder->R);
#endif
*/
			}
		}
	}

        return(GetImageSizeX(acstream));
}

/*------------------------------------------------------*/
/* Exceptional bab data coding.                        */
/*------------------------------------------------------*/
Int EncodeExceptionalBab(Image *bordered_smb,
				Image *acstream, 
				ArCoder *ar_coder)
{
	Int i,j,curr;
	Int context,prob;
	Int width=GetImageSizeX(bordered_smb),
    	    height=GetImageSizeY(bordered_smb); 
        SInt *smb_data;

	smb_data = (SInt *)GetImageData(bordered_smb) + 2*width + 2;

/*
#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>EncodeExceptionalBab()==>\n");
fprintf(stderr,">>>>>Vertical scanning()\n");
#endif
*/
	/* Vertical scanning */
        for(j=0;j<(width-4);j+=2) {
                for(i=1;i<(height-4);i+=2) {
                       	curr=*(smb_data+i*width+j);
/*
#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>(%d,%d)\n",i,j);
#endif
*/
			context = GetContextSI_VS(smb_data, 
						j, 
						i, 
							0,
							width-5,
							0,
							height-5,	
						width); 

                       
			prob = si_intra_vs_prob[context];
			ArCodeSymbol(curr,prob,ar_coder,acstream);
/*
#ifdef	_DEBUG_OBSS_
fprintf(stdout,">>>>>curr=%d, context=%d, prob=%d\n",curr,context,prob);
fprintf(stdout,">>>>>ar_coder->L:%d,ar_coder->R:%d\n",ar_coder->L,ar_coder->R);
#endif
*/
		}
	}

/*
#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>Horizontal scanning()\n");
#endif
*/
	/* Horizontal scanning */
        for(i=0;i<(height-4);i+=2) {
                for(j=0;j<(width-4);j++) {
/*
#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>(%d,%d)\n",i,j);
#endif
*/
                       	curr=*(smb_data+i*width+j);
			context = GetContextSI_HS(smb_data, 
						j, 
						i, 
							0,
							width-5,
							0,
							height-5,	
						width); 
			prob = si_intra_hs_prob[context];
			ArCodeSymbol(curr,prob,ar_coder,acstream);
/*
#ifdef	_DEBUG_OBSS_
fprintf(stdout,">>>>>curr=%d, context=%d, prob=%d\n",curr,context,prob);
fprintf(stdout,">>>>>ar_coder->L:%d,ar_coder->R:%d\n",ar_coder->L,ar_coder->R);
#endif
*/
		}
	}

        return(GetImageSizeX(acstream));
}


/*------------------------------------------------------*/
/* Transitional bab data coding.                        */
/*------------------------------------------------------*/
Int EncodeTransitionalBab_AR (Image *bordered_smb,
				UChar *alpha_mb_spos_h,
				UChar *alpha_mb_spos_v,
				Int hor_loop_num,
				Int ver_loop_num,
				Int hor_res_scan,
				Int ver_res_scan,
				Image *acstream, 
				ArCoder *ar_coder)
{
	Int i,j, prev,next,curr;
	Int context,prob;
	Int width=GetImageSizeX(bordered_smb), /* 20 */
    	    height=GetImageSizeY(bordered_smb);  /* 20 */
	Int skip_upper=0,skip_bottom=0,skip_hor=0,skip_ver=0,skip_left=0,skip_right=0;
	Int start_v=0,start_h=0;
	Int h_scan_freq=1,v_scan_freq=1;	
	Int tmp;

        SInt *smb_data;

	smb_data = (SInt *)GetImageData(bordered_smb) + 2*width + 2;

#ifdef  _DEBUG_OBSS_
fprintf(stderr,">>>EncodeTransitionalBab_AR()==>\n");
fprintf(stderr,">>>horizontal sampling position !\n");
for(i=0; i<MB_SIZE; i++) fprintf(stderr,"%d",alpha_mb_spos_h[i]);
fprintf(stderr,"\n>>>vertical sampling position !\n");
for(i=0; i<MB_SIZE; i++) fprintf(stderr,"%d",alpha_mb_spos_v[i]);
fprintf(stderr,"\n");
fprintf(stderr,">>>>>Vertical scanning()\n");
#endif

	/* Vertical scanning */
	if(hor_res_scan) {
#ifdef  _DEBUG_OBSS_
fprintf(stderr,">>>>>residual encoding\n");
#endif
	   i=0;
	   while(alpha_mb_spos_h[i]==1) i++;
	   while(alpha_mb_spos_h[i]==0) i++;
	   tmp = i;

	   start_v = start_h = 0;
	   h_scan_freq = skip_hor = 1 << hor_loop_num;
	   v_scan_freq = 1 << ver_loop_num;

	   if (tmp-(1<<hor_loop_num)>=0)		start_h = tmp - (1<<hor_loop_num);
	   else if (tmp+(1<<hor_loop_num)<MB_SIZE) 	start_h = tmp + (1<<hor_loop_num);
	   else {
		fprintf(stderr,"Error:out of sampling ratio !\n");
		exit(1);
	   }
	
	   i=0;
	   while(alpha_mb_spos_v[i]==0) i++;
	   start_v = i;	
	   	
           for(j=start_h;j<(width-4);j+=h_scan_freq) {
		if ( alpha_mb_spos_h[j] == 0 ) { 
		   skip_upper = start_v+1;
                   for(i=start_v;i<(height-4);i+=v_scan_freq) {
		      if ( alpha_mb_spos_v[i] == 1 ) {
			if (i+v_scan_freq > 15) skip_bottom=MB_SIZE+1-i;
			else {
			   if ( alpha_mb_spos_v[i+v_scan_freq] == 0 ) {
				if (i+v_scan_freq*2 > 15) skip_bottom=MB_SIZE+1-i;
				else  skip_bottom=v_scan_freq*2;
			   } else  skip_bottom = v_scan_freq;
			}

		   	if (j-h_scan_freq < -2) prev = *(smb_data+i*width-2);
		   	else	prev = *(smb_data+i*width+j-h_scan_freq);
		   	if (j+h_scan_freq > 17) next = *(smb_data+i*width+17); 
		   	else	next = *(smb_data+i*width+j+h_scan_freq);

		   	if (j-skip_hor<0)	skip_left = j+2;
		   	else			skip_left = skip_hor;
		   	if (j+skip_hor>15)	skip_right = MB_SIZE+1-j;
		   	else			skip_right = skip_hor;

			if(prev!=next) {
                        	curr=*(smb_data+i*width+j);
				context = GetContextSI_VS_AR (smb_data, 
								j, 
								i, 
								skip_right, 
								skip_left, 
								skip_upper, 
								skip_bottom,
                        					width);

                                prob = si_intra_vs_prob[context];
                                ArCodeSymbol(curr,prob,ar_coder,acstream);

#ifdef  _DEBUG_OBSS_
fprintf(stderr,">>>>>(%d,%d):prev=%d,next=%d --curr=%d, context=%d, prob=%d (skip_right=%d left=%d upper=%d bottom=%d)\n",i,j,prev,next,curr,context,prob,skip_right,skip_left,skip_upper,skip_bottom);
/*fprintf(stderr,">>>>>ar_coder->L:%d,ar_coder->R:%d\n",ar_coder->L,ar_coder->R);*/
#endif

			}
#ifdef  _DEBUG_OBSS_
else {
fprintf(stderr,">>>>>(%d,%d):prev=%d,next=%d\n",i,j,prev,next);
}
#endif		

		   	if (skip_bottom == v_scan_freq) skip_upper = v_scan_freq;
		   	else				skip_upper = 2*v_scan_freq;
		      } 
		   }
		} 
	   }
	}

	i=0;
	while(alpha_mb_spos_h[i]==1) i++;
	while(alpha_mb_spos_h[i]==0) i++;
	if(i>start_h && hor_res_scan) tmp=start_h;
	else	tmp = i;

#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>k-loop encoding - hor_loop_num=%d!\n", hor_loop_num);
getchar();
#endif

	while (hor_loop_num>0) {

	   start_v = start_h = 0;
	   h_scan_freq = 1<<hor_loop_num;
	   v_scan_freq = 1<<ver_loop_num;
	   skip_hor = 1<<(hor_loop_num-1);

	   if (tmp-(1<<(hor_loop_num-1))>=0)	tmp = start_h = tmp - (1<<(hor_loop_num-1));
	   else				start_h = tmp + (1<<(hor_loop_num-1));
		
	   i=0;
	   while (alpha_mb_spos_v[i]==0)	i++;
	   start_v = i;

	   for (j=start_h; j<(width-4); j+=h_scan_freq) {
		skip_upper = start_v + 1;
		for (i=start_v; i<(height-4); i+=v_scan_freq) {
		   if (alpha_mb_spos_v[i]==1) {
		      if (i+v_scan_freq > 15) skip_bottom=MB_SIZE+1-i;
		      else {
			  if ( alpha_mb_spos_v[i+v_scan_freq] == 0 ) {
				if (i+v_scan_freq*2 > 15) skip_bottom=MB_SIZE+1-i;
				else			  skip_bottom=v_scan_freq*2;
			  } else	skip_bottom = v_scan_freq;
		      }

		      if (j-(1<<(hor_loop_num-1))<-2) prev = *(smb_data+i*width-2);
		      else	prev = *(smb_data+i*width+j-(1<<(hor_loop_num-1)));
		      if (j+(1<<(hor_loop_num-1))>17) next = *(smb_data+i*width+17); 
		      else	next = *(smb_data+i*width+j+(1<<(hor_loop_num-1)));

		      if (j-skip_hor<0)	skip_left = j+2;
		      else		skip_left = skip_hor;
		      if (j+skip_hor>15)	skip_right = MB_SIZE+1-j;
		      else  		skip_right = skip_hor;

		      if(prev!=next) {
                           curr=*(smb_data+i*width+j);
			   context = GetContextSI_VS_AR (smb_data, 
								j, 
								i, 
								skip_right, 
								skip_left, 
								skip_upper, 
								skip_bottom,
                        					width);
                           prob = si_intra_vs_prob[context];
                           ArCodeSymbol(curr,prob,ar_coder,acstream);

#ifdef  _DEBUG_OBSS_
fprintf(stderr,">>>>>(%d,%d):prev=%d,next=%d --curr=%d, context=%d, prob=%d (skip_right=%d left=%d upper=%d bottom=%d)\n",i,j,prev,next,curr,context,prob,skip_right,skip_left,skip_upper,skip_bottom);
/*fprintf(stderr,">>>>>ar_coder->L:%d,ar_coder->R:%d\n",ar_coder->L,ar_coder->R);*/
#endif
		      }
#ifdef  _DEBUG_OBSS_
else {
fprintf(stderr,">>>>>(%d,%d):prev=%d,next=%d\n",i,j,prev,next);
}
#endif		
		
		      if (skip_bottom == v_scan_freq)	skip_upper = v_scan_freq;
		      else				skip_upper = 2*v_scan_freq;
		   } 
		}
	   }
	   hor_loop_num--;
	}

	/* Horizontal scanning */

#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>Horizontal scanning()\n");
#endif

	if(ver_res_scan) {

#ifdef  _DEBUG_OBSS_
fprintf(stderr,">>>>>residual encoding\n");
#endif
	   i=0;
	   while(alpha_mb_spos_v[i]==1) i++;
	   while(alpha_mb_spos_v[i]==0) i++;
	   tmp = i;

	   start_v = start_h = 0;
	   v_scan_freq = 1 << ver_loop_num;
	   h_scan_freq = 1;
	   skip_ver = v_scan_freq;

	   if (tmp-(1<<ver_loop_num)>=0)		start_v = tmp - (1<<ver_loop_num);
	   else if (tmp+(1<<ver_loop_num)<MB_SIZE) 	start_v = tmp + (1<<ver_loop_num);
	   else {
		fprintf(stderr,"Error:out of sampling ratio !\n");
		exit(1);
	   }
	
           for(i=start_v;i<(height-4);i+=v_scan_freq) {
		if( alpha_mb_spos_v[i]== 0 ) {
                   for(j=start_h;j<(width-4);j+=h_scan_freq) {
   
		      if (i-v_scan_freq < -2)	prev = *(smb_data-2*width+j);
		      else			prev = *(smb_data+(i-v_scan_freq)*width+j);
		      if (i+v_scan_freq > 17)	next = *(smb_data+17*width+j); 
		      else			next = *(smb_data+(i+v_scan_freq)*width+j);

		      if (i-skip_ver<0)		skip_upper = i+2;
		      else			skip_upper = skip_ver;
		      if (i+skip_ver>15)	skip_bottom = MB_SIZE+1-i;
		      else 	      		skip_bottom = skip_ver;

		      if(prev!=next) {
			curr=*(smb_data+i*width+j);
			context = GetContextSI_HS_AR (smb_data,
							j,
							i,
							skip_upper,
							skip_bottom,
							width);
			prob = si_intra_hs_prob[context];
			ArCodeSymbol(curr,prob,ar_coder,acstream);
#ifdef  _DEBUG_OBSS_
fprintf(stderr,">>>>>(%d,%d):prev=%d,next=%d --curr=%d, context=%d, prob=%d (skip_upper=%d bottom=%d)\n",i,j,prev,next,curr,context,prob,skip_upper,skip_bottom);
/*fprintf(stderr,">>>>>ar_coder->L:%d,ar_coder->R:%d\n",ar_coder->L,ar_coder->R);*/
#endif
		      }
#ifdef  _DEBUG_OBSS_
else {
fprintf(stderr,">>>>>(%d,%d):prev=%d,next=%d\n",i,j,prev,next);
}
#endif		
		   }
		}
	   }
	}

	i=0;
	while(alpha_mb_spos_v[i]==1) i++;
	while(alpha_mb_spos_v[i]==0) i++;
	if(i>start_v && ver_res_scan) tmp=start_v;
	else	tmp = i;

#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>k-loop encoding - ver_loop_num=%d!\n", ver_loop_num);
getchar();
#endif

	while (ver_loop_num>0) {
	   start_v = start_h = 0;
	   v_scan_freq = 1<<ver_loop_num;
	   h_scan_freq = 1;

	   if (tmp-(1<<(ver_loop_num-1))>=0)	tmp = start_v = tmp - (1<<(ver_loop_num-1));
	   else				start_v = tmp + (1<<(ver_loop_num-1));
	   skip_ver = 1<<(ver_loop_num-1);

	   for (i=start_v; i<(height-4); i+=v_scan_freq) {
	   	for (j=start_h; j<(width-4); j+=h_scan_freq) {
		   if (i-skip_ver<-2) prev = *(smb_data-2*width+j);
		   else	prev = *(smb_data+(i-skip_ver)*width+j);
		   if (i+skip_ver>17) next = *(smb_data+17*width+j); 
		   else	next = *(smb_data+(i+skip_ver)*width+j);

		   if (i-skip_ver<0)	skip_upper = i+2;
		   else			skip_upper = skip_ver;
		   if (i+skip_ver>15)	skip_bottom = MB_SIZE+1-i;
		   else 	      	skip_bottom = skip_ver;

		   if(prev!=next) {
                           curr=*(smb_data+i*width+j);
			   context = GetContextSI_HS_AR (smb_data,
							j,
							i,
							skip_upper,
							skip_bottom,
							width);
			   prob = si_intra_hs_prob[context];
			   ArCodeSymbol(curr,prob,ar_coder,acstream);

#ifdef  _DEBUG_OBSS_
fprintf(stderr,">>>>>(%d,%d):prev=%d,next=%d --curr=%d, context=%d, prob=%d (skip_upper=%d bottom=%d)\n",i,j,prev,next,curr,context,prob,skip_upper,skip_bottom);
/*fprintf(stderr,">>>>>ar_coder->L:%d,ar_coder->R:%d\n",ar_coder->L,ar_coder->R);*/
#endif

		   }
#ifdef  _DEBUG_OBSS_
else {
fprintf(stderr,">>>>>(%d,%d):prev=%d,next=%d\n",i,j,prev,next);
}
#endif		
		}
	   }
	   ver_loop_num--;
	}

        return(GetImageSizeX(acstream));
}


/*------------------------------------------------------*/
/* Transitional bab data coding.                        */
/*------------------------------------------------------*/
Int EncodeExceptionalBab_AR (Image *bordered_smb,
				UChar *alpha_mb_spos_h,
				UChar *alpha_mb_spos_v,
				Int hor_loop_num,
				Int ver_loop_num,
				Int hor_res_scan,
				Int ver_res_scan,
				Image *acstream, 
				ArCoder *ar_coder)
{
	Int i,j,curr;
	Int context,prob;
	Int width=GetImageSizeX(bordered_smb), /* 20 */
    	    height=GetImageSizeY(bordered_smb);  /* 20 */
	Int skip_upper=0,skip_bottom=0,skip_hor=0,skip_ver=0,skip_left=0,skip_right=0;
	Int start_v=0,start_h=0;
	Int h_scan_freq=1,v_scan_freq=1;	
	Int tmp;

        SInt *smb_data;

	smb_data = (SInt *)GetImageData(bordered_smb) + 2*width + 2;

#ifdef  _DEBUG_OBSS_
fprintf(stderr,">>>EncodeExceptionalBab_AR ()==>\n");
fprintf(stderr,">>>horizontal sampling position !\n");
for(i=0; i<MB_SIZE; i++) fprintf(stderr,"%d",alpha_mb_spos_h[i]);
fprintf(stderr,"\n>>>vertical sampling position !\n");
for(i=0; i<MB_SIZE; i++) fprintf(stderr,"%d",alpha_mb_spos_v[i]);
fprintf(stderr,"\n");
fprintf(stderr,">>>>>Vertical scanning()\n");
#endif
	/* Vertical scanning */
	if(hor_res_scan) {
#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>residual encoding !\n");
#endif
	   i=0;
	   while(alpha_mb_spos_h[i]==1) i++;
	   while(alpha_mb_spos_h[i]==0) i++;
	   tmp = i;

	   start_v = start_h = 0;
	   h_scan_freq = skip_hor = 1 << hor_loop_num;
	   v_scan_freq = 1 << ver_loop_num;

	   if (tmp-(1<<hor_loop_num)>=0)		start_h = tmp - (1<<hor_loop_num);
	   else if (tmp+(1<<hor_loop_num)<MB_SIZE) 	start_h = tmp + (1<<hor_loop_num);
	   else {
		fprintf(stderr,"Error:out of sampling ratio !\n");
		exit(1);
	   }
	
	   i=0;
	   while(alpha_mb_spos_v[i]==0) i++;
	   start_v = i;	
	   	
           for(j=start_h;j<(width-4);j+=h_scan_freq) {
		if ( alpha_mb_spos_h[j] == 0 ) { 
		   skip_upper = start_v+1;
                   for(i=start_v;i<(height-4);i+=v_scan_freq) {
		      if ( alpha_mb_spos_v[i] == 1 ) {
			if (i+v_scan_freq > 15) skip_bottom=MB_SIZE+1-i;
			else {
			   if ( alpha_mb_spos_v[i+v_scan_freq] == 0 ) {
				if (i+v_scan_freq*2 > 15) skip_bottom=MB_SIZE+1-i;
				else  skip_bottom=v_scan_freq*2;
			   } else  skip_bottom = v_scan_freq;
			}
		   	if (j-skip_hor<0)	skip_left = j+2;
		   	else			skip_left = skip_hor;
		   	if (j+skip_hor>15)	skip_right = MB_SIZE+1-j;
		   	else			skip_right = skip_hor;
		
                       	curr=*(smb_data+i*width+j);
			context = GetContextSI_VS_AR (smb_data, 
								j, 
								i, 
								skip_right, 
								skip_left, 
								skip_upper, 
								skip_bottom,
                        					width);

			prob = si_intra_vs_prob[context];
			ArCodeSymbol(curr,prob,ar_coder,acstream);

#ifdef  _DEBUG_OBSS_
fprintf(stderr,">>>>>(%d,%d)- curr=%d, context=%d, prob=%d (skip_right=%d left=%d upper=%d bottom=%d)\n",i,j,curr,context,prob,skip_right,skip_left,skip_upper,skip_bottom);
/*fprintf(stdout,">>>>>ar_coder->L:%d,ar_coder->R:%d\n",ar_coder->L,ar_coder->R);*/
#endif

		   	if (skip_bottom == v_scan_freq) skip_upper = v_scan_freq;
		   	else				skip_upper = 2*v_scan_freq;
		      } 
		   }
		} 
	   }
	}

	i=0;
	while(alpha_mb_spos_h[i]==1) i++;
	while(alpha_mb_spos_h[i]==0) i++;
	if(i>start_h && hor_res_scan) tmp=start_h;
	else	tmp = i;
#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>k-loop encoding - hor_loop_num=%d!\n", hor_loop_num);
getchar();
#endif
	while (hor_loop_num>0) {
	   start_v = start_h = 0;
	   h_scan_freq = 1<<hor_loop_num;
	   v_scan_freq = 1<<ver_loop_num;
	   skip_hor = 1<<(hor_loop_num-1);

	   if (tmp-(1<<(hor_loop_num-1))>=0)	tmp = start_h = tmp - (1<<(hor_loop_num-1));
	   else				start_h = tmp + (1<<(hor_loop_num-1));
		
	   i=0;
	   while (alpha_mb_spos_v[i]==0)	i++;
	   start_v = i;

	   for (j=start_h; j<(width-4); j+=h_scan_freq) {
		skip_upper = start_v + 1;
		for (i=start_v; i<(height-4); i+=v_scan_freq) {
		   if (alpha_mb_spos_v[i]==1) {
			if (i+v_scan_freq > 15) skip_bottom=MB_SIZE+1-i;
			else {
			      if ( alpha_mb_spos_v[i+v_scan_freq] == 0 ) {
				if (i+v_scan_freq*2 > 15) skip_bottom=MB_SIZE+1-i;
				else			  skip_bottom=v_scan_freq*2;
			      } else	skip_bottom = v_scan_freq;
			}

		   	if (j-skip_hor<0)	skip_left = j+2;
		   	else			skip_left = skip_hor;
		   	if (j+skip_hor>15)	skip_right = MB_SIZE+1-j;
		   	else  			skip_right = skip_hor;

			curr=*(smb_data+i*width+j);
			context = GetContextSI_VS_AR (smb_data, 
							j, 
							i, 
							skip_right, 
							skip_left, 
							skip_upper, 
							skip_bottom,
                        				width);
                                
			prob = si_intra_vs_prob[context];
			ArCodeSymbol(curr,prob,ar_coder,acstream);

#ifdef  _DEBUG_OBSS_
fprintf(stderr,">>>>>(%d,%d)- curr=%d, context=%d, prob=%d (skip_right=%d left=%d upper=%d bottom=%d)\n",i,j,curr,context,prob,skip_right,skip_left,skip_upper,skip_bottom);
/*fprintf(stdout,">>>>>ar_coder->L:%d,ar_coder->R:%d\n",ar_coder->L,ar_coder->R);*/
#endif

		   	if (skip_bottom == v_scan_freq)	skip_upper = v_scan_freq;
		   	else				skip_upper = 2*v_scan_freq;
		   } 
		}
	   }
	   hor_loop_num--;
	}

	/* Horizontal scanning */
#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>Horizontal scanning()\n");
#endif
	if(ver_res_scan) {
#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>residual encoding !\n");
#endif
	   i=0;
	   while(alpha_mb_spos_v[i]==1) i++;
	   while(alpha_mb_spos_v[i]==0) i++;
	   tmp = i;

	   start_v = start_h = 0;
	   v_scan_freq = 1 << ver_loop_num;
	   h_scan_freq = 1;
	   skip_ver = v_scan_freq;

	   if (tmp-(1<<ver_loop_num)>=0)		start_v = tmp - (1<<ver_loop_num);
	   else if (tmp+(1<<ver_loop_num)<MB_SIZE) 	start_v = tmp + (1<<ver_loop_num);
	   else {
		fprintf(stderr,"Error:out of sampling ratio !\n");
		exit(1);
	   }
	
           for(i=start_v;i<(height-4);i+=v_scan_freq) {
		if( alpha_mb_spos_v[i]== 0 ) {
		   for(j=start_h;j<(width-4);j+=h_scan_freq) {
   
		      if (i-skip_ver<0)		skip_upper = i+2;
		      else			skip_upper = skip_ver;
		      if (i+skip_ver>15)	skip_bottom = MB_SIZE+1-i;
		      else 	      		skip_bottom = skip_ver;

                      curr=*(smb_data+i*width+j);
		      context = GetContextSI_HS_AR (smb_data,
							j,
							i,
							skip_upper,
							skip_bottom,
							width);
		      prob = si_intra_hs_prob[context];
		      ArCodeSymbol(curr,prob,ar_coder,acstream);

#ifdef  _DEBUG_OBSS_
fprintf(stderr,">>>>>(%d,%d)- curr=%d, context=%d, prob=%d (skip_upper=%d bottom=%d)\n",i,j,curr,context,prob,skip_upper,skip_bottom);
/*fprintf(stdout,">>>>>ar_coder->L:%d,ar_coder->R:%d\n",ar_coder->L,ar_coder->R);*/
#endif

		   }
		}
	   }
	}

	i=0;
	while(alpha_mb_spos_v[i]==1) i++;
	while(alpha_mb_spos_v[i]==0) i++;
	if(i>start_v && ver_res_scan) tmp=start_v;
	else	tmp = i;
#ifdef	_DEBUG_OBSS_
fprintf(stderr,">>>>>k-loop encoding - ver_loop_num=%d!\n", ver_loop_num);
getchar();
#endif
	while (ver_loop_num>0) {

	   start_v = start_h = 0;
	   v_scan_freq = 1<<ver_loop_num;
	   h_scan_freq = 1;

	   if (tmp-(1<<(ver_loop_num-1))>=0)	tmp = start_v = tmp - (1<<(ver_loop_num-1));
	   else				start_v = tmp + (1<<(ver_loop_num-1));
	   skip_ver = 1<<(ver_loop_num-1);

	   for (i=start_v; i<(height-4); i+=v_scan_freq) {
		for (j=start_h; j<(width-4); j+=h_scan_freq) {
		   	
		   if (i-skip_ver<0)	skip_upper = i+2;
		   else			skip_upper = skip_ver;
		   if (i+skip_ver>15)	skip_bottom = MB_SIZE+1-i;
		   else 	     	skip_bottom = skip_ver;
                        
		   curr=*(smb_data+i*width+j);
		   context = GetContextSI_HS_AR (smb_data,
							j,
							i,
							skip_upper,
							skip_bottom,
							width);
		   prob = si_intra_hs_prob[context];
		   ArCodeSymbol(curr,prob,ar_coder,acstream);

#ifdef  _DEBUG_OBSS_
fprintf(stderr,">>>>>(%d,%d)- curr=%d, context=%d, prob=%d (skip_upper=%d bottom=%d)\n",i,j,curr,context,prob,skip_upper,skip_bottom);
/*fprintf(stdout,">>>>>ar_coder->L:%d,ar_coder->R:%d\n",ar_coder->L,ar_coder->R);*/
#endif

		}
	   }
	   ver_loop_num--;
	}

        return(GetImageSizeX(acstream));
}


