/* $Id: ztscan_dec.c,v 1.88 1998/11/10 16:42:41 hatrack Exp $ */
/****************************************************************************/
/*   MPEG4 Visual Texture Coding (VTC) Mode Software                        */
/*                                                                          */
/*   This software was jointly developed by the following participants:     */
/*                                                                          */
/*   Single-quant,  multi-quant and flow control                            */
/*   are provided by  Sarnoff Corporation                                   */
/*     Iraj Sodagar   (iraj@sarnoff.com)                                    */
/*     Hung-Ju Lee    (hjlee@sarnoff.com)                                   */
/*     Paul Hatrack   (hatrack@sarnoff.com)                                 */
/*     Shipeng Li     (shipeng@sarnoff.com)                                 */
/*     Bing-Bing Chai (bchai@sarnoff.com)                                   */
/*     B.S. Srinivas  (bsrinivas@sarnoff.com)                               */
/*                                                                          */
/*   Bi-level is provided by Texas Instruments                              */
/*     Jie Liang      (liang@ti.com)                                        */
/*                                                                          */
/*   Shape Coding is provided by  OKI Electric Industry Co., Ltd.           */
/*     Zhixiong Wu    (sgo@hlabs.oki.co.jp)                                 */
/*     Yoshihiro Ueda (yueda@hlabs.oki.co.jp)                               */
/*     Toshifumi Kanamaru (kanamaru@hlabs.oki.co.jp)                        */
/*                                                                          */
/*   OKI, Sharp, Sarnoff, TI and Microsoft contributed to bitstream         */
/*   exchange and bug fixing.                                               */
/*                                                                          */
/*                                                                          */
/* In the course of development of the MPEG-4 standard, this software       */
/* module is an implementation of a part of one or more MPEG-4 tools as     */
/* specified by the MPEG-4 standard.                                        */
/*                                                                          */
/* The copyright of this software belongs to ISO/IEC. ISO/IEC gives use     */
/* of the MPEG-4 standard free license to use this  software module or      */
/* modifications thereof for hardware or software products claiming         */
/* conformance to the MPEG-4 standard.                                      */
/*                                                                          */
/* Those intending to use this software module in hardware or software      */
/* products are advised that use may infringe existing  patents. The        */
/* original developers of this software module and their companies, the     */
/* subsequent editors and their companies, and ISO/IEC have no liability    */
/* and ISO/IEC have no liability for use of this software module or         */
/* modification thereof in an implementation.                               */
/*                                                                          */
/* Permission is granted to MPEG members to use, copy, modify,              */
/* and distribute the software modules ( or portions thereof )              */
/* for standardization activity within ISO/IEC JTC1/SC29/WG11.              */
/*                                                                          */
/* Copyright 1995, 1996, 1997, 1998 ISO/IEC                                 */
/****************************************************************************/

/************************************************************/
/*     Sarnoff Very Low Bit Rate Still Image Coder          */
/*     Copyright 1995, 1996, 1997, 1998 Sarnoff Corporation */
/************************************************************/

/************************************************************/
/*  Filename: ztscan_dec.c                                  */
/*  Author: Bing-Bing CHai                                  */
/*  Date: Dec. 17, 1997                                     */
/*                                                          */
/*  Descriptions:                                           */
/*    This file contains the routines that performs         */
/*    zero tree scanning and entropy decoding.              */
/*                                                          */
/************************************************************/

#include <stdio.h>
#include <stdlib.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <ctype.h>
#include <string.h>
#include <math.h>
#include "typedef.h"
#include "Utils.h"
#include "startcode.h"
#include "dataStruct.h"
#include "states.h"
#include "globals.h"
#include "errorHandler.h"
#include "ac.h"
#include "bitpack.h"
#include "ztscan_common.h"
#include "ztscanUtil.h"
#include "encQM.h"
#include "ztscan_dec.h"

static ac_decoder acd;

/* functions defined */
static Void callc_decode();

static Void cachb_decode_SQ_band(SNR_IMAGE *snr_img);
static Void cachb_decode_SQ_tree();
static Void decode_pixel_SQ(Int h,Int w);
static Void mag_sign_decode_SQ(Int h,Int w);


static Void cachb_decode_MQ_band();
static Void cachb_decode_MQ_tree();
static Void decode_pixel_MQ(Int h,Int w);
static Void mark_ZTR_D(Int h,Int w);
static Void mag_sign_decode_MQ(Int h,Int w);

/* begin: added by Rockwell (99/3/3) */

/* IM: VTC error res */ 
/* error resilience routine prototypes and vars, bbc 11/6/98 */
void mark_not_decodable_TUs(Int start_TU, Int end_TU);
static Int init_ac=0; /*,prev_good_TU,prev_good_height,prev_good_width;*/

static Void init_arith_decoder_model(Int col);
static Void close_arith_decoder_model(Int color);
static Void check_end_of_packet();

/* end: added by Rockwell (99/3/3) */

/******************************************************************/
/****************************  DC  ********************************/
/******************************************************************/

/*******************************************************/
/**************  Inverse DC Prediction  ****************/
/*******************************************************/

/********************************************************
  Function Name
  -------------
  static DATA iDC_pred_pix(Int i, Int j)

  Arguments
  ---------
  Int i, Int j: Index of wavelet coefficient (row, col)
  
  Description
  -----------
  Inverse DPCM prediction for a DC coefficient, refer 
  to syntax for algorithm. 

  Functions Called
  ----------------
  None.

  Return Value
  ------------
    inverse prediction for coeffinfo[i][j].quantized_value
********************************************************/ 
static DATA  iDC_pred_pix(Int i, Int j)
{
  /*  modified by Z. Wu @ OKI */

  Int pred_i, pred_j, pred_d;

  if ( i==0 || coeffinfo[i-1][j].mask == 0 )	
    pred_i = 0;
  else
    pred_i = coeffinfo[i-1][j].quantized_value;

  if ( j==0 || coeffinfo[i][j-1].mask == 0 )	
    pred_j = 0;
  else 
    pred_j = coeffinfo[i][j-1].quantized_value;

  if ( i==0 || j== 0 || coeffinfo[i-1][j-1].mask == 0 )	
    pred_d = 0;
  else	
    pred_d = coeffinfo[i-1][j-1].quantized_value;

  if ( abs(pred_d-pred_j) < abs(pred_d-pred_i))	
    return(pred_i);
  else
    return(pred_j);
}



/*****************************************************
  Function Name
  -------------
  Void iDC_predict()

  Arguments
  ---------
  None
  
  Description
  -----------
  control program for inverse DC prediction

  Functions Called
  ----------------
  iDC_pred_pix(i,j).

  Return Value
  ------------
  none
******************************************************/
static Void iDC_predict(Int color)
{
  Int i,j,dc_h,dc_w,offset_dc;

  dc_h=mzte_codec.dcHeight;
  dc_w=mzte_codec.dcWidth;
 
  coeffinfo=mzte_codec.SPlayer[color].coeffinfo;
  offset_dc=mzte_codec.offset_dc;

  for(i=0;i<dc_h;i++){
    for(j=0;j<dc_w;j++){
      if (coeffinfo[i][j].mask != 0)
	coeffinfo[i][j].quantized_value += offset_dc;
		}
	}

  for(i=0;i<dc_h;i++){
    for(j=0;j<dc_w;j++){
      if (coeffinfo[i][j].mask != 0)
	coeffinfo[i][j].quantized_value+=iDC_pred_pix(i,j);
		}
	}
}




/********************************************************
  Function Name
  -------------
  Void wavelet_dc_decode(Int c)


  Arguments
  ---------
  Int c - color component.
  
  Description
  -----------
  Control program for decode DC information for one 
  color component.

  Functions Called
  ----------------
  None.
  iDC_predict()
  get_param()
  cacll_decode()
  
  Return Value
  ------------
  None.

********************************************************/ 
Void wavelet_dc_decode(Int c)
{
  noteDetail("Decoding DC (wavelet_dc_decode)....");
  color=c;

/* begin: added by Rockwell (99/3/3) */

  /* IM: VTC error res */ 
  /* Maximum TU number excluding dc part, moved from ac part, bbc, 11/18/98 */
  if(color==0){
    /* bbc, 11/19/98 */
    if(mzte_codec.scan_direction==0)  /* TD */
      TU_max=mzte_codec.dcHeight*mzte_codec.dcWidth*9-1;
    else /* BB */
      TU_max=mzte_codec.dcHeight*(3*mzte_codec.wvtDecompLev-2) - 1;
  }

/* end: added by Rockwell (99/3/3) */

  mzte_codec.mean[color] = get_X_bits(8);
  /* mzte_codec.Qdc[color]  = get_X_bits(8); */
  mzte_codec.Qdc[color]  = get_param(7);

  mzte_codec.offset_dc=-get_param(7);
  mzte_codec.max_dc=get_param(7); 
  /* mzte_codec.max_dc=get_param(7)-mzte_codec.offset_dc;*/ /* hjlee */

  callc_decode();
  iDC_predict(color);
  noteDetail("Completed decoding DC.");


}



/********************************************************
  Function Name
  -------------
  static Void cacll_decode()

  Arguments
  ---------
  None.

  
  Description
  -----------
  Decode DC information for one color component.

  Functions Called
  ----------------
  mzte_ac_decoder_init()
  mzte_ac_model_init()
  mzte_ac_decode_symbol()
  mzte_ac_model_done()
  mzte_ac_decoder_done()

  Return Value
  ------------
  None.

********************************************************/ 
static Void callc_decode()
{
  Int dc_h, dc_w,i,j;
  Int numBP, bp;

  dc_w=mzte_codec.dcWidth;
  dc_h=mzte_codec.dcHeight;

  /* init arithmetic model */
  mzte_ac_decoder_init(&acd);

  numBP = ceilLog2(mzte_codec.max_dc+1); /* modified by Sharp(99/2/16) */
  if ((acm_bpdc=(ac_model *)calloc(numBP,sizeof(ac_model)))==NULL)
    errorHandler("Can't allocate memory for prob model.");
  
  for (i=0; i<numBP; i++) {
    acm_bpdc[i].Max_frequency = Bitplane_Max_frequency;
    mzte_ac_model_init(&(acm_bpdc[i]),2,NULL,ADAPT,1);
  }
  coeffinfo=mzte_codec.SPlayer[color].coeffinfo;
  for (bp=numBP-1; bp>=0; bp--) {
    for(i=0;i<dc_h;i++)
      for(j=0;j<dc_w;j++){
	if(coeffinfo[i][j].mask == 1) {
	   coeffinfo[i][j].quantized_value +=
	     (mzte_ac_decode_symbol(&acd,&(acm_bpdc[bp])) << bp);
	}
	else
	  coeffinfo[i][j].quantized_value=-mzte_codec.offset_dc;
      }
  }

  for (i=0; i<numBP; i++) 
    mzte_ac_model_done(&(acm_bpdc[i]));
  free(acm_bpdc);
  mzte_ac_decoder_done(&acd);
}


/*********************************************************************/
/*****************************  AC  **********************************/
/*********************  Single and Multi Quant  **********************/
/*********************************************************************/

  
static Int bitplane_decode(Int l,Int max_bplane)
{
  register Int i,val=0,k=0;

  for(i=max_bplane-1;i>=0;i--,k++)
    val+=mzte_ac_decode_symbol(&acd,&acm_bpmag[l][k])<<i;

  return val;
}


/*********************************************************************/
/*****************************  AC  **********************************/
/*************************  Single quant  ****************************/
/*********************************************************************/

/*******************************************************
  The following single quant routines are for band by
  band scan order.
*******************************************************/
/********************************************************
  Function Name
  -------------
  Void wavelet_higher_bands_decode_SQ_band(Int col)

  Arguments
  ---------
  None.

  Description
  -----------
  Control program for decoding AC information for one 
  color component. Single quant mode.

  Functions Called
  ----------------
  cachb_encode_SQ_band()
  ac_encoder_init()
  mzte_ac_model_init()
  mzte_ac_model_done()
  ac_encoder_done()

  Return Value
  ------------
  None.

********************************************************/ 
Void wavelet_higher_bands_decode_SQ_band(Int col)
{
  SNR_IMAGE *snr_image;
    
  noteDetail("Encoding AC (wavelet_higher_bands_encode_SQ)....");

  color=col;
  snr_image=&(mzte_codec.SPlayer[color].SNRlayer.snr_image);

/* begin: added by Rockwell (99/3/3) */

  /* IM: VTC error res */ 
  if (mzte_codec.error_res_flag)
    {
  /* init arithmetic coder */
	init_arith_decoder_model(color);  /* ph, 11/13/98 */
/*  mzte_ac_decoder_init(&acd);*/
/*  probModelInitSQ(color);*/
    }
  else 
    {
  /* init arithmetic coder */
  mzte_ac_decoder_init(&acd);

  probModelInitSQ(color);
    }

  cachb_decode_SQ_band(snr_image);
  
  /* IM: VTC error res */ 
  if (mzte_codec.error_res_flag)
    {
  /* close arithmetic coder */
	/* ph, 11/13/98 */
	close_arith_decoder_model(color);
/*  probModelFreeSQ(color);*/
/*  mzte_ac_decoder_done(&acd);*/
    }
  else
    {
  /* close arithmetic coder */
  probModelFreeSQ(color);
  
  mzte_ac_decoder_done(&acd);
    }

  noteDetail("Completed encoding AC.");
 
#if 0  
  /* init arithmetic coder */
  mzte_ac_decoder_init(&acd);

  probModelInitSQ(color);
  
  cachb_decode_SQ_band(snr_image);
  
  /* close arithmetic coder */
  probModelFreeSQ(color);

  mzte_ac_decoder_done(&acd);

  noteDetail("Completed encoding AC.");
#endif
/* end: added by Rockwell (99/3/3) */
}


/********************************************************
  Function Name
  -------------
  static Void cachb_decode_SQ_band(SNR_IMAGE *snr_img)

  Arguments
  ---------
  None.

  Description
  -----------
  Decode AC information for single quant mode, tree-depth scan.

  Functions Called
  ----------------
  codeBlocks();
  decode_pixel_SQ_band()

  Return Value
  ------------
  None.

********************************************************/ 
static Void cachb_decode_SQ_band(SNR_IMAGE *snr_image)
{
  Int h,w,ac_h,ac_w,ac_h2,ac_w2;
  Int n; /* layer index - for codeBlocks function */
  Int k; /* block jump for the layer */

  /* ac_h, ac_w init */
  ac_h2=mzte_codec.SPlayer[color].height;
  ac_w2=mzte_codec.SPlayer[color].width;
  ac_h=ac_h2>>1;
  ac_w=ac_w2>>1;

  height=mzte_codec.Image[color].height;
  width=mzte_codec.Image[color].width;

  /* Get layer index - for codeBlocks function */
  n = -1;
  for (w=mzte_codec.dcWidth; w < ac_w2; w<<=1)
    n++;

  setProbModelsSQ(color);
	
  coeffinfo=mzte_codec.SPlayer[color].coeffinfo;

  /* scan each coefficients in the spatial layer */  
  k = 1<<n;

/* begin: added by Rockwell (99/3/3) */

  /* IM: VTC error res */ 
  if (mzte_codec.error_res_flag)
    {
/*  for(h=0;h<ac_h;h+=k)*/
  while (1)  /* ph, 11/17/98 - removed: for(h=0;h<ac_h;h+=k) */
  {
    /* check that current TU and this function are ok */
    get_TU_location(LTU);

    /* IM: VTC error res - I assume this is an hack ... */   
    /* if color or band_height (spatial layer) are mismatched then leave */
    if (TU_color!=color || band_height!=ac_h)
      return;

    /* color and spatial layer are correct test row. Set row of TU */
    h = start_h;

    /*******************************************/

    /* IM: VTC error res */ 
    for(w=ac_w;w<ac_w2;w+=k)
    {
      /* LH */
      decodeBlocks(h,w,n,&decode_pixel_SQ,color);
      if(n>0 && n<5) /* ph, 11/17/98 */
  if (found_segment_error(color)==1)
  {
    /* found error */
  }

      /* HL */
      h += ac_h;
      w -= ac_w;
      decodeBlocks(h,w,n,&decode_pixel_SQ,color);
      if(n>0 && n<5) /* ph, 11/17/98 */
  if (found_segment_error(color)==1)
  {
    /* found error */
  }

      /* HH */
      w += ac_w;
      decodeBlocks(h,w,n,&decode_pixel_SQ,color);
      if(n>0 && n<5) /* ph, 11/17/98 */
  if (found_segment_error(color)==1)
  {
    /* found error */
  }

      /* Set h back to where it started. w is already there */
      h -= ac_h;
    }

    /* IM: VTC error res */ 
    check_end_of_packet(); /* ph, 11/17/98 */
    ++LTU;
  }
    } 
  else /* mzte_codec.error_res_flag == 0 */
    {
  for(h=0;h<ac_h;h+=k)
    for(w=ac_w;w<ac_w2;w+=k)
    {
      /* LH */
      codeBlocks(h,w,n,&decode_pixel_SQ,color);

      /* HL */
      h += ac_h;
      w -= ac_w;
      codeBlocks(h,w,n,&decode_pixel_SQ,color);

      /* HH */
      w += ac_w;
      codeBlocks(h,w,n,&decode_pixel_SQ,color);

      /* Set h back to where it started. w is already there */
      h -= ac_h;
    }
    }

#if 0
  for(h=0;h<ac_h;h+=k)
    for(w=ac_w;w<ac_w2;w+=k)
    {
      /* LH */
      codeBlocks(h,w,n,&decode_pixel_SQ);

      /* HL */
      h += ac_h;
      w -= ac_w;
      codeBlocks(h,w,n,&decode_pixel_SQ);

      /* HH */
      w += ac_w;
      codeBlocks(h,w,n,&decode_pixel_SQ);

      /* Set h back to where it started. w is already there */
      h -= ac_h;
    }
#endif
/* end: added by Rockwell (99/3/3) */
}

/*******************************************************
  The following single quant routines are for tree
  depth scan order.
*******************************************************/
/********************************************************
  Function Name
  -------------
  Void wavelet_higher_bands_decode_SQ_tree()

  Arguments
  ---------
  None.

  Description
  -----------
  Control program for decoding AC information for single quant mode.
  All colors decoded. 

  Functions Called
  ----------------
  cachb_decode_SQ_tree()
  mzte_ac_decoder_init()
  mzte_ac_model_init()
  mzte_ac_model_done()
  mzte_ac_decoder_done()

  Return Value
  ------------
  None.

********************************************************/ 
Void wavelet_higher_bands_decode_SQ_tree()
{
  noteDetail("Decoding AC band (wavelet_higher_bands_decode_SQ)....");

/* begin: added by Rockwell (99/3/3) */

  /* IM: VTC error res */ 
  if (mzte_codec.error_res_flag)
    {
  /* init arithmetic coder */
/*  mzte_ac_decoder_init(&acd);*/
  
/*  for (color=0; color<mzte_codec.colors; color++) */
/*    probModelInitSQ(color);*/
  init_arith_decoder_model(color); /* bbc, 11/6/98 */
    }
  else
    {
  /* init arithmetic coder */
  mzte_ac_decoder_init(&acd);
  
  for (color=0; color<mzte_codec.colors; color++) 
    probModelInitSQ(color);
    }
      
  cachb_decode_SQ_tree();

  if (mzte_codec.error_res_flag)
    {
  close_arith_decoder_model(color); /* bbc, 11/6/98 */

/*  for (color=0; color<mzte_codec.colors; color++) */
    /* close arithmetic coder */
/*    probModelFreeSQ(color);*/

/*  mzte_ac_decoder_done(&acd);*/
    }
  else
    {
  for (color=0; color<mzte_codec.colors; color++) 
    /* close arithmetic coder */
    probModelFreeSQ(color);

  mzte_ac_decoder_done(&acd);
    }

  noteDetail("Completed decoding AC band.");

#if 0  
  /* init arithmetic coder */
  mzte_ac_decoder_init(&acd);
  
  for (color=0; color<mzte_codec.colors; color++) 
    probModelInitSQ(color);
  
  cachb_decode_SQ_tree();

  for (color=0; color<mzte_codec.colors; color++) 
    /* close arithmetic coder */
    probModelFreeSQ(color);

  mzte_ac_decoder_done(&acd);
 
  noteDetail("Completed decoding AC band.");
#endif
/* end: added by Rockwell (99/3/3) */
}



/********************************************************
  Function Name
  -------------
  static Void cachb_decode_SQ_tree()

  Arguments
  ---------
  None.

  Description
  -----------
  Decode AC information for one color component. 
  Single quant mode, tree-depth scan

  Functions Called
  ----------------
  decode_pixel_SQ_tree()

  Return Value
  ------------
  None.

********************************************************/ 
static Void cachb_decode_SQ_tree()
{
  Int h,w,dc_h,dc_w,dc_h2,dc_w2;

/* begin: added by Rockwell (99/3/3) */
  /* IM: VTC error res */ 
  Int tw,sw,sh,n;
/* end: added by Rockwell (99/3/3) */

  dc_h=mzte_codec.dcHeight;
  dc_w=mzte_codec.dcWidth;

/* begin: added by Rockwell (99/3/3) */

  /* IM: VTC error res */ 
  if (mzte_codec.error_res_flag) 
    {
  /* rewrote for error resilience, bbc, 11/9/98 */
  while(LTU<=TU_max){
    get_TU_location(LTU);
    height=mzte_codec.SPlayer[TU_color].height;
    width=mzte_codec.SPlayer[TU_color].width;

    setProbModelsSQ(TU_color);
    coeffinfo=mzte_codec.SPlayer[TU_color].coeffinfo;
    color=TU_color;

    /* decoding one TU */
    n = 0;
    for (tw=mzte_codec.dcWidth; tw < width; tw<<=1){
      sh = start_h << n;
      sw = start_w << n;
      decodeBlocks(sh,sw,n,&decode_pixel_SQ,TU_color);
      if(n>0 && n<5)
  found_segment_error(TU_color);
      n++;
    }

    check_end_of_packet();  /* error resilience code, bbc, 11/9/98 */
    LTU++;
  }
    }
  else {

/* end: added by Rockwell (99/3/3) */

  dc_h2=dc_h<<1;
  dc_w2=dc_w<<1;

  for(h=0;h<dc_h;h++)
    for(w=0;w<dc_w;w++)      
      for (color=0; color<mzte_codec.colors; color++) 
      {  	
	SNR_IMAGE *snr_image;
	int tw,sw,sh,n;

	snr_image=&(mzte_codec.SPlayer[color].SNRlayer.snr_image);
 
	height=mzte_codec.SPlayer[color].height;
	width=mzte_codec.SPlayer[color].width;
	
	setProbModelsSQ(color);

	coeffinfo=mzte_codec.SPlayer[color].coeffinfo;
	
	/* LH */
	n = 0;
	for (tw=mzte_codec.dcWidth; tw < width; tw<<=1)
	{
	  sh = h << n;
	  sw = (w+dc_w) << n;
	  /* begin: added by Rockwell (99/3/3) */
	  /* codeBlocks(sh,sw,n,&decode_pixel_SQ); */
	  codeBlocks(sh,sw,n,&decode_pixel_SQ,color);
	  /* end: added by Rockwell (99/3/3) */
	  n++;
	}
	/* HL */
	n = 0;
	for (tw=mzte_codec.dcWidth; tw < width; tw<<=1)
	{
	  sh = (h+dc_h) << n;
	  sw = w << n;
	  /* begin: added by Rockwell (99/3/3) */
	  /* codeBlocks(sh,sw,n,&decode_pixel_SQ); */
	  codeBlocks(sh,sw,n,&decode_pixel_SQ,color);
	  /* end: added by Rockwell (99/3/3) */
	  n++;
	}
	/* HH */
	n = 0;
	for (tw=mzte_codec.dcWidth; tw < width; tw<<=1)
	{
	  sh = (h+dc_h) << n;
	  sw = (w+dc_w) << n;
	  /* begin: added by Rockwell (99/3/3) */
	  /* codeBlocks(sh,sw,n,&decode_pixel_SQ); */
	  codeBlocks(sh,sw,n,&decode_pixel_SQ,color);
	  /* end: added by Rockwell (99/3/3) */
	  n++;
	}
      }
  }
}


/*************************************************************
  The following single quant routines are for both scan orders.
  ************************************************************/

/********************************************************
  Function Name
  -------------
  static Void decode_pixel_SQ(Int h,Int w)

  Arguments
  ---------
  Int h,Int w - position of a pixel in height and width
  
  Description
  -----------
  Decoding the type and/or value of a coefficient, a
  recursive function.

  Functions Called
  ----------------
  mag_sign_decode_SQ()
  mzte_ac_decode_symbol()

  Return Value
  ------------
  None.

********************************************************/ 
static Void decode_pixel_SQ(Int h,Int w)
{
  UChar zt_type;
  Int l;

  if(coeffinfo[h][w].type == ZTR_D)
    return;

  l=xy2wvtDecompLev(w,h);

  /* decode leave coefficients, value only */  
  if(IS_STATE_LEAF(coeffinfo[h][w].state)){ /* zero value. no sign */

      /* Map leaf code word to type 0->ZTR, 1->VZTR */
      coeffinfo[h][w].type = 
	mzte_ac_decode_symbol(&acd,acm_type[l][CONTEXT_LINIT]) ? VZTR : ZTR;
      if (coeffinfo[h][w].type==VZTR)
	mag_sign_decode_SQ(h,w);
      else
	coeffinfo[h][w].quantized_value = 0;

      return;
  }
  
  
  /* decode zero tree symbols */
    coeffinfo[h][w].type=zt_type=
    mzte_ac_decode_symbol(&acd,acm_type[l][CONTEXT_INIT]);

  /* code magnitude and sign */
  switch(zt_type){
    case IZ :
      break;
    case VZTR:
      mag_sign_decode_SQ(h,w);
    case ZTR:
      mark_ZTR_D(h,w);  /* necessary for checking purpose bandwise scan */
      return;
    case VAL:
      mag_sign_decode_SQ(h,w);
      break;
    default: 
      errorHandler("Invalid zerotree symbol in single quant decode");
  }
}
     

/********************************************************
  Function Name
  -------------
  static Void  mag_sign_decode_SQ(Int h,Int w)

  Arguments
  ---------
  Int h,Int w - position of a pixel

  Description
  -----------
  Decode the value of a coefficient.

  Functions Called
  ----------------
  mzte_ac_decode_symbol()

  Return Value
  ------------
  None.

********************************************************/ 


static Void  mag_sign_decode_SQ(Int h,Int w)
{
  Int value,v_sign;
  Int l;
    
  l=xy2wvtDecompLev(w,h);
    

  value=bitplane_decode(l,WVTDECOMP_NUMBITPLANES(color,l))+1;
  v_sign=mzte_ac_decode_symbol(&acd,acm_sign[l]);
  coeffinfo[h][w].quantized_value=(v_sign==0) ? value : -value;
  

}




/*********************************************************************/
/******************************  AC  *********************************/
/**************************  Multi quant  ****************************/
/*********************************************************************/

static Int bitplane_res_decode(Int l,Int max_bplane)
{
  register Int i,val=0,k=0;

  for(i=max_bplane-1;i>=0;i--,k++)
    val+=mzte_ac_decode_symbol(&acd,&acm_bpres[l][k])<<i;

  return val;
}

/********************************************************
  Function Name
  -------------
  Void wavelet_higher_bands_decode_MQ(Int scanDirection)

  Arguments
  ---------
  Int scanDirection - 0 <=> tree, 1 <=> band
  
  Description
  -----------
  Control program for decoding AC information for one 
  color component. Multi quant mode.

  Functions Called
  ----------------
  cachb_decode_MQ_band()
  mzte_ac_decoder_init()
  mzte_ac_model_init()
  mzte_ac_model_done()
  mzte_ac_decoder_done()

  Return Value
  ------------
  None.

********************************************************/ 
Void wavelet_higher_bands_decode_MQ(Int scanDirection)
{
  noteDetail("Decoding AC band (wavelet_higher_bands_decode_MQ)....");

  /* init arithmetic coder */
  mzte_ac_decoder_init(&acd);
  
  if (scanDirection==0)
    cachb_decode_MQ_tree();
  else
    cachb_decode_MQ_band();

  /* close arithmetic coder */
  mzte_ac_decoder_done(&acd);
}



/**********************************************************************/
/***************       MQ BAND         ********************************/
/**********************************************************************/


/********************************************************
  Function Name
  -------------
  static Void cachb_decode_MQ_band()

  Arguments
  ---------
  None.

  Description
  -----------
  Decode AC information for one color component. 
  Multiple quant, bandwise scan.

  Functions Called
  ----------------
  clear_ZTR_D();
  codeBlocks();
  decode_pixel_MQ_band()

  Return Value
  ------------
  None.

********************************************************/ 
static Void cachb_decode_MQ_band()
{
  Int h,w;
  Int ac_h,ac_w,ac_h2,ac_w2;
  Int acH,acW,acH2,acW2;
  Int layer, nCol;
  Int n; /* layer index - for codeBlocks function */
  Int k; /* block jump for the layer */

  /* clear the ZTR_D type from the previous pass */
  for (color=0; color<NCOL; ++color)
  {      
    coeffinfo=mzte_codec.SPlayer[color].coeffinfo;
    height=mzte_codec.SPlayer[color].height;
    width=mzte_codec.SPlayer[color].width;

    clear_ZTR_D(coeffinfo, width, height);
  }

  for (color=0; color<NCOL; ++color)
    probModelInitMQ(color);
  
  acH=mzte_codec.dcHeight;
  acW=mzte_codec.dcWidth;
  acH2=acH<<1;
  acW2=acW<<1;

  /* scan each coefficients in the spatial layer */
  /* assume luma dimensions are >= chroma dimensions */
  layer=0;
  while(acH2<=mzte_codec.SPlayer[0].height 
	&& acW2<=mzte_codec.SPlayer[0].width)
  {
    nCol = (layer==0) ? 1 : NCOL;
    for (color=0; color < nCol; ++color)
    {      
      SNR_IMAGE *snr_image;

      noteProgress("  Coding Layer %d, Color %d", layer - (color!=0), color);

      ac_h2=acH2;
      ac_w2=acW2;
      ac_h=acH;
      ac_w=acW;

      if (color)
      {
	ac_h2>>=1;
	ac_w2>>=1;
	ac_h>>=1;
	ac_w>>=1;
      }

      snr_image=&(mzte_codec.SPlayer[color].SNRlayer.snr_image);

      coeffinfo=mzte_codec.SPlayer[color].coeffinfo;
      height=mzte_codec.SPlayer[color].height;
      width=mzte_codec.SPlayer[color].width;

      setProbModelsMQ(color);

      /* Go through bands */
      n = layer - (color>0);
      k = 1<<n;
      for(h=0;h<ac_h;h+=k)
	for(w=ac_w;w<ac_w2;w+=k)
	{
	  /* LH */
	  /* begin: added by Rockwell (99/3/3) */
	  /* codeBlocks(h,w,n,&decode_pixel_MQ); */
	   decodeBlocks(h,w,n,&decode_pixel_MQ,color); 
	  /* end: added by Rockwell (99/3/3) */	  

	  /* HL */
	  h += ac_h;
	  w -= ac_w;
	  /* begin: added by Rockwell (99/3/3) */
	  /* codeBlocks(h,w,n,&decode_pixel_MQ); */
	  decodeBlocks(h,w,n,&decode_pixel_MQ,color);
	  /* end: added by Rockwell (99/3/3) */	  

	  /* HH */
	  w += ac_w;
	  /* begin: added by Rockwell (99/3/3) */
	  /* codeBlocks(h,w,n,&decode_pixel_MQ); */
	  decodeBlocks(h,w,n,&decode_pixel_MQ,color);
	  /* end: added by Rockwell (99/3/3) */	  
	  
	  /* Set h back to where it started. w is already there */
	  h -= ac_h;
	}

    }

    /* update ranges */
    acH=acH2;
    acW=acW2;
    acW2<<=1;
    acH2<<=1;

    layer++;
  }

  for (color=0; color<NCOL; ++color)
    probModelFreeMQ(color);
}

/**********************************************************************/
/***************       MQ TREE         ********************************/
/**********************************************************************/

/********************************************************
  Function Name
  -------------
  static Void cachb_decode_MQ_tree()

  Arguments
  ---------
  None.

  Description
  -----------
  Decode AC information for one color component. 
  Multiple quant, treewise scan.

  Functions Called
  ----------------
  clear_ZTR_D();
  decode_pixel_MQ_tree()

  Return Value
  ------------
  None.

********************************************************/ 
static Void cachb_decode_MQ_tree()
{
  Int h,w, dc_h, dc_w;
     
  /* clear the ZTR_D type from the previous pass */
  for (color=0; color<NCOL; ++color)
  {      
    coeffinfo=mzte_codec.SPlayer[color].coeffinfo;
    height=mzte_codec.SPlayer[color].height;
    width=mzte_codec.SPlayer[color].width;

    clear_ZTR_D(coeffinfo, width, height);
  }
     
  for (color=0; color<NCOL; ++color)
    probModelInitMQ(color);

  /* ac_h, ac_w init */
  dc_h=mzte_codec.dcHeight;
  dc_w=mzte_codec.dcWidth;

  for (h=0; h<dc_h; ++h)
    for (w=0; w<dc_w; ++w)
    {
      for (color=0; color<NCOL; ++color)
      {      
	SNR_IMAGE *snr_image;
	int tw,sw,sh,n;
	
	snr_image=&(mzte_codec.SPlayer[color].SNRlayer.snr_image);
	
	coeffinfo=mzte_codec.SPlayer[color].coeffinfo;
	height=mzte_codec.SPlayer[color].height;
	width=mzte_codec.SPlayer[color].width;
      
	setProbModelsMQ(color);

	/* LH */
	n = 0;
	for (tw=mzte_codec.dcWidth; tw < width; tw<<=1)
	{
	  sh = h << n;
	  sw = (w+dc_w) << n;
	  /* begin: added by Rockwell (99/3/3) */
	  /* codeBlocks(sh,sw,n,&decode_pixel_MQ);*/
	  decodeBlocks(sh,sw,n,&decode_pixel_MQ,color);
	  /* end: added by Rockwell (99/3/3) */
	  n++;
	}
	/* HL */
	n = 0;
	for (tw=mzte_codec.dcWidth; tw < width; tw<<=1)
	{
	  sh = (h+dc_h) << n;
	  sw = w << n;
	  /* begin: added by Rockwell (99/3/3) */
	  /* codeBlocks(sh,sw,n,&decode_pixel_MQ); */
	  decodeBlocks(sh,sw,n,&decode_pixel_MQ,color);
	  /* end: added by Rockwell (99/3/3) */
	  n++;
	}
	/* HH */
	n = 0;
	for (tw=mzte_codec.dcWidth; tw < width; tw<<=1)
	{
	  sh = (h+dc_h) << n;
	  sw = (w+dc_w) << n;
	  /* begin: added by Rockwell (99/3/3) */
	  /* codeBlocks(sh,sw,n,&decode_pixel_MQ); */
	  decodeBlocks(sh,sw,n,&decode_pixel_MQ,color);
	  /* end: added by Rockwell (99/3/3) */
	  n++;
	}
      }
    }


  for (color=0; color<NCOL; ++color)
    probModelFreeMQ(color);
}



/**********************************************************************/
/***************    MQ TREE & BAND     ********************************/
/**********************************************************************/


/********************************************************
  Function Name
  -------------
  static Void decode_pixel_MQ(Int h,Int w)

  Arguments
  ---------
  Int h,Int w - position of a pixel in height and width
  
  Description
  -----------
  Decoding the type and/or value of a coefficient, a
  recursive function, multi quant mode.

  Functions Called
  ----------------
  mzte_ac_decode_symbol()
  mark_ZTR_D()
  mag_sign_decode_MQ()

  Return Value
  ------------
  None.

********************************************************/ 

static Void decode_pixel_MQ(Int h,Int w)
{
  Int zt_type;

  /*~~~~~~~~~~~~~~~~~ zerotree descendent ~~~~~~~~~~~~~~~~~~~*/  
  if(coeffinfo[h][w].type==ZTR_D)
    return;

  /*~~~~~~~~~~~~~~ decode zero tree symbol ~~~~~~~~~~~~~~~~~~*/
  if (IS_RESID(w,h,color))
  {
    zt_type=VAL; /* tmp assign. for next switch statement */
  }
  else
  {
    Int czt_type; /* what to put on bitstream */
    Int l;

    l=xy2wvtDecompLev(w,h);  

    zt_type = coeffinfo[h][w].type;
    if(coeffinfo[h][w].mask==1) {	/* added by D.-S.Cho, Samsung AIT (99/04/13) */
      switch(coeffinfo[h][w].state)
      {
	case S_INIT:
	  czt_type=mzte_ac_decode_symbol(&acd,acm_type[l][CONTEXT_INIT]);
	  coeffinfo[h][w].type=zt_type=czt_type;
	  break;
	case S_ZTR:
	  czt_type=mzte_ac_decode_symbol(&acd,acm_type[l][CONTEXT_ZTR]);
	  coeffinfo[h][w].type=zt_type=czt_type;
	  break;
	case S_ZTR_D:
	  czt_type=mzte_ac_decode_symbol(&acd,acm_type[l][CONTEXT_ZTR_D]);
	  coeffinfo[h][w].type=zt_type=czt_type;
	  break;
	case S_IZ:
	  czt_type=mzte_ac_decode_symbol(&acd,acm_type[l][CONTEXT_IZ]);
	  coeffinfo[h][w].type=zt_type = czt_type ? VAL : IZ;
	  break;
	case S_LINIT: 
	  czt_type=mzte_ac_decode_symbol(&acd,acm_type[l][CONTEXT_LINIT]);
	  coeffinfo[h][w].type=zt_type = czt_type ? VZTR : ZTR;
	  break;
	case S_LZTR:
	  czt_type=mzte_ac_decode_symbol(&acd,acm_type[l][CONTEXT_LZTR]);
	  coeffinfo[h][w].type=zt_type = czt_type ? VZTR : ZTR;
	  break;
	case S_LZTR_D:
	  czt_type=mzte_ac_decode_symbol(&acd,acm_type[l][CONTEXT_LZTR_D]);
	  coeffinfo[h][w].type=zt_type = czt_type ? VZTR : ZTR;
	  break;
	default:
	  errorHandler("Invalid state (%d) in multi-quant encoding.", 
		       coeffinfo[h][w].state);
      }
/* begin - added by D.-S.Cho, Samsung AIT (99/04/13) */
    }
    else /* treat out-node as isolated zero for decoding purpose */
    {
      switch(coeffinfo[h][w].state)
      {
      case S_INIT:
      case S_ZTR:
      case S_ZTR_D:
      case S_IZ:
        zt_type = coeffinfo[h][w].type = IZ;
        break;
      case S_LINIT:
      case S_LZTR:
      case S_LZTR_D:
        zt_type = coeffinfo[h][w].type = ZTR;
        break;
      default:
        errorHandler("Invalid state (%d) in multi-quant encoding.",
                     coeffinfo[h][w].state);
      }
    }
/* end  */
  }

  /*~~~~~~~~~~~~~~~~ mark ztr_d and encode magnitudes ~~~~~~~~~~~~~~~~~*/
  switch(zt_type)
  {
    case ZTR:
      if(coeffinfo[h][w].mask!=1) return;	/* added by D.-S.Cho, Samsung AIT (99/04/13) */
    case ZTR_D:
      mark_ZTR_D(h,w);
    case IZ:
      coeffinfo[h][w].quantized_value=0;
      return;
    case VZTR:
      mark_ZTR_D(h,w);
    case VAL:
      if(coeffinfo[h][w].mask==1)	/* added by D.-S.Cho, Samsung AIT (99/04/13) */
	mag_sign_decode_MQ(h,w);
      break;
    default:
      errorHandler("Invalid type in multi quant decoding.");     
  }
}



/********************************************************
  Function Name
  -------------
  static Void mark_ZTR_D(Int h,Int w)

  Arguments
  ---------
  Int h,Int w - position of a pixel

  
  Description
  -----------
  Mark the coefficient at (h,w) and its descendents as
  zerotree descendents. 

  Functions Called
  ----------------
  mark_ZTR_D()


  Return Value
  ------------
  None.

********************************************************/ 
static Void mark_ZTR_D(Int h,Int w)
{
  Int i,j;

  i=h<<1; j=w<<1;

  if(i<height && j<width)
  {
    coeffinfo[i][j].quantized_value     = 0;
    coeffinfo[i+1][j].quantized_value   = 0;
    coeffinfo[i][j+1].quantized_value   = 0;
    coeffinfo[i+1][j+1].quantized_value = 0;
    
    coeffinfo[i][j].type     = ZTR_D; 
    coeffinfo[i+1][j].type   = ZTR_D; 
    coeffinfo[i][j+1].type   = ZTR_D; 
    coeffinfo[i+1][j+1].type = ZTR_D; 
    
    mark_ZTR_D(i,j);
    mark_ZTR_D(i+1,j);
    mark_ZTR_D(i,j+1);
    mark_ZTR_D(i+1,j+1);
  }
}




/********************************************************
  Function Name
  -------------
  static Void  mag_sign_decode_MQ(Int h,Int w)

  Arguments
  ---------
  Int h,Int w - position of a pixel

  Description
  -----------
  Decode the value of a coefficient.

  Functions Called
  ----------------
  mzte_ac_decode_symbol()

  Return Value
  ------------
  None.

********************************************************/

static Void mag_sign_decode_MQ(Int h,Int w)
{
  Int val,v_sign;
  Int l;

  if(coeffinfo[h][w].skip)
  {
    coeffinfo[h][w].quantized_value=0;
    return;
  }
    
  l=xy2wvtDecompLev(w,h);


  if (IS_RESID(w,h,color))
  {
    val=bitplane_res_decode(l,WVTDECOMP_RES_NUMBITPLANES(color));
    coeffinfo[h][w].quantized_value=val;
  }
  else
  {
    val=bitplane_decode(l,WVTDECOMP_NUMBITPLANES(color,l))+1;
    v_sign=mzte_ac_decode_symbol(&acd,acm_sign[l]);
    coeffinfo[h][w].quantized_value=(v_sign==0) ? val : -val;
  }


}




/********************************************************
  Function Name
  -------------
  Void write_coeff_to_file(Char *fname,WVT_CODEC *mzte_codec)

  Arguments
  ---------
  Char *fname - output file name
  WVT_CODEC *mzte_codec - structure for the mzte coder.

  
  Description
  -----------
  write coeff position, value, type, state, and ac_model to
  a file in the band-wise order.

  Functions Called
  ----------------
  None.


  Return Value
  ------------
  None.

********************************************************/ 
Void write_coeff_to_file(Char *fname,WVT_CODEC *mzte_codec)
{
  Int h,w,ac_h,ac_w,ac_h2,ac_w2;
  File *fp;
  COEFFINFO **coeffinfo;

  coeffinfo=mzte_codec->SPlayer[color].coeffinfo;
  fp=fopen(fname,"w");

  ac_h=mzte_codec->dcHeight;
  ac_w=mzte_codec->dcWidth;
  ac_h2=ac_h<<1; ac_w2=ac_w<<1;
  for(h=0;h<ac_h;h++)
    for(w=0;w<ac_w;w++)
      fprintf(fp,"%4d,%4d: %6d %3d %3d\n",h,w,
        coeffinfo[h][w].quantized_value,coeffinfo[h][w].type,
        coeffinfo[h][w].state);

  while(ac_h2<=mzte_codec->Image[0].height && 
        ac_w2<=mzte_codec->Image[0].width){
    /* LH */
    for(h=0;h<ac_h;h++)
      for(w=ac_w;w<ac_w2;w++)
      fprintf(fp,"%4d,%4d: %6d %3d %3d\n",h,w,
        coeffinfo[h][w].quantized_value,coeffinfo[h][w].type,
        coeffinfo[h][w].state);

    /* HL */
    for(h=ac_h;h<ac_h2;h++)
      for(w=0;w<ac_w;w++)
      fprintf(fp,"%4d,%4d: %6d %3d %3d\n",h,w,
        coeffinfo[h][w].quantized_value,coeffinfo[h][w].type,
        coeffinfo[h][w].state);

    /* HH */
    for(h=ac_h;h<ac_h2;h++)
      for(w=ac_w;w<ac_w2;w++)
      fprintf(fp,"%4d,%4d: %6d %3d %3d\n",h,w,
        coeffinfo[h][w].quantized_value,coeffinfo[h][w].type,
        coeffinfo[h][w].state);

    /* update ranges */
    ac_h=ac_h2;
    ac_w=ac_w2;
    ac_w2<<=1;
    ac_h2 <<=1;
  }
  fclose(fp);
}

/* begin: added by Rockwell (99/3/3) */

/* ----------------------------------------------------------------- */
/* ----------------- Error resilience related routines ------------- */
/* ----------------------------------------------------------------- */

/* IM: VTC error res */ 
/* init all ac_decoder and models, bbc, 11/6/98 */
/* ph, 11/13/98 - added color argument for band-by-band */
static Void init_arith_decoder_model(Int color)
{
  if(init_ac!=0) /* check for not closed ac coder. bbc, 7/2/98 */
    errorHandler("didn't close arithmetic decoder before.");
  else
    init_ac=1;

   /* init arithmetic coder */
  mzte_ac_decoder_init(&acd);
  
  if(mzte_codec.scan_direction ==0 ){ /* tree depth */
    for (color=0; color<mzte_codec.colors; color++) 
      probModelInitSQ(color);
  }
  else {   /* band-by-band */
    probModelInitSQ(color); /* ph - 11/13/98 */
  }
  /* mzte_ac_model_init(&acmSGMK,2,ifreq,!ADAPT,1);  bbc, 11/20/98 */
}  

/* IM: VTC error res */ 
/* close all ac_decoder and models, bbc, 11/6/98 */
/* ph, 11/13/98 - added color argument for band-by-band */
static Void close_arith_decoder_model(Int color)
{
  if(init_ac==0) /* didn't init ac before. */
    return;
  else
    init_ac=0;

#if ST_ERROR_RES_DEBUG
  if(errSignal ==0)
    noteProgress("--- ST Error Res: Dec found packet at [TU_%d,TU_%d], l=%d bits ---",TU_first,
                 TU_last,packet_size-16);
#endif

  if(mzte_codec.scan_direction==0){ /* TD */
    for (color=0; color<mzte_codec.colors; color++) 
      /* close arithmetic coder */
      probModelFreeSQ(color);
  }
  else{    /* BB */
      probModelFreeSQ(color); /* ph - 11/13/98 */
  }

  mzte_ac_decoder_done(&acd);
}


/* IM: VTC error res */ 
/*****************************************************/
/* to check if a segment in a packet has exceeded a  */
/* threshold. Look for a marker if so. bbc, 11/6/98  */
/*****************************************************/
Int found_segment_error(Int col)
{

  /* segment not long enough, bbc, 11/16/98 */
  if(packet_size-16-prev_segs_size<(Int)mzte_codec.segment_thresh) 
    return 2;

#if ST_ERROR_RES_DEBUG
  noteProgress("\t--- ST Error Res: decode segment marker ---");
#endif

  prev_segs_size=packet_size-16;

  /* search for the marker, bbc, 11/10/98 */
  if(mzte_ac_decode_symbol(&acd,&acmType[col][0][CONTEXT_INIT])==ZTR)
    return 0;

  /* check for the correct marker "0111", bbc, 11/20/98 */
  /* v=mzte_ac_decode_symbol(&acd,&acmSGMK);
  v=(v<<1)+mzte_ac_decode_symbol(&acd,&acmSGMK);
  v=(v<<1)+mzte_ac_decode_symbol(&acd,&acmSGMK);
  v=(v<<1)+mzte_ac_decode_symbol(&acd,&acmSGMK);
  if(v == 7) 
    return 0; */

  prev_segs_size=0;
  return 1;
}


/* IM: VTC error res */ 
/* Check if a new packet will start, bbc, 11/9/98 */
static Void check_end_of_packet()
{
  if(LTU==TU_last){  /* reach the end of a packet */
    close_arith_decoder_model(color);
    align_byte();

    if(TU_last == TU_max){  /* successfully decoded last packet */
      if(end_of_stream())
         error_bits_stat(0);
      else{
        while(!end_of_stream())
          get_X_bits(8);
        rewind_bits(16);
        error_bits_stat(1);
      }
      return;
    }

    packet_size=0; 
    prev_segs_size=0;
    /* start new packet */

    CTU_no=get_err_resilience_header();
    LTU=CTU_no-1;
      
    get_TU_location(TU_first-1);
    if(mzte_codec.scan_direction==0)
    { 
      /* TD */
      if(TU_color==0)
        set_prev_good_TD_segment(TU_first-1,
                               ((start_h+1)<<(mzte_codec.wvtDecompLev-1))-1,
                               ((start_w+1)<<(mzte_codec.wvtDecompLev-1))-1);
      else
        set_prev_good_TD_segment(TU_first-1,
				 ((start_h+1)<<(mzte_codec.wvtDecompLev-2))-1,
				 ((start_w+1)<<(mzte_codec.wvtDecompLev-2))-1);
    }
    else
    {
      /* BB */

    }

    if(CTU_no>TU_max)
      return;
    if(get_X_bits(1) != 0) /* no repeated header info for now, bbc, 6/27/98 */
      /* errorHandler("Error in decoding HEC.") */;

    if (mzte_codec.scan_direction==0)
      init_arith_decoder_model(color);
    else
    {
      /* don't reinitialize if color change */
      if ((LTU-TU_max_dc+1) % mzte_codec.dcHeight != 0)
	init_arith_decoder_model(color);
    }
  }
}

/* end: added by Rockwell (99/3/3) */
