/************************************************************************/
/*                                                     			*/
/* This software module was originally developed by              	*/
/*                                                               	*/
/* Stefan Rauthenberg (HHI / ACTS-MoMuSys).     	              	*/
/*                                                               	*/
/* and edited by                                                	*/
/*                                                               	*/
/* Jan De Lameillieure (HHI / ACTS-MoMuSys).     	              	*/
/*                                                               	*/
/* in the course of development of the MPEG-4 Video (ISO/IEC 14496-2).	*/
/* This software module is an implementation of a part of one or 	*/
/* more MPEG-4 Video (ISO/IEC 14496-2) tools as specified by the        */
/* MPEG-4 Video (ISO/IEC 14496-2). ISO/IEC gives users of the MPEG-4    */
/* Video free license to this software module or modifications thereof 	*/
/* for use in hardware or software products claiming conformance to the */
/* MPEG-4 Video (ISO/IEC 14496-2). Those intending to use this software */
/* module in hardware or software products are advised that its use may */
/* infringe existing patents. The original developer of this software  	*/
/* module and his/her company, the subsequent editors and their     	*/
/* companies, and ISO/IEC have no liability for use of this software    */
/* module or modifications thereof in an implementation. Copyright is   */
/* not released for non MPEG-4 Video (ISO/IEC 14496-2) conforming 	*/
/* products. ACTS-MoMuSys partners retain full right to use  the code   */
/* for their own purposes, assign or donate the code to a third party   */
/* and to inhibit third parties from using the code for non MPEG-4    	*/
/* Video (ISO/IEC 14496-2) conforming products. This copyright notice 	*/
/* must be included in all copies or derivative works.                  */
/* Copyright (c)1997                                            	*/
/*                                                               	*/
/************************************************************************/

/***********************************************************HeaderBegin*******
 *                                                                         
 * File: sadct_nrutil.c 
 * 
 * Author: Stefan Rauthenberg (HHI)
 *
 * Created: 
 *                                                                         
 * Description: 
 *	functions from the freeware part of the numerical recipes
 *
 * Notes:  
 *
 * Modified: 
 *      14-MAR-97 Jan De Lameillieure (HHI) : by means of the variable
 *			RC_MATRIX_NOT_INCLUDED all functions already
 *			defined in rc_matrix.c have been excluded
 *      20-MAR-97 Jan De Lameillieure (HHI) : left out the functions not
 *			necessary for SADCT by means of variable
 *			FUNCTIONS_NOT_FOR_SADCT_NEEDED;
 *                      added this MoMuSys-header
 *      23-APR-97 Jan De Lameillieure (HHI) : reintroduced the functions
 *                      free_smatrix() and free_cmatrix()
 *      23-APR-97 Jan De Lameillieure (HHI) : free_dmatrix() is replaced by
 *                      free_dmatrix_sadct() and dmatrix() by dmatrix_sadct()
 *                      to prevent confusion with similar functions defined
 *                      in the sprite software
 *     16-JUN-97 Jan De Lameillieure : renaming some include files to sadct_*
 *
 ***********************************************************HeaderEnd*********/

/************************    INCLUDE FILES    ********************************/

#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>

#include "sadct_bsnrmem.h"
/* #include "bserror.h" */

#define NR_END 1
#define FREE_ARG Char*

#ifdef RC_MATRIX_NOT_INCLUDED
Void nrerror(Char error_text[])
/* Numerical Recipes standard error handler */
{
  fatal_exit("NR",
"Numerical Recipes run-time error...\n\
%s\n\
...now exiting to system...\n", error_text);

}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Float *fvector(LInt nl, LInt nh)
/* allocate a float vector with subscript range v[nl..nh] */
{
	Float *v;

	v=(Float *)malloc((size_t) ((nh-nl+1+NR_END)*sizeof(Float)));
	if (!v) nrerror("allocation failure in vector()");
	return v-nl+NR_END;
}
#endif

#ifdef RC_MATRIX_NOT_INCLUDED
Int *ivector(LInt nl, LInt nh)
/* allocate an Int vector with subscript range v[nl..nh] */
{
	Int *v;

	v=(Int *)malloc((size_t) ((nh-nl+1+NR_END)*sizeof(Int)));
	if (!v) nrerror("allocation failure in ivector()");
	return v-nl+NR_END;
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
SInt *svector(LInt nl, LInt nh)
/* allocate an Int vector with subscript range v[nl..nh] */
{
	SInt *v;

	v=(SInt *)malloc((size_t) ((nh-nl+1+NR_END)*sizeof(SInt)));
	if (!v) nrerror("allocation failure in svector()");
	return v-nl+NR_END;
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
UChar *cvector(LInt nl, LInt nh)
/* allocate an UChar vector with subscript range v[nl..nh] */
{
	UChar *v;

	v=(UChar *)malloc((size_t) ((nh-nl+1+NR_END)*sizeof(UChar)));
	if (!v) nrerror("allocation failure in cvector()");
	return v-nl+NR_END;
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
UInt *lvector(LInt nl, LInt nh)
/* allocate an unsigned long vector with subscript range v[nl..nh] */
{
	UInt *v;

	v=(UInt *)malloc((size_t) ((nh-nl+1+NR_END)*sizeof(LInt)));
	if (!v) nrerror("allocation failure in lvector()");
	return v-nl+NR_END;
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Double *dvector(LInt nl, LInt nh)
/* allocate a double vector with subscript range v[nl..nh] */
{
	Double *v;

	v=(Double *)malloc((size_t) ((nh-nl+1+NR_END)*sizeof(Double)));
	if (!v) nrerror("allocation failure in dvector()");
	return v-nl+NR_END;
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Float **fmatrix(LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* allocate a float matrix with subscript range m[nrl..nrh][ncl..nch] */
{
	LInt i, nrow=nrh-nrl+1,ncol=nch-ncl+1;
	Float **m;

	/* allocate pointers to rows */
	m=(Float **) malloc((size_t)((nrow+NR_END)*sizeof(Float*)));
	if (!m) nrerror("allocation failure 1 in fmatrix()");
	m += NR_END;
	m -= nrl;

	/* allocate rows and set pointers to them */
	m[nrl]=(Float *) malloc((size_t)((nrow*ncol+NR_END)*sizeof(Float)));
	if (!m[nrl]) nrerror("allocation failure 2 in fmatrix()");
	m[nrl] += NR_END;
	m[nrl] -= ncl;

	for(i=nrl+1;i<=nrh;i++) m[i]=m[i-1]+ncol;

	/* return pointer to array of pointers to rows */
	return m;
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Double **dmatrix(LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* allocate a double matrix with subscript range m[nrl..nrh][ncl..nch] */
{
	LInt i, nrow=nrh-nrl+1,ncol=nch-ncl+1;
	Double **m;

	/* allocate pointers to rows */
	m=(Double **) malloc((size_t)((nrow+NR_END)*sizeof(Double*)));
	if (!m) nrerror("allocation failure 1 in dmatrix()");
	m += NR_END;
	m -= nrl;

	/* allocate rows and set pointers to them */
	m[nrl]=(Double *) malloc((size_t)((nrow*ncol+NR_END)*sizeof(Double)));
	if (!m[nrl]) nrerror("allocation failure 2 in dmatrix()");
	m[nrl] += NR_END;
	m[nrl] -= ncl;

	for(i=nrl+1;i<=nrh;i++) m[i]=m[i-1]+ncol;

	/* return pointer to array of pointers to rows */
	return m;
}
#endif

Double **dmatrix_sadct(LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* allocate a double matrix with subscript range m[nrl..nrh][ncl..nch] */
{
	LInt i, nrow=nrh-nrl+1,ncol=nch-ncl+1;
	Double **m;

	/* allocate pointers to rows */
	m=(Double **) malloc((size_t)((nrow+NR_END)*sizeof(Double*)));
	if (!m) nrerror("allocation failure 1 in dmatrix()");
	m += NR_END;
	m -= nrl;

	/* allocate rows and set pointers to them */
	m[nrl]=(Double *) malloc((size_t)((nrow*ncol+NR_END)*sizeof(Double)));
	if (!m[nrl]) nrerror("allocation failure 2 in dmatrix()");
	m[nrl] += NR_END;
	m[nrl] -= ncl;

	for(i=nrl+1;i<=nrh;i++) m[i]=m[i-1]+ncol;

	/* return pointer to array of pointers to rows */
	return m;
}

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Int **imatrix(LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* allocate a Int matrix with subscript range m[nrl..nrh][ncl..nch] */
{
	LInt i, nrow=nrh-nrl+1,ncol=nch-ncl+1;
	Int **m;

	/* allocate pointers to rows */
	m=(Int **) malloc((size_t)((nrow+NR_END)*sizeof(Int*)));
	if (!m) nrerror("allocation failure 1 in smatrix()");
	m += NR_END;
	m -= nrl;


	/* allocate rows and set pointers to them */
	m[nrl]=(Int *) malloc((size_t)((nrow*ncol+NR_END)*sizeof(Int)));
	if (!m[nrl]) nrerror("allocation failure 2 in smatrix()");
	m[nrl] += NR_END;
	m[nrl] -= ncl;

	for(i=nrl+1;i<=nrh;i++) m[i]=m[i-1]+ncol;

	/* return pointer to array of pointers to rows */
	return m;
}
#endif

SInt **smatrix(LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* allocate a Int matrix with subscript range m[nrl..nrh][ncl..nch] */
{
	LInt i, nrow=nrh-nrl+1,ncol=nch-ncl+1;
	SInt **m;

	/* allocate pointers to rows */
	m=(SInt **) malloc((size_t)((nrow+NR_END)*sizeof(SInt*)));
	if (!m) nrerror("allocation failure 1 in smatrix()");
	m += NR_END;
	m -= nrl;


	/* allocate rows and set pointers to them */
	m[nrl]=(SInt *) malloc((size_t)((nrow*ncol+NR_END)*sizeof(SInt)));
	if (!m[nrl]) nrerror("allocation failure 2 in smatrix()");
	m[nrl] += NR_END;
	m[nrl] -= ncl;

	for(i=nrl+1;i<=nrh;i++) m[i]=m[i-1]+ncol;

	/* return pointer to array of pointers to rows */
	return m;
}

UChar **cmatrix(LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* allocate a UChar matrix with subscript range m[nrl..nrh][ncl..nch] */
{
  LInt i, nrow=nrh-nrl+1,ncol=nch-ncl+1;
  UChar **m;

  /* allocate pointers to rows */
  m=(UChar **) malloc((size_t)((nrow+NR_END)*sizeof(UChar*)));
  if (!m) nrerror("allocation failure 1 in cmatrix()");
  m += NR_END;
  m -= nrl;


  /* allocate rows and set pointers to them */
  m[nrl]=(UChar *) malloc((size_t)((nrow*ncol+NR_END)*
					   sizeof(UChar)));
  if (!m[nrl]) nrerror("allocation failure 2 in cmatrix()");
  m[nrl] += NR_END;
  m[nrl] -= ncl;

  for(i=nrl+1;i<=nrh;i++) m[i]=m[i-1]+ncol;

  /* return pointer to array of pointers to rows */
  return m;
}

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Float **submatrix(Float **a, LInt oldrl, LInt oldrh, LInt oldcl, LInt oldch,
	LInt newrl, LInt newcl)
/* point a submatrix [newrl..][newcl..] to a[oldrl..oldrh][oldcl..oldch] */
{
        LInt i,j,nrow=oldrh-oldrl+1,ncol=oldcl-newcl;
	Float **m;

	/* allocate array of pointers to rows */
	m=(Float **) malloc((size_t) ((nrow+NR_END)*sizeof(Float*)));
	if (!m) nrerror("allocation failure in submatrix()");
	m += NR_END;
	m -= newrl;

	/* set pointers to rows */
	for(i=oldrl,j=newrl;i<=oldrh;i++,j++) m[j]=a[i]+ncol;

	/* return pointer to array of pointers to rows */
	return m;
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Float **convert_fmatrix(Float *a, LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* allocate a float matrix m[nrl..nrh][ncl..nch] that points to the matrix
declared in the standard C manner as a[nrow][ncol], where nrow=nrh-nrl+1
and ncol=nch-ncl+1. The routine should be called with the address
&a[0][0] as the first argument. */
{
	LInt i,j,nrow=nrh-nrl+1,ncol=nch-ncl+1;
	Float **m;

	/* allocate pointers to rows */
	m=(Float **) malloc((size_t) ((nrow+NR_END)*sizeof(Float*)));
	if (!m) nrerror("allocation failure in convert_fmatrix()");
	m += NR_END;
	m -= nrl;

	/* set pointers to rows */
	m[nrl]=a-ncl;
	for(i=1,j=nrl+1;i<nrow;i++,j++) m[j]=m[j-1]+ncol;
	/* return pointer to array of pointers to rows */
	return m;
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Double **convert_dmatrix(Double *a, LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* allocate a double matrix m[nrl..nrh][ncl..nch] that points to the matrix
declared in the standard C manner as a[nrow][ncol], where nrow=nrh-nrl+1
and ncol=nch-ncl+1. The routine should be called with the address
&a[0][0] as the first argument. */
{
	LInt i,j,nrow=nrh-nrl+1,ncol=nch-ncl+1;
	Double **m;

	/* allocate pointers to rows */
	m=(Double **) malloc((size_t) ((nrow+NR_END)*sizeof(Double*)));
	if (!m) nrerror("allocation failure in convert_dmatrix()");
	m += NR_END;
	m -= nrl;

	/* set pointers to rows */
	m[nrl]=a-ncl;
	for(i=1,j=nrl+1;i<nrow;i++,j++) m[j]=m[j-1]+ncol;
	/* return pointer to array of pointers to rows */
	return m;
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Int **convert_imatrix(Int *a, LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* allocate a Int matrix m[nrl..nrh][ncl..nch] that points to the matrix
declared in the standard C manner as a[nrow][ncol], where nrow=nrh-nrl+1
and ncol=nch-ncl+1. The routine should be called with the address
&a[0][0] as the first argument. */
{
	LInt i,j,nrow=nrh-nrl+1,ncol=nch-ncl+1;
	Int **m;

	/* allocate pointers to rows */
	m=(Int **) malloc((size_t) ((nrow+NR_END)*sizeof(Int*)));
	if (!m) nrerror("allocation failure in convert_imatrix()");
	m += NR_END;
	m -= nrl;

	/* set pointers to rows */
	m[nrl]=a-ncl;
	for(i=1,j=nrl+1;i<nrow;i++,j++) m[j]=m[j-1]+ncol;
	/* return pointer to array of pointers to rows */
	return m;
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
SInt **convert_smatrix(SInt *a, LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* allocate a short matrix m[nrl..nrh][ncl..nch] that points to the matrix
declared in the standard C manner as a[nrow][ncol], where nrow=nrh-nrl+1
and ncol=nch-ncl+1. The routine should be called with the address
&a[0][0] as the first argument. */
{
	LInt i,j,nrow=nrh-nrl+1,ncol=nch-ncl+1;
	SInt **m;

	/* allocate pointers to rows */
	m=(SInt **) malloc((size_t) ((nrow+NR_END)*sizeof(SInt*)));
	if (!m) nrerror("allocation failure in convert_smatrix()");
	m += NR_END;
	m -= nrl;

	/* set pointers to rows */
	m[nrl]=a-ncl;
	for(i=1,j=nrl+1;i<nrow;i++,j++) m[j]=m[j-1]+ncol;
	/* return pointer to array of pointers to rows */
	return m;
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
UChar **convert_cmatrix(UChar *a, LInt nrl, LInt nrh, 
				LInt ncl, LInt nch)
/* allocate a unsigned char  matrix m[nrl..nrh][ncl..nch] that points to 
the matrix declared in the standard C manner as a[nrow][ncol], 
where nrow=nrh-nrl+1 and ncol=nch-ncl+1. The routine should be called 
with the address &a[0][0] as the first argument. */
{
	LInt i,j,nrow=nrh-nrl+1,ncol=nch-ncl+1;
	UChar **m;

	/* allocate pointers to rows */
	m=(UChar **) malloc((size_t) ((nrow+NR_END)
					      *sizeof(UChar*)));
	if (!m) nrerror("allocation failure in convert_cmatrix()");
	m += NR_END;
	m -= nrl;

	/* set pointers to rows */
	m[nrl]=a-ncl;
	for(i=1,j=nrl+1;i<nrow;i++,j++) m[j]=m[j-1]+ncol;
	/* return pointer to array of pointers to rows */
	return m;
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Float ***f3tensor(LInt nrl, LInt nrh, LInt ncl, LInt nch, LInt ndl, LInt ndh)
/* allocate a float 3tensor with range t[nrl..nrh][ncl..nch][ndl..ndh] */
{
	LInt i,j,nrow=nrh-nrl+1,ncol=nch-ncl+1,ndep=ndh-ndl+1;
	Float ***t;

	/* allocate pointers to pointers to rows */
	t=(Float ***) malloc((size_t)((nrow+NR_END)*sizeof(Float**)));
	if (!t) nrerror("allocation failure 1 in f3tensor()");
	t += NR_END;
	t -= nrl;

	/* allocate pointers to rows and set pointers to them */
	t[nrl]=(Float **) malloc((size_t)((nrow*ncol+NR_END)*sizeof(Float*)));
	if (!t[nrl]) nrerror("allocation failure 2 in f3tensor()");
	t[nrl] += NR_END;
	t[nrl] -= ncl;

	/* allocate rows and set pointers to them */
	t[nrl][ncl]=(Float *) malloc((size_t)((nrow*ncol*ndep+NR_END)*sizeof(Float)));
	if (!t[nrl][ncl]) nrerror("allocation failure 3 in f3tensor()");
	t[nrl][ncl] += NR_END;
	t[nrl][ncl] -= ndl;

	for(j=ncl+1;j<=nch;j++) t[nrl][j]=t[nrl][j-1]+ndep;
	for(i=nrl+1;i<=nrh;i++) {
		t[i]=t[i-1]+ncol;
		t[i][ncl]=t[i-1][ncl]+ncol*ndep;
		for(j=ncl+1;j<=nch;j++) t[i][j]=t[i][j-1]+ndep;
	}

	/* return pointer to array of pointers to rows */
	return t;
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Void free_fvector(Float *v, LInt nl, LInt nh)
/* free a float vector allocated with vector() */
{
	free((FREE_ARG) (v+nl-NR_END));
}
#endif

#ifdef RC_MATRIX_NOT_INCLUDED
Void free_dvector(Double *v, LInt nl, LInt nh)
/* free a double vector allocated with dvector() */
{
	free((FREE_ARG) (v+nl-NR_END));
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Void free_lvector(UInt *v, LInt nl, LInt nh)
/* free an unsigned long vector allocated with lvector() */
{
	free((FREE_ARG) (v+nl-NR_END));
}
#endif

#ifdef RC_MATRIX_NOT_INCLUDED
Void free_ivector(Int *v, LInt nl, LInt nh)
/* free an Int vector allocated with ivector() */
{
	free((FREE_ARG) (v+nl-NR_END));
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Void free_svector(SInt *v, LInt nl, LInt nh)
/* free an Int vector allocated with svector() */
{
	free((FREE_ARG) (v+nl-NR_END));
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Void free_cvector(UChar *v, LInt nl, LInt nh)
/* free an unsigned char vector allocated with cvector() */
{
	free((FREE_ARG) (v+nl-NR_END));
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Void free_fmatrix(Float **m, LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* free a float matrix allocated by matrix() */
{
	free((FREE_ARG) (m[nrl]+ncl-NR_END));
	free((FREE_ARG) (m+nrl-NR_END));
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Void free_dmatrix(Double **m, LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* free a double matrix allocated by dmatrix() */
{
	free((FREE_ARG) (m[nrl]+ncl-NR_END));
	free((FREE_ARG) (m+nrl-NR_END));
}
#endif

Void free_dmatrix_sadct(Double **m, LInt nrl, LInt nrh, LInt ncl, LInt nch)
{
	free((FREE_ARG) (m[nrl]+ncl-NR_END));
	free((FREE_ARG) (m+nrl-NR_END));
}

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Void free_imatrix(Int **m, LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* free an Int matrix allocated by imatrix() */
{
	free((FREE_ARG) (m[nrl]+ncl-NR_END));
	free((FREE_ARG) (m+nrl-NR_END));
}
#endif

Void free_smatrix(SInt **m, LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* free an Int matrix allocated by smatrix() */
{
	free((FREE_ARG) (m[nrl]+ncl-NR_END));
	free((FREE_ARG) (m+nrl-NR_END));
}

Void free_cmatrix(UChar **m, LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* free an Int matrix allocated by smatrix() */
{
	free((FREE_ARG) (m[nrl]+ncl-NR_END));
	free((FREE_ARG) (m+nrl-NR_END));
}

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Void free_submatrix(Float **b, LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* free a submatrix allocated by submatrix() */
{
	free((FREE_ARG) (b+nrl-NR_END));
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Void free_convert_fmatrix(Float **b, LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* free a matrix allocated by convert_fmatrix() */
{
	free((FREE_ARG) (b+nrl-NR_END));
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Void free_convert_dmatrix(Double **b, LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* free a matrix allocated by convert_dmatrix() */
{
	free((FREE_ARG) (b+nrl-NR_END));
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Void free_convert_imatrix(Int **b, LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* free a matrix allocated by convert_matrix() */
{
	free((FREE_ARG) (b+nrl-NR_END));
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Void free_convert_smatrix(SInt **b, LInt nrl, LInt nrh, LInt ncl, LInt nch)
/* free a matrix allocated by convert_smatrix() */
{
	free((FREE_ARG) (b+nrl-NR_END));
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Void free_convert_cmatrix(UChar **b, LInt nrl, LInt nrh, LInt ncl, 
			  LInt nch)
/* free a matrix allocated by convert_cmatrix() */
{
	free((FREE_ARG) (b+nrl-NR_END));
}
#endif

#ifdef FUNCTIONS_NOT_FOR_SADCT_NEEDED
Void free_f3tensor(Float ***t, LInt nrl, LInt nrh, LInt ncl, LInt nch,
	LInt ndl, LInt ndh)
/* free a float f3tensor allocated by f3tensor() */
{
	free((FREE_ARG) (t[nrl][ncl]+ndl-NR_END));
	free((FREE_ARG) (t[nrl]+ncl-NR_END));
	free((FREE_ARG) (t+nrl-NR_END));
}
#endif

