/* $Id: quant.c,v 1.15 1998/05/14 21:41:40 hjlee Exp $ */
/****************************************************************************/
/*   MPEG4 Visual Texture Coding (VTC) Mode Software                        */
/*                                                                          */
/*   This software was jointly developed by the following participants:     */
/*                                                                          */
/*   Single-quant,  multi-quant and flow control                            */
/*   are provided by  Sarnoff Corporation                                   */
/*     Iraj Sodagar   (iraj@sarnoff.com)                                    */
/*     Hung-Ju Lee    (hjlee@sarnoff.com)                                   */
/*     Paul Hatrack   (hatrack@sarnoff.com)                                 */
/*     Shipeng Li     (shipeng@sarnoff.com)                                 */
/*     Bing-Bing Chai (bchai@sarnoff.com)                                   */
/*     B.S. Srinivas  (bsrinivas@sarnoff.com)                               */
/*                                                                          */
/*   Bi-level is provided by Texas Instruments                              */
/*     Jie Liang      (liang@ti.com)                                        */
/*                                                                          */
/*   Shape Coding is provided by  OKI Electric Industry Co., Ltd.           */
/*     Zhixiong Wu    (sgo@hlabs.oki.co.jp)                                 */
/*     Yoshihiro Ueda (yueda@hlabs.oki.co.jp)                               */
/*     Toshifumi Kanamaru (kanamaru@hlabs.oki.co.jp)                        */
/*                                                                          */
/*   OKI, Sharp, Sarnoff, TI and Microsoft contributed to bitstream         */
/*   exchange and bug fixing.                                               */
/*                                                                          */
/*                                                                          */
/* In the course of development of the MPEG-4 standard, this software       */
/* module is an implementation of a part of one or more MPEG-4 tools as     */
/* specified by the MPEG-4 standard.                                        */
/*                                                                          */
/* The copyright of this software belongs to ISO/IEC. ISO/IEC gives use     */
/* of the MPEG-4 standard free license to use this  software module or      */
/* modifications thereof for hardware or software products claiming         */
/* conformance to the MPEG-4 standard.                                      */
/*                                                                          */
/* Those intending to use this software module in hardware or software      */
/* products are advised that use may infringe existing  patents. The        */
/* original developers of this software module and their companies, the     */
/* subsequent editors and their companies, and ISO/IEC have no liability    */
/* and ISO/IEC have no liability for use of this software module or         */
/* modification thereof in an implementation.                               */
/*                                                                          */
/* Permission is granted to MPEG members to use, copy, modify,              */
/* and distribute the software modules ( or portions thereof )              */
/* for standardization activity within ISO/IEC JTC1/SC29/WG11.              */
/*                                                                          */
/* Copyright 1995, 1996, 1997, 1998 ISO/IEC                                 */
/****************************************************************************/

/************************************************************/
/*     Sarnoff Very Low Bit Rate Still Image Coder          */
/*     Copyright 1995, 1996, 1997, 1998 Sarnoff Corporation */
/************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include "typedef.h"
#include "quant.h"


/* sign */
#define SGN(x) (((x)<0) ? -1 : 1)
/* Integer rounding */
#define IROUND(n,d) ( ((n)/(d)) + (((n)%(d) > (d-1)/2) || (n)<(d))) 
/* Integer ceiling */
#define ICEIL(n,d) ( ((n)/(d)) + ((n)%(d)!=0 || (n)<(d)) ) 

/* For partitionType bit field processing */
#define MASK_fromReduced  ((UChar)'\1')
#define MASK_fromDeadZone ((UChar)'\2')

#define SET_fromReduced(x)  ((x) |= MASK_fromReduced)
#define SET_fromDeadZone(x) ((x) |= MASK_fromDeadZone)

#define CLR_fromReduced(x)  ((x) &= ~(MASK_fromReduced))
#define CLR_fromDeadZone(x) ((x) &= ~(MASK_fromDeadZone))

#define fromReduced(x)  ((x) & MASK_fromReduced)
#define fromDeadZone(x) (((x) & MASK_fromDeadZone)>>1)


/*------------------------- QUANTIZATION --------------------------*/


/* 
   Function:
   ---------
   initQuantSingleStage - Initialization of the single-stage quantizer for
     a given value.

   Arguments:
   ----------
   quantState *state - PoInter to the state data structure.
   Int *statePrevQ - PoInter to previous quantized value state.
   Int initialValue - Value which is to be quantized.

   Return Value:
   -------------
   <None>

   Description:
   ------------
   This must be called prior to single-stage quantization. A seperate
   state structure must be kept for each value that is quantized in parallel.
   Single-stage quantization is just successive calls to quantSingleStage.
   For each value,  need only be called once, before the first call to 
   quantSingleStage.
*/

Void initQuantSingleStage(quantState *state, Int *statePrevQ, Int initialVal)
{
  state->residualValue = initialVal;
  state->partitionType = 0x2; /* fromReduced = 0 and fromDeadZone  = 1 */
  *statePrevQ          = 0;
}

/* 
   Function:
   ---------
   quantSingleStage - Single-stage quantizer.

   Arguments:
   ----------
   Int Q - Quantization value. Represents desired quantization level size.
   quantState *state - State of quantizer.
   Int *statePrevQ - PoInter to previous quantized value state.
   Int updatePrevQ - 0 means don't update the statePrevQ variable. !0 means
     update it.

   Return Value:
   -------------
   New Q index.

   Description:
   ------------
   initQuantSingleStage must be called prior to using this function the
   first time for a given value. It will compute the new quantization index
   based on the current state associated with the value.
*/

Int quantSingleStage(Int Q, quantState *state, Int *statePrevQ,
		     Int updatePrevQ)
{
  Int refLevs;    /* how many refinement levels in new stage */
  Int QIndex;     /* new quantization index to return for this stage */

  /*--------------- INITIAL QUANTIZATION STAGE -------------------*/
  if (*statePrevQ==0)
  {
    QIndex = state->residualValue/Q;

    /* update state */
    if (QIndex)
      state->residualValue = abs(state->residualValue) - (abs(QIndex)*Q);
    CLR_fromReduced(state->partitionType);
    if (QIndex)
      CLR_fromDeadZone(state->partitionType);
    else
      SET_fromDeadZone(state->partitionType);
 
    if (updatePrevQ)
      *statePrevQ = Q;

    return QIndex;
  }


  /*--------------- NON-INITIAL QUANTIZATION STAGES -------------------*/

  /* get the number of refinement levels from lastQUsed state */
  refLevs = IROUND(*statePrevQ,Q);

  /* Catch condition where there's no refinement being done.
     State information is not changed.
  */
  if (refLevs<=1)
    QIndex = 0;
  else
  {
    Int inDeadZone;  /* are we still in the dead zone */
    Int lastQUsed;   /* The "real" Q value last used */
    Int val;         /* value of number to be quantized */
    Int lastLevSize; /* Size of quantization level used last */
    Int newQUsed, newStateQ;
    Int excess;

    /* Initialize the last quant value used */
    lastQUsed = *statePrevQ;
    
    /* update new Q value state */
    newStateQ = newQUsed = ICEIL(lastQUsed,refLevs);
    if (updatePrevQ)
      *statePrevQ = newStateQ;

    /* Get last level size */
    lastLevSize = lastQUsed-fromReduced(state->partitionType);

    /* check if a reduced level can span the last level */
    if (refLevs*(newQUsed-1) >= lastLevSize)
    {
      --newQUsed;

      /* might overshoot (?) but can't reduce anymore */
      excess=0;
#if 0
      if (lastLevSize-refLevs*newQUsed)
	fprintf(stderr,"Excess in reduced partition\n");
#endif
    }
    else
      /* get excess (overshoot) */
      excess=lastLevSize-refLevs*newQUsed;

    /* Set dead zone indicator */
    inDeadZone = fromDeadZone(state->partitionType);
    
    /* Set value of leftovers from last pass */
    val=state->residualValue;

    /*--- Calculate QIndex. Update residualValue and fromReduced states ---*/
    if (excess==0)
    {
      QIndex = val/newQUsed;
      if (newQUsed < newStateQ)
	SET_fromReduced(state->partitionType);
      else
	CLR_fromReduced(state->partitionType);

      
      if (QIndex)
	state->residualValue -= QIndex*newQUsed;
    }
    else
    {
      Int reducedParStart; /* Where the reduced partition starts (magnitude)
			    */
      Int reducedIdx;

      reducedParStart = newQUsed*(refLevs+excess);
      if (abs(val) >= reducedParStart)
      {
	SET_fromReduced(state->partitionType);
	QIndex = SGN(state->residualValue)*(refLevs+excess);
	state->residualValue -= QIndex*newQUsed;

	--newQUsed;
	reducedIdx = 
	  SGN(state->residualValue) * (abs(val)-reducedParStart) / newQUsed;
	QIndex += reducedIdx;
	state->residualValue -= reducedIdx*newQUsed;
      }
      else
      {
	CLR_fromReduced(state->partitionType);
	QIndex = val/newQUsed;
	state->residualValue -= QIndex*newQUsed;
      }
    }
    
    if (inDeadZone && QIndex)
    {
      /* We have the sign info so all residual values from here on in are 
	 positive */
      state->residualValue = abs(state->residualValue);
      CLR_fromDeadZone(state->partitionType);
    }
  }

  return QIndex;
}

#if 0 
/* 
   Function:
   ---------
   quant - Complete quantization of a value.

   Arguments:
   ----------
   Int *QVals - PoInter to array where quantized values are to be stored.
   Int val - The value to be quantized.
   Int numQ - The number of quantization stages.
   Int *QList - The quantization values to be used for quantization. These
     will be modified if not multiples of one another.

   Return Value:
   -------------
   <none>

   Description:
   ------------
   quant will put the sequence of quantized values for the passed value,
   val, in the array poInted to by QVals. The quantization is based on the
   numQ Q values in QList.
*/

Void quant(Int *QVals, Int val, Int numQ, Int *QList)
{
  Int curIdx;
  quantState state;
  Int prevQ;

  /* initialize state */
  initQuantSingleStage(&state, &prevQ, val);
  for (curIdx = 0; curIdx < numQ; ++curIdx)
    QVals[curIdx]=quantSingleStage(QList[curIdx], &state, &prevQ, 1);
}
#endif


/*--------------- INVERSE QUANTIZATION --------------------------*/

/* 
   Function:
   ---------
   initInvQuantSingleStage - Initialization of the single-stage inverse
     quantizer for a given Q index.

   Arguments:
   ----------
   quantState *state - PoInter to the state data structure.
   Int *statePrevQ - PoInter to previous quantized value state.

   Return Value:
   -------------
   <None>

   Description:
   ------------
   This must be called prior to single-stage inverse quantization. A seperate
   state structure must be kept for each value that is quantized in parallel.
   Single-stage inverse quantization is just successive calls to 
   invQuantSingleStage. For each value,  need only be called once, before
   the first call to invQuantSingleStage.
*/

Void initInvQuantSingleStage(quantState *state, Int *statePrevQ)
{
  state->residualValue = 0;
  state->partitionType = 0x2; /* fromReduced = 0 and fromDeadZone  = 1 */
  *statePrevQ          = 0;
}

/* Middle of quantization level */

#define QLEVMID(q) (((q))/2)  /* 1124 */


/* Mapping of start of quantization level, sign, and quantization level size
   to specific value.
*/
#define GETVAL(start, sgn, q) ((start) \
			       ? ((start) + (sgn)*QLEVMID((q))) \
			       : 0)

/* 
   Function:
   ---------
   invQuantSingleStage - Single-stage inverse quantizer.

   Arguments:
   ----------
   QIndex - Quantized value for this stage.
   Q     - Quantization value. Represents desired quantization level size.
   state - State of quantizer.
   Int *statePrevQ - PoInter to previous quantized value state.
   Int updatePrevQ - 0 means don't update the statePrevQ variable. !0 means
     update it.

   Return Value:
   -------------
   Inverse quantization value for this stage.

   Description:
   ------------
   initInvQuantSingleStage must be called prior to using this function the
   first time for a given value. It will compute the new, updated value
   based on the current index and state associated with the value.
*/

Int invQuantSingleStage(Int QIndex, Int Q, quantState *state, Int *statePrevQ,
			Int updatePrevQ)
{
  Int refLevs; /* how many refinement levels in new stage */
  Int val;     /* new value to return for this stage */
  Int sgn;

  /*--------------- INITIAL QUANTIZATION STAGE -------------------*/
  if (*statePrevQ==0)
  {
    val = QIndex*Q + SGN(QIndex)*(QIndex ? QLEVMID(Q) : 0);

    /* update state */
    state->residualValue = QIndex*Q;
    CLR_fromReduced(state->partitionType);
    if (QIndex)
      CLR_fromDeadZone(state->partitionType);
    else
      SET_fromDeadZone(state->partitionType);

    if (updatePrevQ)
      *statePrevQ = Q;

    return val;
  }


  /*--------------- NON-INITIAL QUANTIZATION STAGES -------------------*/

  /* get the number of refinement levels from lastQUsed state */
  refLevs = IROUND(*statePrevQ,Q);

  /* Catch condition where there's no refinement being done.
     State information is not changed.
  */
  sgn = (state->residualValue < 0 || QIndex < 0) ? -1 : 1;      
  if (refLevs<=1)
    val = GETVAL(state->residualValue,sgn,*statePrevQ);
  else
  {
    Int inDeadZone;  /* are we still in the dead zone */
    Int lastQUsed;   /* The "real" Q value last used */
    Int lastLevSize; /* Size of quantization level used last */
    Int newQUsed, newStateQ;
    Int excess;
    Int absQIndex;

    /* Initialize the last quant value used */
    lastQUsed = *statePrevQ;
    
    /* update new Q value state */
    newStateQ = newQUsed = ICEIL(lastQUsed,refLevs);
    if (updatePrevQ)
      *statePrevQ = newStateQ;

    /* Get last level size */
    lastLevSize = lastQUsed-fromReduced(state->partitionType);

    /* check if a reduced level can span the last level */
    if (refLevs*(newQUsed-1) >= lastLevSize)
    {
      --newQUsed;

      /* might overshoot (?) but can't reduce anymore */
      excess=0;
#if 1
      if (lastLevSize-refLevs*newQUsed)
	fprintf(stderr,"Excess in reduced partition\n");
#endif
    }
    else
      /* get excess (overshoot) */
      excess=lastLevSize-refLevs*newQUsed;

    /* Set dead zone indicator */
    inDeadZone = fromDeadZone(state->partitionType);

    /*--- Calculate val. Update residualValue and fromReduced states ---*/
    absQIndex = abs(QIndex);
    if (excess==0)
    {
      if (newQUsed < newStateQ)
	SET_fromReduced(state->partitionType);
      else
	CLR_fromReduced(state->partitionType);
      state->residualValue += sgn*absQIndex*newQUsed;
    }
    else
    {
      Int reducedIdx;
      Int fullLevs;

      fullLevs = refLevs+excess;
      if (absQIndex >= fullLevs)
      {
	SET_fromReduced(state->partitionType);
	state->residualValue += sgn*fullLevs*newQUsed;

	--newQUsed;
	
	reducedIdx = absQIndex-fullLevs;
	state->residualValue += sgn*reducedIdx*newQUsed;
      }
      else
      {
	CLR_fromReduced(state->partitionType);
	state->residualValue += sgn*absQIndex*newQUsed;
      }
    }

    val = GETVAL(state->residualValue, sgn, newQUsed);
    
    if (inDeadZone && QIndex)
      CLR_fromDeadZone(state->partitionType);
  }

  return val;
}

#if 0
/* 
   Function:
   ---------
   invQuant - Complete inverse quantization of a value.

   Arguments:
   ----------
   Int *vals - PoInter to array where de-quantized values are to be stored.
   Int numQ - The number of quantization stages.
   Int *QVals - PoInter to array where all the quantized values are stored.
   Int *QList - The quantization values to be used for quantization. These
     will be modified if not multiples of one another.

   Return Value:
   -------------
   <none>

   Description:
   ------------
   invQuant will put the sequence of de-quantized values for the passed 
   sequence of quantized indices (values) value, QVals, in the array poInted
   to by vals. The inverse quantization is based on the numQ Q values in QList.
*/

Void invQuant(Int *vals, Int numQ, Int *QVals, Int *QList)
{
  Int curIdx;
  quantState state;
  Int prevQ;

  /* initialize state */
  initInvQuantSingleStage(&state, &prevQ);
  for (curIdx = 0; curIdx < numQ; ++curIdx)
    vals[curIdx]=invQuantSingleStage(QVals[curIdx], QList[curIdx], &state,
				     &prevQ, 1);
}
#endif


/*--------- DERIVED QUANTIZATION VALUES AND REFINEMENT LEVELS ----------*/


/* 
   Function:
   ---------
   quantRefLev - Get the number of quantization levels for a given stage and
     the revised Q value.

   Arguments:
   ----------
   Int curQ - Current input Q value.
   Int *lastQUsed - last, revised, Q value. Will be updated.
   Int whichQ - Quantization stage.

   Return Value:
   -------------
   The number of refinement quantization levels.

   Description:
   ------------
   quantRefLev will return the number of refinement quantization levels
   at stage whichQ based on the last Q value used and the current input Q
   value. It will also update the revised Q value by overwriting the argument
   lastQUsed.
*/

Int quantRefLev(Int curQ, Int *lastQUsed, Int whichQ)
{
  Int refLevs;
  Int newQUsed;

  /* get the number of refinement levels */
  refLevs = IROUND(*lastQUsed,curQ);
    
  if (whichQ==0 || refLevs>1)
  {
    /* get new level size */
    newQUsed = ICEIL(*lastQUsed,refLevs);

    /* update the last quant value used */
    *lastQUsed = newQUsed;
  }

  return refLevs;
}

#if 0
/* 
   Function:
   ---------
   quantRevisedQList - Get the revised Q value list for all quantization
     stages.

   Arguments:
   ----------
   Int *QList - Input list of Q values.
   Int numQ - Number of Q values in QList (number of quantization stages).

   Return Value:
   -------------
   PoInter to allocated memory containing numQ revised Q values.

   Description:
   ------------
   quantRevisedQList takes in the input Q values list and returns the 
   revised list. Memory is allocated Internally for the revised list.
*/

Int *quantRevisedQList(Int *QList, Int numQ)
{
  Int lastQUsed;
  Int *revQList;
  Int i;

  if ((revQList = (Int *)malloc(numQ*sizeof(Int))) != NULL)
  {
    lastQUsed=QList[0];
    for (i=0; i<numQ; ++i)
    {
      quantRefLev(QList[i], &lastQUsed, i);
      revQList[i]=lastQUsed;
    }
  }

  return revQList;
} 


/* 
   Function:
   ---------
   quantRefinementLevelList - Get the number of refinement levels for each
     of the quantization stages.

   Arguments:
   ----------
   Int *QList - Input list of Q values.
   Int numQ - Number of Q values in QList (number of quantization stages).

   Return Value:
   -------------
   PoInter to allocated memory containing numQ refinement level values.

   Description:
   ------------
   quantRefinementLevelList takes in the input Q values list and returns
   the number of refinement levels for each stage after revision. Memory is 
   allocated Internally for the list.
*/

Int *quantRefinementLevelList(Int *QList, Int numQ)
{
  Int lastQUsed;
  Int *refLevList;
  Int i;

  if ((refLevList = (Int *)malloc(numQ*sizeof(Int))) != NULL)
  {
    lastQUsed=QList[0];
    for (i=0; i<numQ; ++i)
      refLevList[i]=quantRefLev(QList[i], &lastQUsed, i);
  }

  return refLevList;
} 
#endif

/*-------------- FOR TESTING AS STAND-ALONE PROGRAM ------------------*/

#if 0

/*--------------  Max/Min and Match Test for Range --------------*/

main()
{
  /* Example QLists:

    {70, 30, 1};    
    {16, 8, 4, 2, 1};
    {97, 43, 17, 1};
    {100, 20, 200, 100, 50, 25, 10, 5, 2, 1};
    {37, 25 , 18, 15, 12, 25, 16, 12, 11, 9, 9, 8, 7, 6, 5, 1};
    {140, 25, 5, 18, 9, 7, 6 , 5, 37, 25, 18, 15, 12,
      25, 16, 12, 11, 9, 9, 8, 7, 6, 5, 1};
    {81, 40, 20, 6, 1};
    {128, 64, 32, 16, 8, 4, 2, 1};
    {209, 105, 53, 27, 1};
  */
  Int QList[]=    {100, 20, 200, 100, 50, 25, 10, 5, 2, 1};
  Int *QVals;
  Int val, *iVals;
  Int numQ = sizeof(QList)/sizeof(Int);
  Int i;
  Int *QValMax, *QValMin;
  Int noMatchCnt;
  Int *refLevList;
  Int *revQList;
  Int lastQUsed;

  iVals = (Int *)malloc(sizeof(Int)*numQ);
  QVals = (Int *)malloc(sizeof(Int)*numQ);
  QValMax = (Int *)calloc(sizeof(Int),numQ);
  QValMin = (Int *)calloc(sizeof(Int),numQ);
  refLevList = (Int *)calloc(sizeof(Int),numQ);
  revQList = (Int *)calloc(sizeof(Int),numQ);
  noMatchCnt=0;

  lastQUsed=QList[0];
  for (i=0; i<numQ; ++i)
  {
    refLevList[i] = quantRefLev(QList[i], &lastQUsed, i);
    revQList[i] = lastQUsed;
  }

  for (i=0; i<numQ; ++i)
    fprintf(stderr,"%d: QList=%d, revQList=%d, refLevList=%d\n",
	    i,QList[i],revQList[i],refLevList[i]);

  for (val= -4096; val <= 4096; ++val)
  {
    /* get val */
    vtc_quant(QVals, val, numQ, QList);
    
    for (i=0; i<numQ; ++i)
    {
      if (QValMax[i] < QVals[i])
	QValMax[i] = QVals[i];
      if (QValMin[i] > QVals[i])
	QValMin[i] = QVals[i];

      if (i && abs(QValMax[i]) >= refLevList[i])
	fprintf(stderr,"Error: val=%d, lev=%d, QValMax=%d >= refLevList=%d\n",
		val,i,QValMax[i],refLevList[i]);
      if (i && abs(QValMin[i]) >= refLevList[i])
	fprintf(stderr,"Error: val=%d, lev=%d, QValMin=%d >= refLevList=%d\n",
		val,i,QValMin[i],refLevList[i]);

      invQuant(iVals, numQ, QVals, QList);
    }
    if (iVals[numQ-1]!=val)
      ++noMatchCnt;
  }

  fprintf(stderr,"noMatchCnt=%d\n",noMatchCnt);
  for (i=0; i<numQ; ++i)
    fprintf(stderr,"%d: Max=%d, Min=%d\n",i, QValMax[i], QValMin[i]);

}

#endif




#if 0
/*-------------- User Input ------------------*/

main()
{
  /* Example QLists:

    {70, 30, 1};    
    {16, 8, 4, 2, 1};
    {97, 43, 17, 1};
    {100, 20, 200, 100, 50, 25, 10, 5, 2, 1};
    {97, 43, 17, 112, 40, 30, 69, 21, 8, 1};
  */

  Int QList[]={140, 25, 5, 18, 9, 7, 6 , 5, 37, 25, 18, 15, 12,
      25, 16, 12, 11, 9, 9, 8, 7, 6, 5, 1};

  Int *QVals;
  Int val, *iVals;
  Int numQ = sizeof(QList)/sizeof(Int);
  Int i;
  Int *revQList;

  QVals = (Int *)malloc(sizeof(Int)*numQ);
  iVals = (Int *)malloc(sizeof(Int)*numQ);

  revQList=quantRevisedQList(QList, numQ);
  for (i=0; i<numQ; ++i)
    prIntf("%d: QList=%d, revQList=%d\n",i,QList[i],revQList[i]);
  

  for (;;)
  {
    /* get val */
    prIntf("Enter val => ");
    scanf("%d",&val);
    if (val == 0)
      exit(-1);
    prIntf("val is %d\n",val);

    /*------------  Quantization ----------------*/
    vtc_quant(QVals, val, numQ, QList);

    prIntf("Quantized values are:\n");
    for (i=0; i<numQ; ++i)
      prIntf("QVals[%d] (Q=%3d) = %d\n",i,QList[i],QVals[i]);

 
    /*----------- Inverse Quantization ------------------*/    
    invQuant(Vals, numQ, QVals, QList);

    prIntf("Inverse Quantized values are:\n");
    
    for (i=0; i<numQ; ++i)  
      prIntf("iVal[%d] (Q=%3d) = %d, percent diff = %f\n",
	     i,QList[i],iVals[i], 100.0*(val-iVals[i])/val);

    prIntf("\n");
  }
}

#endif
