/******************************************************************************
 *
 * This software module was originally developed by
 *
 *   Fernando Jaureguizar (UPM-GTI / ACTS-MoMuSys)
 *
 * and edited by
 *
 *   Angel Pacheco (UPM-GTI / ACTS-MoMuSys)
 *
 * in the course of development of the MPEG-4 Video (ISO/IEC 14496-2) standard.
 * This software module is an implementation of a part of one or more MPEG-4
 * Video (ISO/IEC 14496-2) tools as specified by the MPEG-4 Video (ISO/IEC
 * 14496-2) standard.
 *
 * ISO/IEC gives users of the MPEG-4 Video (ISO/IEC 14496-2) standard free
 * license to this software module or modifications thereof for use in hardware
 * or software products claiming conformance to the MPEG-4 Video (ISO/IEC
 * 14496-2) standard.
 *
 * Those intending to use this software module in hardware or software products
 * are advised that its use may infringe existing patents. The original
 * developer of this software module and his/her company, the subsequent
 * editors and their companies, and ISO/IEC have no liability for use of this
 * software module or modifications thereof in an implementation. Copyright is
 * not released for non MPEG-4 Video (ISO/IEC 14496-2) standard conforming
 * products.
 *
 * ACTS-MoMuSys partners retain full right to use the code for his/her own
 * purpose, assign or donate the code to a third party and to inhibit third
 * parties from using the code for non MPEG-4 Video (ISO/IEC 14496-2) standard
 * conforming products. This copyright notice must be included in all copies or
 * derivative works.
 *
 * Copyright (c) 1997
 *
 *****************************************************************************/

/***********************************************************HeaderBegin*******
 *
 * File:        io_debug.c
 *
 * Author:      Fernando Jaureguizar (UPM-GTI)
 * Created:     30.07.97
 *
 * Description: IO-Functions to write a VOP in raw format for debugging
 *              process
 * Modified:
 *      13.11.97 Angel Pacheco (UPM-GTI): formating and new header.
 *      09.03.98 Fernando Jaureguizar: New formating. 
 *
 ***********************************************************HeaderEnd*********/

/************************    INCLUDE FILES    ********************************/

#include <stdio.h>
#include <stdlib.h>

#include "momusys.h"
#include "mom_structs.h"
#include "mom_vop.h"
#include "io_generic.h"

/***********************************************************CommentBegin******
 *
 * -- PrintVopInfo --
 *
 * Author :
 *      Fernando Jaureguizar (UPM-GTI)
 *
 * Created :
 *      30.07.97
 *
 * Description :
 *      Prints some data of a vop. Only for the debugging process of
 *      the Spatial Scalability
 *
 * Arguments in :
 *      Vop  *vop,      Vop with data to be printed
 *      Char *name      Name of stored vop
 *
 * Arguments in/out :
 *      none.
 *
 * Arguments out :
 *      none.
 *
 * Return values :
 *      none.
 *
 * Side effects :
 *
 *
 * See also :
 *
 *
 * Modified :
 *          14.04.1998 Minhua zhou: added shape mode 
 *
 ***********************************************************CommentEnd********/

Void PrintVopInfo(
   Vop    *vop,         /* <- Vop with data to be printed                */
   Char   *name         /* <- Name of stored vop                         */
   )
{
  /* Actual syntax elements for VOP (standard) */
  Int     mod_time_base;       /* VOP modulo time base (absolute) */
  Int     time_inc;            /* VOP time increment (relative to last mtb) */
  Int     prediction_type;     /* VOP prediction type */
  Int     width;               /* VOP height (smallest rectangle) */
  Int     height;              /* VOP width  (smallest rectangle) */
  Int     hor_spat_ref;        /* VOP horizontal ref. (for composition) */
  Int     ver_spat_ref;        /* VOP vertical ref.   (for composition) */

  /* Actual syntax elements for VOP (additional for scalability) */
  Int     ref_sel_code;        /* VOP refereence for P- and B-VOPs */
  Int     for_temp_ref;        /* VOP forward temporal reference */
  Int     back_temp_ref;       /* VOP backward_temporal_reference */

  /* Syntax elements copied from VOL (standard) */
  Int     vo_id;               /* VO Id */
  Int     ident;               /* VOL identifier (for tracking) */
  Int     arbitrary_shape;     /* VOL shape: 0-rect., 1-bin., 2-greyl. */
  Int     scalability;         /* VOL scalability (flag) */

  /* Syntax elements copied from VOL (additional for scalability) */
  Int     ref_ident;           /* VOL id of reference VOL */
  Int     enhancement_type;    /* VOL type of enhancement layer */

  /* Pointers to the images (YUVA) and to related VOPs */
  Image   *a_chan;             /* The alpha plane */
  Image   *a_uv_chan;          /* The alpha  plane for uv */
  Image   *y_chan;             /* Y component of the VOP texture */
  Image   *u_chan;             /* U component of the VOP texture */
  Image   *v_chan;             /* V component of the VOP texture */
  Image   *QP_chan;            /* Storage of QP values */
  Image   *shape_mode; 
  Vop     *pnext_spat;         /* List of VOPs within same time instance */
  Vop     *pnext_temp;         /* List of VOPs with same ID along time-axis */
  Vop     *pprev_temp;         /* Previously decoded I or P VOP */

  Int     display_time;

  fprintf(stdout,"\n@@START@@ VOP Data: %s (0x%lX) @@@@@@@@@\n",name,
          (ULInt)vop);
  if (vop != NULL)
    {
    mod_time_base = GetVopModTimeBase(vop);
    time_inc = GetVopTimeInc(vop);

    display_time = mod_time_base*1000 + time_inc;

    prediction_type  = GetVopPredictionType(vop);
    width = GetVopWidth(vop);
    height = GetVopHeight(vop);
    hor_spat_ref = GetVopHorSpatRef(vop);
    ver_spat_ref = GetVopVerSpatRef(vop);

    ref_sel_code = GetVopRefSelCode(vop);
    for_temp_ref = GetVopForTempRef(vop);
    back_temp_ref = GetVopBackTempRef(vop);

    vo_id = GetVopVOId(vop);
    ident = GetVopId(vop);
    arbitrary_shape = GetVopArbitraryShape(vop);
    scalability = GetVopScalability(vop);

    ref_ident = GetVopRefId(vop);
    enhancement_type = GetVopEnhanceType(vop);

    a_chan = GetVopA(vop);
    a_uv_chan = GetVopAuv(vop);
    y_chan = GetVopY(vop);
    u_chan = GetVopU(vop);
    v_chan = GetVopV(vop);
    QP_chan = GetVopQP(vop);
    shape_mode = GetVopShapeMode(vop);

    pnext_spat = GetVopNextSpat(vop);
    pnext_temp = GetVopNextTemp(vop);
    pprev_temp = GetVopPrevTemp(vop);


    fprintf(stdout,"Time> Mod Base   : %d \n",mod_time_base);
    fprintf(stdout,"Time> Increment  : %d \n",time_inc);
    fprintf(stdout,"Time> Display    : %d \n",display_time);
    fprintf(stdout,"Prediction Type  : %d \n",prediction_type);
    fprintf(stdout,"Size             : %d x %d\n",width,height);
    fprintf(stdout,"Spat_Ref         : %d x %d\n",hor_spat_ref,ver_spat_ref);
    fprintf(stdout,"Ref Sel Code     : %d \n",ref_sel_code);
    fprintf(stdout,"For/Back Temp Ref: %d/%d\n",for_temp_ref,back_temp_ref);
    fprintf(stdout,"VO Id            : %d \n",vo_id);
    fprintf(stdout,"VOL Id           : %d \n",ident);
    fprintf(stdout,"Shape            : %d \n",arbitrary_shape);
    fprintf(stdout,"Scalability      : %d \n",scalability);
    fprintf(stdout,"Ref VOL Id       : %d \n",ref_ident);
    fprintf(stdout,"Enhan Type       : %d \n\n",enhancement_type);

    if (a_chan != NULL)
      {
      fprintf(stdout,"A   Image Size   : %d x %d\n",
            (Int)GetImageSizeX(a_chan),(Int)GetImageSizeY(a_chan));
      fprintf(stdout,"Y   Image Size   : %d x %d\n",
            (Int)GetImageSizeX(y_chan),(Int)GetImageSizeY(y_chan));
      fprintf(stdout,"    Size in MB   : %d x %d\n",
            (Int)GetImageSizeX(y_chan)/16,(Int)GetImageSizeY(y_chan)/16);
      fprintf(stdout,"Auv Image Size   : %d x %d\n",
            (Int)GetImageSizeX(a_uv_chan),(Int)GetImageSizeY(a_uv_chan));
      fprintf(stdout,"U   Image Size   : %d x %d\n",
            (Int)GetImageSizeX(u_chan),(Int)GetImageSizeY(u_chan));
      fprintf(stdout,"V   Image Size   : %d x %d\n",
            (Int)GetImageSizeX(v_chan),(Int)GetImageSizeY(v_chan));
      fprintf(stdout,"    Size in MB   : %d x %d\n",
            (Int)GetImageSizeX(v_chan)/16,(Int)GetImageSizeY(v_chan)/16);
      fprintf(stdout,"QP  Image Size   : %d x %d\n\n",
            (Int)GetImageSizeX(QP_chan),(Int)GetImageSizeY(QP_chan));
      fprintf(stdout,"ShapeMode  Image Size   : %d x %d\n\n",
            (Int)GetImageSizeX(shape_mode),(Int)GetImageSizeY(shape_mode));

      }

    fprintf(stdout,"Next Spat Vop    : ");
    if (pnext_spat == NULL)
      fprintf(stdout,"NO\n");
    else
      {
      fprintf(stdout,"(0x%lX)\n    Size         : %d x %d\n",(ULInt)pnext_spat,
              GetVopWidth(pnext_spat), GetVopHeight(pnext_spat));
      }
    fprintf(stdout,"Next Temp Vop    : ");
    if (pnext_temp == NULL)
      fprintf(stdout,"NO\n");
    else
      {
      fprintf(stdout,"(0x%lX)\n    Size         : %d x %d\n",(ULInt)pnext_temp,
              GetVopWidth(pnext_temp), GetVopHeight(pnext_temp));
      }
    fprintf(stdout,"Prev Temp Vop    : ");
    if (pprev_temp == NULL)
      fprintf(stdout,"NO\n");
    else
      {
      fprintf(stdout,"(0x%lX)\n    Size         : %d x %d\n",(ULInt)pprev_temp,
              GetVopWidth(pprev_temp), GetVopHeight(pprev_temp));
      }
    }

  fprintf(stdout,"@@@END@@@ %s @@@@@@@@@@@@@@@@@@@@@@@\n\n",name);
  fflush(stdout);
}

/***********************************************************CommentBegin******
 *
 * -- WriteVopDebug --
 *
 * Author :
 *      Fernando Jaureguizar (UPM-GTI)
 *
 * Created :
 *      30.07.97
 *
 * Description :
 *      Write to disk the current, next and previous vops.  Only for
 *      the debugging process of the Spatial Scalability
 *
 * Arguments in :
 *      Vop  *vop,         vop with data to be printed
 *      Char *curr,        name of the current vop
 *      Char *next,        name of the next vop
 *      Char *prev,        name of the orevious vop
 *      Int intern         with a static numering
 *
 * Arguments in/out :
 *      none.
 *
 * Arguments out :
 *      none.
 *
 * Return values :
 *      none.
 *
 * Side effects :
 *
 *
 * See also :
 *
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/

Void WriteVopDebug(
   Vop    *vop,   /* <- vop with data to be printed                          */
   Char   *curr,  /* <- name of the current vop                              */
   Char   *next,  /* <- name of the next vop                                 */
   Char   *prev,  /* <- name of the orevious vop                             */
   Int   intern   /* <- with a static numering                               */
   )
{
  Char         n[80];
  static Int   aux = 0;

  if (vop != NULL)
    {
    if (strlen(curr)) /* the name of the file is not a null string */
      {
      if(intern)
        sprintf(n,"curr%s%d.yuv",curr,aux);
      else
        sprintf(n,"curr%s.yuv",curr);
      if (GetVopA(vop) != NULL)
        WriteVopGeneric(vop,n,n,n,"",NULL,0,aux,IO_FORMAT,IO_OVERWRITE,0);
      }

    if (strlen(next) && GetVopNextTemp(vop) != NULL)
      {
      if(intern)
        sprintf(n,"next%s%d.yuv",next,aux);
      else
        sprintf(n,"next%s.yuv",next);
      if (GetVopA(GetVopNextTemp(vop)) != NULL)
        WriteVopGeneric(GetVopNextTemp(vop),n,n,n,"",NULL,0,
          aux,IO_FORMAT,IO_OVERWRITE,0);
      }

    if (strlen(prev) && GetVopPrevTemp(vop) != NULL)
      {
      if(intern)
        sprintf(n,"prev%s%d.yuv",prev,aux);
      else
        sprintf(n,"prev%s.yuv",prev);
      if (GetVopA(GetVopPrevTemp(vop)) != NULL)
        WriteVopGeneric(GetVopPrevTemp(vop),n,n,n,"",NULL,0,
          aux,IO_FORMAT,IO_OVERWRITE,0);
      }
    }

  if(intern)
    aux++;
}

/***********************************************************CommentBegin******
 *
 * -- WriteVopAlphas --
 *
 * Author :
 *      Fernando Jaureguizar (UPM-GTI)
 *
 * Created :
 *      30-JUL-1997
 *
 * Description :
 *      Write to disk the Y and UV alpha masks of a vop. Only for the
 *      debugging process of the Spatial Scalability.
 *
 * Arguments in :
 *      Vop *vop,             vop with the data to be printed
 *      Char *alpha,          name of the Y-alpha mask
 *      Char *alphauv,        name of the UV-alpha mask
 *      Int intern            with a static numering
 *
 * Arguments in/out :
 *      none.
 *
 * Arguments out :
 *      none.
 *
 * Return values :
 *      none.
 *
 * Side effects :
 *
 *
 * See also :
 *
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/

Void WriteVopAlphas(
   Vop    *vop,       /* <- vop with the data to be printed                   */
   Char   *alpha,    /* <- name of the Y-alpha mask                          */
   Char   *alphauv,  /* <- name of the UV-alpha mask                         */
   Int    intern      /* <- with a static numering                            */
   )
{
  Char     n[80];
  static   Int aux = 0;
  File     *fichero;
  Int      height    = (Int)vop->a_chan->y;
  Int      width     = (Int)vop->a_chan->x;
  Int      height_uv = (Int)vop->a_chan->y/2;
  Int      width_uv  = (Int)vop->a_chan->x/2;
  Int      i, j;
  Char     datum;
  SInt     *data;

  if (strlen(alpha)) /* the name of the file is not a null string */
    {
    if(intern)
      sprintf(n,"alpha%s%d.a",alpha,aux);
    else
      sprintf(n,"alpha%s.a",alpha);

    if((fichero = fopen (n,"wb"))==NULL)
      {
      fprintf(stderr," - ERROR: can not open Alpha file\n");
      return;
      }

    data = vop->a_chan->data->s;
    for(i = 0; i < height; i++)
      for(j = 0; j < width; j++)
        {
        datum = (Char)data[i*width+j];
        fwrite (&datum, 1, 1, fichero);
        }
    }

  if (strlen(alphauv))
    {
    if(intern)
      sprintf(n,"alpuv%s%d.a",alphauv,aux);
    else
      sprintf(n,"alpuv%s.a",alphauv);

    if((fichero = fopen (n,"wb"))==NULL)
      {
      fprintf(stderr," - ERROR: can not open Alpha_UV file\n");
      return;
      }

    data = vop->a_uv_chan->data->s;
    for(i = 0; i < height_uv; i++)
      for(j = 0; j < width_uv; j++)
        {
        datum = (Char)data[i*width_uv+j];
        fwrite (&datum, 1, 1, fichero);
        }

    }

  if(intern)
    aux++;
}
