/******************************************************************************
 * This software module was originally developed by
 *	Seishi TAKAMURA (NTT)
 * in the course of development of the <MPEG-4 Video(ISO/IEC 14496-2)>
 * standard.  This software module is an implementation of a part of
 * one or more MPEG-4 Video(ISO/IEC 14496-2) standard tools as
 * specified by the MPEG-4 Video(ISO/IEC 14496-2) standard.  ISO/IEC
 * gives users of the MPEG-4 Video(ISO/IEC 14496-2) standard free
 * license to this software module or modifications thereof for use in
 * hardware or software products claiming conformance to the MPEG-4
 * Video(ISO/IEC 14496-2) standard.  Those intending to use this
 * software module in hardware or software products are advised that
 * its use may infringe existing patents.  THE ORIGINAL DEVELOPER OF
 * THIS SOFTWARE MODULE AND HIS/HER COMPANY, THE SUBSEQUENT EDITORS
 * AND THEIR COMPANIES, AND ISO/IEC HAVE NO LIABILITY FOR USE OF THIS
 * SOFTWARE MODULE OR MODIFICATIONS THEREOF.  No license to this
 * software module is granted for non MPEG-4 Video(ISO/IEC 14496-2)
 * standard conforming products. NTT retains full right to use the
 * software module for his/her own purpose, assign or donate the
 * software module to a third party and to inhibit third parties from
 * using the code for non MPEG-4 Video(ISO/IEC 14496-2) standard
 * conforming products.  This copyright notice must be included in all
 * copies or derivative works of the software module.
 *
 * Copyright (C) 1999.
 *****************************************************************************/

/*************************************************HeaderBegin****************
 *
 * File:	globalMC.c
 *
 * Author:	Seishi TAKAMURA (NTT)
 * Created:	03.03.99
 *                                                                          
 * Description: GMC coding tool
 *
 * Notes: 	
 *
 * Modified:	
 *	18.08.99 Seishi TAKAMURA (NTT): modified GMC coding (rounding control)
 *
 *************************************************HeaderEnd******************/

/************************    INCLUDE FILES    ********************************/
#include <sys/stat.h>
#include "globalMC.h"

/***********************************************************CommentBegin******
 *
 * -- GetPred_GMC -- make GMC prediction image data for one MB
 *
 * Author :		
 *	Seishi TAKAMURA (NTT)
 *
 * Created :		
 *	03.03.99
 *
 * Purpose :		
 *	make GMC prediction image data for one MB
 * 
 * Arguments in : 	
 *	Int	Col		MB left top coodinate (x)
 *      Int	Row		MB left top coodinate (y)
 *	Vop	*comp		compensation VOP data
 *	Image	*rec_prev	reference VOP data (luminance-signal)
 *	Int	offset_x	offset for edge
 *	Int	offset_y	offset for edge
 *
 * Arguments in/out :	
 *
 * Arguments out :	
 *	SInt	pred[][16]	GMC predict image (this MB)
 *
 * Return values :	
 *	Int			warping succeed=1/failed=0
 *
 * Side effects :	
 *	
 * Description :	
 *
 * See also :
 *
 * Modified :		
 *	18.08.99 Seishi TAKAMURA (NTT): modified GMC coding (rounding control)
 *
 ***********************************************************CommentEnd********/
Int
GetPred_GMC(SInt	pred[][16],
	    Int		Col,
	    Int		Row,
	    Vop		*comp,
	    Image	*rec_prev,
	    Int		offset_x,
	    Int		offset_y)
{
    Int		i, j;
    Int		C[10];		/* for warping caluculation */
    Double	Cd[10];		/* for warping caluculation */
    Int		col, row;
    Int		I_, J_;
    Int		px, py;
    Int		x0, x1, y0, y1;
    Int		P00, P01, P10, P11, P;
    Int		s			= 2 << GetVopWarpingAccuracy(comp);
    Int		no_of_sprite_points	= GetVopNoOfSpritePoints(comp);
    Int		br_x			= GetVopHorSpatRef(comp);
    Int		br_y			= GetVopVerSpatRef(comp);
    SInt	*padvop			= (SInt*) GetImageData(rec_prev);
    Int		sizex			= GetImageSizeX(rec_prev);
    Int		sizey			= GetImageSizeY(rec_prev);
    Int		RoundType		= GetVopRoundingType(comp);

    /* setting warping Consts */
    set_warping_para(C, Cd, comp);

    for (j = 0, row = Row; j < MB_SIZE; j++, row++) {
	for (i = 0, col = Col; i < MB_SIZE; i++, col++) {
	    if (! VOP_synthesis_Lumin(col+br_x, row+br_y, &I_, &J_, C, Cd,
				      br_x, br_y, no_of_sprite_points)) {
		/* warping Error check */
		return 0;
	    }

	    px = I_ & (s - 1);
	    py = J_ & (s - 1);

	    x0 = Round(floor((Double) I_ / s));
	    y0 = Round(floor((Double) J_ / s));
	    x1 = x0 + 1;
	    y1 = y0 + 1;

	    x0 -= offset_x;
	    y0 -= offset_y;
	    x1 -= offset_x;
	    y1 -= offset_y;

	    x0 = limitter(x0, 0, sizex - 1);	/* UMV */
	    x1 = limitter(x1, 0, sizex - 1);
	    y0 = limitter(y0, 0, sizey - 1);
	    y1 = limitter(y1, 0, sizey - 1);

	    P00 = *(padvop + y0 * sizex + x0);
	    P01 = *(padvop + y0 * sizex + x1);
	    P10 = *(padvop + y1 * sizex + x0);
	    P11 = *(padvop + y1 * sizex + x1);

	    P = ((s - py) * ((s - px) * P00 + px * P01) +
		 py       * ((s - px) * P10 + px * P11) + (s * s / 2) - RoundType) / (s * s);
	    pred[j][i] = limitter(P, 0, 255);
	}
    }

    return 1;
}


/***********************************************************CommentBegin******
 *
 * -- GetPred_GMC_Chrom -- make GMC prediction image data for one chrominance MB
 *
 * Author :		
 *	Seishi TAKAMURA (NTT)
 *
 * Created :		
 *	03.03.99
 *
 * Purpose :		
 *	make GMC prediction image data for one chrominance MB
 * 
 * Arguments in : 	
 *	Int	Col			MB left top coodinate (x)
 *      Int	Row			MB left top coodinate (y)
 *	Vop	*prevvop		previous Vop data
 *	Int	offset_x		offset for edge
 *	Int	offset_y		offset for edge
 *
 * Arguments in/out :	
 *	Vop	*compvop		Vop to be filled with compensated data
 *
 * Arguments out :	
 *
 * Return values :	
 *	
 * Side effects :	
 *	
 * Description :	
 *
 * See also :
 *
 * Modified :		
 *	18.08.99 Seishi TAKAMURA (NTT): modified GMC coding (rounding control)
 *
 ***********************************************************CommentEnd********/
Void
GetPred_GMC_Chrom(Int	Col,
		  Int	Row,
		  Vop	*prevvop,
		  Vop	*compvop,
		  Int	offset_x,
		  Int	offset_y)
{
    Int		i, j;
    Int		C[10];
    Double	Cd[10];
    Int		col, row;
    Int		Ic_, Jc_;
    Int		px, py;
    Int		x0, x1, y0, y1;
    Int		P00, P01, P10, P11, P;
    Int		s		= 2 << GetVopWarpingAccuracy(compvop);
    Image	*imgU		= GetVopU(prevvop);
    Image	*imgV		= GetVopV(prevvop);
    SInt   	*padimgU	= (SInt *) GetImageData(imgU);
    SInt   	*padimgV	= (SInt *) GetImageData(imgV);
    Int		sizex		= GetImageSizeX(imgU);
    Int		sizey		= GetImageSizeY(imgU);
    Int		nosp		= GetVopNoOfSpritePoints(compvop);
    Int		br_x		= GetVopHorSpatRef(compvop) >> 1;
    Int 	br_y		= GetVopVerSpatRef(compvop) >> 1;
    Int		width		= GetVopWidth(compvop) >> 1;
    Int		RoundType	= GetVopRoundingType(compvop);

    Col		>>= 1;
    Row		>>= 1;
    offset_x	>>= 1;
    offset_y	>>= 1;

    set_warping_para(C, Cd, compvop);

    for (j = 0, row = Row; j < MB_SIZE/2; j++, row++) {
	for (i = 0, col = Col; i < MB_SIZE/2; i++, col++) {
	    /* warping */
	    if (! VOP_synthesis_Chrom(col+br_x, row+br_y, &Ic_, &Jc_, C, Cd,
				      br_x*2, br_y*2, nosp)) {
		fprintf(stderr, "GMC warping error\n");
		exit(-1);
	    }

	    x0 = Round(floor((Double) Ic_ / s));
	    y0 = Round(floor((Double) Jc_ / s));
	    x1 = x0 + 1;
	    y1 = y0 + 1;

	    px = Ic_ - (Round(floor((Double) Ic_ / s)) * s);
	    py = Jc_ - (Round(floor((Double) Jc_ / s)) * s);

	    x0 -= offset_x;
	    y0 -= offset_y;
	    x1 -= offset_x;
	    y1 -= offset_y;

	    /* UMV */
	    x0 = limitter(x0, 0, sizex - 1);
	    x1 = limitter(x1, 0, sizex - 1);
	    y0 = limitter(y0, 0, sizey - 1);
	    y1 = limitter(y1, 0, sizey - 1);

	    /* U-signal */
	    P00 = *(padimgU + y0 * sizex + x0);
	    P01 = *(padimgU + y0 * sizex + x1);
	    P10 = *(padimgU + y1 * sizex + x0);
	    P11 = *(padimgU + y1 * sizex + x1);

	    P = ((s - py) * ((s - px) * P00 + px * P01) +
		 py       * ((s - px) * P10 + px * P11) + (s * s / 2) - RoundType) / (s * s);
	    compvop->u_chan->f[row * width + col] = limitter(P, 0, 255);

	    /* V-signal */
	    P00 = *(padimgV + y0 * sizex + x0);
	    P01 = *(padimgV + y0 * sizex + x1);
	    P10 = *(padimgV + y1 * sizex + x0);
	    P11 = *(padimgV + y1 * sizex + x1);

	    P = ((s - py) * ((s - px) * P00 + px * P01) +
		 py       * ((s - px) * P10 + px * P11) + (s * s / 2) - RoundType) / (s * s);
	    compvop->v_chan->f[row * width + col] = limitter(P, 0, 255);
	}
    }
}


/***********************************************************CommentBegin******
 *
 * -- calc_gmc_vector -- caluculation GMC vector data for all MB in Vop
 *
 * Author :		
 *	Seishi TAKAMURA (NTT)
 *
 * Created :		
 *	03.03.99
 *
 * Purpose :		
 *	caluculation GMC vector data for all MB in Vop
 * 
 * Arguments in : 	
 *	Vop	*curr		current Vop data
 *
 * Arguments in/out :	
 *
 * Arguments out :	
 *	Image	**mot_gmc_x	GMC vector x
 *	Image	**mot_gmc_y	GMC vector y
 *
 * Return values :	
 *	
 * Side effects :	
 *	
 * Description :	
 *
 * See also :
 *
 * Modified :		
 *
 ***********************************************************CommentEnd********/
Void
calc_gmc_vector(Image	**mot_gmc_x,
		Image	**mot_gmc_y,
		Vop	*curr)
{
    Int		i, j;
    Int		col, row;
    Int		C[10];
    Double	Cd[10];
    Float	*vec_x, *vec_y;
    Float	vx, vy;
    Int		vop_width	= GetImageSizeX(GetVopY(curr));
    Int		vop_height	= GetImageSizeY(GetVopY(curr));
    Int		mv_w		= vop_width  / MB_SIZE;
    Int		mv_h		= vop_height / MB_SIZE;
    Int		hsr		= GetVopHorSpatRef(curr);
    Int		vsr		= GetVopVerSpatRef(curr);

    /* initialization gmc vector */
    *mot_gmc_x = AllocImage(mv_w * 2, mv_h * 2, FLOAT_TYPE);
    *mot_gmc_y = AllocImage(mv_w * 2, mv_h * 2, FLOAT_TYPE);
    vec_x = (Float*)GetImageData(*mot_gmc_x);
    vec_y = (Float*)GetImageData(*mot_gmc_y);

    set_warping_para(C, Cd, curr);

    for (j = 0, row = vsr; j < mv_h; j++, row += MB_SIZE) {
	for (i = 0, col = hsr; i < mv_w; i++, col += MB_SIZE) {
	    calc_gmc_vector_sub(&vx, &vy, col, row, C, Cd, curr);

	    vec_x[(j*2) * (mv_w*2) + (i*2)] = vx;
	    vec_y[(j*2) * (mv_w*2) + (i*2)] = vy;
	}
    }
}


/***********************************************************CommentBegin******
 *
 * -- calc_gmc_vector_MB -- caluculation GMC vector data for one MB
 *
 * Author :		
 *	Seishi TAKAMURA (NTT)
 *
 * Created :		
 *	03.03.99
 *
 * Purpose :		
 *	caluculation GMC vector data for one MB
 * 
 * Arguments in : 	
 *	Int	Col		MB left top coodinate (x)
 *      Int	Row		MB left top coodinate (y)
 *	Vop	*curr		current Vop data
 *
 * Arguments in/out :	
 *
 * Arguments out :	
 *	Float	*vec_x		GMC vector x
 *	Float	*vec_y		GMC vector y
 *
 * Return values :	
 *	
 * Side effects :	
 *	
 * Description :	
 *
 * See also :
 *
 * Modified :		
 *
 ***********************************************************CommentEnd********/
Void
calc_gmc_vector_MB(Int		Col,
		   Int		Row,
		   Float	*vec_x,
		   Float	*vec_y,
		   Vop		*curr)
{
    Int		C[10];
    Double	Cd[10];

    /* setting warping parameter */
    set_warping_para(C, Cd, curr);

    /* vector calculate */
    calc_gmc_vector_sub(vec_x, vec_y,
			Col + GetVopHorSpatRef(curr),
			Row + GetVopVerSpatRef(curr), C, Cd, curr);
}


/***********************************************************CommentBegin******
 *
 * -- calc_gmc_vector_sub -- calculated average pel-wise vector from GMC parameter
 *
 * Author :		
 *	Seishi TAKAMURA (NTT)
 *
 * Created :		
 *	03.03.99
 *
 * Purpose :		
 *	calculated median pel-wise vector from GMC parameter
 * 
 * Arguments in : 	
 *	Int	Col		MB left top coodinate (x)
 *      Int	Row		MB left top coodinate (y)
 *	Int	C[]		constants value for calculation
 *	Double	Cd[]		constants value for calculation
 *	Vop	*curr		current Vop data
 *
 * Arguments in/out :	
 *
 * Arguments out :	
 *	Float	*gmc_vx		median pel-wise vector from GMC parameter
 *	Float	*gmc_vy		median pel-wise vector from GMC parameter
 *
 * Return values :	
 *	
 * Side effects :	
 *	
 * Description :	
 *
 * See also :
 *
 * Modified :		
 *
 ***********************************************************CommentEnd********/
Void
calc_gmc_vector_sub(Float	*gmc_vx,
		    Float	*gmc_vy,
		    Int		Col,
		    Int		Row,
		    Int		C[],
		    Double	Cd[],
		    Vop		*curr)
{
    Int		i, j;
    Int		I_, J_;
    Int		vx, vy;
    Int		col, row;
    Double	sum_vx, sum_vy;
    Double	ave_vx, ave_vy;
    Int		s		= 2 << GetVopWarpingAccuracy(curr);
    Float	min_mv, max_mv;
    Float	vec;

    if (GetVopQuarterPel(curr)) {
	min_mv	= -( 8 << (GetVopFCodeFor(curr) - 1));
	max_mv	=  ( 8 << (GetVopFCodeFor(curr) - 1)) - 0.25;
    } else {
	min_mv	= -(16 << (GetVopFCodeFor(curr) - 1));
	max_mv	=  (16 << (GetVopFCodeFor(curr) - 1)) - 0.5;
    }

    sum_vx = sum_vy = 0;
    for (j = 0, row = Row; j < MB_SIZE; j++, row++) {
	for (i = 0, col = Col; i < MB_SIZE; i++, col++) {
	    /* calculated pel-wise vector */
	    if (! VOP_synthesis_Lumin(col, row, &I_, &J_, C, Cd,
				      GetVopHorSpatRef(curr), GetVopVerSpatRef(curr), 
				      GetVopNoOfSpritePoints(curr))) {
		fprintf(stderr, "GMC warping error\n");
		exit(-1);
	    }
	    vx = I_ - (col * s);
	    vy = J_ - (row * s);

	    /* summed pel-wise vector */
	    sum_vx += (Double) vx;
	    sum_vy += (Double) vy;
	}
    }
    /* calculated average pel-wise vector */
    ave_vx = sum_vx / (MB_SIZE * MB_SIZE);
    ave_vy = sum_vy / (MB_SIZE * MB_SIZE);

    if (GetVopQuarterPel(curr)) {
	vec = ((Float) Round((ave_vx / s) * 4)) / 4.0;
	*gmc_vx = limitter(vec, min_mv, max_mv);
	vec = ((Float) Round((ave_vy / s) * 4)) / 4.0;
	*gmc_vy = limitter(vec, min_mv, max_mv);
    } else {
	vec = ((Float) Round((ave_vx / s) * 2)) / 2.0;
	*gmc_vx = limitter(vec, min_mv, max_mv);
	vec = ((Float) Round((ave_vy / s) * 2)) / 2.0;
	*gmc_vy = limitter(vec, min_mv, max_mv);
    }
}


/***********************************************************CommentBegin******
 *
 * -- VOP_synthesis_Lumin -- warping luminance signal for one pixel
 *
 * Author :		
 *	Seishi TAKAMURA (NTT)
 *
 * Created :		
 *	03.03.99
 *
 * Purpose :		
 *	warping luminance signal for one pixel
 * 
 * Arguments in : 	
 *	Int	i		pixel axis
 *	Int	j		pixel axis
 *	Int	C[]		constants value for calculation
 *	Double	Cd[]		constants value for calculation
 *	Int	VOP_horizontal_mc_spatial_ref
 *	Int	VOP_vertical_mc_spatial_ref
 *	Int	no_of_sprite_points
 *
 * Arguments in/out :	
 *
 * Arguments out :	
 *	Int	*I_		after warping pixel axis
 *	Int	*J_		after warping pixel axis 
 *
 * Return values :
 *	Int			warping succeed=1/failed=0
 *	
 * Side effects :	
 *	
 * Description :	
 *
 * See also :
 *
 * Modified :		
 *
 ***********************************************************CommentEnd********/
Int
VOP_synthesis_Lumin(Int		i,
		    Int		j,
		    Int		*I_,
		    Int		*J_,
		    Int		C[],
		    Double 	Cd[],
		    Int		VOP_horizontal_mc_spatial_ref,
		    Int		VOP_vertical_mc_spatial_ref,
		    Int		no_of_sprite_points)
{
    Int		I = i - VOP_horizontal_mc_spatial_ref;
    Int		J = j - VOP_vertical_mc_spatial_ref;
    Double	temp, denom;

    switch (no_of_sprite_points) {
      case 0:	/* Stationary transform */
	/* I' = si,	J' = sj
		C[2] = s
	*/
	*I_ = C[2] * i;
	*J_ = C[2] * j;
	break;
      case 1:	/* Translational transform */
	/* I' = i0' + sI,  J' = j0' + sJ
		C[0] = i0'
		C[1] = j0'
		C[2] = s
	*/
	*I_ = C[0] + C[2] * I;
	*J_ = C[1] + C[2] * J;
	break;
      case 2:	/* Isotropic transform */
	/* I' = (i0' + ((-r i0' + i1")I + ( r j0' - j1")J) /// (W' r)
	   J' = (j0' + ((-r j0' + j1")I + (-r i0' - i1")J) /// (W' r)
		C[0] = i0'
		C[1] = -r i0' + i1"
		C[2] =  r j0' - j1"
		C[3] = j0'
		C[4] = -r j0' + j1"
		C[5] = -r i0' - i1"
		C[6] = W' r
	*/
      case 3:	/* Affine transform */
	/* I' = (i0' + ((-r i0' + i1")H' I + (-r i0' + i2")W' J) /// (W' H' r)
	   J' = (j0' + ((-r j0' + j1")H' I + (-r j0' + j2")W' J) /// (W' H' r)
		C[0] = i0'
		C[1] = (-r i0' + i1")H'
		C[2] = (-r i0' + i2")W'
		C[3] = j0'
		C[4] = (-r j0' + j1")H'
		C[5] = (-r j0' + j2")W'
		C[6] = W' H' r
	*/
	temp = (Double) C[0] + ((Double) (C[1] * I + C[2] * J) / (Double) C[6]);
	*I_ = round_toward_positive(temp);
	temp = (Double) C[3] + ((Double) (C[4] * I + C[5] * J) / (Double) C[6]);
	*J_ = round_toward_positive(temp);
	break;
      case 4:	/* Perspective transform */
	/* I' = (a i + b j + c) /// (g i + h j + D W H)
	   J' = (d i + e j + f) /// (g i + h j + D W H)
		Cd[0] = a,	Cd[1] = b,	Cd[2] = c
		Cd[3] = d,	Cd[4] = e,	Cd[5] = f
		Cd[6] = g,	Cd[7] = h,	Cd[8] = D W H
	*/
	denom = Cd[6] * I + Cd[7] * J + Cd[8];
	/* Error check */
	if (denom == 0.0) {
	    fprintf(stderr, "Warning: perspective transform %d, %d\n", i, j);
	    return 0;
	}

	temp = (Cd[0] * I + Cd[1] * J + Cd[2]) / denom;
	*I_ = round_toward_positive(temp);
	temp = (Cd[3] * I + Cd[4] * J + Cd[5]) / denom;
	*J_ = round_toward_positive(temp);
	break;
      default:
	fprintf(stderr, "VOP_synthesis_Lumin: no_of_sprite_warping_points error: %d\n",
		no_of_sprite_points);
	exit(-1);
    }

    return 1;
}


/***********************************************************CommentBegin******
 *
 * -- VOP_synthesis_Chrom -- warping chrominance signal for one pixel
 *
 * Author :
 *	Seishi TAKAMURA (NTT)
 *
 * Created :		
 *	03.03.99
 *
 * Purpose :		
 *	warping chrominance signal for one pixel
 * 
 * Arguments in : 	
 *	Int	ic		pixel axis
 *	Int	jc		pixel axis
 *	Int	C[]		constants value for calculation
 *	Double	Cd[]		constants value for calculation
 *	Int	VOP_horizontal_mc_spatial_ref
 *	Int	VOP_vertical_mc_spatial_ref
 *	Int	no_of_sprite_points
 *
 * Arguments in/out :	
 *
 * Arguments out :	
 *	Int	*Ic_		after warping pixel axis
 *	Int	*Jc_		after warping pixel axis 
 *
 * Return values :	
 *	Int			warping succeed=1/failed=0
 *	
 * Side effects :	
 *	
 * Description :	
 *
 * See also :
 *
 * Modified :		
 *
 ***********************************************************CommentEnd********/
Int
VOP_synthesis_Chrom(Int		ic,
		    Int		jc,
		    Int		*Ic_,
		    Int		*Jc_,
		    Int		C[],
		    Double	Cd[],
		    Int		VOP_horizontal_mc_spatial_ref,
		    Int		VOP_vertical_mc_spatial_ref,
		    Int		no_of_sprite_points)
{
    Int		Ic = 4 * ic - 2 * VOP_horizontal_mc_spatial_ref + 1;
    Int		Jc = 4 * jc - 2 * VOP_vertical_mc_spatial_ref + 1;
    Double	temp, denom;

    switch (no_of_sprite_points) {
      case 0:	/* Stationary transform */
	/* Ic' = s ic,	J' = s jc
		C[2] = s
		C[3] = C[4] = 0
	*/
	*Ic_ = C[3] + C[2] * ic;
	*Jc_ = C[4] + C[2] * jc;
	break;
      case 1:	/* Translational transform */
	/* Ic' = (((i0' >> 1) | (i0' & 1)) + s (ic - i0 / 2),
	   Jc' = (((j0' >> 1) | (j0' & 1)) + s (jc - j0 / 2),
		C[2] = s
		C[3] = ((i0' >> 1) | (i0' & 1))
		C[4] = ((j0' >> 1) | (j0' & 1))
	*/
	*Ic_ = C[3] + C[2] * (ic - VOP_horizontal_mc_spatial_ref / 2);
	*Jc_ = C[4] + C[2] * (jc - VOP_vertical_mc_spatial_ref   / 2);
	break;
      case 2:	/* Isotropic transform */
	/* Ic' = (((-r i0' + i1")Ic + ( r j0' - j1")Jc + 2 W' r i0' - 16 W') /// (4 W' r)
	   Jc' = (((-r j0' + j1")Ic + (-r i0' - i1")Jc + 2 W' r j0' - 16 W') /// (4 W' r)
		C[1] = -r i0' + i1"
		C[2] =  r j0' - j1"
		C[4] = -r j0' + j1"
		C[5] = -r i0' - i1"
		C[7] = 4 W' r
		C[8] = 2 W' r i0' - 16 W'
		C[9] = 2 W' r j0' - 16 W'
	*/
      case 3:	/* Affine transform */
	/* I' = (((-r i0' + i1")H' Ic + (-r i0' + i2")W' Jc + 2 W' H' r i0' - 16 W' H') /// (4 W' H' r)
	   J' = (((-r j0' + j1")H' Ic + (-r j0' + j2")W' Jc + 2 W' H' r j0' - 16 W' H') /// (4 W' H' r)
		C[1] = (-r i0' + i1")H'
		C[2] = (-r i0' + i2")W'
		C[4] = (-r j0' + j1")H'
		C[5] = (-r j0' + j2")W'
		C[7] = 4 W' H' r
		C[8] = 2 W' H' r i0' - 16 W' H'
		C[9] = 2 W' H' r j0' - 16 W' H'
	*/
	temp = ((Double) C[1]*Ic + (Double) C[2]*Jc + (Double) C[8]) / (Double) C[7];
	*Ic_ = round_toward_positive(temp);
	temp = ((Double) C[4]*Ic + (Double) C[5]*Jc + (Double) C[9]) / (Double) C[7];
	*Jc_ = round_toward_positive(temp);
	break;
      case 4:	/* Perspective transform */
	/* I' = (2 a Ic + b Jc + 4 c - (g Ic + h Jc + 2 D W H)s) /// (4 g Ic + 4 h Jc + 8 D W H)
	   J' = (2 d Ic + e Jc + 4 f - (g Ic + h Jc + 2 D W H)s) /// (4 g Ic + 4 h Jc + 8 D W H)
		Cd[0] = a,	Cd[1] = b,	Cd[2] = c
		Cd[3] = d,	Cd[4] = e,	Cd[5] = f
		Cd[6] = g,	Cd[7] = h,	Cd[8] = D W H,	Cd[9] = s
	*/
	denom = 4*Cd[6]*Ic + 4*Cd[7]*Jc + 8*Cd[8];
	/* Error check */
	if (denom == 0.0) {
	    fprintf(stderr, "Warning: perspective transform %d, %d\n", ic, jc);
	    return 0;
	}

	temp = (2*Cd[0]*Ic + 2*Cd[1]*Jc + 4*Cd[2] - (Cd[6]*Ic + Cd[7]*Jc + 2*Cd[8])*Cd[9]) / denom;
	*Ic_ = round_toward_positive(temp);
	temp = (2*Cd[3]*Ic + 2*Cd[4]*Jc + 4*Cd[5] - (Cd[6]*Ic + Cd[7]*Jc + 2*Cd[8])*Cd[9]) / denom;
	*Jc_ = round_toward_positive(temp);
	break;
      default:
	fprintf(stderr, "VOP_synthesis_Chrom: no_of_sprite_warping_points error: %d\n",
		no_of_sprite_points);
	exit(-1);
    }

    return 1;
}


/***********************************************************CommentBegin******
 *
 * -- set_warping_para -- setting constants value for GMC caluculation
 *
 * Author :		
 *	Seishi TAKAMURA (NTT)
 *
 * Created :		
 *	03.03.99
 *
 * Purpose :		
 *	setting constants value for GMC caluculation
 * 
 * Arguments in : 	
 *	Vop	*sprite		sprite Vop data
 *
 * Arguments in/out :	
 *
 * Arguments out :	
 *	Int	C[]		constants value for GMC caluculation
 *	Double	Cd[]		constants value for GMC caluculation
 *
 * Return values :	
 *	
 * Side effects :	
 *	
 * Description :	
 *
 * See also :
 *
 * Modified :		
 *	18.08.99 Seishi TAKAMURA (NTT): modified GMC coding (rounding control)
 *
 ***********************************************************CommentEnd********/
Void
set_warping_para(Int		C[],
		 Double		Cd[],
		 Vop		*sprite)
{
    Int		s	= 2 << GetVopWarpingAccuracy(sprite);
    Int		r	= 16 / s;
    Int		W, H;
    Int		W_, H_;
    Int		i_[4], j_[4];
    Int		i__[4], j__[4];
    Int		x[4], y[4];
    TrajPoint	*Traj;
    Int		Idx;

    if (GetVopNoOfSpritePoints(sprite) == 0) {
	C[0] = C[1] = 0;
	C[2] = s;
	C[3] = C[4] = 0;

	return;
    }

    W = GetVopRefPointCoord(sprite)[1].x - GetVopRefPointCoord(sprite)[0].x;
    H = GetVopRefPointCoord(sprite)[2].y - GetVopRefPointCoord(sprite)[0].y;

    /* W' */
    for (W_ = 1; W_ < W; ) {
	W_ <<= 1;
    }

    /* H' */
    for (H_ = 1; H_ < H; ) {
	H_ <<= 1;
    }

    /* x, y */
    x[0] = GetVopHorSpatRef(sprite);	y[0] = GetVopVerSpatRef(sprite);
    x[1] = x[0] + W;			y[1] = y[0];
    x[2] = x[0];			y[2] = y[0] + H;
    x[3] = x[1];			y[3] = y[2];

    /* ir'(n), jr'(n) */
    Traj = GetVopTrajPointCoord(sprite);
    for (Idx = 0; Idx < GetVopNoOfSpritePoints(sprite); Idx++) {
	i_[Idx] = (s/2) * (2 * x[Idx] + Traj[Idx].x);
	j_[Idx] = (s/2) * (2 * y[Idx] + Traj[Idx].y);
    }

    /* ir'', jr'' */
    if (GetVopNoOfSpritePoints(sprite) >= 2) {
	i__[1] = 16 * (x[0] + W_) +
	    Round((Double) ((W - W_) * (r * i_[0] - 16 * x[0]) +
			    W_      * (r * i_[1] - 16 * x[1])) / (Double) W);
	j__[1] = 16 * y[0] +
	    Round((Double) ((W - W_) * (r * j_[0] - 16 * y[0]) +
			    W_      * (r * j_[1] - 16 * y[1])) / (Double) W);

	if (GetVopNoOfSpritePoints(sprite) >= 3) {
	    i__[2] = 16 * x[0] +
		Round((Double) ((H - H_) * (r * i_[0] - 16 * x[0]) +
				H_      * (r * i_[2] - 16 * x[2])) / (Double) H);
	    j__[2] = 16 * (y[0] + H_) +
		Round((Double) ((H - H_) * (r * j_[0] - 16 * y[0]) +
				H_      * (r * j_[2] - 16 * y[2])) / (Double) H);
	}
    }

    switch (GetVopNoOfSpritePoints(sprite)) {
      case 0:	/* Stationary transform */
	C[0] = C[1] = 0;
	C[2] = s;
	C[3] = C[4] = 0;
      case 1:	/* Translational transform */
	C[0] = i_[0];
	C[1] = j_[0];
	C[2] = s;
	C[3] = ((i_[0] >> 1) | (i_[0] & 1));
	C[4] = ((j_[0] >> 1) | (j_[0] & 1));
	break;
      case 2:	/* Isotropic transform */
	C[0] = i_[0];
	C[1] = (-r * i_[0] + i__[1]);
	C[2] = ( r * j_[0] - j__[1]);
	C[3] = j_[0];
	C[4] = (-r * j_[0] + j__[1]);
	C[5] = (-r * i_[0] + i__[1]);
	C[6] = W_ * r;
	C[7] = 4 * W_ * r;
	C[8] = (2 * W_ * r * i_[0]) - (16 * W_);
	C[9] = (2 * W_ * r * j_[0]) - (16 * W_);
	break;
      case 3:	/* Affine transform */
/*
	C[0] = i_[0];
	C[1] = (-r * i_[0] + i__[1]) * H_;
	C[2] = (-r * i_[0] + i__[2]) * W_;
	C[3] = j_[0];
	C[4] = (-r * j_[0] + j__[1]) * H_;
	C[5] = (-r * j_[0] + j__[2]) * W_;
	C[6] = W_ * H_ * r;
	C[7] = 4 * W_ * H_ * r;
	C[8] = (2 * W_ * H_ * r * i_[0]) - (16 * W_ * H_);
	C[9] = (2 * W_ * H_ * r * j_[0]) - (16 * W_ * H_);
*/

	C[0] = i_[0];
	C[3] = j_[0];
	if (W_ > H_) {
	    C[1] = (-r * i_[0] + i__[1]);
	    C[2] = (-r * i_[0] + i__[2]) * (W_ / H_);
	    C[4] = (-r * j_[0] + j__[1]);
	    C[5] = (-r * j_[0] + j__[2]) * (W_ / H_);
	    C[6] = W_ * r;
	    C[7] = 4 * W_ * r;
	    C[8] = (2 * W_ * r * i_[0]) - (16 * W_ );
	    C[9] = (2 * W_ * r * j_[0]) - (16 * W_ );
	} else {
	    C[1] = (-r * i_[0] + i__[1]) * (H_ / W_);
	    C[2] = (-r * i_[0] + i__[2]);
	    C[4] = (-r * j_[0] + j__[1]) * (H_ / W_);
	    C[5] = (-r * j_[0] + j__[2]);
	    C[6] = H_ * r;
	    C[7] = 4 * H_ * r;
	    C[8] = (2 * H_ * r * i_[0]) - (16 * H_);
	    C[9] = (2 * H_ * r * j_[0]) - (16 * H_);
	}

	break;
      case 4:	/* Perspective transform */
	Cd[6] = ((Double) (i_[0] - i_[1] - i_[2] + i_[3]) * (j_[2] - j_[3]) - 
		 (Double) (i_[2] - i_[3]) * (j_[0] - j_[1] - j_[2] + j_[3])) * H;
	Cd[7] = ((Double) (i_[1] - i_[3]) * (j_[0] - j_[1] - j_[2] + j_[3]) -
		 (Double) (i_[0] - i_[1] - i_[2] + i_[3]) * (j_[1] - j_[3])) * W;
	Cd[8] = ((Double) (i_[1] - i_[3]) * (j_[2] - j_[3]) -
		 (Double) (i_[2] - i_[3]) * (j_[1] - j_[3]));
	Cd[0] = Cd[8] * (i_[1] - i_[0]) * H + Cd[6] * i_[1];
	Cd[1] = Cd[8] * (i_[2] - i_[0]) * W + Cd[7] * i_[2];
	Cd[2] = Cd[8] * i_[0] * W * H;
	Cd[3] = Cd[8] * (j_[1] - j_[0]) * H + Cd[6] * j_[1];
	Cd[4] = Cd[8] * (j_[2] - j_[0]) * W + Cd[7] * j_[2];
	Cd[5] = Cd[8] * j_[0] * W * H;
	Cd[8] *= (W * H);
	Cd[9] = s;
	break;
      default:
	fprintf(stderr, "set_warping_para: no_of_sprite_warping_points error: %d\n",
		GetVopNoOfSpritePoints(sprite));
	exit(-1);
    }
}
