/******************************************************************************
 *                                                                          
 * This software module was originally developed by 
 *
 *   Fujitsu Laboratories Ltd. (contact: Eishi Morimatsu)
 *
 * in the course of development of the MPEG-4 Video (ISO/IEC 14496-2) standard.
 * This software module is an implementation of a part of one or more MPEG-4 
 * Video (ISO/IEC 14496-2) tools as specified by the MPEG-4 Video (ISO/IEC 
 * 14496-2) standard. 
 *
 * ISO/IEC gives users of the MPEG-4 Video (ISO/IEC 14496-2) standard free 
 * license to this software module or modifications thereof for use in hardware
 * or software products claiming conformance to the MPEG-4 Video (ISO/IEC 
 * 14496-2) standard. 
 *
 * Those intending to use this software module in hardware or software products
 * are advised that its use may infringe existing patents. THE ORIGINAL 
 * DEVELOPER OF THIS SOFTWARE MODULE AND HIS/HER COMPANY, THE SUBSEQUENT 
 * EDITORS AND THEIR COMPANIES, AND ISO/IEC HAVE NO LIABILITY FOR USE OF THIS 
 * SOFTWARE MODULE OR MODIFICATIONS THEREOF. No license to this software
 * module is granted for non MPEG-4 Video (ISO/IEC 14496-2) standard 
 * conforming products.
 *
 * Fujitsu Laboratories Ltd. retains full right to use the software module for 
 * his/her own purpose, assign or donate the software module to a third party 
 * and to inhibit third parties from using the code for non MPEG-4 Video 
 * (ISO/IEC 14496-2) standard conforming products. This copyright notice must 
 * be included in all copies or derivative works of the software module. 
 *
 * Copyright (c) 1999.  
 *
 *****************************************************************************/

/***********************************************************HeaderBegin*******
 *
 * File:        drc_util_common.c
 *
 * Author:      Fujitsu Laboratories - Eishi Morimatsu
 * Created:     06-SEP-1999
 *
 * Description: modules for DRC
 *
 * Notes:
 *
 * Modified:    
 *    14.12.99 Eishi Morimatsu (Fujitsu Labs.): bug fix for DRC
 *    28.12.99 Eishi Morimatsu (Fujitsu Labs.): bug fix for MeanUpSamplingMB
 *
 ***********************************************************HeaderEnd*********/


/************************    INCLUDE FILES    ********************************/
#include <stdlib.h>
#include <stdio.h>
#include "momusys.h"
#include "vm_common_defs.h"
#include "drc_util_common.h"

#define	RRU_BLOCK_SIZE	16
#define Max(x,y) (((x)>(y))?(x):(y))
#define Min(x,y) (((x)<(y))?(x):(y))

/***********************************************************CommentBegin******
 *
 * -- UpSamplingTexture --
 *
 * Author :
 *      Fujitsu Laboratories - Eishi Morimatsu
 *
 * Created :
 *      06-SEP-1999
 *
 * Purpose :
 *	DRC - upsampling of Reduced Resolution VOP's texturet data
 *
 * Arguments in :
 *	Vop *lowres_vop - reduced resolution VOP(input)
 *	Vop *normal_vop - normal resolution VOP(output)
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Void 
UpSamplingTexture(Vop *lowres_vop, Vop *normal_vop)
{
  Image *Y_low, *U_low, *V_low, *Y_norm, *U_norm, *V_norm;

  Y_low  = (Image *)GetVopY(lowres_vop);
  U_low  = (Image *)GetVopU(lowres_vop);
  V_low  = (Image *)GetVopV(lowres_vop);

  Y_norm = (Image *)GetVopY(normal_vop);
  U_norm = (Image *)GetVopU(normal_vop);
  V_norm = (Image *)GetVopV(normal_vop);

  UpSamplingImage(Y_low, Y_norm);
  UpSamplingImage(U_low, U_norm);
  UpSamplingImage(V_low, V_norm);
}

/***********************************************************CommentBegin******
 *
 * -- UpSamplingImage --
 *
 * Author :
 *      Fujitsu Laboratories - Eishi Morimatsu
 *
 * Created :
 *      06-SEP-1999
 *
 * Purpose :
 *	DRC - upsampling of Reduced Resolution VOP's image data
 *
 * Arguments in :
 *	Image *lowres_image - image data of reduced resolution VOP(input)
 *	Image *normal_image - image data of normal resolution VOP(output)
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Void 
UpSamplingImage(Image *lowres_image, Image *normal_image)
{
  Int low_sizex = GetImageSizeX(lowres_image);
  Int nor_sizex = GetImageSizeX(normal_image);
  Int nor_sizey = GetImageSizeY(normal_image);
  SInt *lowres_data, *normal_data;
  SInt block_low[64], block_norm[256];
  Int bx,by,x,y, pos, cnt;

  lowres_data = (SInt *)GetImageData(lowres_image);
  normal_data = (SInt *)GetImageData(normal_image);

  for(by = 0; by < (nor_sizey/2); by+=8){  
    for(bx = 0; bx < (nor_sizex/2); bx+=8){

      for(y = by, cnt = 0; y < by + 8; y ++){
        pos = bx + (y) * low_sizex; 
        for(x = bx; x < bx + 8; x++){
	  block_low[cnt] = lowres_data[pos];
          cnt++;
          pos++;
	}   
      }    

      MeanUpSamplingMB(block_low, block_norm, 8, 8);

      for(y = 2 * by, cnt = 0; y < 2 * by + 16; y ++){
        pos = 2 * bx + (y) * (nor_sizex);
        for(x = 2 * bx; x < 2 * bx + 16; x++){
	  normal_data[pos] =  block_norm[cnt];
          cnt++;
          pos++;
        }
      }  
    }
  }
  return;
}

/***********************************************************CommentBegin******
 *
 * -- MeanUpSamplingMB --
 *
 * Author :
 *      Fujitsu Laboratories - Eishi Morimatsu
 *
 * Created :
 *      06-SEP-1999
 *
 * Purpose :
 *	DRC - upsampling of a block in the Reduced Resolution VOP
 *
 * Arguments in :
 *      SInt *pSrcPic - pointer to the block data to be upsampled 
 *      SInt *pDestPic - pointer to the block data upsampled 
 *      Int   width - width of the block to be upsampled
 *      Int   height - height of the block to be upsampled
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *    27.12.99 Eishi Morimatsu (Fujitsu Labs.): bug fix for block corner filtering
 *
 ***********************************************************CommentEnd********/
Void
MeanUpSamplingMB(
    SInt       *pSrcPic,
    SInt       *pDestPic,
    Int         width,
    Int         height )
{
    int         i, j, x0, y0, x1, y1, x2, y2;
    int         val;

    for (j = 0; j < height; j++){
        for (i = 0; i < width; i++){
            x0 = i;
            y0 = j;
            x1 = Max(0, (i - 1));
            y1 = Max(0, (j - 1));
            x2 = Min((width - 1), (i + 1));
            y2 = Min((height - 1), (j + 1));

            if(i==0 && j==0) {   /* DRC 1999.12.27 */
              pDestPic[i * 2 + j * 2 * width * 2] = pSrcPic[x0 + y0 * width]; /* DRC 1999.12.27 */
	    } else {  /* DRC 1999.12.27 */
            val = 9 * pSrcPic[x0 + y0 * width] +
                  3 * pSrcPic[x1 + y0 * width] +
                  3 * pSrcPic[x0 + y1 * width] +
                  1 * pSrcPic[x1 + y1 * width];
            pDestPic[i * 2 + j * 2 * width * 2] = (val + 8) / 16;
	    }  /* DRC 1999.12.27 */
	    
            if(i==(width-1) && j==0) {   /* DRC 1999.12.27 */
              pDestPic[i * 2 + 1 + j * 2 * width * 2] = pSrcPic[x0 + y0 * width]; /* DRC 1999.12.27 */
	    } else {  /* DRC 1999.12.27 */
            val = 9 * pSrcPic[x0 + y0 * width] +
                  3 * pSrcPic[x2 + y0 * width] +
                  3 * pSrcPic[x0 + y1 * width] +
                  1 * pSrcPic[x2 + y1 * width];
            pDestPic[i * 2 + 1 + j * 2 * width * 2] = (val + 8) / 16;
	    }  /* DRC 1999.12.27 */

            if(i==0 && j==(height-1)) {   /* DRC 1999.12.27 */
              pDestPic[i * 2  + (j * 2 + 1) * width * 2] = pSrcPic[x0 + y0 * width]; /* DRC 1999.12.27 */
	    } else {  /* DRC 1999.12.27 */
            val = 9 * pSrcPic[x0 + y0 * width] +
                  3 * pSrcPic[x1 + y0 * width] +
                  3 * pSrcPic[x0 + y2 * width] +
                  1 * pSrcPic[x1 + y2 * width];
            pDestPic[i * 2  + (j * 2 + 1) * width * 2] = (val + 8) / 16;
	    }  /* DRC 1999.12.27 */

            if(i==(width-1) && j==(height-1)) {   /* DRC 1999.12.27 */
              pDestPic[i * 2 + 1 + (j * 2 + 1)* width * 2] = pSrcPic[x0 + y0 * width]; /* DRC 1999.12.27 */
	    } else {  /* DRC 1999.12.27 */
            val = 9 * pSrcPic[x0 + y0 * width] +
                  3 * pSrcPic[x2 + y0 * width] +
                  3 * pSrcPic[x0 + y2 * width] +
                  1 * pSrcPic[x2 + y2 * width];
            pDestPic[i * 2 + 1 + (j * 2 + 1)* width * 2] = (val + 8) / 16;
	    }  /* DRC 1999.12.27 */
        }
    }

}

/***********************************************************CommentBegin******
 *
 * -- BlockBoundaryFilter --
 *
 * Author :
 *      Fujitsu Laboratories - Eishi Morimatsu
 *
 * Created :
 *      06-SEP-1999
 *
 * Purpose :
 *	DRC - block boundary filtering of Reduced Resolution VOP
 *
 * Arguments in :
 *	Vop *rec_vop - reduced resolution VOP
 *	Vop *cod_flag_array - COD flags of the reduced resolution VOP
 *	Int *vp_num_array - video packet number for each Macroblock
 *	Int vp_edge_enable - flag for enabling the filter at video packet edges
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *    14.12.99 Eishi Morimatsu (Fujitsu Labs.): added vp_num_array and vp_edge_enable
 *
 ***********************************************************CommentEnd********/
Void 
BlockBoundaryFilter(Vop *rec_vop, Int *cod_flag_array, Int *vp_num_array, Int vp_edge_enable) /* DRC 1999.12.14 */
{
  Image *imageY, *imageU, *imageV;
  SInt *dataY, *dataU, *dataV;
  Int numMBx, numMBy;
  Int sizex, sizey;
  Int pMB, pPreMB;
  Int xcount, ycount; 
  Int curMB  = 0;

  imageY = (Image *)GetVopY(rec_vop);
  imageU = (Image *)GetVopU(rec_vop);
  imageV = (Image *)GetVopV(rec_vop);

  sizex = GetImageSizeX(imageY);
  sizey = GetImageSizeY(imageY);

  numMBx = sizex / RRU_BLOCK_SIZE / 2;
  numMBy = sizey / RRU_BLOCK_SIZE / 2;
  
  dataY = (SInt *)GetImageData(imageY);
  dataU = (SInt *)GetImageData(imageU);
  dataV = (SInt *)GetImageData(imageV);

#ifdef _DRC_DEBUG_
  /* Check cod-frag */
  fprintf (stdout, "Checking cod flag...\n");
  for (i = 0; i < numMBy; i++)
    {
      for (j = 0; j < numMBx; j++)
	{
	  fprintf (stdout, "%d ", cod_flag_array[i*numMBx+j]);
	}
      fprintf (stdout, "\n");
    } 

  /* Check vp-number */                                              /* DRC 1999.12.14 */
  fprintf (stdout, "Checking vp edge enable = %d\n",vp_edge_enable); /* DRC 1999.12.14 */
  fprintf (stdout, "Checking vp number...\n");                       /* DRC 1999.12.14 */
  for (i = 0; i < numMBy; i++)                                       /* DRC 1999.12.14 */
    {                                                                /* DRC 1999.12.14 */
      for (j = 0; j < numMBx; j++)                                   /* DRC 1999.12.14 */
	{                                                            /* DRC 1999.12.14 */
	  fprintf (stdout, "%2d ", vp_num_array[i*numMBx+j]);        /* DRC 1999.12.14 */
	}                                                            /* DRC 1999.12.14 */
      fprintf (stdout, "\n");                                        /* DRC 1999.12.14 */
    }                                                                /* DRC 1999.12.14 */
#endif

/*--- Filtering Macro-Block Borderline (Lum part) ---*/
/*    printf("Filtering Macro-Block Horizontal Borderline (Lum)\n"); */
    for (ycount=1; ycount<(numMBy * 2); ycount++) {
        curMB = (ycount / 2) * numMBx;

        for (xcount=0; xcount<(numMBx * 2); xcount+=2, curMB++) {
            pMB = curMB;
	
            if (ycount % 2) {
                pPreMB = pMB;
            }
            else {
                pPreMB = (pMB - numMBx);
            }

            if (((cod_flag_array[pMB] == 1) || (cod_flag_array[pPreMB] == 1))&&
                ((vp_num_array[pMB] == vp_num_array[pPreMB])||(vp_edge_enable))) { /* DRC 1999.12.14 */
                FilterMBHorBorder( dataY, sizex, xcount, ycount );
                FilterMBHorBorder( dataY, sizex, (xcount+1), ycount );
            }
        }
    }
/*    printf("Done\n"); */

/*    printf("Filtering Macro-Block Vertical Borderline (Lum)\n"); */
    for (ycount=0; ycount<(numMBy * 2); ycount+=2) {
        curMB = (ycount / 2) * numMBx;

        for (xcount=1; xcount<(numMBx * 2); xcount++) {
            pMB = curMB + (xcount / 2);
            if (xcount % 2) {
                pPreMB = pMB;
            }
            else {
                pPreMB = pMB - 1;
            }

            if (((cod_flag_array[pMB] == 1) || (cod_flag_array[pPreMB] == 1))&&
                ((vp_num_array[pMB] == vp_num_array[pPreMB])||(vp_edge_enable))) { /* DRC 1999.12.14 */
                FilterMBVerBorder( dataY, sizex, xcount, ycount );
                FilterMBVerBorder( dataY, sizex, xcount, (ycount+1) );
            }
        }
    }
/*    printf("Done\n"); */

/*--- Filtering Macro-Block Borderline (Cb & Cr part) ---*/
/*    printf("Filtering Macro-Block Horizontal Borderline (Cb & Cr)\n"); */
    for (ycount=1; ycount<numMBy; ycount++) {
        curMB = ycount * numMBx;

        for (xcount=0; xcount<numMBx; xcount++, curMB++) {
            pMB    = curMB; 
            pPreMB = pMB - numMBx;

            if (((cod_flag_array[pMB] == 1) || (cod_flag_array[pPreMB] == 1))&&
                ((vp_num_array[pMB] == vp_num_array[pPreMB])||(vp_edge_enable))) { /* DRC 1999.12.14 */
                FilterMBHorBorder( dataU, sizex/2, xcount, ycount );
                FilterMBHorBorder( dataV, sizex/2, xcount, ycount );
            }
        }
    }
/*    printf("Done\n"); */

/*    printf("Filtering Macro-Block Vertical Borderline (Cb & Cr)\n"); */
    for (ycount=0; ycount<numMBy; ycount++) {
        curMB = ycount * numMBx;

        for (xcount=1; xcount<numMBx; xcount++) {
            curMB++;
            pMB    = curMB;
            pPreMB = pMB - 1;

            if (((cod_flag_array[pMB] == 1) || (cod_flag_array[pPreMB] == 1))&&
                ((vp_num_array[pMB] == vp_num_array[pPreMB])||(vp_edge_enable))) { /* DRC 1999.12.14 */
                FilterMBVerBorder( dataU, sizex/2, xcount, ycount );
                FilterMBVerBorder( dataV, sizex/2, xcount, ycount );
            }
        }
    }
/*    printf("Done\n"); */

}

/***********************************************************CommentBegin******
 *
 * -- FilterMBVerBorder --
 *
 * Author :
 *      Fujitsu Laboratories - Eishi Morimatsu
 *
 * Created :
 *      06-SEP-1999
 *
 * Purpose :
 *	DRC - block boundary filtering of Reduced Resolution VOP(vertical border)
 *
 * Arguments in :
 *      SInt *pPicTop - pointer to the image data
 *      Int  sizex - width of the image
 *      Int  curX - X position of current MB
 *      Int  curY - Y position of current MB
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Void
FilterMBVerBorder(
    SInt       *pPicTop,
    Int         sizex,
    Int         curX,
    Int         curY)
{
    SInt      *pPixel1, *pPixel2;
    int        val1, val2;
    int i;

    curX *= RRU_BLOCK_SIZE;
    curY *= RRU_BLOCK_SIZE;

    /*--- Read Cb Data ---*/
    pPixel1 = pPicTop + ((curX - 1) + curY * sizex);
    pPixel2 = pPicTop + ( curX      + curY * sizex);

    for (i=0; i<RRU_BLOCK_SIZE; i++) {
        val1 = *pPixel1;
        val2 = *pPixel2;

        *pPixel1 = (3 * val1 + val2 + 2) / 4;
        *pPixel2 = (val1 + 3 * val2 + 2) / 4;

        pPixel1 += sizex;
        pPixel2 += sizex;
    }

}


/***********************************************************CommentBegin******
 *
 * -- FilterMBHorBorder --
 *
 * Author :
 *      Fujitsu Laboratories - Eishi Morimatsu
 *
 * Created :
 *      06-SEP-1999
 *
 * Purpose :
 *	DRC - block boundary filtering of Reduced Resolution VOP(horizontal border)
 *
 * Arguments in :
 *      SInt *pPicTop - pointer to the image data
 *      Int  sizex - width of the image
 *      Int  curX - X position of current MB
 *      Int  curY - Y position of current MB
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Void
FilterMBHorBorder(
    SInt       *pPicTop,
    Int         sizex,
    Int         curX,
    Int         curY)
{
    SInt      *pPixel1, *pPixel2;
    int        val1, val2;
    int	       i;

    curX *= RRU_BLOCK_SIZE;
    curY *= RRU_BLOCK_SIZE;

    /*--- Read Data ---*/
    pPixel1 = pPicTop + (curX + (curY - 1) * sizex);
    pPixel2 = pPicTop + (curX +  curY      * sizex);

    for (i=0; i<RRU_BLOCK_SIZE; i++) {
        val1 = *pPixel1;
        val2 = *pPixel2;

        *pPixel1 = (3 * val1 + val2 + 2) / 4;
        *pPixel2 = (val1 + 3 * val2 + 2) / 4;

        pPixel1++;
        pPixel2++;
    }

}

/***********************************************************CommentBegin******
 *
 * -- CopyQPStore --
 *
 * Author :
 *      Fujitsu Laboratories - Eishi Morimatsu
 *
 * Created :
 *      06-SEP-1999
 *
 * Purpose :
 *	DRC - copying the stored QP data from reduced resolution VOP to normal resolution VOP
 *
 * Arguments in :
 *      Vop *lowres_vop - reduced resolution vop
 *      Vop *normal_vop - normal resolution vop
 *
 * Arguments in/out :
 *
 * Arguments out :
 *
 * Return values :
 *
 * Side effects :
 *
 * Description :
 *
 * See also :
 *
 * Modified :
 *
 ***********************************************************CommentEnd********/
Void 
CopyQPStore(Vop *lowres_vop, Vop *normal_vop)
{
  Image *Q_low, *Q_norm;

  Q_low  = (Image *)GetVopQP(lowres_vop);
  Q_norm = (Image *)GetVopQP(normal_vop);

  CopyImageI_RR(Q_low, Q_norm);
} 





