/****************************************************************************/
/*   MPEG4 Visual Texture Coding (VTC) Mode Software                        */
/*                                                                          */
/*   This software was jointly developed by the following participants:     */
/*                                                                          */
/*   Single-quant,  multi-quant and flow control                            */
/*   are provided by  Sarnoff Corporation                                   */
/*     Iraj Sodagar   (iraj@sarnoff.com)                                    */
/*     Hung-Ju Lee    (hjlee@sarnoff.com)                                   */
/*     Paul Hatrack   (hatrack@sarnoff.com)                                 */
/*     Shipeng Li     (shipeng@sarnoff.com)                                 */
/*     Bing-Bing Chai (bchai@sarnoff.com)                                   */
/*     B.S. Srinivas  (bsrinivas@sarnoff.com)                               */
/*                                                                          */
/*   Bi-level is provided by Texas Instruments                              */
/*     Jie Liang      (liang@ti.com)                                        */
/*                                                                          */
/*   Shape Coding is provided by  OKI Electric Industry Co., Ltd.           */
/*     Zhixiong Wu    (sgo@hlabs.oki.co.jp)                                 */
/*     Yoshihiro Ueda (yueda@hlabs.oki.co.jp)                               */
/*     Toshifumi Kanamaru (kanamaru@hlabs.oki.co.jp)                        */
/*                                                                          */
/*   Scalable shape coding is provided by Samsung AIT			    */
/*     Dae-Sung Cho (dscho@sait.samsung.co.kr)				    */
/*     Se-Hoon Son (shson@sait.samsung.co.kr)				    */
/*									    */
/*   OKI, Sharp, Sarnoff, TI and Microsoft contributed to bitstream         */
/*   exchange and bug fixing.                                               */
/*                                                                          */
/*                                                                          */
/* In the course of development of the MPEG-4 standard, this software       */
/* module is an implementation of a part of one or more MPEG-4 tools as     */
/* specified by the MPEG-4 standard.                                        */
/*                                                                          */
/* The copyright of this software belongs to ISO/IEC. ISO/IEC gives use     */
/* of the MPEG-4 standard free license to use this  software module or      */
/* modifications thereof for hardware or software products claiming         */
/* conformance to the MPEG-4 standard.                                      */
/*                                                                          */
/* Those intending to use this software module in hardware or software      */
/* products are advised that use may infringe existing  patents. The        */
/* original developers of this software module and their companies, the     */
/* subsequent editors and their companies, and ISO/IEC have no liability    */
/* and ISO/IEC have no liability for use of this software module or         */
/* modification thereof in an implementation.                               */
/*                                                                          */
/* Permission is granted to MPEG members to use, copy, modify,              */
/* and distribute the software modules ( or portions thereof )              */
/* for standardization activity within ISO/IEC JTC1/SC29/WG11.              */
/*                                                                          */
/* Copyright 1995, 1996, 1997, 1998 ISO/IEC                                 */
/****************************************************************************/

#include <stdio.h>
#include <memory.h>
#include <string.h>
#include "typedef.h"

#include "ShapeCodec.h"
#include "ShapeCommon.h"
#include "shape_def.h"
#include "ShapeUtil.h"

#include "dwt.h"
#include "BinArCodec.h"
#include "bitpack.h"
#include "errorHandler.h"

#include "dwtmask.h"
#include "idwtmask.h"

#define Error -1

/* begin : added by D.-S.Cho, Samsung AIT (99/04/13) */
Int ShapeEnCoding_V1(UChar *inmask, 
		  Int object_width, 
		  Int object_height, 
		  Int alphaTH,
		  Int change_CR_disable,
		  Int constAlpha,
		  UChar constAlphaValue)
{
  Int	blks		= 4;
  Int	blkn		= 1<<blks;
  Int	blkx		= (object_width+15)/16;
  Int	blky		= (object_height+15)/16;

  Int	ret,bitstream_start,bitstream_length;
  Int	**shape_mode = malloc_2d_Int(blky,blkx);

  UChar **BAB		= malloc_2d_Char(blkn,blkn);
  UChar **BABdown	= malloc_2d_Char(blkn,blkn);
  UChar **BAB_ext	= malloc_2d_Char(blkn+4,blkn+4);
  UChar **shape	= malloc_2d_Char(object_height,object_width);
  Int	i, j, k, l, status;
  Int	bsize;
  Int	CR;

  Int	shape_bitstream_length;
  BSS	*shape_bitstream;	

  /* Initialize shape bitstream buffer */
  shape_bitstream = AllocBitstream(object_width*object_height);
  InitBitstream(1,shape_bitstream);
  
  fprintf(stderr,"Coding Shape Header Syntax...\n");
  PutBitstoStream(1,change_CR_disable,shape_bitstream);
  PutBitstoStream(1,(constAlpha?1:0),shape_bitstream);
  if(constAlpha)
    PutBitstoStream(8,(Int)constAlphaValue,shape_bitstream);
  
  bitstream_start= shape_bitstream -> cnt;

  for(i=0;i<object_height;i++)
      for(j=0;j<object_width;j++)
                shape[i][j]= *(inmask + i*object_width+j);

  for ( j=0; j<blky; j++ ) {
      for ( i=0; i<blkx; i++ ) {
	/* get BAB. */
    	for ( l=0; l<blkn; l++ ) {
	  for ( k=0; k<blkn; k++ ) {
            BAB[l][k] = ((((j<<blks)+l)<object_height)
			 && (((i<<blks)+k) <object_width))?
	      ( shape[(j<<blks)+l][(i<<blks)+k]!=0 ):0;
	  }
    	}
	/* check content of BAB. */
        status = decide_CR(i,j,blkn,
				object_width,object_height,
				blkx, BAB,BABdown,change_CR_disable,alphaTH,shape);

        if ( status==ALL0 || status==ALL255 ) {
		shape_mode[j][i] = status;
		CR = 1;
	} else {				
		shape_mode[j][i] = BORDER;
	    	CR = ( 1<<(status-CR1_1) );
	}

	ret = ShapeBaseLayerModeEncode(shape_mode,i,j,blkx,CR,change_CR_disable,shape_bitstream);
        if ( ret==Error ) {
                fprintf(stderr,"\n CAE arithmetic coding Error !\n");
                return  Error;
        }

        if ( status==ALL0 || status==ALL255 ) {
	    /* All 0 or 255, hence no CAE is needed.  */
            for ( l=0; l<blkn; l++ ) {
              for ( k=0; k<blkn; k++ ) {
	        if((j<<blks)+l < object_height && (i<<blks)+k < object_width)
                  shape[(j<<blks)+l][(i<<blks)+k] = ( status==ALL0 ? 0 : 1 );
              }
            }
	} else {
	    /* Add surrounding pixels for CAE. */
            AddBorderToBAB(i,j,blkn,object_width,object_height,
			CR,blkx,BABdown,BAB_ext,shape,1);

            bsize = blkn/CR;

	    ret = ShapeBaseLayerContentEncode(i,j,bsize,BAB_ext,shape_bitstream);

	    if( ret == Error ) {
                fprintf(stderr,"\n CAE arithmetic coding Error !\n");
                return  Error;
            }

            if ( CR!=1 ) {
		/* Reconstruct shape from BAB */
                UpSampling_Still(i,j,blkn,
				object_width,object_height,
				CR,blkx,BABdown,BAB,shape);
            }

            for ( l=0; l<blkn; l++ ) {
              for ( k=0; k<blkn; k++ ) {
	        if((j<<blks)+l < object_height && (i<<blks)+k < object_width)
                  shape[(j<<blks)+l][(i<<blks)+k] = ( BAB[l][k]==0 ? 0 : 1 );
              }
            }
	}
      }
  }
  bitstream_length = shape_bitstream -> cnt - bitstream_start + 1;

  fprintf(stderr, "\tshape bits = %d\n", bitstream_length);

  /* merge shape bitstream into main bit stream */
  if(shape_bitstream==NULL) {
    fprintf(stderr,"shape_bitstream Not Available\n");
    exit(1);
  }
  shape_bitstream_length = shape_bitstream->cnt;

  InitBitstream(0,shape_bitstream);
  BitStreamMerge(shape_bitstream_length, shape_bitstream);
  ByteAlignmentEncCopy(shape_bitstream);

  FreeBitstream(shape_bitstream);

  free_2d_Char ( BAB, blkn );
  free_2d_Char ( BABdown, blkn );
  free_2d_Char ( BAB_ext, blkn+4 );

  free_2d_Int (shape_mode, blky);
  free_2d_Char (shape, object_height);

  return(shape_bitstream_length);
}
/* end : added by D.-S.Cho, Samsung AIT (99/04/13) */

Int ShapeEnCoding(UChar *inmask, 
		  Int width, 
		  Int height, 
		  Int levels, 
		  Int alphaTH,
		  Int change_CR_disable,
		  Int constAlpha,
		  UChar constAlphaValue, 
/*		  Int shapeScalable,*/	/* FPDAM : deleted by SAIT (99/09/03) */
		  Int startCodeEnable,
		  FILTER **filter)
{
  UChar *outmask, *recmask;
  Int i,j, k;
  Int object_height, object_width;
  Int minbsize;
  Int base_bits, enh_bits, shape_bitstream_length;
  BSS *shape_bitstream;	

/*  if(!shapeScalable) levels = 0; */	/* FPDAM : deleted by SAIT (99/09/03) */

  /* begin : added by D.-S.Cho, Samsung AIT (99/04/13) */
/* FPDAM begin: deleted by SAIT (99/09/03) */
  /* if(shapeScalable && !change_CR_disable) {
	fprintf(stderr,"Changing CR is not allowable in base layer shape for scalability !\n");
	fprintf(stderr," The change_CR_disable is always set to 1 !\n");
	change_CR_disable=1;
  } */
/* FPDAM end: deleted by SAIT (99/09/03) */
  /* end : added by D.-S.Cho, Samsung AIT (99/04/13) */

  minbsize = 1<< levels;
  object_width = (((width+minbsize-1)>>levels)<<levels);
  object_height = (((height+minbsize-1)>>levels)<<levels);
  if(object_width!=width || object_height!=height) {
    printf("Object width or height is not multiples of 2^levels\n");
    exit(1);
  }

  /* Initialize shape bitstream buffer */
  shape_bitstream = AllocBitstream(object_width*object_height);
  InitBitstream(1,shape_bitstream);
  
  outmask = (UChar *)
    malloc(object_width*object_height*sizeof(UChar));
  recmask = (UChar *)
    malloc(object_width*object_height*sizeof(UChar));

  if(outmask == NULL || recmask == NULL) {
    errorHandler("Memory allocation failed\n");
  }
  
  memset(recmask, 0, object_width*object_height);
  for(i=0;i<height;i++) 
    for(j=0;j<width;j++)
      recmask[i*object_width+j] = inmask[i*width+j]!=0?1:0;

  do_DWTMask(recmask, outmask, object_width, object_height, levels, filter);

  fprintf(stderr,"Coding Shape Header Syntax...\n");
/* FPDAM begin: modified by SAIT (99/09/03) */
  /* EncodeShapeHeader(change_CR_disable, constAlpha, constAlphaValue, 
			shapeScalable, shape_bitstream); */
  EncodeShapeHeader(change_CR_disable, constAlpha, constAlphaValue, shape_bitstream);
/* FPDAM end: modified by SAIT (99/09/03) */
  
  fprintf(stderr,"Coding Shape Base Layer...\n");
  base_bits = EncodeShapeBaseLayer(outmask, alphaTH, change_CR_disable,	
			object_width, object_height, levels, shape_bitstream);

  fprintf(stderr, "\tBase Layer bits = %d\n", base_bits);

/*  if(shapeScalable) { */	/* FPDAM : deleted by SAIT (99/09/03) */
    if(!startCodeEnable) {
      PutBitstoStream(4, levels, shape_bitstream);
      PutBitstoStream(1,MARKER_BIT, shape_bitstream);
    }
    for(k=levels;k>0;k--) {
      fprintf(stderr,"Coding Shape Enhanced Layer %d...\n", k);
      enh_bits = EncodeShapeEnhancedLayer(outmask, object_width, object_height, k,
                                startCodeEnable, filter[k-1], shape_bitstream);

      fprintf(stderr,"Enhanced Layer %d bits = %d\n", levels-k, enh_bits);
    }
    if(startCodeEnable) {
      ByteAlignmentEncCopy(shape_bitstream);
      /****
      shape_bitstream_length = shape_bitstream->cnt;
      InitBitstream(0,shape_bitstream);
      BitStreamMerge(shape_bitstream_length, shape_bitstream);
      ByteAlignmentEnc();
      InitBitstream(1,shape_bitstream);
      ****/

      PutBitstoStream(32,TEXTURE_SPATIAL_START_CODE, shape_bitstream);
      PutBitstoStream(5, 0, shape_bitstream);
      PutBitstoStream(1, MARKER_BIT, shape_bitstream);    
    }
/*  }	*/ /* FPDAM : deleted by SAIT (99/09/03) */

  /* merge shape bitstream into main bit stream */
  if(shape_bitstream==NULL) {
    fprintf(stderr,"shape_bitstream Not Available\n");
    exit(1);
  }
  shape_bitstream_length = shape_bitstream->cnt;

  InitBitstream(0,shape_bitstream);
  BitStreamMerge(shape_bitstream_length, shape_bitstream);

  FreeBitstream(shape_bitstream);

  free(outmask);
  free(recmask);

  return(shape_bitstream_length);
}

Int EncodeShapeHeader(Int change_CR_disable,
			Int constAlpha, 
			UChar constAlphaValue, 
/*			Int shapeScalable, */	/* FPDAM : deleted by SAIT (99/09/03) */
			BSS *shape_bitstream)
{
  PutBitstoStream(1,change_CR_disable,shape_bitstream);
  PutBitstoStream(1,(constAlpha?1:0),shape_bitstream);
  if(constAlpha)
    PutBitstoStream(8,(Int)constAlphaValue,shape_bitstream);
/*  PutBitstoStream(1,shapeScalable,shape_bitstream); */	/* FPDAM : deleted by SAIT (99/09/03) */
  PutBitstoStream(1,MARKER_BIT,shape_bitstream);
  return(0);
}

Int
EncodeShapeBaseLayer(UChar *outmask, Int alphaTH, Int change_CR_disable,	
			Int object_width, Int object_height, Int levels, BSS *shape_bitstream)
{
    Int	blks		= 4;
    Int	blkn		= 1<<blks;

    Int coded_width     = object_width >> levels;
    Int coded_height    = object_height >> levels;

    Int	blkx		= (coded_width+15)/16;
    Int	blky		= (coded_height+15)/16;

    Int ret,bitstream_start,bitstream_length;
    Int **shape_mode = malloc_2d_Int(blky,blkx);

    UChar **BAB		= malloc_2d_Char(blkn,blkn);
    UChar **BABdown	= malloc_2d_Char(blkn,blkn);
    UChar **BAB_ext	= malloc_2d_Char(blkn+4,blkn+4);
    UChar **shape	= malloc_2d_Char(coded_height,coded_width);
    Int	i, j, k, l, status;
    Int	bsize;
    Int	CR;

    bitstream_start= shape_bitstream -> cnt;

    for(i=0;i<coded_height;i++)
        for(j=0;j<coded_width;j++)
                shape[i][j]= *(outmask + i*object_width+j);

    for ( j=0; j<blky; j++ ) {
      for ( i=0; i<blkx; i++ ) {
	/* get BAB. */
    	for ( l=0; l<blkn; l++ ) {
	  for ( k=0; k<blkn; k++ ) {
            BAB[l][k] = ((((j<<blks)+l)<coded_height)
			 && (((i<<blks)+k) <coded_width))?
	      ( shape[(j<<blks)+l][(i<<blks)+k]!=0 ):0;
	  }
    	}
	/* check content of BAB. */
        status = decide_CR(i,j,blkn,
				coded_width,coded_height,
				blkx, BAB,BABdown,change_CR_disable,alphaTH,shape);

        if ( status==ALL0 || status==ALL255 ) {
		shape_mode[j][i] = status;
		CR = 1;
	} else {				
		shape_mode[j][i] = BORDER;
	    	CR = ( 1<<(status-CR1_1) );
	}

	ret = ShapeBaseLayerModeEncode(shape_mode,i,j,blkx,CR,change_CR_disable,shape_bitstream);
        if ( ret==Error ) {
                fprintf(stderr,"\n CAE arithmetic coding Error !\n");
                return  Error;
        }

        if ( status==ALL0 || status==ALL255 ) {
	    /* All 0 or 255, hence no CAE is needed.  */
            for ( l=0; l<blkn; l++ ) {
              for ( k=0; k<blkn; k++ ) {
	        if((j<<blks)+l < coded_height && (i<<blks)+k < coded_width)
                  shape[(j<<blks)+l][(i<<blks)+k] = ( status==ALL0 ? 0 : 1 );
              }
            }
	} else {
	    /* Add surrounding pixels for CAE. */
            AddBorderToBAB(i,j,blkn,coded_width,coded_height,
			CR,blkx,BABdown,BAB_ext,shape,1); /* modified by Z. Wu @ OKI */

            bsize = blkn/CR;

	    ret = ShapeBaseLayerContentEncode(i,j,bsize,BAB_ext,shape_bitstream);

	    if( ret == Error ) {
                fprintf(stderr,"\n CAE arithmetic coding Error !\n");
                return  Error;
            }

            if ( CR!=1 ) {
		/* Reconstruct shape from BAB */
                UpSampling_Still(i,j,blkn,
				coded_width,coded_height,
				CR,blkx,BABdown,BAB,shape);
            }

            for ( l=0; l<blkn; l++ ) {
              for ( k=0; k<blkn; k++ ) {
	        if((j<<blks)+l < coded_height && (i<<blks)+k < coded_width)
                  shape[(j<<blks)+l][(i<<blks)+k] = ( BAB[l][k]==0 ? 0 : 1 );
              }
            }
	}
      }
    }
    PutBitstoStream(1,MARKER_BIT,shape_bitstream);

    bitstream_length = shape_bitstream -> cnt - bitstream_start + 1;

    free_2d_Char ( BAB, blkn );
    free_2d_Char ( BABdown, blkn );
    free_2d_Char ( BAB_ext, blkn+4 );

    free_2d_Int (shape_mode, blky);
    free_2d_Char (shape, coded_height);

/*    free(shape);*/

    return (bitstream_length);
}

Int
ShapeBaseLayerModeEncode(Int **shape_mode, 
			Int i,
			Int j,
			Int blkx, 
			Int CR,
			Int change_CR_disable,
			BSS *shape_bitstream)
{
    Int		mode	= shape_mode[j][i];
    Int		dis_CR	= change_CR_disable;
    Int		UL = ( (i == 0      || j == 0) ? 0 : shape_mode[j-1][i-1] );
    Int		UR = ( (i == blkx-1 || j == 0) ? 0 : shape_mode[j-1][i+1] );
    Int		U  = ( (j == 0) ? 0 : shape_mode[j-1][i] );
    Int		L  = ( (i == 0) ? 0 : shape_mode[j][i-1] );
    Int		index	= ( 27 * UL + 9 * U + 3 * UR + L );
    Int		bits;
    U_Int	code;

    bits = LMMR_first_shape_code_I [index][mode];
    code = CMMR_first_shape_code_I [index][mode];

    PutBitstoStream ( bits, code, shape_bitstream );

    if ( mode == BORDER ) {
        if ( dis_CR == 0 ) { 
            bits = LCR [CR];
            code = CCR [CR];

            PutBitstoStream ( bits, code, shape_bitstream );
        }
    }

    return ( 0 );
}

Int
ShapeBaseLayerContentEncode(Int i,Int j,Int bsize,UChar **BAB,BSS *shape_bitstream)
{
    Int		k, l;
    Int		dir, dirMin=0;
    Int		len, lenMin = INT_MAX;
    Int		context;
    Char	target;
    Int		ctx_max = 1023;
    BSS		*tmp_bitstream[2];
    ArCoder	coder;

    for ( dir=0; dir<2; dir++ ) {

        tmp_bitstream[dir]  = (BSS *)malloc( sizeof( BSS ) );
        tmp_bitstream[dir] -> bs  = (UChar *) malloc(bsize*bsize*sizeof(UChar));

	memset( tmp_bitstream[dir] -> bs, (UChar)0, bsize*bsize); 

        InitBitstream ( 1, tmp_bitstream[dir] );

        if ( dir == 0 ) {

            StartArCoder_Still ( &coder );

            for ( l=0; l<bsize; l++ ) {
              for ( k=0; k<bsize; k++ ) {
                target  = BAB [l+2][k+2];
                context =	( BAB [l+2+(+0)][k+2+(-1)] << 0 ) +
		  ( BAB [l+2+(+0)][k+2+(-2)] << 1 ) +
		  ( BAB [l+2+(-1)][k+2+(+2)] << 2 ) +
		  ( BAB [l+2+(-1)][k+2+(+1)] << 3 ) +
		  ( BAB [l+2+(-1)][k+2+(+0)] << 4 ) +
		  ( BAB [l+2+(-1)][k+2+(-1)] << 5 ) +
		  ( BAB [l+2+(-1)][k+2+(-2)] << 6 ) +
		  ( BAB [l+2+(-2)][k+2+(+1)] << 7 ) +
		  ( BAB [l+2+(-2)][k+2+(+0)] << 8 ) +
		  ( BAB [l+2+(-2)][k+2+(-1)] << 9 );
		
                if ( context > ctx_max ) {
		  fprintf ( stderr, "\n Shape context Error !\n" );
		  return  Error;
                }
                ArCodeSymbol_Still( &coder, tmp_bitstream[dir], target, intra_prob[context] );	
              }
            }
            StopArCoder_Still( &coder, tmp_bitstream[dir] );
	    
            if ( (len  = tmp_bitstream[dir] -> cnt) < lenMin ) {
                lenMin = len;
                dirMin = dir;
            }

        } else {

            StartArCoder_Still ( &coder );

            for ( k=0; k<bsize; k++ ) {
              for ( l=0; l<bsize; l++ ) {
                target  = BAB [l+2][k+2];
                context = ( BAB [l+2+(-1)][k+2+(+0)] << 0 ) +
			( BAB [l+2+(-2)][k+2+(+0)] << 1 ) +
			( BAB [l+2+(+2)][k+2+(-1)] << 2 ) +
			( BAB [l+2+(+1)][k+2+(-1)] << 3 ) +
			( BAB [l+2+(+0)][k+2+(-1)] << 4 ) +
			( BAB [l+2+(-1)][k+2+(-1)] << 5 ) +
			( BAB [l+2+(-2)][k+2+(-1)] << 6 ) +
			( BAB [l+2+(+1)][k+2+(-2)] << 7 ) +
			( BAB [l+2+(+0)][k+2+(-2)] << 8 ) +
			( BAB [l+2+(-1)][k+2+(-2)] << 9 );

                if ( context > ctx_max ) {
                    fprintf ( stderr, "\n Shape context Error !\n" );
                    return  Error;
                }
                ArCodeSymbol_Still( &coder, tmp_bitstream[dir], target, intra_prob[context] );	

              }
            }
            StopArCoder_Still( &coder, tmp_bitstream[dir] );

            if ( (len  = tmp_bitstream[dir] -> cnt) < lenMin ) {
                lenMin = len;
                dirMin = dir;
            }

        }

    }

    PutBitstoStream ( LST, (U_Int)dirMin, shape_bitstream );
    InitBitstream ( 0, tmp_bitstream[dirMin] );
    BitStreamCopy ( lenMin, tmp_bitstream[dirMin], shape_bitstream );

    for ( dir=0; dir<2; dir++ ) {
        free ( tmp_bitstream[dir] -> bs );
        free ( tmp_bitstream[dir] );
    }

    return ( 0 );
}

Int
CheckBABstatus(Int blkn,UChar **BAB1,UChar **BAB2,Int alphaTH)
{
    Int	i, j, k, l;
    Int	SAD;
    Int	all0   = 0;
    Int	all255 = 0;

    for ( j=0; j<blkn; j+=4 ) {
      for ( i=0; i<blkn; i+=4 ) {

        SAD = 0;
        if ( BAB2==(UChar **)NULL ) {

            for ( l=0; l<4; l++ )
              for ( k=0; k<4; k++ )
                SAD += ( BAB1[j+l][i+k]!=0 );

            if ( 16 *     SAD  > alphaTH ) all0   = 1;
            if ( 16 * (16-SAD) > alphaTH ) all255 = 1;

            if ( all0==1 && all255==1 )
                return (BORDER);

        } else {

            for ( l=0; l<4; l++ )
              for ( k=0; k<4; k++ )
                SAD += ( BAB1[j+l][i+k]!=BAB2[j+l][i+k] );

            if ( 16 * SAD > alphaTH )
                return (BORDER);

        }
      }
    }
    if ( all0==0 ) return ( ALL0   );
    else           return ( ALL255 );
}

Int
CheckMB_Gray(Int BABsize,UChar **BAB)
{
    Int		i, j;
    Int		all0   = 0;
    Int		all255 = 0;

    for (j=0; j<BABsize; j++) {
      for (i=0; i<BABsize; i++) {

        if ( BAB [j][i] == 0 )	all255 = 1;
        else			all0   = 1;

        if ( all0 == 1 && all255 == 1 )	return (BORDER);
      }
    }
    if ( all0 == 0 ) return ( ALL0   );
    else             return ( ALL255 );
}

Int
decide_CR(Int x,
	Int y,
	Int blkn,
	Int coded_width,
	Int coded_height,
	Int blkx, 
	UChar **BAB_org,
	UChar **BAB_dwn,
	Int change_CR_disable,
	Int alphaTH,
	UChar **shape)
{
    Int		status;
    Int		i, j;

    /* estimation of all_0 or all_255 or gray at 16x16 first. */
    if ( (status=CheckBABstatus(blkn,BAB_org,(UChar **)NULL,alphaTH))!=BORDER )
        return (status);

    if ( change_CR_disable == 0 ) {
        UChar	**BAB_up = malloc_2d_Char ( blkn, blkn );

	/* estimation of CR = 1/4 conversion error. */
        DownSampling_Still(BAB_org,BAB_dwn,blkn/4,4);
    	UpSampling_Still(x,y,blkn,
			coded_width,coded_height,
			4,blkx,BAB_dwn,BAB_up,shape);

        if ( CheckBABstatus(blkn,BAB_org,BAB_up,alphaTH)!=BORDER ) {
            free_2d_Char(BAB_up, blkn);
            return (CR1_4 );
        }

	/* estimation of CR = 1/2 conversion error. */
        DownSampling_Still ( BAB_org, BAB_dwn, blkn/2, 2 );
        UpSampling_Still ( x, y, blkn, 
			coded_width,coded_height,
			2, blkx, BAB_dwn, BAB_up, shape );

        if ( CheckBABstatus ( blkn, BAB_org, BAB_up, alphaTH ) != BORDER ) {
            free_2d_Char ( BAB_up, blkn );
            return (CR1_2 );
        }
	free_2d_Char( BAB_up, blkn );
    }
    for ( j=0; j<blkn; j++ ) 
      for ( i=0; i<blkn; i++ ) 
        BAB_dwn[j][i] = BAB_org[j][i];

    return (CR1_1);
}

Int QuantizeShape(UChar *inmask, Int width, Int height, Int alphaTH)
{
  /* quantize the mask according to a 4x4 block alphaTH, should always favour zeros
   since it will reduce the coefficients to be coded*/
 
  return(0);
}

/***********************************************************CommentBegin******
 *
 * -- EncodeShapeEnhancedLayer -- Encodes a binary shape in the enhancement layer. 
 *
 * Author :
 *      Dae-Sung Cho (Samsung AIT)
 *
 * Created :
 *      02-Dec-98
 *
 * Arguments:
 *
 *
 * Return values :
 *
 * Side effects :
 *      -
 *
 * Description :        Alpha blocks in the enhancement layer will be coded by bac.
 *                      For the bab coding there are two kinds of coding modes
 *                      for the coded bab, i.e. transitional bab and exceptional bab coding. 
 *			- The transitional bab coding uses scan interleaving (SI) method
 *			  proposed by Samsung AIT. 
 *			- The exceptional bab coding is performed by the method proposed 
 *			  by Sarnoff Corp.
 *
 *                      These bits are added to the <shape_stream>.
 *
 * See also :
 *
 ***********************************************************CommentEnd********/

Int EncodeShapeEnhancedLayer(UChar *outmask,		/* shape mask in the layer (k) */
				Int object_width,	/* object_width in the current layer */
				Int object_height,	/* object_height in the current layer */
				Int levels,
				Int startCodeEnable,
				FILTER *filter,
				BSS *shape_bitstream)
{
    Int i, j, k, l, p, q, x, y, x2, y2;
    Int bab_type, scan_order, ret;

    Int coded_width	= object_width >> (levels-1),
	coded_height	= object_height >> (levels-1);
    Int width2		= coded_width,
	height2		= coded_height,
	width		= width2 >> 1,
	height		= height2 >> 1;

    Int NB		= (coded_width>=1024 || coded_height>=1024) ? 2 :
			  (coded_width>=512 || coded_height>=512) ? 1: 0;
    Int mborder		= MBORDER,
    	mblks           = 4+NB,
	mbsize          = 1<<mblks,		/* bab size in the current layer : 16 */
	mbsize_ext      = mbsize+(mborder<<1);  /* bordered bab size in the current layer: 20 */
 
    Int border		= BBORDER,
    	blks            = mblks-1,
	bsize           = 1<<blks,		/* bab size in the lower layer : 8 */
	bsize_ext       = bsize+(border<<1);	/* bordered bab size in the lower layer: 8 */
 
    Int blkx            = (coded_width+mbsize-1)/mbsize,
	blky            = (coded_height+mbsize-1)/mbsize;

    Int bitstream_length, bitstream_start;

    UChar *low_mask;
    UChar *cur_mask;
    UChar *half_mask;			/* shape mask at the half-higher layer */

    UChar *lower_bab;			/* alpha block in the lower layer */
    UChar *bordered_lower_bab;		/* bordered alpha block in the lower layer */
    UChar *half_bab;			/* alpha block in the half-higher layer */
    UChar *bordered_half_bab;		/* bordered alpha block in the half-higher layer */
    UChar *curr_bab;			/* alpha mb in the current layer */
    UChar *bordered_curr_bab;		/* bordered alpha mb in the current layer */

    ArCoder coder; 

    low_mask = (UChar *) calloc(width*height, sizeof(UChar));
    cur_mask = (UChar *) calloc(width2*height2, sizeof(UChar));
    half_mask = (UChar *) calloc(width*height2, sizeof(UChar));

    lower_bab = (UChar *) calloc(bsize*bsize, sizeof(UChar));
    bordered_lower_bab = (UChar *) calloc(bsize_ext*bsize_ext, sizeof(UChar));
    half_bab = (UChar *) calloc(bsize*mbsize, sizeof(UChar));
    bordered_half_bab = (UChar *) calloc(bsize_ext*mbsize_ext, sizeof(UChar));
    curr_bab = (UChar *) calloc(mbsize*mbsize, sizeof(UChar));
    bordered_curr_bab = (UChar *) calloc(mbsize_ext*mbsize_ext, sizeof(UChar));

    for(j=0;j<height;j++)
	for(i=0;i<width;i++)
		low_mask[j*width+i]= outmask[j*object_width+i];

    /* vertical first */
    ret=SynthesizeMaskHalfLevel(outmask, object_width, object_height, 
				levels, filter, NONZERO_HIGH, VERTICAL);
    if(ret!=0) {
    	errorHandler("Error Code=%d\n", ret);
    }
    for(j=0;j<height2;j++)
	for(i=0;i<width;i++)
		half_mask[j*width+i]= outmask[j*object_width+i];


    ret=SynthesizeMaskHalfLevel(outmask, object_width, object_height, 
				levels, filter, NONZERO_HIGH, HORIZONTAL);
    if(ret!=0) {
        errorHandler("Error Code=%d\n", ret);
    }
 
    for(j=0;j<height2;j++)
        for(i=0; i<width2;i++)
      		cur_mask[j*width2+i]= outmask[j*object_width+i];

    bitstream_start= shape_bitstream -> cnt;

    if(startCodeEnable) {
	ByteAlignmentEncCopy(shape_bitstream);
	/********
        bitstream_length = shape_bitstream->cnt;
        InitBitstream(0,shape_bitstream);
        BitStreamMerge(bitstream_length, shape_bitstream);
        ByteAlignmentEnc();

        InitBitstream(1,shape_bitstream);
        bitstream_start= shape_bitstream -> cnt;
	*********/

    	PutBitstoStream(32,TEXTURE_SHAPE_START_CODE,shape_bitstream);
    	PutBitstoStream(5,k,shape_bitstream);
    	PutBitstoStream(1,MARKER_BIT,shape_bitstream);
    }

    StartArCoder_Still(&coder);   

    for ( j=y=y2=0; j<blky; j++, y+=bsize, y2+=mbsize ) {
      for ( i=x=x2=0; i<blkx; i++, x+=bsize, x2+=mbsize ) {

	/* Initialize BABs */
	q = y*width;
	for ( l=p=0; l<bsize; l++, q+=width ) {
	  for ( k=0; k<bsize; k++, p++ ) {
              if(  y+l < height && x+k < width )
                lower_bab[p] = (low_mask[ q+x+k ] != 0);
              else
                lower_bab[p] = 0;
          }
	}
	q = y2*width;
	for ( l=p=0; l<mbsize; l++, q+=width ) {
	  for ( k=0; k<bsize; k++, p++ ) {
              if(  y2+l < height2 && x+k < width )
                half_bab[p] = (half_mask[ q+x+k ] != 0);
              else
                half_bab[p] = 0;
          }
	}
	q = y2*width2;
        for ( l=p=0; l<mbsize; l++, q+=width2 ) {
          for ( k=0; k<mbsize; k++, p++ ) {
	      if( y2+l < height2 && x2+k < width2)
                curr_bab[p]= (cur_mask[ q+x2+k ] != 0);
	      else
                curr_bab[p]= 0;
	  }
	}

	AddBorderToLowerBAB (low_mask, 
				lower_bab, bordered_lower_bab, 
				width, height,
				i, j, bsize, blkx);

	AddBorderToEnhBAB (low_mask, cur_mask,
				curr_bab, bordered_curr_bab,
				coded_width, coded_height,
				i, j, mbsize, blkx);

	AddBorderToHalfHigherBAB (low_mask, half_mask,
					half_bab, bordered_half_bab,
					coded_width, coded_height,
					i, j, mbsize, blkx);

        scan_order = DecideScanOrder(bordered_lower_bab, mbsize);

	bab_type = DecideEnhLayerBabType(bordered_lower_bab, 
					bordered_half_bab,
					bordered_curr_bab, 
					mbsize, 
					scan_order);

	/* Encode BAB mode and alpha values in the enhancement layer */
	ret = EncodeEnhancedLayerBAB(&coder, 
					bordered_lower_bab, 
					bordered_half_bab,
					bordered_curr_bab, 
					bab_type, 
					scan_order,	
					mbsize, 
					filter,
					shape_bitstream);
	if( ret == Error ) {
          fprintf(stderr,"\n SI arithmetic coding Error !\n");
          return  Error;
        }
      }
    }
    StopArCoder_Still(&coder,shape_bitstream); 
    PutBitstoStream(1,MARKER_BIT,shape_bitstream);

    bitstream_length= shape_bitstream -> cnt - bitstream_start + 1;

    free(low_mask);
    free(half_mask);
    free(cur_mask);
 
    free(lower_bab);
    free(bordered_lower_bab);
    free(half_bab);
    free(bordered_half_bab);
    free(curr_bab);
    free(bordered_curr_bab);

    return (bitstream_length);
}

Int DecideEnhLayerBabType(UChar *bordered_lower_bab, 	
				UChar *bordered_half_bab,
				UChar *bordered_curr_bab, 
				Int mbsize,
				Int scan_order)
{
	Int	i,j,i2,j2,k,l;
	Int	mborder = MBORDER;
	Int	mbsize_ext = mbsize+(mborder<<1);
	Int	border = BBORDER;
	Int	bsize = mbsize >> 1;
	Int	bsize_ext = bsize+(border<<1);

	Int	b_differ,b_except,curr,prev,next;
	Int	bab_type;

	UChar	*lower_bab_data,
		*half_bab_data,
		*curr_bab_data;
        UChar   *curr_bab_data_tr;
	
	lower_bab_data = bordered_lower_bab + border * bsize_ext + border;
	half_bab_data = bordered_half_bab + mborder * bsize_ext + border; 
	curr_bab_data= bordered_curr_bab + mborder * mbsize_ext + mborder;
 
	/* P0, P1,
	   P2, P3 */

	/* compare P0 pixels in the current layer 
	   with the pixels in the lower layer and half layer  */
	b_differ=0;
        for(j=j2=k=l=0; j<bsize; j++, j2+=2, k+=(mbsize_ext<<1), l+=bsize_ext) {
           for(i=i2=0; i<bsize; i++, i2+=2) {
		if( curr_bab_data[k+i2] != lower_bab_data[l+i] ) {
			b_differ = 1;	/* 010 case */
			break;
		}
	   }
	   if( b_differ ) break;
	}

        /* SL added for every line*/
	if(!b_differ) {
	   for(j=k=l=0; j<bsize; j++, k+=mbsize_ext, l+=bsize_ext) {
		for(i=i2=0; i<bsize; i++, i2+=2) {
                  if( curr_bab_data[k+i2] != half_bab_data[l+i] ) {
                      b_differ = 1;     /* 010 case */
                      break;
                  }
		}
		if( b_differ ) break;
	   }
	}

        /* Current BAB transposing */
        if (scan_order == 1) {
                curr_bab_data_tr = (UChar*)calloc(mbsize_ext*mbsize_ext, sizeof(UChar));
                for(j=0; j<mbsize_ext; j++)
                   for(i=0; i<mbsize_ext; i++)
                        curr_bab_data_tr[j*mbsize_ext+i]=bordered_curr_bab[i*mbsize_ext+j];
                curr_bab_data= curr_bab_data_tr + mborder * mbsize_ext + mborder;
        }

	/* 000 111 110 100 001 011 and 101 cases */
	if(!b_differ) {
	   /* check wheather there are some exceptional samples or not */
	   b_except=0;
	   /* check for P1 pixel decoding */
	   for(j2=k=0; j2<mbsize; j2+=2, k+=(mbsize_ext<<1) ) {
	      for(i2=1; i2<mbsize; i2+=2) {
		prev= curr_bab_data[k+i2-1];
		curr= curr_bab_data[k+i2];
                next= curr_bab_data[k+i2+1];

		if((prev==next) && (curr!=prev)) {
			b_except = 1;   	/*101 case*/
			break;
		}
	      }
	      if(b_except) break;
	   }

	   if(!b_except) {
		/* check for P2/P3 pixel decoding */
		for(j2=1,k=mbsize_ext; j2<mbsize; j2+=2, k+=(mbsize_ext<<1) ) {
		   for(i2=0; i2<mbsize; i2++) {
			prev= curr_bab_data[k-mbsize_ext+i2];
			curr= curr_bab_data[k+i2]; 
			next= curr_bab_data[k+mbsize_ext+i2];
 
		        if((prev==next) && (curr!=prev)) {
			   b_except = 1; 	/*101 case */
			   break;
			}
		    }
		    if(b_except) break;
		}
	   }
	} else {
	   b_except = 1;
	} 

	if(b_except)	bab_type = EXCEPTIONAL; /* exceptional BAB coding */
	else		bab_type = TRANSITIONAL; /* transitional BAB coding */
		
	return bab_type;
}


/***********************************************************CommentBegin******
 *
 * -- EncodeEnhancedLayerBAB -- Encodes a binary alpha block using SI.
 *
 * Author :		
 *	Dae-Sung Cho (Samsung AIT)
 *
 * Created :		
 *	02-Dec-98
 * 
 * Arguments: 	
 *
 *
 * Return values :	
 *
 * Side effects :	
 *	-
 *
 * Description :	A binary alpha block will be coded using bac.
 *			These bits are added to the <shape_stream>. 
 *
 * See also :
 *
 ***********************************************************CommentEnd********/

Int EncodeEnhancedLayerBAB(ArCoder *coder,
				UChar *bordered_lower_bab, 
				UChar *bordered_half_bab,
				UChar *bordered_curr_bab, 
				Int bab_type, 
				Int scan_order,	
				Int mbsize,
				FILTER *filter,
				BSS *shape_bitstream)
{
	Int prob;

	/* Bab mode coding */
	if(filter->Class==ODD_SYMMETRIC){
		prob=scalable_bab_type_prob[0];
	} else if(filter->Class==EVEN_SYMMETRIC) {
		prob=scalable_bab_type_prob[1];
	} else {
		fprintf(stderr,"Error: filter type in BAB_TYPE encoding!\n");
		exit(0);
	}
	ArCodeSymbol_Still(coder, shape_bitstream, bab_type, prob);

	if (bab_type==TRANSITIONAL) {  /* Transitional BAB coding */
	      EncodeTransitionalBAB (coder,
					bordered_lower_bab,
					bordered_curr_bab,
					mbsize,
					scan_order,
					shape_bitstream
					);

	} else if(bab_type==EXCEPTIONAL) {	/* Exceptional BAB coding */
	      EncodeExceptionalBAB (coder,
					bordered_lower_bab,
					bordered_half_bab,
					bordered_curr_bab,
					mbsize, 
					filter,
					shape_bitstream 
					);
	} else {
	      fprintf(stderr,"BAB type[%d] ERROR in Enhancement layer coding!\n", bab_type);
	}

        return 0;
}
	     
/* Encoding of transitional BAB (Proposed by Samsung AIT) */
Void EncodeTransitionalBAB (ArCoder *coder,
				UChar *bordered_lower_bab,
				UChar *bordered_curr_bab,
				Int mbsize,
				Int scan_order,
				BSS *shape_bitstream)
{
	Int             i,j,i2,j2,k,curr,prev,next;
	Int             mborder = MBORDER;
	Int             mbsize_ext = mbsize+(mborder<<1);
	Int             border = BBORDER;
	Int             bsize = mbsize >> 1;
	Int             bsize_ext = bsize+(border<<1);
	Int             context, prob=0;
	UChar           *lower_bab_data,
                        *curr_bab_data;
	UChar           *curr_bab_data_tr;
 
        lower_bab_data = bordered_lower_bab + border * bsize_ext + border;

        if (scan_order==1) {
                curr_bab_data_tr = (UChar*)calloc(mbsize_ext*mbsize_ext, sizeof(UChar));
                for(j=0; j<mbsize_ext; j++)
                   for(i=0; i<mbsize_ext; i++)
                        curr_bab_data_tr[j*mbsize_ext+i]=bordered_curr_bab[i*mbsize_ext+j];
                curr_bab_data=curr_bab_data_tr+mborder * mbsize_ext + mborder;
        } else curr_bab_data= bordered_curr_bab + mborder * mbsize_ext + mborder;

        /*** P0 pixels reconstructing 
        for(j=j2=0,k=l=0; j<bsize; j++, j2+=2, k+=(mbsize_ext<<1), l+=bsize_ext)
           for(i=i2=0; i<bsize; i++, i2+=2)
                curr_bab_data[k+i2]=lower_bab_data[l+i];
	***/

	/* P1 pixels encoding: vertical scanning of the pixel */
	for(i2=1; i2<mbsize; i2+=2) {
	   for(j2=k=0; j2<mbsize; j2+=2, k+=(mbsize_ext<<1) ) {
		curr= curr_bab_data[k+i2];
		prev= curr_bab_data[k+i2-1];
		next= curr_bab_data[k+i2+1];
                       
		if(prev!=next) {
		   context = GetTransitionalBabContext(curr_bab_data,
					i2, 
					j2, 
					mbsize_ext, 
					0); /* pixel type : 0-P1, 1-P2/P3 */

		   prob=scalable_xor_prob_1[context];
		   ArCodeSymbol_Still(coder, shape_bitstream, curr, prob);

		} else {
		   if(prev!=curr) {
			fprintf(stderr, "Error: BAB coding mode mismatch in XOR coding : P1!\n");
			fprintf(stderr, "Error: P1[%d,%d,%d]!\n",prev,curr,next);
			fprintf(stderr,"1, j2=%d i2=%d prev=%d curr=%d next=%d context=%d\n", 
				j2, i2, prev, curr, next, context);
			exit(0);
		   }
		}
	   }
	}

	/* P2/P3 pixel coding: horizontal scanning of the pixels */
        for(j2=1,k=mbsize_ext; j2<mbsize; j2+=2, k+=(mbsize_ext<<1) ) {
	   for(i2=0; i2<mbsize; i2++) {
		curr= curr_bab_data[k+i2];
		prev= curr_bab_data[k-mbsize_ext+i2];	
		next= curr_bab_data[k+mbsize_ext+i2];	

		if(prev!=next) {
		   context = GetTransitionalBabContext(curr_bab_data,
					i2, 
					j2, 
					mbsize_ext, 
					1);	/* pixel type : 0-P1, 1-P2/P3 */

		   prob=scalable_xor_prob_23[context];
		   ArCodeSymbol_Still(coder, shape_bitstream, curr, prob);
		} else {
		   if(prev!=curr) {
			fprintf(stderr, "Error: BAB coding mode mismatch in XOR coding : P2, P3!\n");
			exit(0);
		   }
		}
	   }
	}

        if(scan_order==1) free(curr_bab_data_tr);
}


/* Encoding of exceptional BAB (Proposed by Sarnoff Corp.) */
Void EncodeExceptionalBAB (ArCoder *coder,
				UChar *bordered_lower_bab,
				UChar *bordered_half_bab,
				UChar *bordered_curr_bab,
				Int mbsize,
				FILTER *filter,
				BSS *shape_bitstream)
{
	Int		i,j,i2,j2,k,m;
	Int		mborder = MBORDER;
	Int		mbsize_ext = mbsize+(mborder<<1);
	Int		border = BBORDER;
	Int		bsize = mbsize >> 1;
	Int		bsize_ext = bsize+(border<<1);
	Int		half_value, curr_value;
	Int		context, prob=0;
	UChar		*lower_bab_data,
			*half_bab_data,		/* half higher bab data */
			*curr_bab_data;

	lower_bab_data = bordered_lower_bab + border * bsize_ext + border;
	half_bab_data = bordered_half_bab + mborder * bsize_ext + border;
	curr_bab_data = bordered_curr_bab + mborder * mbsize_ext + mborder;

	/* First pass: horizontal scanning of 1x2 block */
	for(i2=k=0; i2<mbsize; i2+=2, k+=(bsize_ext<<1)) {
	   for(j=0; j<bsize; j++) {
		for(m=0; m<2; m++) {
			if(m==0) half_value = half_bab_data[k+j];		/* T1 pixel */
			else 	 half_value = half_bab_data[k+bsize_ext+j];	/* T0 pixel */
			
			context = GetExceptionalBabContext_HalfHigher(lower_bab_data,
                                			half_bab_data,
                                			j,
                                			i2+m,
                                			bsize_ext,
                                			bsize_ext,
                                			m); 
		   
			if(filter->Class==ODD_SYMMETRIC) {
		   	   prob= (m==0) ? sto_enh_odd_prob0[context] 
					   : sto_enh_odd_prob1[context];
			} else if(filter->Class==EVEN_SYMMETRIC) {
		   	   prob= (m==0) ? sto_enh_even_prob0[context]
					   : sto_enh_even_prob1[context];
			} else {
		   	   fprintf(stderr,"Error: filter type in EncodeExceptionalBAB() !\n");
		   	   exit(0);
			}
			if(prob!=0 && prob!=65536 && prob!=65537) {
			    ArCodeSymbol_Still(coder, shape_bitstream, half_value, prob);
			} else {
			    if( (prob==0 && half_value!=1) || 
				  (prob==65536 && half_value!=0) || (prob==65537)) {
				fprintf(stderr,"Error: Incorrect special case for arithmetic coding for half-higher BAB!\n");
				fprintf(stderr,"prob=%d half_value=%d\n",prob,half_value);
				exit(0);
			    }
			}
		}
	   }
	}

	/* Second pass: vertical scanning of 2x1 block */
	for(j2=0; j2<mbsize; j2+=2){
	   for(i=k=0; i<mbsize; i++, k+=mbsize_ext) {
		for(m=0; m<2; m++) {
			curr_value = curr_bab_data[k+j2+m];
			
			context = GetExceptionalBabContext(half_bab_data,
                                			curr_bab_data,
                                			j2+m,
                                			i,
                                			bsize_ext,
							mbsize_ext,
                                			m); 
		   
			if(filter->Class==ODD_SYMMETRIC) {
		   	   prob= (m==0) ? sto_enh_odd_prob0[context] 
					   : sto_enh_odd_prob1[context];
			} else if(filter->Class==EVEN_SYMMETRIC) {
		   	   prob= (m==0) ? sto_enh_even_prob0[context]
					   : sto_enh_even_prob1[context];
			} else {
		   	   fprintf(stderr,"Error: filter type in EncodeExceptionalBAB() !\n");
		   	   exit(0);
			}

			if(prob!=0 && prob!=65536 && prob!=65537) {
			    ArCodeSymbol_Still(coder, shape_bitstream, curr_value, prob);
			} else {
			    if( (prob==0 && curr_value!=1) || 
				  (prob==65536 && curr_value!=0) || (prob==65537)) {
				fprintf(stderr,"Error: Incorrect special case for arithmetic coding for higher BAB!\n");
				fprintf(stderr,"prob=%d curr_value=%d\n",prob,curr_value);
				exit(0);
			    }
			}
		}
	   }
	}
}
