/****************************************************************************/
/*   MPEG4 Visual Texture Coding (VTC) Mode Software                        */
/*                                                                          */
/*   This software was jointly developed by the following participants:     */
/*                                                                          */
/*   Single-quant,  multi-quant and flow control                            */
/*   are provided by  Sarnoff Corporation                                   */
/*     Iraj Sodagar   (iraj@sarnoff.com)                                    */
/*     Hung-Ju Lee    (hjlee@sarnoff.com)                                   */
/*     Paul Hatrack   (hatrack@sarnoff.com)                                 */
/*     Shipeng Li     (shipeng@sarnoff.com)                                 */
/*     Bing-Bing Chai (bchai@sarnoff.com)                                   */
/*     B.S. Srinivas  (bsrinivas@sarnoff.com)                               */
/*                                                                          */
/*   Bi-level is provided by Texas Instruments                              */
/*     Jie Liang      (liang@ti.com)                                        */
/*                                                                          */
/*   Shape Coding is provided by  OKI Electric Industry Co., Ltd.           */
/*     Zhixiong Wu    (sgo@hlabs.oki.co.jp)                                 */
/*     Yoshihiro Ueda (yueda@hlabs.oki.co.jp)                               */
/*     Toshifumi Kanamaru (kanamaru@hlabs.oki.co.jp)                        */
/*                                                                          */
/*   Scalable shape coding is provided by Samsung AIT                       */
/*     Dae-Sung Cho (dscho@sait.samsung.co.kr)                              */
/*									    */
/*   OKI, Sharp, Sarnoff, TI and Microsoft contributed to bitstream         */
/*   exchange and bug fixing.                                               */
/*                                                                          */
/*                                                                          */
/* In the course of development of the MPEG-4 standard, this software       */
/* module is an implementation of a part of one or more MPEG-4 tools as     */
/* specified by the MPEG-4 standard.                                        */
/*                                                                          */
/* The copyright of this software belongs to ISO/IEC. ISO/IEC gives use     */
/* of the MPEG-4 standard free license to use this  software module or      */
/* modifications thereof for hardware or software products claiming         */
/* conformance to the MPEG-4 standard.                                      */
/*                                                                          */
/* Those intending to use this software module in hardware or software      */
/* products are advised that use may infringe existing  patents. The        */
/* original developers of this software module and their companies, the     */
/* subsequent editors and their companies, and ISO/IEC have no liability    */
/* and ISO/IEC have no liability for use of this software module or         */
/* modification thereof in an implementation.                               */
/*                                                                          */
/* Permission is granted to MPEG members to use, copy, modify,              */
/* and distribute the software modules ( or portions thereof )              */
/* for standardization activity within ISO/IEC JTC1/SC29/WG11.              */
/*                                                                          */
/* Copyright 1995, 1996, 1997, 1998 ISO/IEC                                 */
/****************************************************************************/
#include <stdio.h>
#include "typedef.h"
#include "ShapeUtil.h"
#include "ShapeCodec.h"
#include "ShapeCommon.h"
#include "shape_def.h"

Void
UpSampling_Still(Int x,
		Int y,
		Int blkn,
		Int width, 
		Int height,
		Int cr,
		Int blkx,
		UChar **buff,
		UChar**data,
		UChar **shape)
{
    Int	i, j;

    Int b_size=blkn/cr;

    if ( b_size==16 ) {
        for ( j=0; j<b_size; j++ ) {
        for ( i=0; i<b_size; i++ ) {
            data[j][i] = buff[j][i];
        }
        }
    } else if ( b_size==8 ) {
        UChar **down_tmp = (UChar**) malloc_2d_Char( blkn + 4, blkn + 4);
        AddBorderToBAB(x,y,blkn,width,height,cr,blkx,buff,down_tmp,shape,0);
        AdaptiveUpSampling_Still(data,down_tmp,8);
        free_2d_Char(down_tmp, blkn + 4 );
    } else if ( b_size==4 ) {
        UChar **down_tmp = (UChar **)malloc_2d_Char( blkn + 4, blkn + 4);
        UChar **up_tmp   = (UChar **)malloc_2d_Char( blkn    , blkn    );
        AddBorderToBAB(x,y,blkn,width,height,cr,blkx,buff,down_tmp,shape,0);
        AdaptiveUpSampling_Still(down_tmp,up_tmp,4);
        AddBorderToBAB(x,y,blkn,width,height,cr/2,blkx,up_tmp,down_tmp,shape,0);
        AdaptiveUpSampling_Still(down_tmp,data,8);
        free_2d_Char(down_tmp, blkn + 4 );
        free_2d_Char(up_tmp, blkn);
    } else {
        fprintf(stderr,"UpSampling():Wrong UpSampling Size (%d->16)\n",b_size);
        exit( 0 );
    }
}

static Int	GetContext ( Char a, Char b, Char c, Char d, Char e, Char f, Char g, Char h )
{
	return (Int) ( a + (b<<1) + (c<<2) + (d<<3) + (e<<4) + (f<<5) + (g<<6) + (h<<7) );
}

static Char	GetShapeVL ( Char a, Char b, Char c, Char d, Char e, Char f, 
				   Char g, Char h, Char i, Char j, Char k, Char l, Int t )
{
	return (Char) ( ( (Int)(a<<2) + (Int)((b+c+d)<<1) + (Int)(e+f+g+h+i+j+k+l) ) > t );
}

Void
AdaptiveUpSampling_Still(UChar **BABdown,UChar **BABresult,Int sub_size)
{
  static Int
    Th[256] = {
      3, 6, 6, 7, 4, 7, 7, 8, 6, 7, 5, 8, 7, 8, 8, 9,
      6, 5, 5, 8, 5, 6, 8, 9, 7, 6, 8, 9, 8, 7, 9,10,
      6, 7, 7, 8, 7, 8, 8, 9, 7,10, 8, 9, 8, 9, 9,10,
      7, 8, 6, 9, 6, 9, 9,10, 8, 9, 9,10,11,10,10,11,
      6, 9, 5, 8, 5, 6, 8, 9, 7,10,10, 9, 8, 7, 9,10,
      7, 6, 8, 9, 8, 7, 7,10, 8, 9, 9,10, 9, 8,10, 9,
      7, 8, 8, 9, 6, 9, 9,10, 8, 9, 9,10, 9,10,10, 9,
      8, 9,11,10, 7,10,10,11, 9,12,10,11,10,11,11,12,
      6, 7, 5, 8, 5, 6, 8, 9, 5, 6, 6, 9, 8, 9, 9,10,
      5, 8, 8, 9, 6, 7, 9,10, 6, 7, 9,10, 9,10,10,11,
      7, 8, 6, 9, 8, 9, 9,10, 8, 7, 9,10, 9,10,10,11,
      8, 9, 7,10, 9,10, 8,11, 9,10,10,11,10,11, 9,12,
      7, 8, 6, 9, 8, 9, 9,10,10, 9, 7,10, 9,10,10,11,
      8, 7, 7,10, 7, 8, 8, 9, 9,10,10,11,10,11,11,12,
      8, 9, 9,10, 9,10,10, 9, 9,10,10,11,10,11,11,12,
      9,10,10,11,10,11,11,12,10,11,11,12,11,12,12,13 };
  /* --  E  F -- */
  /*  L  A  B  G */
  /*  K  D  C  H */
  /* --  J  I -- */
  
  /*   A   B   C   D   E   F   G   H   I   J   K   L	*/
  static Int	xx [12] = { -1, +0, +0, -1, -1, +0, +1, +1, +0, -1, -2, -2, };
  static Int	yy [12] = { -1, -1, +0, +0, -2, -2, -1, +0, +1, +1, +0, -1, };
  
  register Int	i, j, k, l, m;
  
  Char		val[12];
  Int			blk_size  = sub_size;
  Int			context;
  Int			iup, jup;
  Int			i2, j2;
  Int			ks, ke, ls, le;
  
  
  for	( j=0; j<=blk_size; j++ ) {
    for	( i=0; i<=blk_size; i++ ) {
      
      for( m=0; m<12; m++ )
	val[m] = ( BABdown [yy[m]+j+2] [xx[m]+i+2] != 0 );
      
      if ( i == 0        ) ks =  0;
      else                 ks = -1;
      if ( i == blk_size ) ke =  0; 
      else                 ke =  1;
      
      if ( j == 0        ) ls =  0;
      else                 ls = -1;
      if ( j == blk_size ) le =  0; 
      else                 le =  1;
      
      i2 = i << 1;
      j2 = j << 1;
      
      for ( l=ls; l<le; l++ ) {
	for ( k=ks; k<ke; k++ ) {
	  
	  iup = i2 + k;
	  jup = j2 + l;
	  
	  if( (iup & 1) == 1 && (jup & 1) == 1 ) { /* P1 case : */
	    
	    context = GetContext (	val[5],val[4],val[11],val[10],val[9],val[8],val[7],val[6] );
	    
	    BABresult [jup][iup]
	      = GetShapeVL (	val[0],
				val[1],val[2],val[3],
				val[4],val[5],val[6],val[7],val[8],val[9],val[10],val[11],
				Th [context] );
	    
	  } else
	    if( (iup & 1) == 0 && (jup & 1) == 1 ) { /* P2 case : */
	      
	      context	= GetContext (	val[7],val[6],val[5],val[4],val[11],val[10],val[9],val[8] );
	      
	      BABresult [jup][iup]
		= GetShapeVL (	val[1],
				val[0],val[2],val[3],
				val[4],val[5],val[6],val[7],val[8],val[9],val[10],val[11],
				Th [context] );
	      
	    } else
	      if( (iup & 1) == 1 && (jup & 1) == 1 ) { /* P3 case : */
		
		context	= GetContext  (	val[9],val[8],val[7],val[6],val[5],val[4],val[11],val[10] );
		
		BABresult [jup][iup]
		  = GetShapeVL (	val[2],
					val[0],val[1],val[3],
					val[4],val[5],val[6],val[7],val[8],val[9],val[10],val[11],
					Th [context] );
		
	      } else {								/* P4 case : */
		
		context = GetContext  (	val[11],val[10],val[9],val[8],val[7],val[6],val[5],val[4] );
		
		BABresult [jup][iup]
		  = GetShapeVL (	val[3],
					val[0],val[1],val[2],
					val[4],val[5],val[6],val[7],val[8],val[9],val[10],val[11],
					Th [context] );
		
	      }
	}
      }
    }
  }
  return;
}


Void
DownSampling_Still(UChar **buff,UChar **data,Int b_size,Int s_size)
{
    Int	i, j, k, l;
    Int	dat;
 
    if ( s_size == 1 ) {;
        for ( j=0; j<b_size; j++ ) 
          for ( i=0; i<b_size; i++ ) 
            data[j][i] = buff[j][i];
    } else {
        for ( j=0; j<b_size; j++ ) {
          for ( i=0; i<b_size; i++ ) {
            dat = 0;
            for ( l=0; l<s_size; l++ ) 
              for ( k=0; k<s_size; k++ ) 
                dat += buff[j*s_size+l][i*s_size+k];
            data[j][i] = ( (dat*2)<(s_size*s_size) ) ? 0 : 1;
          }
        }
    }
    return;
}

Void
AddBorderToBAB(Int blkx,
		Int blky,
		Int blksize,
		Int object_width, 
		Int object_height,
		Int cr_size,
		Int MAX_blkx,
		UChar **BABinput,
		UChar **BABresult,
		UChar **shape, 
		Int flag)
{
    Int	blkn = blksize/cr_size;
    Int	i, j, k, tmp;

    for ( j=0; j<blkn+4; j++ )
      for ( i=0; i<blkn+4; i++ )
        BABresult[j][i] = 0;

    for ( j=0; j<blkn; j++ )
      for ( i=0; i<blkn; i++ )
        BABresult[j+2][i+2] = BABinput[j][i];

    /*  Up-left  */
    if ( blkx!=0 && blky!=0 ) {
        Int	X = blkx * blksize - 2;
        Int	Y = blky * blksize - 2;
        for ( j=0; j<2; j++ ) {
          for ( i=0; i<2; i++ ) {
	    if(Y+j < object_height && X+i < object_width)
              BABresult[j][i] = ( shape[Y+j][X+i]!=0 );
          }
        }
    }

    /*  up  */
    if ( blky!=0 ) {
        Int	X = blkx * blksize;
        Int	Y = blky * blksize - 2;
        for ( j=0; j<2   ; j++ ) {
          for ( i=0; i<blkn; i++ ) {
            tmp = 0;
            for ( k=0; k<cr_size; k++ ) 
	      if(Y+j < object_height && X+i*cr_size+k < object_width)
                tmp += ( shape[Y+j][X+i*cr_size+k]!=0 );
	    
            BABresult[j][i+2] = ( (tmp*2 < cr_size) ? 0 : 1 );
          }
        }
    }

    /*  left  */
    if ( blkx!=0 ) {
        Int	X = blkx * blksize - 2;
        Int	Y = blky * blksize;
        for ( j=0; j<blkn; j++ ) {
          for ( i=0; i<2   ; i++ ) {
            tmp = 0;
            for ( k=0; k<cr_size; k++ ) 
	      if(Y+j*cr_size+k < object_height && X+i < object_width)
                tmp += ( shape[Y+j*cr_size+k][X+i]!=0 );
            BABresult[j+2][i] = ( (tmp*2 < cr_size) ? 0 : 1 );
          }
        }
    }

    /*  up-right  */
    if ( blky!=0 && blkx<MAX_blkx-1 ) {
        Int	X = (blkx + 1) * blksize;
        Int	Y = blky * blksize - 2;
        for ( j=0; j<2; j++ ) {
          for ( i=0; i<2; i++ ) {
	    if(Y+j < object_height && X+i < object_width)
              BABresult[j][blkn+2+i] = ( shape[Y+j][X+i]!=0 );
          }
        }
    }

    /* padding right & bottom border pixels unknown at decoding time. */
    if ( flag != 2 ) {		/*  for up sampling & encoding */
	for ( i=0; i<blkn; i++ ) {
		BABresult [i+2][blkn+3] =
		BABresult [i+2][blkn+2] = BABresult [i+2][blkn+1];
		BABresult [blkn+3][i+2] =
		BABresult [blkn+2][i+2] = BABresult [blkn+1][i+2];
	}
    }
    if ( flag == 0 ) {		/* for up sampling */
	for ( j=0; j<2; j++ ) {
		BABresult [blkn+j+2][0] =
		BABresult [blkn+j+2][1] = BABresult [blkn+j+2][2];
		BABresult [blkn+j+2][blkn+3] =
		BABresult [blkn+j+2][blkn+2] = BABresult [blkn+j+2][blkn+1];
	}
    } else {
	for ( i=0; i<2; i++ ) {
		BABresult [blkn+3][i] =
		BABresult [blkn+2][i] = BABresult [blkn+1][i];
		BABresult [blkn+i+2][blkn+3] =
		BABresult [blkn+i+2][blkn+2] = 0;
	}
    }

    return;
}

/***********************************************************HeaderBegin*******
 *
 * Author:      Dae-Sung Cho (Samsung AIT)
 * Created:     09-Feb-98
 * Modified by: Shipeng Li (Sarnoff Corporation), Dec 2, 1998.
 * Description: Contains functions used to implement scan interleaving
 *                               coding of spatial scalable binary alpha blocks.
 *
 ***********************************************************HeaderEnd*********/

Int GetTransitionalBabContext(UChar *curr_bab_data,
				Int x2, 
				Int y2,
				Int width2,
				Int pixel_type)	/* pixel type : 0-P1, 1-P2/P3 */
{
   Int		t=0;
   UChar	*p;

   if(pixel_type==0) {	/* P1 pixel */
	p=curr_bab_data + (y2-2) * width2 + x2-1;

	t <<= 1; t |= *p; p++;			/* c6 */
	t <<= 1; t |= *p; p++;			/* c5 */
	t <<= 1; t |= *p; p+=((width2<<1)-2);	/* c4 */  
	t <<= 1; t |= *p; p+=2;			/* c3 */
	t <<= 1; t |= *p; p+=((width2<<1)-2);	/* c2 */
	t <<= 1; t |= *p; p+=2;			/* c1 */
	t <<= 1; t |= *p;			/* c0 */
   } else {		/* P2/P3 pixels */
	p=curr_bab_data + (y2-1) * width2 + x2-1;

	t <<= 1; t |= *p; p++;			/* c6 */
	t <<= 1; t |= *p; p++;			/* c5 */
	t <<= 1; t |= *p; p+=(width2-2);	/* c4 */  
	t <<= 1; t |= *p; p+=width2;		/* c3 */
	t <<= 1; t |= *p; p++;			/* c2 */ 
	t <<= 1; t |= *p; p++;			/* c1 */ 
	t <<= 1; t |= *p;			/* c0 */
   } 
   return t;
}

Int GetContextEnhBAB_FULL(UChar *lower_bab_data,
				UChar *curr_bab_data,
				Int x2, 
				Int y2,
				Int width,
				Int width2)
{
   Int		t=0;
   Int		x = x2 >> 1, 
		y = y2 >> 1;
   UChar	*p, *q;

   p = lower_bab_data + y*width + x;
   q = curr_bab_data + (y2-1) * width2 + (x2-1);

   t <<= 1; t |= *p; p++;			/* c7 */
   t <<= 1; t |= *p; p+= width-1;		/* c6 */
   t <<= 1; t |= *p; p++;			/* c5 */
   t <<= 1; t |= *p; 				/* c4 */

   t <<= 1; t |= *q; q++;			/* c3 */
   t <<= 1; t |= *q; q++;			/* c2 */
   t <<= 1; t |= *q; q+=(width2-2);		/* c1 */
   t <<= 1; t |= *q; 				/* c0 */

   return t;
}

Int GetExceptionalBabContext_HalfHigher(UChar *lower_bab_data,
				UChar *half_bab_data,
				Int x2, 
				Int y2,
				Int width, 
				Int width2,
				Int pixel_type)	/* 0-T1, 1-T0 */
{
   Int		t=0;
   Int		x = x2,
		y = (y2 >> 1);
   UChar	*p, *q;

   if(pixel_type==0) {	/* T1 pixel */
	if(y2%2!=0) {
		fprintf(stderr,"Error: Pixel type mismatch !\n");
		exit(0);
	}
   	p = lower_bab_data + y*width + x;
   	q = half_bab_data + (y2-1)*width2 + x2;

   	t <<= 1; t |= *q; q++;			/* c7<-T9 */
   	t <<= 1; t |= *q; q+= (width2<<1)-2;	/* c6<-T8 */
   	t <<= 1; t |= *q; q-= width2;		/* c5<-T7 */
   	t <<= 1; t |= *q; 			/* c4<-T6 */

   	t <<= 1; t |= *p; p++; 			/* c3<-T5 */
   	t <<= 1; t |= *p; p+= (width-1); 	/* c2<-T4 */
   	t <<= 1; t |= *p; p++; 			/* c1<-T3 */
   	t <<= 1; t |= *p;  			/* c0<-T2 */
   } else {			/* T0 pixel */
	if(y2%2!=1) {
		fprintf(stderr,"Error: Pixel type mismatch !\n");
		exit(0);
	}
   	p = lower_bab_data + (y+1)*width + x-1;
   	q = half_bab_data + (y2-1)*width2 + x2;

   	t <<= 1; t |= *q; q+= (width2-1);	/* c7<-T1 */

   	t <<= 1; t |= *p; p-= (width-1); 	/* c6<-T10 */

   	t <<= 1; t |= *q; q-= width2;		/* c5<-T7 */
   	t <<= 1; t |= *q; 			/* c4<-T6 */

   	t <<= 1; t |= *p; p++; 			/* c3<-T5 */
   	t <<= 1; t |= *p; p+= width-1; 		/* c2<-T4 */
   	t <<= 1; t |= *p; p++; 			/* c1<-T3 */
   	t <<= 1; t |= *p; 			/* c0<-T2 */
   }
   return t;
}

Int GetExceptionalBabContext(UChar *half_bab_data,
				UChar *curr_bab_data,
				Int x2, 
				Int y2,
				Int width,
				Int width2,
				Int pixel_type)	/* 0-T1, 1-T0 */
{
   Int		t=0;
   Int		x = (x2 >> 1),
		y = y2;
   UChar	*p, *q;

   if(pixel_type==0) {	/* T1 pixel */
	if(x2%2!=0) {
		fprintf(stderr,"Error: Pixel type mismatch !\n");
		exit(0);
	}
   	p = half_bab_data + y*width + x;
   	q = curr_bab_data + y2*width2 + x2-1;

   	t <<= 1; t |= *q; q+= (width2);		/* c7<-T9 */
   	t <<= 1; t |= *q; q-= ((width2<<1)-2);	/* c6<-T8 */
   	t <<= 1; t |= *q; q--;			/* c5<-T7 */
   	t <<= 1; t |= *q; 			/* c4<-T6 */

   	t <<= 1; t |= *p; p+= width; 		/* c3<-T5 */
   	t <<= 1; t |= *p; p-= (width-1); 	/* c2<-T4 */
   	t <<= 1; t |= *p; p+= width; 		/* c1<-T3 */
   	t <<= 1; t |= *p;  			/* c0<-T2 */
   } else {		/* T0 pixel */
	if(x2%2!=1) {
		fprintf(stderr,"Error: Pixel type mismatch !\n");
		exit(0);
	}
   	p = half_bab_data + (y-1)*width + x+1;
   	q = curr_bab_data + y2*width2 + x2-1;

   	t <<= 1; t |= *q; q-= (width2-1);	/* c7<-T1 */

   	t <<= 1; t |= *p; p+= (width-1); 	/* c6<-T10 */

   	t <<= 1; t |= *q; q--;			/* c5<-T7 */
   	t <<= 1; t |= *q; 			/* c4<-T6 */

   	t <<= 1; t |= *p; p+= width; 		/* c3<-T5 */
   	t <<= 1; t |= *p; p-= (width-1);	/* c2<-T4 */
   	t <<= 1; t |= *p; p+= width;		/* c1<-T3 */
   	t <<= 1; t |= *p; 			/* c0<-T2 */
   }

   return t;
}

Int DecideScanOrder(UChar *bordered_lower_bab, Int mbsize)
{
        Int scan_order;
        Int border = BBORDER;
        Int bsize = mbsize >> 1;
        Int bsize_ext = bsize+(border<<1);
        Int i,j;
        Int num_right=0, num_bottom=0;

        UChar   *lower_bab_data;

        lower_bab_data = bordered_lower_bab + border * bsize_ext + border;

        for(j=0;j<bsize;j++){
	   for(i=0;i<bsize;i++){
		if( *(lower_bab_data+j*bsize_ext + i) != *(lower_bab_data+j*bsize_ext + i+1))
                                num_right++;
                
		if( *(lower_bab_data+j*bsize_ext + i) != *(lower_bab_data+(j+1)*bsize_ext + i))
                                num_bottom++;
	   }
        }

        if(num_bottom <= num_right)     scan_order = 0;
        else                            scan_order = 1;

        return(scan_order);
}

Void AddBorderToLowerBAB(UChar *low_shape, 
			UChar *lower_bab, 
			UChar *bordered_lower_bab, 
			Int width, 	/* width in the lower layer */
			Int height,	/* height in the lower layer */
			Int blkx, 
			Int blky, 
			Int blksize, 
			Int max_blkx)

{
    Int i, j, x, y, k, l;

    Int border		= BBORDER,
	blksize_ext	= blksize+(border<<1);	/* bordered bab size in the lower layer: 8 */
 
    /* Initialize bordered babs */
    for ( j=k=0; j<blksize_ext; j++)
	for( i=0; i<blksize_ext; i++, k++)
		bordered_lower_bab[k] = 0;

    /* Copy original babs */
    for ( j=k=0, l=border*blksize_ext; j<blksize; j++, l+=blksize_ext ) 
        for ( i=0; i<blksize; i++, k++ ) 
	   bordered_lower_bab[l+i+border] = lower_bab[k];

    /* Top border */
    x = blkx * blksize - border;
    y = blky * blksize - border;
    l = 0;
    k = y*width;
    for( j=0; j<border; j++, k+=width, l+=blksize_ext ) {
          for( i=0; i<blksize_ext; i++ ) {
	    if( (0 <= y+j && y+j < height) && (0 <= x+i && x+i < width) ) 
	    	bordered_lower_bab[l+i] = (low_shape[k+x+i] != 0);
	  }
    }

    /* Left border */
    x = blkx * blksize - border;
    y = blky * blksize;
    l = border * blksize_ext; 
    k = y * width;
    for( j=0; j<blksize; j++, k+=width, l+=blksize_ext) {
          for( i=0; i<border; i++ ) {
	    if( (0 <= y+j && y+j < height) && (0 <= x+i && x+i < width) ) 
	    	bordered_lower_bab[l+i] = (low_shape[k+x+i] != 0);
	  }
    }

    /* Right border */
    x = (blkx + 1) * blksize;
    y = blky * blksize;
    l = border * blksize_ext;
    k = y * width;
    for ( j=0; j<blksize; j++, k+=width, l+=blksize_ext) {
          for ( i=0; i<border; i++ ) {
	    if( (0 <= y+j && y+j < height) && (0 <= x+i && x+i < width) ) 
	    	bordered_lower_bab[l+i+blksize+border] = (low_shape[k+x+i] != 0);
          }
    }

    /* Bottom border */
    x = blkx * blksize - border;
    y = (blky + 1) * blksize;
    l = (blksize+border) * blksize_ext;
    k = y * width;
    for ( j=0; j<border; j++, k+=width, l+=blksize_ext) {
          for ( i=0; i<blksize_ext; i++ ) {
	    if( (0 <= y+j && y+j < height) && (0 <= x+i && x+i < width) ) 
	    	bordered_lower_bab[l+i] = (low_shape[k+x+i] != 0);
          }
    }

    return;
}

Void AddBorderToHalfHigherBAB(UChar *low_shape, 
				UChar *half_shape,
				UChar *half_bab, 
				UChar *bordered_half_bab,
				Int object_width, 	/* object_width in the current layer */
				Int object_height,	/* object_height in the current layer */
				Int blkx, 
				Int blky, 
				Int mblksize, 
				Int max_blkx)

{
    Int i, j, j2, x, y, y2, k, l;

    Int mborder		= MBORDER,
	mblksize_ext	= mblksize+(mborder<<1);/* bordered bab size in the current layer: 20 */
    Int border		= BBORDER,
	blksize		= mblksize>>1,		/* bab size in the lower layer : 8 */
	blksize_ext	= blksize+(border<<1);	/* bordered bab size in the lower layer: 8 */
    Int width2 = object_width,
	height2 = object_height,
	width = width2 >> 1,
	height = height2 >> 1;
 
    /* Initialize bordered babs */
    for ( j=k=0; j<mblksize_ext; j++)
	for( i=0; i<blksize_ext; i++, k++)
		bordered_half_bab[k] = 0;

    /* Copy original babs */
    for ( j=k=0, l=mborder*blksize_ext; j<mblksize; j++, l+=blksize_ext ) 
        for ( i=0; i<blksize; i++, k++ ) 
	   bordered_half_bab[l+i+border] = half_bab[k];

    /* Top border */
    x = blkx * blksize - border;
    y2 = blky * mblksize - mborder; 
    l = 0;
    k = y2*width;
    for( j2=0; j2<mborder; j2++, k+=width, l+=blksize_ext ) {
          for( i=0; i<blksize_ext; i++ ) {
	    if( (0 <= y2+j2 && y2+j2 < height2) && (0 <= x+i && x+i < width) ) 
	    	bordered_half_bab[l+i] = (half_shape[k+x+i] != 0);
	  }
    }

    /* Left border */
    x = blkx * blksize - border;
    y2 = blky * mblksize;
    l = mborder * blksize_ext;
    k = y2 * width;
    for(j2=0; j2<mblksize; j2++, k+=width, l+=blksize_ext) {
          for( i=0; i<border; i++ ) {
	    if( (0 <= y2+j2 && y2+j2 < height2) && (0 <= x+i && x+i < width) ) 
	    	bordered_half_bab[l+i] = (half_shape[k+x+i] != 0);
	  }
    }

    /* Right border */
    x = (blkx + 1) * blksize;
    y = blky * blksize;
    l = mborder * blksize_ext;
    for ( j2=0; j2<mblksize; j2++, l+=blksize_ext) {
	  j = j2>>1;
	  k = (y+j)*width;
          for ( i=0; i<border; i++ ) {
	    if( (0 <= y+j && y+j < height) && (0 <= x+i && x+i < width) ) {
                bordered_half_bab[l+i+blksize+border] = (low_shape[k+x+i] != 0);
                if(y+j+1 < height && j2%2==1) { /* avoid the odd subsampling mishap */
                  bordered_half_bab[l+i+blksize+border] =  (low_shape[k+x+i] != 0) | (low_shape[k+width+x+i] != 0);
                }
	    }
          }
    }
   
    /* Bottom border */
    x = blkx * blksize - border;
    y = (blky + 1) * blksize;
    l = (mblksize+mborder)*blksize_ext;
    for ( j2=0; j2<mborder; j2++, l+=blksize_ext) {
	  j = j2>>1;
	  k = (y+j)*width;
          for ( i=0; i<blksize_ext; i++ ) {
	    if( (0 <= y+j && y+j < height) && (0 <= x+i && x+i < width) ) 
	    	bordered_half_bab[l+i] = (low_shape[k+x+i] != 0);
          }
    }

    return;
}

Void AddBorderToEnhBAB(UChar *low_shape, 
			UChar *cur_shape,
			UChar *curr_bab,
			UChar *bordered_curr_bab,
			Int object_width, 	/* object_width in the current layer */
			Int object_height,	/* object_height in the current layer */
			Int blkx, 
			Int blky, 
			Int mblksize, 
			Int max_blkx)

{
    Int i, j, i2, j2, x, y, x2, y2, k, l;

    Int mborder		= MBORDER,
	mblksize_ext	= mblksize+(mborder<<1);/* bordered bab size in the current layer: 20 */
    Int border		= BBORDER,
	blksize		= mblksize>>1;		/* bab size in the lower layer : 8 */
    Int width2 = object_width,
	height2 = object_height,
	width = width2 >> 1,
	height = height2 >> 1;
 
    /* Initialize bordered babs */
    for ( j=k=0; j<mblksize_ext; j++)
	for( i=0; i<mblksize_ext; i++, k++)
		bordered_curr_bab[k] = 0;
	
    /* Copy original babs */
    for ( j=k=0, l=mborder*mblksize_ext; j<mblksize; j++, l+=mblksize_ext ) 
        for ( i=0; i<mblksize; i++, k++ ) 
	   bordered_curr_bab[l+i+mborder] = curr_bab[k];

    /* Top border */
    x2 = blkx * mblksize - mborder;
    y2 = blky * mblksize - mborder;
    l = 0;
    k = y2 * width2;
    for( j2=0; j2<mborder; j2++, k+=width2, l+=mblksize_ext ) {
          for( i2=0; i2<mblksize_ext; i2++ ) {
	    if( (0 <= y2+j2 && y2+j2 < height2) && (0 <= x2+i2 && x2+i2 < width2) ) 
	    	bordered_curr_bab[l+i2] = (cur_shape[k+x2+i2] != 0);
	  }
    }

    /* Left border */
    x2 = blkx * mblksize - mborder;
    y2 = blky * mblksize;
    l = mborder * mblksize_ext; 
    k = y2 * width2;
    for( j2=0; j2<mblksize; j2++, k+=width2, l+=mblksize_ext) {
          for( i2=0; i2<mborder; i2++ ) {
	    if( (0 <= y2+j2 && y2+j2 < height2) && (0 <= x2+i2 && x2+i2 < width2) ) 
	    	bordered_curr_bab[l+i2] = (cur_shape[k+x2+i2] != 0);
	  }
    }

    /* Right border */
    x = (blkx + 1) * blksize;
    y = blky * blksize;
    l = mborder * mblksize_ext;
    for ( j2=0; j2<mblksize; j2++, l+=mblksize_ext) {
	  j = j2>>1;
	  k = (y+j)*width;
          for ( i2=0; i2<mborder; i2++ ) {
	    i = i2 >> 1;
	    if( (0 <= y+j && y+j < height) && (0 <= x+i && x+i < width) ) 
	    	bordered_curr_bab[l+i2+mblksize+mborder] = (low_shape[k+x+i] != 0);
          }
    }
   
    /* Bottom border */
    x = blkx * blksize - border;
    y = (blky + 1) * blksize;
    l = (mblksize + mborder) * mblksize_ext;
    for ( j2=0; j2<mborder; j2++, l+=mblksize_ext) {
	  j = j2>>1;
	  k = (y+j)*width;
          for ( i2=0; i2<mblksize_ext; i2++ ) {
	    i = i2>>1;
	    if( (0 <= y+j && y+j < height) && (0 <= x+i && x+i < width) ) 
	    	bordered_curr_bab[l+i2] = (low_shape[k+x+i] != 0);
          }
    }

    return;
}
