/******************************************************************************/
// File: ChildView.cpp [scope = APPS/SHOW]
// Version: Kakadu, V2.2
// Author: David Taubman
// Last Revised: 20 June, 2001
/******************************************************************************/
// Copyright 2001, David Taubman, The University of New South Wales (UNSW)
// The copyright owner is Unisearch Ltd, Australia (commercial arm of UNSW)
// Neither this copyright statement, nor the licensing details below
// may be removed from this file or dissociated from its contents.
/******************************************************************************/
// Licensee: Book Owner
// License number: 99999
// The Licensee has been granted a NON-COMMERCIAL license to the contents of
// this source file, said Licensee being the owner of a copy of the book,
// "JPEG2000: Image Compression Fundamentals, Standards and Practice," by
// Taubman and Marcellin (Kluwer Academic Publishers, 2001).  A brief summary
// of the license appears below.  This summary is not to be relied upon in
// preference to the full text of the license agreement, which was accepted
// upon breaking the seal of the compact disc accompanying the above-mentioned
// book.
// 1. The Licensee has the right to Non-Commercial Use of the Kakadu software,
//    Version 2.2, including distribution of one or more Applications built
//    using the software, provided such distribution is not for financial
//    return.
// 2. The Licensee has the right to personal use of the Kakadu software,
//    Version 2.2.
// 3. The Licensee has the right to distribute Reusable Code (including
//    source code and dynamically or statically linked libraries) to a Third
//    Party, provided the Third Party possesses a license to use the Kakadu
//    software, Version 2.2, and provided such distribution is not for
//    financial return.
/*******************************************************************************
Description:
   Implementation of the child view window of the interactive JPEG2000 image
viewer application, "kdu_show".  Client-area windows messages are
processed (at least initially) here.
*******************************************************************************/

#include "stdafx.h"
#include "kdu_show.h"
#include "ChildView.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CChildView

/******************************************************************************/
/*                          CChildView::CChildView                            */
/******************************************************************************/

CChildView::CChildView()
{
  app = NULL;
  max_view_size = kdu_coords(10000,10000);
  sizing = false;
  last_size = kdu_coords(0,0);
  scroll_step = kdu_coords(1,1);
  scroll_page = scroll_step;
  scroll_end = kdu_coords(100000,100000);
}

/******************************************************************************/
/*                          CChildView::~CChildView                           */
/******************************************************************************/

CChildView::~CChildView()
{
}

/******************************************************************************/
/*                       CChildView::set_max_view_size                        */
/******************************************************************************/

void
  CChildView::set_max_view_size(kdu_coords size)
{
  max_view_size = size;
  if ((last_size.x > 0) && (last_size.y > 0))
    { /* Otherwise, the windows message loop is bound to send us a
         WM_SIZE message some time soon, which will call the
         following function out of the 'OnSize' member function. */
      check_and_report_size();
    }
}

/******************************************************************************/
/*                     CChildView::check_and_report_size                      */
/******************************************************************************/

void
  CChildView::check_and_report_size()
  /* The code here is very delicate.  The problem is that the framework
     enters into its own layout recalculations whenever the SetWindowPos
     function is called on the frame window.  This asserts certain internal
     guards against recursive re-entry into the layout recalculation code.
     To skirt around some of these complications, we override the frame
     object's OnSize function, letting it process an initial layout for the
     child and then calling this function here, which has to recover the
     layout generated by the frame (i.e., the parent).  This allows the code
     to re-enter the layout generation process -- unless somebody goes and
     modifies the implementation of some parts of the MFC framework.  Shame
     that MFC does not really protect the user against hidden implementation
     choices like this. */
{
  if (sizing)
    return; // We are already in the process of negotiating new dimensions.

  kdu_coords inner_size, outer_size, border_size;
  RECT rect;
  GetClientRect(&rect);
  inner_size.x = rect.right-rect.left;
  inner_size.y = rect.bottom-rect.top;
  kdu_coords target_size = inner_size;
  if (target_size.x > max_view_size.x)
    target_size.x = max_view_size.x;
  if (target_size.y > max_view_size.y)
    target_size.y = max_view_size.y;
  if (inner_size != target_size)
    { // Need to resize windows -- ugh.
      sizing = true; // Guard against unwanted recursion.

      CWnd *parent = GetParent();
      do {
          // First determine the difference between the active child client area
          // and the frame window's outer boundary.  This total border area may
          // change during resizing, which is the reason for the loop.
          parent->GetWindowRect(&rect);
          outer_size.x = rect.right-rect.left;
          outer_size.y = rect.bottom-rect.top;
          
          border_size = outer_size - inner_size;
          outer_size = target_size + border_size; // Assuming no border change
          parent->SetWindowPos(NULL,0,0,outer_size.x,outer_size.y,
                               SWP_NOZORDER | SWP_NOMOVE | SWP_SHOWWINDOW);

          GetClientRect(&rect);
          inner_size.x = rect.right-rect.left;
          inner_size.y = rect.bottom-rect.top;
        } while ((inner_size.x < target_size.x) ||
                 (inner_size.y < target_size.y));
      target_size = inner_size; // If we could not achieve our original target,
      sizing = false;           // the child region will have to be a bit larger
    }

  last_size = inner_size;
  if (app != NULL)
    {
      CWnd *parent = GetParent();
      app->set_view_size(last_size);
    }
}


/////////////////////////////////////////////////////////////////////////////
// CChildView message handlers

BEGIN_MESSAGE_MAP(CChildView,CWnd )
	//{{AFX_MSG_MAP(CChildView)
	ON_WM_PAINT()
	ON_WM_KEYDOWN()
	ON_WM_HSCROLL()
	ON_WM_VSCROLL()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


/******************************************************************************/
/*                        CChildView::PreCreateWindow                         */
/******************************************************************************/

BOOL CChildView::PreCreateWindow(CREATESTRUCT& cs) 
{
  if (!CWnd::PreCreateWindow(cs))
    return FALSE;

  cs.style &= ~WS_BORDER;
  cs.lpszClass =
    AfxRegisterWndClass(CS_HREDRAW|CS_VREDRAW|CS_DBLCLKS, 
                        ::LoadCursor(NULL, IDC_ARROW),
                        HBRUSH(COLOR_WINDOW+1), NULL);
  return TRUE;
}

/******************************************************************************/
/*                            CChildView::OnPaint                             */
/******************************************************************************/

void CChildView::OnPaint() 
{
  CPaintDC dc(this); // device context for painting
  // Convert PAINTSTRUCT region to a `kdu_dims' structure.  The windows
  // RECT structure is poorly defined, but it turns out that the `bottom'
  // and `right' members refer to the point immediately beyond the lower right
  // hand corner of the rectangle (not the corner itself).  To minimize
  // confusion, we avoid using RECT structures ourself.
  kdu_dims region;
  region.pos.x = dc.m_ps.rcPaint.left;
  region.pos.y = dc.m_ps.rcPaint.top;
  region.size.x = dc.m_ps.rcPaint.right - region.pos.x;
  region.size.y = dc.m_ps.rcPaint.bottom - region.pos.y;
  if (app != NULL)
    app->paint_region(&dc,region);
}

/******************************************************************************/
/*                           CChildView::OnHScroll                            */
/******************************************************************************/

void CChildView::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
  if (app == NULL)
    return;
  if (nSBCode == SB_THUMBPOSITION)
    app->set_hscroll_pos(nPos);
  else if (nSBCode == SB_LINELEFT)
    app->set_hscroll_pos(-scroll_step.x,true);
  else if (nSBCode == SB_LINERIGHT)
    app->set_hscroll_pos(scroll_step.x,true);
  else if (nSBCode == SB_PAGELEFT)
    app->set_hscroll_pos(-scroll_page.x,true);
  else if (nSBCode == SB_PAGERIGHT)
    app->set_hscroll_pos(scroll_page.x,true);
  else if (nSBCode == SB_LEFT)
    app->set_hscroll_pos(0);
  else if (nSBCode == SB_RIGHT)
    app->set_hscroll_pos(scroll_end.x);
}

/******************************************************************************/
/*                           CChildView::OnVScroll                            */
/******************************************************************************/

void CChildView::OnVScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
  if (app == NULL)
    return;
  if (nSBCode == SB_THUMBPOSITION)
    app->set_vscroll_pos(nPos);
  else if (nSBCode == SB_LINEUP)
    app->set_vscroll_pos(-scroll_step.y,true);
  else if (nSBCode == SB_LINEDOWN)
    app->set_vscroll_pos(scroll_step.y,true);
  else if (nSBCode == SB_PAGEUP)
    app->set_vscroll_pos(-scroll_page.y,true);
  else if (nSBCode == SB_PAGEDOWN)
    app->set_vscroll_pos(scroll_page.y,true);
  else if (nSBCode == SB_TOP)
    app->set_vscroll_pos(0);
  else if (nSBCode == SB_BOTTOM)
    app->set_vscroll_pos(scroll_end.y);
}

/******************************************************************************/
/*                           CChildView::OnKeyDown                            */
/******************************************************************************/

void CChildView::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
  if (app != NULL)
    { // Look out for keys which are not bound to the menu -- currently the only
      // keys of this form are those with obvious scrolling functions.
      if (nChar == VK_LEFT)
        app->set_hscroll_pos(-scroll_step.x,true);
      else if (nChar == VK_RIGHT)
        app->set_hscroll_pos(scroll_step.x,true);
      else if (nChar == VK_UP)
        app->set_vscroll_pos(-scroll_step.y,true);
      else if (nChar == VK_DOWN)
        app->set_vscroll_pos(scroll_step.y,true);
      else if (nChar == VK_PRIOR)
        app->set_vscroll_pos(-scroll_page.y,true);
      else if (nChar == VK_NEXT)
        app->set_vscroll_pos(scroll_page.y,true);
    }
  CWnd::OnKeyDown(nChar, nRepCnt, nFlags);
}

